/* Copyright (C) 1995
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "printer.h"
#include <stdio.h>

int lowres = 0;
int scale_x=0, scale_y=0;
int offs_x=0, offs_y=0;

#define XMAX 360*8		/* 360 dpi * 8" paperwidth */

#define PRBUFSIZ (XMAX*6)	/* XMAX * 6 bytes/column */


extern char *myname;

static int byte,line,thisbuf;
static int omask, obyte, obuf, usecolor, device,truncwarn;

static unsigned char mask;
static unsigned char prbuf[4][2][PRBUFSIZ];
static char dbl[48];

static void mask_up(void);
static void print_line(void);
static void double_line(void);

typedef struct printer_t {
  const char *const init;
  const char *const graphics;
  const char *const color1;
  const char *const color2;
  const char *const skiprel;
  const char *const skipabs;
  const char *const feed;
  const char *const finish;
  const char coltab[4];  
  int graphics_offset;
  int graphics_scale;
  int skipwidth;
  int feedwidth;
} printer_t;

const printer_t devices[2] = {
  {
    "\033[K\004%c%c\044%c%c"		/* init */
    "\033[\\\004%c%c%c\x68\x01",	/* vertical spacing */

    "\x1B\x5Bg%c%c\x10",		/* Command for graphics */

    "\033[M\001%c", "%c",		/* Command for color selection */
  
    "\033d%c%c", NULL,			/* Command to skip over empty space */

    "\033J%c",				/* Command to feed a line */

    "\033[K\002%c%c\044",		/* init */

    { 5, 3, 6, 0 },
    1, 1,
    18,
    48,
  },
  {
    "\033@"				/* init */
    "\033P"				/* 10cpi, needed for tabbing */
    "\033l%c"				/* clear left margin */
    "\033Q\377"				/* clear right margin */
    "\033U%c"				/* bidirectional is fine for fax */
    "\r",

    "\033*H%c%c",			/* Command for graphics */

    "\033r", "%c",			/* Command for color selection */
  
    NULL, "\033D%c%c\t",		/* Command to skip over empty space */

    "\033J%c",				/* Command to feed a line */

    "\033@",				/* init */

    { 2, 1, 4, 0 },
    0, 6,
    216,
    24,
  }
};

void init_printer(int printer, int color) {
  int i;

  usecolor=color;
  device=printer;

  /*
   * Using the Bresenham linedrawing algorythm seems a neat idea to
   * calculate which lines must be doubled for proper scaling.
   *
   */
  {
    int x,xe,ye,dsum,steps;

    memset(dbl, 0, 48);

    x=0;
    xe=47;
    ye = (48 * scale_y / 1000) - 48;
    if ((ye<0) || (ye>47)) {
      fprintf(stderr, "%s: bad vertical scale factor\n", myname);
      exit(1);
    }

    dsum=xe>>1;
    do {
      dsum+=ye;
      if (dsum>=xe) {
	dsum-=xe;
	dbl[x]=1; /* Double this line */
      }
      x++;
    } while(x<48);
  }

  printf(devices[device].init, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

  if (offs_y) {
    int ofs = devices[device].feedwidth * 15 * offs_y / 2000;
    while (ofs>255) {
      printf(devices[device].feed, 255, 0, 0, 0, 0, 0, 0, 0);
      ofs-=255;
    }
    if (ofs)
      printf(devices[device].feed, ofs, 0, 0, 0, 0, 0, 0, 0);
  }

  byte=0;
  line=0;
  thisbuf=0;
  truncwarn=0;
  mask=128;
  memset(prbuf, 0, PRBUFSIZ*2*4);
}

/* 6 bytes per column:
 *
 *    <--TOP                                      BOTTOM-->
 *    76543210 76543210 76543210 76543210 76543210 76543210
 *
 */

void prt_draw(int color, int x, int len) {
  int ptr;

  if (!usecolor) color=3;

  x=x*scale_x/1000;
  len=len*scale_x/1000;
  ptr=x*6+byte;
  
  while(len--) {
    if (x++>XMAX) break;
    prbuf[color][thisbuf][ptr]|=mask;
    ptr+=6;
  }

  if (truncwarn || x<XMAX) return;
  truncwarn=1;
  fprintf(stderr,
	  "%s: Warning, output is wider than 8\" and will be truncated\n",
	  myname);
}

void prt_next(void) {
  register int p;

  omask=mask; obyte=byte; obuf=thisbuf;

  if (lowres) {
    mask_up();
    double_line();
    if (++line>=48) line=0;

    if (dbl[line]) {
      double_line();
      double_line();
    }
    if (++line>=48) line=0;

  } else {
    mask_up();
    if (dbl[line]) double_line();
    if (++line>=48) line=0;
  }
}


static void double_line(void) {
  register int p, c;

  for (c=usecolor?0:3; c<4; c++) {
    for (p=0; p<PRBUFSIZ; p+=6) {
      if (prbuf[c][obuf][p+obyte]&omask) prbuf[c][thisbuf][p+byte]|=mask;
    }
  }
  mask_up();
}

static void mask_up(void) {
  mask=mask>>1;
  if (!mask) {
    mask=128;
    byte++;
  }
  if (byte>=6) print_line();
}


/*
 * Let's do four runs over the line, one for cyan, magenta, yellow and black
 */
static void print_line(void) {
  register int from,to,q,i,c,offs;

  for (c=usecolor?0:3; c<4; c++) {
    from=to=0;
    if (usecolor) {
      printf (devices[device].color1, 0, 0, 0, 0, 0, 0, 0, 0);
      printf (devices[device].color2, devices[device].coltab[c]);
    }

    if (offs_x) {
      offs=(2160 / devices[device].skipwidth) * offs_x / 1000;
      if (devices[device].skiprel) {
	printf(devices[device].skiprel, offs&255, offs>>8);
      } else {
	printf(devices[device].skipabs, q, 0, 0, 0, 0);
      }
    } else {
      offs=0;
    }

    while(from<PRBUFSIZ) {
      to=from;
      do {
	q=0;
	for (i=0; i<devices[device].skipwidth; i++) q|=prbuf[c][thisbuf][to++];
      } while((!q) && (to<PRBUFSIZ));
      if (to>=PRBUFSIZ) break;		/* all empty */
      to-=devices[device].skipwidth;

      if (to>from) {
	if (devices[device].skiprel) {
	  q=(to-from)/devices[device].skipwidth;
	  printf(devices[device].skiprel, q&255, q>>8);
	} else {
	  printf(devices[device].skipabs,
		 offs+(to/devices[device].skipwidth), 0, 0, 0, 0);
	}
	from=to;
      }

      do {
	q=0;
	for (i=0; i<devices[device].skipwidth; i++) q|=prbuf[c][thisbuf][to++];
      } while((q) && (to<PRBUFSIZ));
      if (to<PRBUFSIZ) to-=devices[device].skipwidth;
      
      if (to>from) {
	int w;
	q=(to-from);
	w = (q+devices[device].graphics_offset)/devices[device].graphics_scale;
	printf(devices[device].graphics, w&255, w>>8);
	fwrite(&prbuf[c][thisbuf][from], 1, q, stdout);
	from=to;
      }
    }
    printf("\r");
    memset(&prbuf[c][thisbuf], 0, PRBUFSIZ);
  }
  printf(devices[device].feed, devices[device].feedwidth, 0, 0, 0, 0, 0, 0, 0);

  if (thisbuf) thisbuf=0; else thisbuf=1;
  byte=0;
}




void prt_feedup(void) {
  while((byte) || (mask!=128)) prt_next();

  if (usecolor) {
    printf (devices[device].color1, 0, 0, 0, 0, 0, 0, 0, 0);
    printf (devices[device].color2, 0);
  }
  printf(devices[device].finish, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
}



