/*
linux/fs/dmsdos/dmsdos_spc.c

DMSDOS filesystem: special routines.

This file implements the DMSDOS filesystem. It is based on the linux msdos 
filesystem routines and partially a complete rewrite of them. 

IMPORTANT: read docs before compiling or access may fail and damage your data!
	   (some patches required)
	   
Author: Frank Gockel
Email:  gockel@etecs4.uni-duisburg.de

Note: 

MSDOS inode numbering:
bit nr.
  3222222222221111111111            u=unused (zero) due to DOS 512MB limit 
  10987654321098765432109876543210  s=physical sector no of dir entry
  uuuuuuuusssssssssssssssssssseeee  e=position of dir entry in dir sector >> 5
  
DMSDOS inode numbering:
bit nr.
  3222222222221111111111            u=unused
  10987654321098765432109876543210  s=relative sector no of dir entry
  ucccccccsssssssssssssssssssseeee  e=position of dir entry in dir sector >> 5
                                    c=cvfnr+1
  'relative sector' means sector counted from the beginning of the CVF, i.e.
  position in compressed volume file % SECTOR_SIZE.
  If the c-bits are all zeros, the inode is a real msdos one, not in a CVF.
  
*/

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#define DMSDOS_VERSION "DMSDOS filesystem version 0.4.4 (alpha)\n"

#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/major.h>
#include <linux/blkdev.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/locks.h>
#include <asm/segment.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/msdos_fs.h>
#include <linux/dmsdos_fs.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/shm.h>
#include <linux/mman.h>
#include <asm/system.h>

#include "../msdos/msbuffer.h"

#define PRINTK(X)
#define CLPOS 26

#define CHS(i) ((unsigned short)i[0]|(unsigned short)i[1]<<8)
#define CHL(i) ((unsigned long)i[0]|(unsigned long)i[1]<<8|(unsigned long)i[2]<<16|(unsigned long)i[3]<<24)

int dbl_cvf_inos[MAXDBLFILES];
int dbl_cvf_startcluster[MAXDBLFILES];

/*
typedef struct {
  int s_dcluster;/[45-46]/
  int s_mdfatstart;/[36-37]+1/
  int s_fatstart;/[39-40]+[14-15]/ 
  int s_rootdir;/[41-42]+[39-40]/
  int s_rootdiranzentry;
  int s_sectperclust;
  int s_first[MAXFRAGMENT];
  int s_anzahl[MAXFRAGMENT];
  int s_16bitfat;
} Dblsb;
*/

Dblsb dblsb[MAXDBLFILES];

Acache mdfat[MDFATCACHESIZE];
Acache dfat[DFATCACHESIZE];

int dbl_sb_dev[MAXDBLFILES];

int lowest_never_used_cvfnr=0;

struct buffer_head* read_real_sector(struct super_block* sb,int sector)
{ struct buffer_head*bh;
  /*printk("DMSDOS: read_real_sector %d\n",sector);*/
  bh=msdos_bread(sb,sector);
  if(bh==NULL)panic("DMSDOS: read_real_sector %d failed (bh=NULL)\n",sector);
  else if(bh->b_data==NULL)
        panic("DMSDOS: read_real_sector %d failed (bh->data=NULL)\n",sector);
  return bh;
}

void bh_free(struct super_block* sb,struct buffer_head*bh)
{ msdos_brelse(sb,bh);
}

int calc_real_sector(int dblsector, int cvfnr)
{ int i;
  int sz;
  
  /* calculate physical sector */
  i=0;sz=dblsector;
  do
  { sz-=dblsb[cvfnr].s_anzahl[i];
    ++i;
  }
  while(sz>=0&&i<MAXFRAGMENT);
  --i;
  sz+=dblsb[cvfnr].s_anzahl[i]+dblsb[cvfnr].s_first[i];
  return sz;
}

int dos_cluster2sector(struct super_block * sb,int clusternr)
{ return (clusternr-2)*MSDOS_SB(sb)->cluster_size+MSDOS_SB(sb)->data_start;
}

struct buffer_head* read_dbl_sector(struct super_block*sb,int dblsector,
                                    int cvfnr)
{ int sector;

/*printk("DMSDOS: read_dbl_sector: sector=%d of CVF %d\n",dblsector,cvfnr+1);*/
  sector=calc_real_sector(dblsector,cvfnr);
  return read_real_sector(sb,sector);
}

int setup_fragment(struct super_block*sb, int cvfnr)
{ int fragmentzaehler;
  int clusterzaehler;
  int akt_cluster;
  int folge_cluster;
  int i;
  
  /*printk("DMSDOS: setup_fragment CVF %d\n",cvfnr+1);*/
  fragmentzaehler=0;

  folge_cluster=dbl_cvf_startcluster[cvfnr];
  
  do
  {
    clusterzaehler=0;
    dblsb[cvfnr].s_first[fragmentzaehler]=folge_cluster;
    do
    { akt_cluster=folge_cluster;
      folge_cluster=fat_access(sb,akt_cluster,-1);
      ++clusterzaehler;
    }
    while(folge_cluster==akt_cluster+1);
  
    dblsb[cvfnr].s_anzahl[fragmentzaehler]=clusterzaehler;
    /*printk("DMSDOS: firstclust=%d anz=%d\n",
           dblsb[cvfnr].s_first[fragmentzaehler],
           dblsb[cvfnr].s_anzahl[fragmentzaehler]);
    */
           
    ++fragmentzaehler;  
  }
  while(folge_cluster>0&&fragmentzaehler<MAXFRAGMENT);
  if(fragmentzaehler==MAXFRAGMENT&&folge_cluster>0)
  { /* zu fragmentiert, raus */
    dbl_cvf_inos[cvfnr]=0;
    dbl_sb_dev[cvfnr]=0;
    printk("DMSDOS: CVF %d too fragmented, not mounted!\n",cvfnr+1);
    return -1; 
  }
  /*printk("DMSDOS: CVF %d has %d fragment(s)\n",cvfnr+1,fragmentzaehler);*/
  
  /* convert cluster-oriented numbers into sector-oriented ones */
  for(i=0;i<fragmentzaehler;++i)
  { /*printk("DMSDOS: umrechnen 1\n");*/
    dblsb[cvfnr].s_first[i]=dos_cluster2sector(sb,dblsb[cvfnr].s_first[i]);
    /*printk("DMSDOS: umrechnen 2\n");*/
    dblsb[cvfnr].s_anzahl[i]*=MSDOS_SB(sb)->cluster_size;
    /*printk("DMSDOS: umrechnen 3\n");*/                                          
  }
  return 0;
}

void setup_mdfat(struct super_block*sb,int cvfnr)
{ 
  struct buffer_head*bh;
  int i;
  unsigned char * pp;

   /*printk("DMSDOS: setup_mdfat with CVF %d\n",cvfnr+1);*/
    for(i=0;i<MAXFRAGMENT;++i)
    { dblsb[cvfnr].s_first[i]=0;
      dblsb[cvfnr].s_anzahl[i]=0;
    }
    if(setup_fragment(sb,cvfnr)<0)return; 
    
    /*printk("DMSDOS: reading %d. CVF boot block...\n",cvfnr+1);*/
    bh=read_dbl_sector(sb,0,cvfnr);
    /*printk("DMSDOS: boot block read finished\n");*/
    pp=&(bh->b_data[45]);
    dblsb[cvfnr].s_dcluster=CHS(pp);
    pp=&(bh->b_data[36]);
    dblsb[cvfnr].s_mdfatstart=CHS(pp)+1;
    pp=&(bh->b_data[17]);
    dblsb[cvfnr].s_rootdiranzentry=CHS(pp);
    dblsb[cvfnr].s_sectperclust=((unsigned long)(bh->b_data[13]));
    pp=&(bh->b_data[39]);i=CHS(pp);
    pp=&(bh->b_data[14]);
    dblsb[cvfnr].s_fatstart=i+CHS(pp);
    pp=&(bh->b_data[41]);
    dblsb[cvfnr].s_rootdir=i+CHS(pp);
    pp=&(bh->b_data[57]);
    if(CHL(pp)==0x20203631)dblsb[cvfnr].s_16bitfat=1;
    else if(CHL(pp)==0x20203231)dblsb[cvfnr].s_16bitfat=0;
    else
    { pp=&(bh->b_data[62]);
      dblsb[cvfnr].s_16bitfat=(CHS(pp)>32) ? 1 : 0;
      printk("DMSDOS: FAT bit size of CVF %d not recognized, guessed %d bit\n",
             cvfnr+1,CHS(pp)>32 ? 16 : 12 );
    }
    bh_free(sb,bh);
 /*
    printk("DMSDOS: dcluster=%d\n",dblsb[cvfnr].s_dcluster);
    printk("DMSDOS: mdfatstart=%d\n",dblsb[cvfnr].s_mdfatstart);
    printk("DMSDOS: rootdiranzentry=%d\n",dblsb[cvfnr].s_rootdiranzentry);
    printk("DMSDOS: sectperclust=%d\n",dblsb[cvfnr].s_sectperclust);
    printk("DMSDOS: fatstart=%d\n",dblsb[cvfnr].s_fatstart);
    printk("DMSDOS: rootdir=%d\n",dblsb[cvfnr].s_rootdir);
    printk("DMSDOS: %d bit FAT\n",dblsb[cvfnr].s_16bitfat ? 16 : 12);
 */  
}

void read_mdfat_sector(struct super_block*sb,int area,int merk_i,int cvfnr)
{
/*printk("DMSDOS: reading dbl_mdfat sector %d of CVF %d\n",area,cvfnr+1);*/
  
  if(mdfat[merk_i].a_buffer!=NULL)bh_free(sb,mdfat[merk_i].a_buffer);
  mdfat[merk_i].a_buffer=read_dbl_sector(sb,area,cvfnr);
  mdfat[merk_i].a_area=area;
  mdfat[merk_i].a_used=1;
  mdfat[merk_i].a_cvfnr=cvfnr;
}

void read_fat_sector(struct super_block*sb,int area,int merk_i,int cvfnr)
{
  /*printk("DMSDOS: reading dbl_fat sector %d of CVF %d\n",area,cvfnr+1);*/
  
  if(dfat[merk_i].a_buffer!=NULL)bh_free(sb,dfat[merk_i].a_buffer);
  dfat[merk_i].a_buffer=read_dbl_sector(sb,area,cvfnr);
  dfat[merk_i].a_area=area;
  dfat[merk_i].a_used=1;
  dfat[merk_i].a_cvfnr=cvfnr;
  
  /*printk("cluster 2 fat entry=%d\n",mdfat[cvfnr].fat_data[merk_i][2]);*/ 
}

int dbl_mdfat_value(struct super_block*sb,int clusternr,int cvfnr)
{ int i;
  int area;
  int pos;
  int offset;
  int min_used;
  int merk_i;
  unsigned char * pp;
  
  pos=(dblsb[cvfnr].s_dcluster+clusternr)*4+512*dblsb[cvfnr].s_mdfatstart;
  area=pos/SECTOR_SIZE;
  offset=(pos%SECTOR_SIZE);                           
  
  min_used=mdfat[0].a_used;
  merk_i=0;
  /* find area and cvfnr in cache */
  for(i=0;i<MDFATCACHESIZE;++i)
  { if(mdfat[i].a_used<min_used)
    { min_used=mdfat[i].a_used;
      merk_i=i;
    }
    if(mdfat[i].a_used>0&&area==mdfat[i].a_area&&cvfnr==mdfat[i].a_cvfnr)
    { /* found */
      ++mdfat[i].a_used;
      pp=&(mdfat[i].a_buffer->b_data[offset]);
      return CHL(pp);
    }
  }
  /* merk_i = least used entry number */
  read_mdfat_sector(sb,area,merk_i,cvfnr);
  pp=&(mdfat[merk_i].a_buffer->b_data[offset]);
  return CHL(pp);
}

int dbl_mdfat_cluster2sector(struct super_block*sb,int clusternr,int cvfnr)
{ return (dbl_mdfat_value(sb,clusternr,cvfnr)&0x1fffff)+1;
}

int dbl_fat_nextcluster(struct super_block*sb,int clusternr,int cvfnr)
{ int i;
  int area;
  int pos;
  int offset;
  int min_used;
  int merk_i;
  int res;
  unsigned char * pp;
  
  pos= dblsb[cvfnr].s_fatstart*512;
  pos+= dblsb[cvfnr].s_16bitfat ? (clusternr*2) :
       ( (clusternr&1)? (clusternr-1)*3/2 + 1 : clusternr*3/2 );
       
  area=pos/SECTOR_SIZE;
  offset=pos%SECTOR_SIZE;                           
  
  min_used=dfat[0].a_used;
  merk_i=0;
  /* find area and cvfnr in cache */
  for(i=0;i<DFATCACHESIZE;++i)
  { if(dfat[i].a_used<min_used)
    { min_used=dfat[i].a_used;
      merk_i=i;
    }
    if(dfat[i].a_used>0&&area==dfat[i].a_area&&cvfnr==dfat[i].a_cvfnr)
    { /* found */
      ++dfat[i].a_used;
      pp=&(dfat[i].a_buffer->b_data[offset]);
      res=CHS(pp);
      if(dblsb[cvfnr].s_16bitfat)return res>=0xFFF7 ? -1 : res;
      if(clusternr&1)res>>=4; else res&=0xfff;
      return res>=0xff7 ? -1 : res;
    }
  }
  /* merk_i = least used entry */
  read_fat_sector(sb,area,merk_i,cvfnr);
  pp=&(dfat[merk_i].a_buffer->b_data[offset]);
  res=CHS(pp);
  if(dblsb[cvfnr].s_16bitfat)return res>=0xFFF7 ? -1 : res;
  if(clusternr&1)res>>=4; else res&=0xfff;
  return res>=0xff7 ? -1 : res;
}

int dbltest(struct inode* inode)
{ int i;

  if(lowest_never_used_cvfnr==0)return 0; /* dmsdos was never in action, 
                                             values are undefined
                                             (this is only for umsdos since
                                             umsdos may call this without
                                             dbl_init being called before) */

  /* am I the inode of a CVF? */
  if(inode->i_ino==0)return 0; /* there is no zero ino for dmsdos */
  for(i=0;i<MAXDBLFILES;++i) 
  { if(dbl_cvf_inos[i]==inode->i_ino&&dbl_sb_dev[i]==inode->i_sb->s_dev)
       return (i+1)*0x1000000;
  }
  return 0;
}
                                           
int dostest(struct inode* inode)
{ /* am I a real DOS inode (=1) or an inode of or in a CVF (=0) */
  if(inode->i_ino<0x1000000&&dbltest(inode)==0)return 1;
  return 0;
}

int find_free_cvfnr(void)
{ int i;

  for(i=0;i<MAXDBLFILES;++i)
  { if(dbl_cvf_inos[i]==0)return i;
  }
  return -1;
}

void dbl_init(struct super_block* sb, void* datai, int silent)
{ int i,j;
  int cvfnr;
  struct buffer_head * bh;
  struct msdos_dir_entry * data;
  char dname[]="        .   ";
  
  printk(DMSDOS_VERSION);

  if(lowest_never_used_cvfnr==0)
  { /* first call of DMSDOS filesystem, initialising variables */
    for(i=0;i<MAXDBLFILES;++i)
    {  dbl_cvf_inos[i]=0;
       dbl_sb_dev[i]=0;
    }
    for(i=0;i<MDFATCACHESIZE;++i){mdfat[i].a_used=0;mdfat[i].a_buffer=NULL;}
    for(i=0;i<DFATCACHESIZE;++i){dfat[i].a_used=0;dfat[i].a_buffer=NULL;}
    lowest_never_used_cvfnr=MAXDBLFILES;
  }
  
  for(i=0;i<MSDOS_SB(sb)->dir_entries/MSDOS_DPS;++i)
  { bh=read_real_sector(sb,MSDOS_SB(sb)->dir_start+i);
    data=(struct msdos_dir_entry*) bh->b_data;
    
    for(j=0;j<MSDOS_DPS;++j)
    { if(strncmp(data[j].name,"DRVSPACE",8)==0 
          || strncmp(data[j].name,"DBLSPACE",8)==0 )
      { if(data[j].name[8]>='0'&&data[j].name[8]<='9'
           &&data[j].name[9]>='0'&&data[j].name[9]<='9'
           &&data[j].name[10]>='0'&&data[j].name[10]<='9'
          ){ /* it is a CVF */
             if((cvfnr=find_free_cvfnr())>=0)
             { dbl_cvf_startcluster[cvfnr]=data[j].start;
               dbl_cvf_inos[cvfnr]=(MSDOS_SB(sb)->dir_start+i)*MSDOS_DPS+j;
               dbl_sb_dev[cvfnr]=sb->s_dev;
               strncpy(dname,data[j].name,8);
               strncpy(&(dname[9]),&(data[j].name[8]),3);
               printk("DMSDOS: CVF %s mounted as CVF %d on device %d ino=%d\n",
                      dname,cvfnr+1,dbl_sb_dev[cvfnr],dbl_cvf_inos[cvfnr]);
               setup_mdfat(sb,cvfnr);
             }
             else printk("DMSDOS: too many CVFs, ignoring\n");
           } 
      }
    }
    bh_free(sb,bh);
  }
/*printk("DMSDOS: msdos_read_super reports data_start=%d cluster_size=%d\n",
         MSDOS_SB(sb)->data_start,MSDOS_SB(sb)->cluster_size);*/
}

/* returns ino or -ENOENT */
int read_dbl_direntry(struct super_block*sb,
                      int dbl_clusterno,int cvfnr,int entrynr,
                      unsigned char*buf)
{ int dbl_sector;
  int dbl_offset;
  int ino;
  struct buffer_head*bh;
  int i;
  
  /* Hmm ...  error test: */
  if(entrynr>65535*16*dblsb[cvfnr].s_sectperclust)
  { /* impossible, there are max 65535 clusters a 16*sectperclust entries */
    /* this is most likely a fat loop (code prevents hang) */
    printk("DMSDOS: read_dbl_direntry: entrynr too large, aborting!\n");
    return -ENOENT;
  }
  
  /* get sector and offset */
  if(dbl_clusterno==0) /*it is a dbl rootdir*/
  { dbl_sector=dblsb[cvfnr].s_rootdir+(entrynr>>4);
    if(entrynr>=dblsb[cvfnr].s_rootdiranzentry)return -ENOENT;
  }
  else
  { while(entrynr/(16*dblsb[cvfnr].s_sectperclust))
    { entrynr-=(16*dblsb[cvfnr].s_sectperclust);
      dbl_clusterno=dbl_fat_nextcluster(sb,dbl_clusterno,cvfnr);
      if(dbl_clusterno==-1)return -ENOENT;
      if(entrynr<0)
      { printk("DMSDOS: read_dbl_direntry: error calculating clusterno\n");
        return -ENOENT;
      }
    }
    dbl_sector=dbl_mdfat_cluster2sector(sb,dbl_clusterno,cvfnr);
    dbl_sector+=entrynr>>4;
  }
  /* always the same */
  dbl_offset=entrynr&0xf;
  ino=((cvfnr+1)<<24)+(dbl_sector<<4)+dbl_offset;
  /* read sector */
  bh=read_dbl_sector(sb,dbl_sector,cvfnr);
  for(i=0;i<32;++i)buf[i]=bh->b_data[i+32*dbl_offset];
  bh_free(sb,bh);
  return ino;
}

int scan_dbl_dir_4_clusterno(struct super_block*sb,
                             int dirstartclust, int tofind,int cvfnr)
{ int ino;
  int ent=0;
  unsigned char buf[32];
  unsigned char * pp;

  while((ino=read_dbl_direntry(sb,dirstartclust,cvfnr,ent,buf))>0)
  { ++ent;
    pp=&(buf[CLPOS]);
    if(CHS(pp)==tofind)return ino;
  }
  return ino;
}

/* does *not* look for . or .. */
int scan_dbl_dir_4_filename(struct super_block*sb, int dirstartclust,
                            unsigned const char*tofind,int len,
                            int cvfnr)
{ int ino;
  int ent=0;
  unsigned char buf[32];
  unsigned char msdosname[]="           ";
  int i;
  int j;
  int warpunkt=0;
  
  for(i=0,j=0;i<len&&j<11;++i)
  { if(tofind[i]=='.')
    { if(j>8)return -ENOENT;
      if(warpunkt)return -ENOENT;
      warpunkt=1;
      j=8;
      continue;
    }
    if(tofind[i]>='a'&&tofind[i]<='z')msdosname[j++]=tofind[i]-'a'+'A';
    else msdosname[j++]=tofind[i];
  }
/*  
  printk("DMSDOS: scan_dbl_dir_4_filename: name=%s dosname=%s\n",
         tofind,msdosname);
  printk("DMSDOS: scan_dbl_dir_4_filename: dirstartclust=%d cvfnr=%d\n",
         dirstartclust,cvfnr+1);
*/
  while((ino=read_dbl_direntry(sb,dirstartclust,cvfnr,ent,buf))>0)
  { ++ent;
    if(strncmp(buf,msdosname,11)==0)
    { /*printk("DMSDOS: scan_dbl_dir_4_filename: found with ino=%d\n",ino);*/
      return ino;
    }
  }
/*
  printk("DMSDOS: scan_dbl_dir_4_filename: not found, error=%d, ent=%d\n",
         ino,ent);
*/
  return ino;
}

/* startcluster=0 -> dbl_rootdir */
/* unused...
int dbl_inode2startcluster(struct inode * inode)
{ int dbl_sector;
  int dbl_offset;
  int cvfnr;
  int cluster;
  struct buffer_head*bh;
  unsigned char * pp;
  
  if(dbltest(inode))return 0;
  cvfnr=(inode->i_ino>>24)-1;
  dbl_sector=(inode->i_ino&0xffffff)>>4;
  dbl_offset=inode->i_ino&0xf;
  
  bh=read_dbl_sector(inode->i_sb,dbl_sector,cvfnr);
  
  pp=&(bh->b_data[dbl_offset*32+CLPOS]);
  cluster=CHS(pp);
  bh_free(inode->i_sb,bh);
  return cluster;
}
*/

/* Returns the inode number of the directory entry at offset pos.
   Pos is incremented. */

int dmsdos_get_entry(struct inode *dir, loff_t *pos,unsigned char * buf)
{
	int entrynr;
	int ino;
	int clust;
	int cvfnr;

		entrynr = (*pos)>>5;
		*pos += 32;
		if((cvfnr=dbltest(dir))!=0)cvfnr=(cvfnr>>24)-1;
		else cvfnr=(dir->i_ino>>24)-1;
		clust=MSDOS_I(dir)->i_start;/*dbl_inode2startcluster(dir);*/
		ino=read_dbl_direntry(dir->i_sb,clust,cvfnr,entrynr,buf);
		return ino;
}

int dmsdos_parent_ino(struct inode*dir)
{  struct buffer_head*bh;
   int cvfnr;
   int parent_dbl_cluster;
   int parent_dbl_sector;
   int parent_parent_dbl_cluster;
   unsigned char * pp;

   if(dbltest(dir))return MSDOS_ROOT_INO;

   cvfnr=(dir->i_ino>>24)-1;
   if(cvfnr<0)return -ENOENT;

/* read first sector of my own cluster */
   bh=read_dbl_sector(dir->i_sb,
                      dbl_mdfat_cluster2sector(dir->i_sb,
                                               MSDOS_I(dir)->i_start,cvfnr),
                      cvfnr);
/* and get second entry (it is ..) */
   pp=&(bh->b_data[32+CLPOS]);
   parent_dbl_cluster=CHS(pp);
   bh_free(dir->i_sb,bh);
    
   if(parent_dbl_cluster==0)
   { /* Verweis auf dbl_rootdir */
     return dbl_cvf_inos[cvfnr];
   }
   
   parent_dbl_sector=dbl_mdfat_cluster2sector(dir->i_sb,parent_dbl_cluster,
                                              cvfnr);
   
   bh=read_dbl_sector(dir->i_sb,parent_dbl_sector,cvfnr);
   pp=&(bh->b_data[32+CLPOS]);
   parent_parent_dbl_cluster=CHS(pp);
   bh_free(dir->i_sb,bh);
   return scan_dbl_dir_4_clusterno(dir->i_sb,
                         parent_parent_dbl_cluster, /*dir start cluster*/
                         parent_dbl_cluster,        /* to find */ 
                         cvfnr);
}

#define NAME_OFFSET(de) ((int) ((de)->d_name - (char *) (de)))
#define ROUND_UP(x) (((x)+3) & ~3)

int dmsdos_readdirx( 
	struct inode *inode,
	struct file *filp,
	struct dirent *dirent,	/* dirent in user space */
	int count)
{	
	int ino,i,i2,last;
	char c,*walk;
	unsigned char buf[32];
	struct msdos_dir_entry *de;

	if (!inode || !S_ISDIR(inode->i_mode)) return -EBADF;
	if (dbltest(inode)) {
	        /* printk("DMSDOS: readdirx reading dblrootdir ino=%ld\n",
	                                                     inode->i_ino);
	        */
/* Fake . and .. for the root directory. */
		if (filp->f_pos == 2) filp->f_pos = 0;
		else if (filp->f_pos < 2) {
			walk = filp->f_pos++ ? ".." : ".";
			for (i = 0; *walk; walk++)
				put_fs_byte(*walk,dirent->d_name+i++);
			put_fs_long(filp->f_pos==1?inode->i_ino:MSDOS_ROOT_INO,&dirent->d_ino);
			put_fs_byte(0,dirent->d_name+i);
			put_fs_word(i,&dirent->d_reclen);
			return ROUND_UP(NAME_OFFSET(dirent) + i + 1);
		}
	}
	/*else printk("DMSDOS: readdirx ??? ino=%ld\n",inode->i_ino);*/
	if (filp->f_pos & (sizeof(struct msdos_dir_entry)-1)) return -ENOENT;

	while ((ino = dmsdos_get_entry(inode,&filp->f_pos,buf)) > -1) {
	        de=(struct msdos_dir_entry*) buf;
		if (!IS_FREE(de->name) && !(de->attr & ATTR_VOLUME)) {
			char bufname[13];
			char *ptname = bufname;
			for (i = last = 0; i < 8; i++) {
				if (!(c = de->name[i])) break;
				if (c >= 'A' && c <= 'Z') c += 32;
				if (c != ' ')
					last = i+1;
				ptname[i] = c;
			}
			i = last;
			ptname[i] = '.';
			i++;
			for (i2 = 0; i2 < 3; i2++) {
				if (!(c = de->ext[i2])) break;
				if (c >= 'A' && c <= 'Z') c += 32;
				if (c != ' ')
					last = i+1;
				ptname[i] = c;
                               i++;
			}
			if ((i = last) != 0) {
				if (!strcmp(de->name,MSDOS_DOT))
					ino = inode->i_ino;
				else if (!strcmp(de->name,MSDOS_DOTDOT))
						ino = dmsdos_parent_ino(inode);
				bufname[i] = '\0';
				put_fs_long(ino,&dirent->d_ino);
				memcpy_tofs(dirent->d_name,bufname,i+1);
				put_fs_word(i,&dirent->d_reclen);
				
				return ROUND_UP(NAME_OFFSET(dirent) + i + 1);
			}
		}
	}
	return 0;
}

int dmsdos_file_readx(
	struct inode *inode,
	struct file *filp,
	char *buf,
	int count)
{      
        int clusternr;
        unsigned char*clusterd;
        int cvfnr;
        int offset;
        int gelesen;
        int membytes;
        int ret;
        char * b;
        int toread;
        /*int trix;*/
               
        cvfnr=(inode->i_ino>>24)-1;
        if(cvfnr<0)
        { printk("DMSDOS: file_readx: invalid cvfnr ino=%ld\n",inode->i_ino);
          return -EIO;
        }
        
        /*printk("DMSDOS: file_readx: ino=%ld, size=0x%08x, fpos=0x%08x, count=0x%08x\n",
               inode->i_ino,inode->i_size,filp->f_pos,count); 
        
          trix=count;
          printk("count=0x%08x trix=0x%08x \n",count,trix);
        */
        if(count<=0)return 0;
        /*if(trix<=0)return 0;*/
        
        if(filp->f_pos>=inode->i_size)return 0;
        
        if(filp->f_pos+count>inode->i_size)count=inode->i_size-filp->f_pos;
        
        membytes=SECTOR_SIZE*dblsb[cvfnr].s_sectperclust;
        
        /* calculate clusternr for cluster to read */
        clusternr=MSDOS_I(inode)->i_start;
        offset=filp->f_pos;
        while(offset>=membytes&&clusternr>0)
        {  offset-=membytes;
           clusternr=dbl_fat_nextcluster(inode->i_sb,clusternr,cvfnr);
        }
        if(clusternr<=0)
        {  printk("DMSDOS: file_readx: FAT mismatches file size for ino=%ld\n",
                   inode->i_ino);
           return 0;        
        }
        
        gelesen=0;
        b=buf;
        
        clusterd=(unsigned char*)kmalloc(membytes,GFP_KERNEL);
        if(clusterd==NULL)
        { printk("DMSDOS: file_readx: no memory!\n");
          return -EIO;
        }
        
        do
        {  ret=dmsdos_read_cluster(inode->i_sb,clusterd,clusternr,cvfnr);
           if(ret==0)
           { toread=(membytes-offset>count) ? count : membytes-offset;
             /*printk("DMSDOS file_readx: memcpy_tofs(0x%08x,0x%08x,0x%08x)\n",
                     b,clusterd+offset,toread);*/
             memcpy_tofs(b,clusterd+offset,toread);
             gelesen+=toread;
             count-=toread;
             if(count>0)
             { b+=toread;
               offset=0;
               clusternr=dbl_fat_nextcluster(inode->i_sb,clusternr,cvfnr);
               if(clusternr<=0)
               { ret=-1;
                 printk("DMSDOS: file_readx: FAT mismatches file size for ino=%ld\n",
                         inode->i_ino);
               }
             }
           }
        }
        while(count>0&&ret==0);
        
        kfree_s(clusterd,membytes);
        
        filp->f_pos+=gelesen;
        return gelesen;
}

int dmsdos_subdirs(struct inode*inode)
{ int cvfnr;
  int startclust;
  int ent=0;
  int res=0;
  unsigned char buf[32];
  
  /*printk("DMSDOS: subdirs, ino=%ld\n",inode->i_ino);*/
  
  if((cvfnr=dbltest(inode))!=0)cvfnr=(cvfnr>>24)-1;
  else cvfnr=(inode->i_ino>>24)-1;
  startclust=MSDOS_I(inode)->i_start;
  
  while(read_dbl_direntry(inode->i_sb,startclust,cvfnr,ent,buf)>0)
  { ++ent;
    if(buf[0]!=0xe5&&buf[0]!=0&&(buf[11]&0x10)!=0)++res;
  }
  return res;
}

void dmsdos_read_inodex(struct inode *inode)
{	
        int dblsec;
        int offs;
        int cvfnr;
        struct super_block *sb = inode->i_sb;
	struct buffer_head *bh;
	struct msdos_dir_entry *raw_entry;
	int nr;
	int zaehler;

/* printk("read inode %d\n",inode->i_ino); */
	MSDOS_I(inode)->i_busy = 0;
	MSDOS_I(inode)->i_depend = MSDOS_I(inode)->i_old = NULL;
	MSDOS_I(inode)->i_binary = 1;
	inode->i_uid = MSDOS_SB(inode->i_sb)->fs_uid;
	inode->i_gid = MSDOS_SB(inode->i_sb)->fs_gid;
	if ((cvfnr=dbltest(inode))!=0) {
	        cvfnr=(cvfnr>>24)-1;
		inode->i_mode = ((S_IRUGO | S_IXUGO)& 
		       ~MSDOS_SB(inode->i_sb)->fs_umask) | S_IFDIR;
		inode->i_op = &dmsdos_dir_inode_operations;
		MSDOS_I(inode)->i_start=0;/*muss bekannt sein fr _subirs*/
		inode->i_nlink = dmsdos_subdirs(inode)+2;
		    /* subdirs (neither . nor ..) plus . and "self" */
		inode->i_size = dblsb[cvfnr].s_rootdiranzentry*
		    sizeof(struct msdos_dir_entry);
		inode->i_blksize = dblsb[cvfnr].s_sectperclust*
		    SECTOR_SIZE;
		inode->i_blocks = (inode->i_size+inode->i_blksize-1)/
		    inode->i_blksize*dblsb[cvfnr].s_sectperclust;
		MSDOS_I(inode)->i_attrs = 0;
		
		/* read it */
		if(!(bh=read_real_sector(inode->i_sb,inode->i_ino>>4)))
		{
			printk("dev = 0x%04X, ino = %ld\n",inode->i_dev,inode->i_ino);
			panic("dmsdos_read_inode: unable to read i-node block");
		}
		raw_entry = &((struct msdos_dir_entry *) (bh->b_data))
	    		              [inode->i_ino & (MSDOS_DPB-1)];
		inode->i_mtime = inode->i_atime = inode->i_ctime = 
		        date_dos2unix(CF_LE_W(raw_entry->time),
		                      CF_LE_W(raw_entry->date));
		return;
	}
	
	dblsec=(inode->i_ino&0xffffff)>>4;
	offs=inode->i_ino&0xf;
	cvfnr=(inode->i_ino>>24)-1;

	if (!(bh = read_dbl_sector(inode->i_sb,dblsec,cvfnr))) {
		printk("dev = 0x%04X, ino = %ld\n",inode->i_dev,inode->i_ino);
		panic("dmsdos_read_inode: unable to read i-node block");
	}
	raw_entry = &((struct msdos_dir_entry *) (bh->b_data))
	    [inode->i_ino & (MSDOS_DPB-1)];
	if ((raw_entry->attr & ATTR_DIR) && !IS_FREE(raw_entry->name)) {
		inode->i_mode = MSDOS_MKMODE(raw_entry->attr,(S_IRUGO|S_IXUGO)
		               & ~MSDOS_SB(inode->i_sb)->fs_umask) | S_IFDIR;
		inode->i_op = &dmsdos_dir_inode_operations;
		MSDOS_I(inode)->i_start = CF_LE_W(raw_entry->start);
		inode->i_nlink = dmsdos_subdirs(inode);
		    /* includes .., compensating for "self" */
#ifdef DEBUG
		if (!inode->i_nlink) {
			printk("directory %d: i_nlink == 0\n",inode->i_ino);
			inode->i_nlink = 1;
		}
#endif
		inode->i_size = 0;
		
		zaehler=0;
		if ((nr = CF_LE_W(raw_entry->start)) != 0)
			while (nr != -1) {
				inode->i_size += 
				            dblsb[cvfnr].s_sectperclust*512;
				if (!(nr = dbl_fat_nextcluster(inode->i_sb,
				                               nr,cvfnr))) {
					printk("DMSDOS: readinodex: bad FAT (unexspected zero), ino=%ld\n",inode->i_ino);
					break;
				}
				
				++zaehler;
				if(zaehler>65535)
				{ printk("DMSDOS: read_inodex: loop in FAT, ino=%ld\n",inode->i_ino);
				  break;
				}
			}
	}
	else {
		inode->i_mode = MSDOS_MKMODE(raw_entry->attr,(IS_NOEXEC(inode)
		    ? S_IRUGO : S_IRUGO|S_IXUGO) &
		        ~MSDOS_SB(inode->i_sb)->fs_umask) | S_IFREG;
		inode->i_op = sb->s_blocksize == 1024
			? &dmsdos_file_inode_operations_1024
			: &dmsdos_file_inode_operations;
		MSDOS_I(inode)->i_start = CF_LE_W(raw_entry->start);
		inode->i_nlink = 1;
		inode->i_size = CF_LE_L(raw_entry->size);
	}
	MSDOS_I(inode)->i_binary = is_binary(MSDOS_SB(inode->i_sb)->conversion,
	    raw_entry->ext);
	MSDOS_I(inode)->i_attrs = raw_entry->attr & ATTR_UNUSED;
	/* this is as close to the truth as we can get ... */
	inode->i_blksize = dblsb[cvfnr].s_sectperclust*SECTOR_SIZE;
	inode->i_blocks = (inode->i_size+inode->i_blksize-1)/
	    inode->i_blksize*dblsb[cvfnr].s_sectperclust;
	inode->i_mtime = inode->i_atime = inode->i_ctime =
	    date_dos2unix(CF_LE_W(raw_entry->time),CF_LE_W(raw_entry->date));
	bh_free(inode->i_sb,bh);

        /*printk("DMSDOS: readinodex reading inode ino=%ld\n",inode->i_ino);*/
}

int dmsdos_lookupx(struct inode *dir,const char *name,int len,
    struct inode **result)
{
	int ino;
	int cvfnr;
	int startclust;
	
	PRINTK (("dmsdos_lookup\n"));

	*result = NULL;
	if (!dir) return -ENOENT;
	if (!S_ISDIR(dir->i_mode)) {
		iput(dir);
		return -ENOENT;
	}
	PRINTK (("dmsdos_lookup 2\n"));
	if (len == 1 && name[0] == '.') {
		*result = dir;
		return 0;
	}
	if (len == 2 && name[0] == '.' && name[1] == '.') {
	        if(dbltest(dir))
	        { /*printk("DMSDOS: lookupx '..' result MDSOS_ROOT\n");*/
	          ino=MSDOS_ROOT_INO;
	        }
		else
		{ /*printk("DMSDOS: lookupx '..' inode ino=%ld\n",dir->i_ino);*/
		  ino=dmsdos_parent_ino(dir);
		}
		iput(dir);
		if (ino < 0) return ino;
		if (!(*result = iget(dir->i_sb,ino))) return -EACCES;
		return 0;
	}
	if((cvfnr=dbltest(dir))!=0)cvfnr=(cvfnr>>24)-1;
	else cvfnr=(dir->i_ino>>24)-1;
	startclust=MSDOS_I(dir)->i_start;/*dbl_inode2startcluster(dir);*/
	ino=scan_dbl_dir_4_filename(dir->i_sb,startclust,name,len,cvfnr);
	if(ino<0)
	{ iput(dir);
	  return ino;
	}
	if(!(*result=iget(dir->i_sb,ino))) return -EACCES;
	iput(dir);
	return 0;
}

void exit_dbl(struct super_block*sb)
{ int cvfnr,j;

  if(lowest_never_used_cvfnr==0)return;
  
  for(cvfnr=0;cvfnr<MAXDBLFILES;++cvfnr)
  { if(dbl_sb_dev[cvfnr]==sb->s_dev)
    { /*printk("DMSDOS: CVF %d ino=%d on device %d unmounted.\n",
             cvfnr+1,dbl_cvf_inos[cvfnr],sb->s_dev);*/
      
      /* zero out special infos about unmounted cvf */
      dbl_sb_dev[cvfnr]=0;
      dbl_cvf_inos[cvfnr]=0;
      
      /* kill buffers used by unmounted cvf */
      for(j=0;j<MDFATCACHESIZE;++j)
      { if(mdfat[j].a_cvfnr==cvfnr)
        { if(mdfat[j].a_buffer!=NULL)
          { bh_free(sb,mdfat[j].a_buffer); 
            mdfat[j].a_buffer=NULL;
          }
          mdfat[j].a_used=0;
        }
      }
      for(j=0;j<DFATCACHESIZE;++j)
      { if(dfat[j].a_cvfnr==cvfnr)
        { if(dfat[j].a_buffer!=NULL)
          { bh_free(sb,dfat[j].a_buffer);
            dfat[j].a_buffer=NULL;
          }
          dfat[j].a_used=0;
        }
      }
      
    }
  }
}