/*
 * netstat	This file contains an implementation of the command
 *		that helps in debugging the networking modules.
 *
 * Usage:	netstat [-acnortu]
 *               -a all sockets (tcp+udp)
 *		 -c continous listing
 *		 -n show network numbers instead of names
 *		 -o show timer states
 *		 -r show kernel routing table
 *		 -t show all tcp connections
 *		 -u show all udp ports
 *
 * Version:	@(#)netstat.c	0.63	09/03/93
 *
 * Author:	Fred Baumgarten, <dc6iq@insu1.etec.uni-karlsruhe.de>
 *		Copyright (c) 1993  Fred Baumgarten
 *
 * Bugs:        Did a coredump for port>10000 (tcp). Corrected it by
 *              reading the portnames into a list and search trough
 *              it. Fixed 09/03/93 Pauline Middelink
 */
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <linux/route.h>
#include <linux/tcp.h>
#include <linux/net.h>
#include "pathnames.h"


char *Version = "@(#)netstat.c 0.63 (09/03/93)";
#define Signature "(c) 1993, Fred Baumgarten <dc6iq@insu1.etec.uni-karlsruhe.de>"
#define Signatur_ "(b) 1993, Pauline Middelink <middelin@calvin.iaf.nl>"


#define E_READ -1
#define E_PARA -2
#define	PROTO_TCP 2
#define PROTO_UDP 1
#define PROTO_RAW 0

FILE *procinfo;
char *line[2000];

int flag_cnt = 0;
int flag_deb = 0;
int flag_not = 0;
int flag_opt = 0;
int flag_raw = 0;
int flag_rou = 0;
int flag_tcp = 0;
int flag_udp = 0;
int flag_unx = 0;

struct porttoname {
 unsigned int port;
 char proto;
 char *name;
 struct porttoname *next;
};

#define NULLPORT (struct porttoname *) 0

struct porttoname *pn = NULLPORT;

struct numbtoname {
  unsigned long addr;
  char *name;
  struct numbtoname *next;
};

#define NULLNN (struct numbtoname *) 0

struct numbtoname *nn = NULLNN;

void query_host(char *h, unsigned long ad) {
  struct hostent *ent = (struct hostent *) 0;
  struct numbtoname *pn = NULLNN;

  if (!ad) {
    strcpy(h, "*");
  } else {
    pn = nn;
    *h = '\0';
    while (pn != NULLNN) {
      if (pn->addr == ad) {
	strcpy(h, pn->name);
	break;
      }
      pn = pn->next;
    }
    if (*h == '\0') {
      if (!flag_not) ent = gethostbyaddr((char *) &ad, 4, AF_INET);
      if (ent) {
	strcpy(h, ent->h_name);
      } else {
	sprintf(h, "%d.%d.%d.%d", (int) (ad & 0xff), (int) ((ad >> 8) & 0xff),
		(int) ((ad >> 16) & 0xff), (int) ((ad >> 24) & 0xff));
      }
      pn = (struct numbtoname *)malloc(sizeof(struct numbtoname));
      pn->addr = ad;
      pn->next = nn;
      pn->name = malloc(strlen(h)+1);
      strcpy(pn->name, h);
      nn = pn;
    }
  }
}

int read_services(void) {
  char buffer[2048], name[32], protocol[16], dummy[1024];
  int number;
  FILE *serv;
  struct porttoname *pp;

  serv = fopen(_PATH_SERVICES, "r");
  if (!serv) {
    perror(_PATH_SERVICES);
    return E_READ;
  }

  while (!feof(serv)) {
    fgets(buffer, sizeof(buffer)-1, serv);
    if (sscanf(buffer, "%s%d/%3s%s\n", name, &number, protocol, dummy) >= 3) {
      if (!strcmp(protocol, "tcp")) {
	pp = malloc(sizeof(struct porttoname));
 	pp->port  = number;
	pp->proto = PROTO_TCP;
	pp->name  = strdup(name);
	pp->next  = pn;
	pn = pp;
      } else {
	if (!strcmp(protocol, "udp")) {
	  pp = malloc(sizeof(struct porttoname));
 	  pp->port  = number;
	  pp->proto = PROTO_UDP;
	  pp->name  = strdup(name);
	  pp->next  = pn;
	  pn = pp;
	} else {
	  if (!strcmp(protocol, "raw")) {
	    pp = malloc(sizeof(struct porttoname));
 	    pp->port  = number;
	    pp->proto = PROTO_RAW;
	    pp->name  = strdup(name);
	    pp->next  = pn;
	    pn = pp;
	  }
	}
      }
    }
  }
  fclose(serv);
  return 0;
}

char *get_sname(int socknumber, char proto) {
  static char buffer[64];
  struct porttoname *pp;

  if (flag_not) {
    sprintf(buffer, "%d", socknumber);
    return buffer;
  }
  if (socknumber == 0)
    return "*";
  pp = pn;
  while (pp!=NULLPORT) {
    if (pp->proto == proto && pp->port == socknumber) {
       sprintf(buffer, "%s", pp->name);
       break;
    }
    pp = pp->next;
  }
  if (pp==NULLPORT) sprintf(buffer, "%d", socknumber);
  return buffer;
}

int route_info(void) {
  char buffer[1024], iface[16], net_addr[64], gate_addr[64], flags[16];
  int anz, iflags, refcnt, use, zeile = 0;
  unsigned long net, gate;

  printf("Kernel routing table\n");
  printf("Destination net/address   Gateway address           Flags RefCnt    Use Iface\n");
  procinfo = fopen(_PATH_PROCNET_ROUTE, "r");
  if (!procinfo) {
    perror(_PATH_PROCNET_ROUTE);
    return E_READ;
  }
  fgets(buffer, 1023, procinfo);
  while (!feof(procinfo)) {
    fgets(buffer, 1023, procinfo);
    if ((line[zeile] = (char *)malloc(strlen(buffer)+1)) != NULL) {
      strcpy(line[zeile++], buffer);
      if (flag_deb) {
        fprintf(stderr, "%s", buffer);
      }
    }
  }
  fclose(procinfo);
  zeile--;
  while (zeile>=0) {
    anz = sscanf(line[zeile--], "%s %lX %lX %X %d %d\n",
	   iface, &net, &gate, &iflags, &refcnt, &use);
    if (anz == 6) {
      query_host(net_addr, net);
      net_addr[25]='\0';
      if (net == 0L) strcpy(net_addr, "default");
      query_host(gate_addr, gate);
      gate_addr[25]='\0';
      flags[0]='\0';
      if (iflags & RTF_UP) strcat(flags, "U");
      if (iflags & RTF_GATEWAY) strcat(flags, "G");
      if (iflags & RTF_HOST) {
        strcat(flags, "H");
      } else {
        strcat(flags, "N");
      }
      if (iflags & RTF_REINSTATE) strcat(flags, "R");
      if (iflags & RTF_DYNAMIC) strcat(flags, "D");
      if (iflags & RTF_MODIFIED) strcat(flags, "M");
      printf("%-25s %-25s %-5s %6d %6d %s\n", net_addr, gate_addr, flags, refcnt, use, iface);
    }
  }
  return 0;
}

int tcp_info(void) {
  char buffer[1024], local_addr[64], rem_addr[64], *tcp_state,
       timer_queued, timers[64], more[512];
  int anz, local_port, rem_port, d, state, timer_run, zeile = 0;
  unsigned long rxq, txq, localaddr, remaddr, time_len, retr;
  
  procinfo = fopen(_PATH_PROCNET_TCP, "r");
  if (!procinfo) {
    perror(_PATH_PROCNET_TCP);
    return E_READ;
  }
  fgets(buffer, 1023, procinfo);
  while (!feof(procinfo)) {
    fgets(buffer, 1023, procinfo);
    anz = strlen(buffer)+1;
    if ((line[zeile] = (char *)malloc(strlen(buffer)+1)) != NULL) {
      strcpy(line[zeile++], buffer);
      if (flag_deb) {
        fprintf(stderr, "%s", buffer);
      }
    }
  }
  fclose(procinfo);
  zeile--;
  zeile--;
  while (zeile>=0) {
    more[0] = '\0';
    timer_queued = '\0';
    anz = sscanf(line[zeile--], "%d: %lX:%X %lX:%X %X %X:%X %X:%lX %lX %c %s\n",
		 &d, &localaddr, &local_port, &remaddr, &rem_port,
		 &state, &txq, &rxq, &timer_run, &time_len, &retr,
		 &timer_queued, more);
    if (!flag_opt) more[0]='\0';
    if (flag_deb) fprintf(stderr, "%s -> %d args", line[zeile+1], anz);
    if (anz >= 11) {
      if (state != TCP_ESTABLISHED) {
        txq=0;		/* Here are stupid thigs... Why isn't it zero ? */
      }
      switch (state) {
      case TCP_ESTABLISHED:
	tcp_state = "ESTABLISHED";
	rxq--;
	break;
      case TCP_SYN_SENT:
	tcp_state = "SYN_SENT";
	break;
      case TCP_SYN_RECV:
	tcp_state = "SYN_RECV";
	break;
      case TCP_FIN_WAIT1:
	tcp_state = "FIN_WAIT1";
	break;
      case TCP_FIN_WAIT2:
	tcp_state = "FIN_WAIT2";
	break;
      case TCP_TIME_WAIT:
	tcp_state = "TIME_WAIT";
	break;
      case TCP_CLOSE:
	tcp_state = "CLOSE";
	break;
      case TCP_CLOSE_WAIT:
	tcp_state = "CLOSE_WAIT";
	break;
      case TCP_LAST_ACK:
	tcp_state = "LAST_ACK";
	break;
      case TCP_LISTEN:
	tcp_state = "LISTEN";
	break;
      default:
	tcp_state = "?? ()";
	break;
      }
      query_host(local_addr, localaddr);
      query_host(rem_addr, remaddr);
      if (flag_tcp || rem_port) {
	sprintf(buffer, "%s",get_sname(local_port, PROTO_TCP));
	if ((strlen(local_addr) + strlen(buffer)) >21) {
	  local_addr[21-strlen(buffer)] = '\0';
	}
	strcat(local_addr, ":");
	strcat(local_addr, buffer);
	sprintf(buffer, "%s",get_sname(rem_port, PROTO_TCP));
	if ((strlen(rem_addr) + strlen(buffer)) >21) {
	  rem_addr[21-strlen(buffer)] = '\0';
	}
	strcat(rem_addr, ":");
	strcat(rem_addr, buffer);
	if (flag_opt) {
	  switch (timer_run) {
	    case 0:
            case 9: {
	      sprintf(timers, "off (0.00/%d) %c", retr, timer_queued);
	      break;
	    }
	    case 1:
	    case 2: {
	      sprintf(timers, "queueing (%2.2f/%d) %c", (double)time_len / 100, retr,
		      timer_queued);
	      break;
	    }
	    case 3:
	    case 4:
	    case 5: {
	      sprintf(timers, "on (%2.2f/%d) %c", (double)time_len / 100, retr, timer_queued);
	      break;
	    }
	    default: {
	      if (timer_run > 10) {
	        sprintf(timers, "exp (0.00/%d) %c", retr, timer_queued);
	        break;
	      } else {
	        sprintf(timers, "unkn-%d (%2.2f/%d) %c", timer_run, (double)time_len / 100,
			retr, timer_queued);
	        break;
	      }
	    }
	  }
	} else {
	  timers[0] = '\0';
	}
	printf("tcp   %6d %6d %-22s %-22s %s %s %s\n", rxq, txq, local_addr,
	       rem_addr, tcp_state, timers, more);
      }
    }
  }
  return 0;
}

int udp_info(void) {
  char buffer[1024], local_addr[64], rem_addr[64], *udp_state,
       timer_queued, timers[64], more[512];
  int anz, local_port, rem_port, d, state, timer_run, zeile = 0;
  unsigned long rxq, txq, localaddr, remaddr, time_len, retr;
  
  procinfo = fopen(_PATH_PROCNET_UDP, "r");
  if (!procinfo) {
    perror(_PATH_PROCNET_UDP);
    return E_READ;
  }
  fgets(buffer, 1023, procinfo);
  while (!feof(procinfo)) {
    fgets(buffer, 1023, procinfo);
    if ((line[zeile] = (char *)malloc(strlen(buffer)+1)) != NULL) {
      strcpy(line[zeile++], buffer);
      if (flag_deb) {
        fprintf(stderr, "%s", buffer);
      }
    }
  }
  fclose(procinfo);
  zeile--;
  zeile--;
  while (zeile>=0) {
    more[0] = '\0';
    timer_queued = '\0';
    anz = sscanf(line[zeile--], "%d: %lX:%X %lX:%X %X %X:%X %X:%lX %lX %c %s\n",
		 &d, &localaddr, &local_port, &remaddr, &rem_port,
		 &state, &txq, &rxq, &timer_run, &time_len, &retr,
		 &timer_queued, more);
    retr = 0L;
    if (!flag_opt) more[0]='\0';
    if (flag_deb) fprintf(stderr, "%s -> %d args", line[zeile+1], anz);
    if (anz >= 10) {
      if (state != TCP_ESTABLISHED) {
        txq=0;		/* Here are stupid thigs... Why isnt it zero ? */
      }
      switch (state) {
      case TCP_ESTABLISHED:
	udp_state = "ESTABLISHED";
	rxq--;
	break;
      default:
	udp_state = "";
	break;
      }
      query_host(local_addr, localaddr);
      query_host(rem_addr, remaddr);
      if (flag_udp) {
	sprintf(buffer, "%s",get_sname(local_port, PROTO_UDP));
	if ((strlen(local_addr) + strlen(buffer)) >21) {
	  local_addr[21-strlen(buffer)] = '\0';
	}
	strcat(local_addr, ":");
	strcat(local_addr, buffer);
	sprintf(buffer, "%s",get_sname(rem_port, PROTO_UDP));
	if ((strlen(rem_addr) + strlen(buffer)) >21) {
	  rem_addr[21-strlen(buffer)] = '\0';
	}
	strcat(rem_addr, ":");
	strcat(rem_addr, buffer);
	if (flag_opt) {
	  switch (timer_run) {
	    case 0:
            case 9: {
	      sprintf(timers, "off (0.00/%d) %c", retr, timer_queued);
	      break;
	    }
	    case 1:
	    case 2: {
	      sprintf(timers, "queueing (%2.2f/%d) %c", (double)time_len / 100, retr,
		      timer_queued);
	      break;
	    }
	    case 3:
	    case 4:
	    case 5: {
	      sprintf(timers, "on (%2.2f/%d) %c", (double)time_len / 100, retr, timer_queued);
	      break;
	    }
	    default: {
	      if (timer_run > 10) {
	        sprintf(timers, "exp (0.00/%d) %c", retr, timer_queued);
	        break;
	      } else {
	        sprintf(timers, "unkn-%d (%2.2f/%d) %c", timer_run, (double)time_len / 100,
			retr, timer_queued);
	        break;
	      }
	    }
	  }
	} else {
	  timers[0] = '\0';
	}
	printf("udp   %6d %6d %-22s %-22s %s %s %s\n", rxq, txq, local_addr,
	       rem_addr, udp_state, timers, more);
      }
    }
  }
  return 0;
}

int raw_info(void) {
  char buffer[1024], local_addr[64], rem_addr[64], *raw_state,
       timer_queued, timers[64], more[512];
  int anz, local_port, rem_port, d, state, timer_run, zeile = 0;
  unsigned long rxq, txq, localaddr, remaddr, time_len, retr;
  
  procinfo = fopen(_PATH_PROCNET_RAW, "r");
  if (!procinfo) {
    perror(_PATH_PROCNET_RAW);
    return E_READ;
  }
  fgets(buffer, 1023, procinfo);
  while (!feof(procinfo)) {
    fgets(buffer, 1023, procinfo);
    if ((line[zeile] = (char *)malloc(strlen(buffer)+1)) != NULL) {
      strcpy(line[zeile++], buffer);
      if (flag_deb) {
        fprintf(stderr, "%s", buffer);
      }
    }
  }
  fclose(procinfo);
  zeile--;
  zeile--;
  while (zeile>=0) {
    more[0] = '\0';
    timer_queued = '\0';
    anz = sscanf(line[zeile--], "%d: %lX:%X %lX:%X %X %X:%X %X:%lX %lX %c %s\n",
		 &d, &localaddr, &local_port, &remaddr, &rem_port,
		 &state, &txq, &rxq, &timer_run, &time_len, &retr,
		 &timer_queued, more);
    retr = 0L;
    if (!flag_opt) more[0]='\0';
    if (flag_deb) fprintf(stderr, "%s -> %d args", line[zeile+1], anz);
    if (anz >= 10) {
      raw_state = "";
      txq=0;		/* Here are stupid thigs... Why isnt it zero ? */
      query_host(local_addr, localaddr);
      query_host(rem_addr, remaddr);
      if (flag_raw) {
	sprintf(buffer, "%s",get_sname(local_port, PROTO_RAW));
	if ((strlen(local_addr) + strlen(buffer)) >21) {
	  local_addr[21-strlen(buffer)] = '\0';
	}
	strcat(local_addr, ":");
	strcat(local_addr, buffer);
	sprintf(buffer, "%s",get_sname(rem_port, PROTO_RAW));
	if ((strlen(rem_addr) + strlen(buffer)) >21) {
	  rem_addr[21-strlen(buffer)] = '\0';
	}
	strcat(rem_addr, ":");
	strcat(rem_addr, buffer);
	if (flag_opt) {
	  switch (timer_run) {
	    case 0:
            case 9: {
	      sprintf(timers, "off (0.00/%d) %c", retr, timer_queued);
	      break;
	    }
	    case 1:
	    case 2: {
	      sprintf(timers, "queueing (%2.2f/%d) %c", (double)time_len / 100, retr,
		      timer_queued);
	      break;
	    }
	    case 3:
	    case 4:
	    case 5: {
	      sprintf(timers, "on (%2.2f/%d) %c", (double)time_len / 100, retr, timer_queued);
	      break;
	    }
	    default: {
	      if (timer_run > 10) {
	        sprintf(timers, "exp (0.00/%d) %c", retr, timer_queued);
	        break;
	      } else {
	        sprintf(timers, "unkn-%d (%2.2f/%d) %c", timer_run, (double)time_len / 100,
			retr, timer_queued);
	        break;
	      }
	    }
	  }
	} else {
	  timers[0] = '\0';
	}
	printf("raw   %6d %6d %-22s %-22s %s %s %s\n", rxq, txq, local_addr,
	       rem_addr, raw_state, timers, more);
      }
    }
  }
  return 0;
}

int unix_info(void) {
  char buffer[1024], path[MAXPATHLEN], ss_flags[32], *ss_proto, *ss_state, *ss_type;
  int anz, d, state, type, zeile = 0;
  unsigned long refcnt, proto, flags;
  
  procinfo = fopen(_PATH_PROCNET_UNIX, "r");
  if (!procinfo) {
    perror(_PATH_PROCNET_UNIX);
    return E_READ;
  }
  fgets(buffer, 1023, procinfo);
  while (!feof(procinfo)) {
    fgets(buffer, 1023, procinfo);
    if ((line[zeile] = (char *)malloc(strlen(buffer)+1)) != NULL) {
      strcpy(line[zeile++], buffer);
      if (flag_deb) {
        fprintf(stderr, "%s", buffer);
      }
    }
  }
  fclose(procinfo);
  zeile--;
  zeile--;
  printf("Unix internal communications\n");
  printf("Proto RefCnt Flags      Type            State           Path\n");
  while (zeile>=0) {
    path[0] = '\0';
    anz = sscanf(line[zeile--], "%d: %lX %lX %lX %X %X %s\n",
		 &d, &refcnt, &proto, &flags, &type, &state, path);
    if (flag_deb) fprintf(stderr, "%s -> %d args", line[zeile+1], anz);
    if (anz >= 6) {
      switch(proto) {
      case 0:
	ss_proto = "unix";
	break;
      default:
	ss_proto = "??";
      }
      switch(type) {
      case SOCK_STREAM:
	ss_type = "SOCK_STREAM";
	break;
      case SOCK_DGRAM:
	ss_type = "SOCK_DGRAM";
	break;
      case SOCK_RAW:
	ss_type = "SOCK_RAW";
	break;
      case SOCK_RDM:
	ss_type = "SOCK_RDM";
	break;
      case SOCK_SEQPACKET:
	ss_type = "SOCK_SEQPACKET";
	break;
      case SOCK_PACKET:
	ss_type = "SOCK_PACKET";
	break;
      default:
	ss_type = "UNKNOWN";
      }
      switch(state) {
      case SS_FREE:
	ss_state = "FREE";
	break;
      case SS_UNCONNECTED:
	/*
	 * Unconnected sockets may be listening
	 * for something.
	 */
	if (flags & SO_ACCEPTCON) {
	  ss_state = "LISTENING";
	} else {
	  ss_state = "UNCONNECTED";
	}
	break;
      case SS_CONNECTING:
	ss_state = "CONNECTING";
	break;
      case SS_CONNECTED:
	ss_state = "CONNECTED";
	break;
      case SS_DISCONNECTING:
	ss_state = "DISCONNECTING";
	break;
      default:
	ss_state = "UNKNOWN";
      }
      strcpy(ss_flags, "[");
      if (flags & SO_ACCEPTCON) {
	strcat(ss_flags, " ACC ");
      }
      if (ss_flags[strlen(ss_flags)-1] != ' ') strcat(ss_flags, " ");
      strcat(ss_flags, "]");
      printf("%-5s %-6d %-10s %-15s %-15s %s\n", ss_proto, refcnt, ss_flags,
	     ss_type, ss_state, path);
    }
  }
  return 0;
}

void usage(void) {
  fprintf(stderr, "Usage:\tnetstat [options]\n");
  fprintf(stderr, "\t-a all sockets (tcp+udp)\n");
  fprintf(stderr, "\t-c continous listing\n");
  fprintf(stderr, "\t-n show network numbers instead of names\n");
  fprintf(stderr, "\t-o show timer states\n");
  fprintf(stderr, "\t-r show kernel routing table\n");
  fprintf(stderr, "\t-t show all tcp connections\n");
  fprintf(stderr, "\t-u show all udp ports\n");
  fprintf(stderr, "\t-w show all raw ports\n");
}

int main (int argc, char *argv[]) {
  int erg;
  char c;

  while ((c = getopt(argc, argv, "acdnortuvwx")) != EOF)
    switch(c) {
    case 'a':
      flag_tcp++;
      flag_udp++;
      flag_raw++;
      break;
    case 'c':
      flag_cnt++;
      break;
    case 'd':
      flag_deb++;
      break;
    case 'n':
      flag_not++;
      break;
    case 'o':
      flag_opt++;
      break;
    case 'r':
      flag_rou++;
      break;
    case 't':
      flag_tcp++;
      break;
    case 'u':
      flag_udp++;
      break;
    case 'v':
      printf("%s\n%s\n", Version, Signature);
      return 0;
      break;
    case 'w':
      flag_raw++;
      break;
    case 'x':
      flag_unx++;
      break;
    case '?':
      usage();
      return E_PARA;
    }
  
  if (flag_rou) {
    for (;; ) {
      erg = route_info();
      if (!flag_cnt || erg) break;
      sleep(1);
    }
    return erg;
  }
  
  if (flag_unx) {
    for (;; ) {
      erg = unix_info();
      if (!flag_cnt || erg) break;
      sleep(1);
    }
    return erg;
  }
  
  if ((erg = read_services()) != 0) return erg;

  for (;; ) {
    printf("Active Internet connections\n");
    printf("Proto Recv-Q Send-Q Local Address          Foreign Address        (State)\n");
    if ((!flag_udp && !flag_raw) || flag_tcp) {
      erg = tcp_info();
      if (erg) return erg;
    }
    if (flag_udp) {
      erg = udp_info();
      if (erg) return erg;
    }
    if (flag_raw) {
      erg = raw_info();
      if (erg) return erg;
    }
    if (!flag_cnt || erg) break;
    sleep(1);
  }
  return erg;
}
