/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  tarlist.c                                                    */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <ntape.h>

void show_tar_list(void);
void draw_tar_listbox(void);
void highlight_tar_list_element(int,int);
void clear_tar_list(void);

static tar_list_el *tar_head = NULL,*tar_tail = NULL;
static int active = 0;
static int first_item;
static int focus_item;
static int no_listitems;
static int clear;
static WINDOW *tarlistbox;


/*************************************************************************/
/* Name:     clear_tar_list                                              */
/* Purpose:  Clears an already allocated list                            */
/*************************************************************************/
void clear_tar_list()
{
    tar_list_el *s,*t;

    s = tar_head;
    while (s != NULL)
    {
	t = s->next;
#ifdef DEBUG
	fprintf(stderr,"Clear %s!\n",s->name);
#endif
	free(s->name);
	free(s);
	s = t;
    }
    tar_head = NULL;
    no_listitems = 0;
  
    delwin(tarlistbox);
}


/*************************************************************************/
/* Name:     draw_tar_listbox                                            */
/* Purpose:  Initialize listbox, draw listbox                            */
/*************************************************************************/
void draw_tar_listbox()
{
    first_item = 0;
    focus_item = 0;
    active = 0;

    tarlistbox = newwin(LINES-9,COLS-8,4,4);
    leaveok(tarlistbox,TRUE);
    keypad(tarlistbox,TRUE);

    clear = 1;
    show_tar_list();
}


/*************************************************************************/
/* Name:     insert_tar_list                                             */
/* Purpose:  Inserts an element with the given data at the end of list.  */
/*************************************************************************/
void insert_tar_list(char *name,long size)
{
    tar_list_el *element,*laufelement;

    if ((element = (tar_list_el *)malloc(sizeof(tar_list_el))) == NULL)
    {
	print_footer("Not enough memory. Sorry...");
	nta_exit(-1);
    }
    if ((element->name = (char *)malloc(strlen(name)+1)) == NULL)
    {
	print_footer("Not enough memory. Sorry...");
	nta_exit(-1);
    }
    strcpy(element->name,name);
    element->size = size;
    element->marked = 0;
  
    laufelement = tar_head;

    if (tar_head == NULL)
    {
	element->next = element->prev = NULL;
	tar_head = tar_tail = element;
    }
    else
    {
	tar_tail->next = element;
	element->prev = tar_tail;
	element->next = NULL;
	tar_tail = element;
    }

    no_listitems++;

#ifdef DEBUG
    fprintf(stderr,"%s %ld\tinserted!\t",tar_tail->name,tar_tail->size);
    fprintf(stderr,"%d Element(e):\n",no_listitems);
#endif
}

/*************************************************************************/
/* Name:     show_tar_list                                               */
/* Purpose:  Shows list in tar list box                                  */
/*************************************************************************/
void show_tar_list()
{
    tar_list_el *laufelement;
    int i;
    char line[MAXLEN];
  
    if (clear)
    {
	set_color_pair(tarlistbox,CONTLIST);
	winclr(tarlistbox);
    }
    set_color_pair(tarlistbox,CONTHEADER);
    wmove(tarlistbox,0,0);
    winclrtoeol(tarlistbox);
    wmove(tarlistbox,2,0);
    winclrtoeol(tarlistbox);
    wmove(tarlistbox,1,0);
    winclrtoeol(tarlistbox);
    sprintf(line,"Contents of %s",archive->name);
    mvwaddstr(tarlistbox,1,(COLS - 8 - strlen(line)) / 2,line);
    
    if (tar_head == NULL)
	goto end;
    laufelement = tar_head;
    for (i = 0; i < first_item; i++)
	laufelement = laufelement->next;
  
    for (i = 3; i < LINES - 12; i++)
    {
	if (laufelement->marked)
	    set_color_pair(tarlistbox,CONTLISTSEL);
	else
	    set_color_pair(tarlistbox,CONTLIST);

	strcpy(line,laufelement->name);
	line[COLS - 20] = '\0';
	mvwaddstr(tarlistbox,i,1,line);
	sprintf(line,"%9ld",laufelement->size);
	mvwaddstr(tarlistbox,i,COLS-18,line);

	laufelement = laufelement->next;
	if (laufelement == NULL)
	    break;
    }
    set_color_pair(tarlistbox,CONTHEADER);
    wmove(tarlistbox,LINES - 12,0);
    winclrtoeol(tarlistbox);
    wmove(tarlistbox,LINES - 11,0);
    winclrtoeol(tarlistbox);
    wmove(tarlistbox,LINES - 10,0);
    winclrtoeol(tarlistbox);
  
    if (active == 1)
	set_color_pair(tarlistbox,INV_CONTBTN);    
    else
	set_color_pair(tarlistbox,CONTBTN);
    mvwaddstr(tarlistbox,LINES - 11,4,"   OK   ");
    if (active == 2)
	set_color_pair(tarlistbox,INV_CONTBTN);
    else
	set_color_pair(tarlistbox,CONTBTN);
    mvwaddstr(tarlistbox,LINES - 11,COLS - 20," Cancel ");
  
    if (no_listitems)
	highlight_tar_list_element(-1,focus_item);

end: touchwin(tarlistbox);
    wrefresh(tarlistbox);
}


/*************************************************************************/
/* Name:     hilight_tar_list_element                                    */
/* Purpose:  Inverts the new and old element.                            */
/*************************************************************************/
void highlight_tar_list_element(int old_element,int new_element)
{
    char line[MAXLEN];
    int i;
    tar_list_el *s;

    if (active == 0)
    {
	if (new_element >= 0)
	{
            for (s = tar_head, i = 0; s != NULL && i != new_element; i++)
                s = s->next;
	    if (s->marked)
		set_color_pair(tarlistbox,INV_CONTLISTSEL);
	    else
		set_color_pair(tarlistbox,INV_CONTLIST);
	    for (i = 0; i < COLS - 8; i++)
		line[i] = mvwinch(tarlistbox,new_element - first_item + 3,i);
	    line[i] = '\0';
	    mvwaddstr(tarlistbox,new_element - first_item + 3,0,line);
	}
  
      
	if (old_element >= 0)
	{
	    for (s = tar_head, i = 0; s != NULL && i != old_element; i++)
		s = s->next;
	    if (s->marked)
		set_color_pair(tarlistbox,CONTLISTSEL);
	    else
		set_color_pair(tarlistbox,CONTLIST);
	    for (i = 0; i < COLS - 8; i++)
		line[i] = mvwinch(tarlistbox,old_element - first_item + 3,i);
	    line[i]= '\0';
	    mvwaddstr(tarlistbox,old_element - first_item + 3,0,line);
	}
	touchwin(tarlistbox);
	wrefresh(tarlistbox);
    }
}

/*************************************************************************/
/* Name:     handle_tar_listbox                                          */
/* Purpose:  Manages the listbox.                                        */
/*************************************************************************/
void handle_tar_listbox()
{
    int i, c, fertig = 0,one_marked;
    tar_list_el *s;
    char cmd[MAXLEN],filename[MAXLEN];
    FILE *fp;

    draw_tar_listbox();
    while (!fertig)
    {
	c = getch();
	switch(c)
	{
	case KEY_DOWN:
	    if (active == 0)
	    {
	key_down:
		clear = 1;
		scroll_down(&focus_item,&first_item,LINES - 15,
			    no_listitems,&highlight_tar_list_element,
			    &show_tar_list);
	    }
	    break;
	case KEY_UP:
	    if (active == 0)
	    {
		clear = 1;
		scroll_up(&focus_item,&first_item,LINES - 15,
			  no_listitems,&highlight_tar_list_element,
			  &show_tar_list);
	    }
	    break;
	case KEY_NPAGE:
	    if (active == 0)
	    {
		clear = 1;
		page_down(&focus_item,&first_item,LINES - 15,
			  no_listitems,&highlight_tar_list_element,
			  &show_tar_list);
	    }
	    break;
	case KEY_PPAGE:
	    if (active == 0)
	    {
		clear = 1;
		page_up(&focus_item,&first_item,LINES - 15,
			no_listitems,&highlight_tar_list_element,
			&show_tar_list);
	    }
	    break;
	case KEY_A1:
	case KEY_HOME:
	    if (active == 0)
	    {
		clear = 1;
		top_pos(&focus_item,&first_item,LINES - 15,
                        no_listitems,&highlight_tar_list_element,
                        &show_tar_list);
	    }
	    break;
	case KEY_C1:
	case KEY_END:
	    if (active == 0)
	    {
		clear = 1;
		bottom_pos(&focus_item,&first_item,LINES - 15,
			   no_listitems,&highlight_tar_list_element,
			   &show_tar_list);
	    }
	    break;
	case TAB_KEY:
	case KEY_RIGHT:
	    if (active == 2)
		active = 0;
	    else
		active++;
	    clear = 1;
	    show_tar_list();
	    break;
	case KEY_LEFT:
	    if (active == 0)
		active = 2;
	    else
		active--;
	    clear = 1;
	    show_tar_list();
	    break;
	case CR_KEY:
	    if (active == 2)
		fertig = 1;
	    if (active == 1)
	    {
		tmpnam(filename);
#ifdef DEBUG
		fprintf(stderr,"%s\n",filename);
#endif
		if ((fp = fopen(filename,"w")) == NULL)
		{
		    print_footer("Can't extract archive. Sorry...");
		    break;
		}
		/* fprintf(fp," \n"); */  /* the list file mustn't be empty */
		one_marked = 0;
		for (s = tar_head; s != NULL; s = s->next)
		    if (s->marked)
		    {
			one_marked = 1;
			fprintf(fp,"%s\n",s->name);
		    }
		fclose(fp);

		if (!one_marked)
		{
		    print_footer("No entry marked! Nothing needs to be done!");
		    unlink(filename);
		    break;
		}
		
		if (watch_tape)
		{
		    create_watch_win(" Extract Archive Log ");
		    print_watch_win("Checking if tape is online...");
		}
		if (offline())
		{
		    print_watch_win("\nTape offline! That's bad!");
		    destroy_watch_win();
		    print_footer("Sorry! Tape is offline! Nothing extracted!");
		    unlink(filename);
		    break;
		}
 
		sprintf(cmd,"Ok.\nSearching archive at block %ld...",archive->position);
		if (watch_tape)
		    print_watch_win(cmd);
		else
		    print_footer("Searching archive...");
		if (do_tape_command("seek",archive->position) < 0)
		{
		    print_watch_win("Not found");
		    destroy_watch_win();
 		    print_footer("Can't find archive! Nothing extracted!");
		    unlink(filename);
		    break;
		}
		if (offline())
		{
                    print_watch_win("\nTape offline! That's bad!");
                    destroy_watch_win();
 		    print_footer("Sorry! Tape is offline! Nothing extracted!");
		    unlink(filename);
		    break;
		}

		sprintf(cmd,"tar xvT %s 2>&1",filename);
		if (watch_tape)
		{
		    print_watch_win("Ok.\nExtracting archive using '");
		    print_watch_win(cmd);
		    print_watch_win("'...\n");
		}
		else
		    print_footer("Extracting archive...");
		if ((fp = popen(cmd,"r")) == NULL)
		{
		    print_watch_win("Tape read error!\n");
		    destroy_watch_win();
		    print_footer("Error on extracting archive!");
		    unlink(filename);
		    break;
		}
		while (fgets(cmd, MAXLEN-1, fp) != NULL)
		    print_watch_win(cmd);
		
		pclose(fp);
		destroy_watch_win();
	
                sprintf(cmd,"Archive %s successfully extracted!",archive->name);
                print_footer(cmd);
		unlink(filename);
		fertig = 1;
	    }
	    break;
	default:
	    break;
	case SPACE_KEY:
	    if (active == 0)
	    {
		for (s = tar_head, i = 0; s != NULL && i != focus_item; i++)
		    s = s->next;
		s->marked = 1 - s->marked;
		clear = 0;
		show_tar_list();
		highlight_tar_list_element(-1,focus_item);
		goto key_down;
	    }
	    break;
	}
    }
    clear_tar_list();
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
