/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  options.c                                                    */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <ntape.h>

void print_options_popup(void);
void handle_sort_opts(void);
void handle_other_opts(void);

#define NO_OPTIONS_CELLS 1
static char *options_cell[] = {" Sort options"," Other options"};

static int chosen_cell = 0;
static WINDOW *options_popup;  /* Popup menu */
static WINDOW *sort_options;   /* Sort options window */
static WINDOW *other_options;   /* Other options window */

/*************************************************************************/
/* Name:     options_chosen                                              */
/* Purpose:  Opens the Options popup window                              */
/*************************************************************************/
void options_chosen()
{
    options_popup = newwin(NO_OPTIONS_CELLS+3,17,1,25);
    leaveok(options_popup,TRUE);
    keypad(options_popup,TRUE);
    print_options_popup();
    popup_active = 1;
}


/*************************************************************************/
/* Name:     options_chosen_cell                                         */
/* Purpose:  Executes a menu item                                        */
/*************************************************************************/
void options_chosen_cell()
{
    switch(chosen_cell)
    {
    case 0:
	handle_sort_opts();
	break;
    case 1:
	handle_other_opts();
	break;
    default:
	break;
    }
}


/*************************************************************************/
/* Name:     print_options_popup                                         */
/* Purpose:  prints the contents of the options-popup                    */
/*************************************************************************/
void print_options_popup()
{
    int i;
  
    wmove(options_popup,0,0);
    for (i = 0; i <= NO_OPTIONS_CELLS; i++)
    {
	if (i == chosen_cell)
	    set_color_pair(options_popup,INV_MENU);
	else
	    set_color_pair(options_popup,MENU);
	mvwaddstr(options_popup,i+1,0,options_cell[i]);
	winclrtoeol(options_popup);
    }
    set_color_pair(options_popup,MENU);
    box(options_popup,ACS_VLINE,ACS_HLINE);
    touchwin(options_popup);
    wrefresh(options_popup);
}


/*************************************************************************/
/* Name:     options_scroll_down                                         */
/* Purpose:                                                              */
/*************************************************************************/
void options_scroll_down()
{
    if (chosen_cell == NO_OPTIONS_CELLS)
	chosen_cell = 0;
    else
	chosen_cell++;
    print_options_popup();
}


/*************************************************************************/
/* Name:     options_scroll_up                                           */
/* Purpose:                                                              */
/*************************************************************************/
void options_scroll_up()
{
    if (chosen_cell == 0)
	chosen_cell = NO_OPTIONS_CELLS;
    else
	chosen_cell--;
    print_options_popup();
}


/*************************************************************************/
/* Name:     destroy_options_popup                                       */
/* Purpose:                                                              */
/*************************************************************************/
void destroy_options_popup()
{
    delwin(options_popup);
    popup_active = 0;
    touchwin(tapestatus);
    wrefresh(tapestatus);
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     handle_sort_opts                                            */
/* Purpose:  manage sort options window                                  */
/*************************************************************************/
void handle_sort_opts()
{
    int fertig = 0,active = 0,opt,c,order;
    char *sortopts[] = {"NAME","SIZE","POSITION"};
      
    opt = sort_type;
    order = sort_ascdesc;
        
    sort_options = newwin(10,40,(LINES-3-10)/2+2,(COLS-40)/2);
    leaveok(sort_options,TRUE);
    keypad(sort_options,TRUE);    
    set_color_pair(sort_options,OPTSLABELS);
    winclr(sort_options);
    
    set_color_pair(sort_options,OPTSFRAME);
    wmove(sort_options,0,0);
    winclrtoeol(sort_options);
    mvwaddstr(sort_options,1,0,"              Sort Options              ");
    winclrtoeol(sort_options);
    wmove(sort_options,2,0);
    winclrtoeol(sort_options);
    wmove(sort_options,7,0);
    winclrtoeol(sort_options);
    wmove(sort_options,8,0);
    winclrtoeol(sort_options);
    wmove(sort_options,9,0);
    winclrtoeol(sort_options);

    while(!fertig)
    {
	if (active == 0)
	    set_color_pair(sort_options,INV_OPTSLABELS);
	else
	    set_color_pair(sort_options,OPTSLABELS);
	mvwaddstr(sort_options,4,1," Sort archives by:");
	
	if (active == 1)
	    set_color_pair(sort_options,INV_OPTSLABELS);
	else
	    set_color_pair(sort_options,OPTSLABELS);
	mvwaddstr(sort_options,5,1," Sort order      :");
	
	if (active == 2)
	    set_color_pair(sort_options,INV_OPTSFRAME);
	else
	    set_color_pair(sort_options,OPTSFRAME);
	mvwaddstr(sort_options,8,2,"   OK   ");
	
	if (active == 3)
	    set_color_pair(sort_options,INV_OPTSFRAME);
	else
	    set_color_pair(sort_options,OPTSFRAME);
	mvwaddstr(sort_options,8,30," Cancel ");

	set_color_pair(sort_options,OPTSFIELDS);
	mvwaddstr(sort_options,4,20,"                  ");
	mvwaddstr(sort_options,4,20,sortopts[opt]);
	
	if (order)
	    mvwaddstr(sort_options,5,20,"DESCENDING        ");
	else
	    mvwaddstr(sort_options,5,20,"ASCENDING         ");
	
	touchwin(sort_options);
	wrefresh(sort_options);

	c = getch();
	if (c == TAB_KEY || c == KEY_RIGHT)
	    if (active == 3)
		active = 0;
	    else
		active++;
	else if (c == KEY_LEFT)
	    if (active == 0)
		active = 3;
	    else
		active--;
	else if (c == KEY_DOWN)
	{
	    if (active == 0)
		if (opt == 2)
		    opt = 0;
		else
		    opt++;
	    else if (active == 1)
		order = 1 - order;
	}
	else if (c == KEY_UP)
	{
	    if (active == 0)
		if (opt == 0)
		    opt = 2;
		else
		    opt--;
	    else if (active == 1)
		order = 1 - order;
	}
	else if (c == CR_KEY)
	    if (active == 3)       /* Cancel */
		fertig = 1;
	    else if (active == 2)  /* OK */
	    {
		fertig = 1;
		sort_type = opt;
		sort_ascdesc = order;
		init_list();
	    }
    }	
    delwin(sort_options);
    destroy_options_popup();
    
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     handle_other_opts                                           */
/* Purpose:  manage other options window                                 */
/*************************************************************************/
void handle_other_opts()
{
    int fertig = 0,active = 0,c,wt;

    wt = watch_tape;
        
    other_options = newwin(9,40,(LINES-3-9)/2+2,(COLS-40)/2);
    leaveok(other_options,TRUE);
    keypad(other_options,TRUE);    
    set_color_pair(other_options,OPTSLABELS);
    winclr(other_options);
    
    set_color_pair(other_options,OPTSFRAME);
    wmove(other_options,0,0);
    winclrtoeol(other_options);
    mvwaddstr(other_options,1,0,"              Other Options             ");
    winclrtoeol(other_options);
    wmove(other_options,2,0);
    winclrtoeol(other_options);
    wmove(other_options,6,0);
    winclrtoeol(other_options);
    wmove(other_options,7,0);
    winclrtoeol(other_options);
    wmove(other_options,8,0);
    winclrtoeol(other_options);

    while(!fertig)
    {
	if (active == 0)
	    set_color_pair(other_options,INV_OPTSLABELS);
	else
	    set_color_pair(other_options,OPTSLABELS);
	mvwaddstr(other_options,4,1," Watch tape      :");
	
	if (active == 1)
	    set_color_pair(other_options,INV_OPTSFRAME);
	else
	    set_color_pair(other_options,OPTSFRAME);
	mvwaddstr(other_options,7,2,"   OK   ");
	
	if (active == 2)
	    set_color_pair(other_options,INV_OPTSFRAME);
	else
	    set_color_pair(other_options,OPTSFRAME);
	mvwaddstr(other_options,7,30," Cancel ");

	set_color_pair(other_options,OPTSFIELDS);
	if (wt)
	    mvwaddstr(other_options,4,20,"YES               ");
 	else
	    mvwaddstr(other_options,4,20,"NO                ");
	
	touchwin(other_options);
	wrefresh(other_options);

	c = getch();
	if (c == TAB_KEY || c == KEY_RIGHT)
	    if (active == 2)
		active = 0;
	    else
		active++;
	else if (c == KEY_LEFT)
	    if (active == 0)
		active = 2;
	    else
		active--;
	else if (c == KEY_DOWN)
	{
	    if (active == 0)
		wt = 1 - wt;
	}
	else if (c == KEY_UP)
	{
	    if (active == 0)
		wt = 1 - wt;
	}
	else if (c == CR_KEY)
	    if (active == 2)       /* Cancel */
		fertig = 1;
	    else if (active == 1)  /* OK */
	    {
		fertig = 1;
		watch_tape = wt;
	    }
    }	
    delwin(other_options);
    destroy_options_popup();
    
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/	      
