/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  list.c                                                       */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include <ntape.h>

void insert_list(char *,long,long,long);
void show_list(void);
void highlight_list_element(int,int);
void clear_archive_list(void);

WINDOW *infobox;
static int no_listitems;
static list_el *archive_head = NULL,*archive_tail = NULL;
static int first_item;
static int focus_item;


/*************************************************************************/
/* Name:     clear_archive_list                                          */
/* Purpose:  Clears list.                                                */
/*************************************************************************/
void clear_archive_list()
{
    list_el *s,*t;

    s = archive_head;
    while (s != NULL)
    {
	t = s->next;
	free(s);
	s = t;
    }
    archive_head = NULL;
  
    werase(listbox);
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     init_list                                                   */
/* Purpose:  Init archive list, read archive database file.              */
/*************************************************************************/
void init_list()
{
    FILE *fp;
    file_entry archive_entry;
    
    first_item = 0;
    focus_item = 0;
    clear_archive_list();

    no_listitems = 0;
    if ((fp = fopen(archive_db,"r")) == NULL)
    {
	print_footer("Error opening database!");
	nta_exit(-1);
    }

    if (!check_database_version(ARCHIVE_DATABASE,fp,NULL))
	nta_exit(-1);
    
    while (get_archive_entry(fp,&archive_entry))
    {
	if (!(archive_entry.add_info & DELETED) && 
	    (!tape->id || (archive_entry.tapeno == tape->id)))
	    insert_list(archive_entry.name, archive_entry.tapeno,
			archive_entry.size, archive_entry.position);
    }
    fclose(fp);

    show_list();
    if (no_listitems)
    {
	highlight_list_element(-1,0);
	print_statusline(focus_item+1,no_listitems);
    }
    else
	print_statusline(0,0);
}


/*************************************************************************/
/* Name:     insert_list                                                 */
/* Purpose:  Inserts element in alphabetical order.                      */
/*************************************************************************/
void insert_list(char *name,long tapeno,long size,long pos)
{
    list_el *element,*laufelement;

    if ((element = (list_el *)malloc(sizeof(list_el))) == NULL)
    {
	print_footer("Not enough memory. Sorry...");
	nta_exit(-1);
    }
    strcpy(element->name,name);
    element->tapeno = tapeno;
    element->size = size;
    element->position = pos;
  
    laufelement = archive_head;
    while (laufelement != NULL)      
    {
	if (sort_type == 0) /* Name */
	{
	    if (sort_ascdesc)
	    {
		if (strcmp(laufelement->name,element->name) < 0)
		    break;
	    }
	    else
	    {
		if (strcmp(laufelement->name,element->name) > 0)
		    break;
	    }
	}
	else if (sort_type == 1) /* size */
	{
            if (sort_ascdesc)
            {
		if (laufelement->size < size)
		    break;
	    }
	    else
	    {
		if (laufelement->size > size)
		    break;
	    }
	}
	else if (sort_type == 2) /* position */
	{
	    if (sort_ascdesc)
	    {
		if (laufelement->position < pos)
		    break;
	    }
	    else
	    {
		if (laufelement->position > pos)
		    break;
	    }
	}
	laufelement = laufelement->next;
    }

    if (laufelement == NULL)
    {
	if (archive_head == NULL)
	{
	    element->next = element->prev = NULL;
	    archive_head = archive_tail = element;
	}
	else
	{
	    archive_tail->next = element;
	    element->prev = archive_tail;
	    element->next = NULL;
	    archive_tail = element;
	}
    }
    else
    {
	if (laufelement != archive_head)
	    laufelement->prev->next = element;
	else
	    archive_head = element;
	element->prev = laufelement->prev;
	laufelement->prev = element;
	element->next = laufelement;
    }
    no_listitems++;
}

/*************************************************************************/
/* Name:     show_list                                                   */
/* Purpose:  Shows list in the archive list box.                         */
/*************************************************************************/
void show_list()
{
    list_el *laufelement;

    int i;
    char line[MAXLEN];

    laufelement = archive_head;
    if (archive_head != NULL)
    {
        for (i = 0; i < first_item; i++)
	laufelement = laufelement->next;
    }

    for (i = 0; i < LINES - 3; i++)
    {
	if (laufelement != NULL)
	{
	    sprintf(line," %-40s",laufelement->name);
	    mvwaddstr(listbox,i,0,line);
	    winclrtoeol(listbox);
	    sprintf(line," %9ld  %07ld  %s",
		    laufelement->size,laufelement->position,
		    get_tapename_from_id(laufelement->tapeno,(long *)NULL));
	    mvwaddstr(listbox,i,COLS-(21+TAPENAMELENGTH),line);
	    laufelement = laufelement->next;
	}
	winclrtoeol(listbox);
    }
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     hilight_list_element                                        */
/* Purpose:  Inverts new and old element.                                */
/*************************************************************************/
void highlight_list_element(int old_element,int new_element)
{
    char line[MAXLEN];
    int i;

    set_color_pair(listbox,INV_ARCHIVELIST);

    if (new_element >= 0)
    {
	for (i = 0; i < COLS; i++)
	{
	    line[i] = mvwinch(listbox,new_element - first_item,i);
	}
	line[i] = '\0';
	mvwaddstr(listbox,new_element - first_item,0,line);
    }

    set_color_pair(listbox,ARCHIVELIST);

    if (old_element >= 0)
    {
	for (i = 0; i < COLS; i++)
	{
	    line[i] = mvwinch(listbox,old_element - first_item,i);
	}
	line[i]= '\0';


	mvwaddstr(listbox,old_element - first_item,0,line);
    }
    touchwin(listbox);
    wrefresh(listbox);
}

/*************************************************************************/
/* Name:     top_pos_list                                                */
/* Purpose:  Top element of archive list                                 */
/*************************************************************************/
void top_pos_list()
{
    top_pos(&focus_item,&first_item,LINES - 3,no_listitems,
	    &highlight_list_element,&show_list);
    print_statusline(focus_item + 1,no_listitems);
}

/*************************************************************************/
/* Name:     bottom_pos_list                                             */
/* Purpose:  Last element of archive list                                */
/*************************************************************************/
void bottom_pos_list()
{
    bottom_pos(&focus_item,&first_item,LINES - 3,no_listitems,
	    &highlight_list_element,&show_list);
    print_statusline(focus_item + 1,no_listitems);
}

/*************************************************************************/
/* Name:     scroll_down_list                                            */
/* Purpose:  Scrolls down one element                                    */
/*************************************************************************/
void scroll_down_list()
{
    scroll_down(&focus_item,&first_item,LINES - 3,no_listitems,
		&highlight_list_element,&show_list);
    print_statusline(focus_item + 1,no_listitems);
}

/*************************************************************************/
/* Name:     scroll_up_list                                              */
/* Purpose:  Scrolls up one element                                      */
/*************************************************************************/
void scroll_up_list()
{
    scroll_up(&focus_item,&first_item,LINES - 3,no_listitems,
	      &highlight_list_element,&show_list);
    print_statusline(focus_item + 1,no_listitems);
}

/*************************************************************************/
/* Name:     page_down_list                                              */
/* Purpose:  Pages down                                                  */
/*************************************************************************/
void page_down_list()
{
    page_down(&focus_item,&first_item,LINES - 3,no_listitems,
	      &highlight_list_element,&show_list);
    print_statusline(focus_item + 1, no_listitems);
}

/*************************************************************************/
/* Name:     page_up_list                                                */
/* Purpose:  Pages up                                                    */
/*************************************************************************/
void page_up_list()
{
    page_up(&focus_item,&first_item,LINES - 3,no_listitems,
	    &highlight_list_element,&show_list);
    print_statusline(focus_item + 1, no_listitems);
}


/*************************************************************************/
/* Name:     info_box                                                    */
/* Purpose:  Pops up an info box about the selected archive              */
/*************************************************************************/
void info_box()
{
    list_el *le;
    int i = 0,j,c,x = 0,y = 0, 
	pf = 0,             /* which platform */
	cat = 0,            /* which category */
	fertig = 0,         /* if you've pressed CANCEL */
	changed = 0,        /* if you've edited sth. */
	active = 0,         /* the active field/button */
	edit = 0,           /* if you've pressed Edit Info */
	day,month,year;     /* temporary variables for date */
    FILE *fp;
    char number[LONGLENGTH];
    file_entry archive_entry;

    if (no_listitems == 0)
	return;
    le = archive_head;
    while (i < focus_item)
    {
	if (le == NULL)
	{
	    print_footer("Sorry! Internal error 102! Read the fucking manual!");
	    return;
	}
	le = le->next;
	i++;
    }
  
    if ((fp = fopen(archive_db,"r")) == NULL)
    {
	print_footer("Error opening database!");
	nta_exit(-1);
    }

    if (!check_database_version(ARCHIVE_DATABASE,fp,NULL))
	nta_exit(-1);

    archive->size = -1;

    while (get_archive_entry(fp,&archive_entry))
    {
	if ((!strcmp(le->name,archive_entry.name)) && 
	    (le->tapeno == archive_entry.tapeno) &&
	    (le->position == archive_entry.position))
	{
	    strcpy(archive->name,archive_entry.name);
	    archive->tapeno = archive_entry.tapeno;
	    archive->size = archive_entry.size;
	    archive->position = archive_entry.position;
	    archive->add_info = archive_entry.add_info;
	    strcpy(archive->date,archive_entry.date);
	    strcpy(archive->description,archive_entry.description);
	    for (i = 0; i < no_platforms; i++)
		if (!strcmp(platform[i],archive_entry.pf))
		    pf = i;
	    for (i = 0; i < no_categories; i++)
		if (!strcmp(category[i],archive_entry.category))
		    cat = i;
	    break;
	}
    }
    fclose(fp);

    /* Shouldn't happen, but i'm a little paranoid */
    if (archive->size < 0)
    {
	print_footer("Selected archive not found in database!");
	return;
    }
  
#ifdef DEBUG
    fprintf(stderr,"%s\t%ld\t%ld\t%ld\t%u\n%s\t%s\n%s\n%s\n",archive->name,
	    archive->tapeno,archive->size,archive->position,
	    archive->add_info,archive->date,platform[pf],
	    category[cat],archive->description);
#endif

    infobox = newwin(22,60,(LINES - 21)/2,(COLS-60)/2);
    leaveok(infobox,TRUE);
    keypad(infobox,TRUE);

    set_color_pair(infobox,INFOLABELS);
    winclr(infobox);

    set_color_pair(infobox,INFOFRAME);
    wmove(infobox,0,0);
    winclrtoeol(infobox);
    mvwaddstr(infobox,1,0,"                     Archive Information");
    winclrtoeol(infobox);
    wmove(infobox,2,0);
    winclrtoeol(infobox);
    wmove(infobox,19,0);
    winclrtoeol(infobox);
    wmove(infobox,20,0);
    winclrtoeol(infobox);
    wmove(infobox,21,0);
    winclrtoeol(infobox);
  
    box(infobox,ACS_VLINE,ACS_HLINE);
  
    set_color_pair(infobox,INFOLABELS);
    mvwaddstr(infobox,5,2,"Tape:");
    mvwaddstr(infobox,7,2,"Size:");
    mvwaddstr(infobox,8,2,"Position:");
    mvwaddstr(infobox,9,2,"Archive date:");
    mvwaddstr(infobox,7,36,"Contents saved:");
  
    while (!fertig)
    {
	if (active == 1)
	    set_color_pair(infobox,INV_INFOLABELS);
	else
	    set_color_pair(infobox,INFOLABELS);
	mvwaddstr(infobox,4,2,"Archive name:");
	if (active == 2)
	    set_color_pair(infobox,INV_INFOLABELS);
	else
	    set_color_pair(infobox,INFOLABELS);
	mvwaddstr(infobox,10,2,"Platform:   ");
        if (active == 3)
            set_color_pair(infobox,INV_INFOLABELS);
        else
            set_color_pair(infobox,INFOLABELS);
        mvwaddstr(infobox,11,2,"Category:   ");

	if (active == 5)
	    set_color_pair(infobox,INV_INFOFRAME);
	else
	    set_color_pair(infobox,INFOFRAME);
	if (tape->id)
	{
	    if (edit)
		mvwaddstr(infobox,20,2,"  Edit OK  ");
	    else
		mvwaddstr(infobox,20,2," Edit Info ");
	}
	if (active == 6)
	    set_color_pair(infobox,INV_INFOFRAME);
	else
	    set_color_pair(infobox,INFOFRAME);
	if (tape->id)
	    mvwaddstr(infobox,20,15," Extract all ");
	if (active == 7)
	    set_color_pair(infobox,INV_INFOFRAME);
	else
	    set_color_pair(infobox,INFOFRAME);
	if (tape->id)
	    mvwaddstr(infobox,20,30," Extract selected ");
	if (active == 0)
	    set_color_pair(infobox,INV_INFOFRAME);
	else
	    set_color_pair(infobox,INFOFRAME);
	if (tape->id)
	    mvwaddstr(infobox,20,50," Cancel ");
	else
	    mvwaddstr(infobox,20,27," Cancel ");
      
 	if (active == 4)
	    set_color_pair(infobox,INV_INFOLABELS);
	else
	    set_color_pair(infobox,INFOLABELS);
	mvwaddstr(infobox,13,2,"Note:       ");

	set_color_pair(infobox,INFOFIELDS);
	for (i = 4; i < 18; i++)
	    if (i < 6 || i > 12)
		mvwaddstr(infobox,i,16,"                                        ");
	    else if (i == 10 || i == 11)
		mvwaddstr(infobox,i,16,"                    ");
	    else if (i != 6 && i != 12)
		mvwaddstr(infobox,i,16,"            ");
      
	mvwaddstr(infobox,4,16,archive_entry.name);
	if ((active == 1) && (strlen(archive_entry.name) < ARCHIVENAMELENGTH-1))
	    waddch(infobox,'_');
	mvwaddstr(infobox,5,16,get_tapename_from_id(archive->tapeno,(long *)NULL));
	sprintf(number,"%ld",archive_entry.size);
	mvwaddstr(infobox,7,16,number);
	sprintf(number,"%ld",archive_entry.position);
	mvwaddstr(infobox,8,16,number);
	sscanf(archive->date,"%d %d %d",&day,&month,&year);
	sprintf(number,"%02d.%02d.%04d",day,month,year);
	mvwaddstr(infobox,9,16,number);
	mvwaddstr(infobox,10,16,platform[pf]);
	mvwaddstr(infobox,11,16,category[cat]);
	mvwaddstr(infobox,7,53,
		  archive->add_info & CONTENTS_SAVED ? "Yes" : "No ");
      
	i = 0;
	j = 13;
	wmove(infobox,13,16);
	while (archive_entry.description[i] != '\0')
	{
	    if (archive_entry.description[i++] != '@')
		waddch(infobox,archive_entry.description[i - 1]);
	    else
	    {
		wmove(infobox,++j,16);
		
		if (j == 18)
		    break;
	    }
	}
	getyx(infobox,y,x);
	if (active == 4)
	{
	    if (x < 56)
		waddch(infobox,'_');
	    else
		wmove(infobox,y,x+1);
	}

	touchwin(infobox);
	wrefresh(infobox);

	c = getch();
	if (c == TAB_KEY || c == KEY_RIGHT)
	{
	    if (active == 0)
	    {
		if (tape->id)
		{
		    if (edit)
			active = 1;
		    else
			active = 5;
		}
	    }
	    else if (active == 7)
		active = 0;
	    else
		active++;
	}
	else if (c == KEY_LEFT)
	{
	    if (active == 0)
	    {
		if (tape->id)
		    active = 7;
	    }
	    else if (active == 5)
		if (edit)
		    active = 4;
		else
		    active = 0;
	    else
		active--;
	}
	else if (c == CR_KEY)
	{
	    if (active == 4)
	    {
		getyx(infobox,y,x);
		if (y == 17)
		    continue;
		c ='@';
		goto cr;
	    }
	    if (active == 5)
	    {
		if (edit)
		{
		    change_info(CHANGE_NAME_DESC_PF_CAT,archive_entry.name,
				archive->tapeno,archive->position,platform[pf],
				category[cat],archive_entry.description);
		    changed = 1;
		    edit = 0;
		}
		else
		{
		    edit = 1;
		    active = 1;
		}
	    }
	    if (active == 6)
	    {
		FileDlg(NTA_EXTRACT);
		fertig = 1;
	    }
	    if (active == 7)
	    {
		FileDlg(NTA_EXTRACT_SELECTED);
		fertig = 1;
	    }
	    if (active == 0)
		fertig = 1;
	}
        else if (c == KEY_UP)
        {
            if (active == 2)
            {
                pf--;
                if (pf < 0)
                    pf = no_platforms - 1;
            }
	    if (active == 3)
	    {
		cat--;
		if (cat < 0)
		    cat = no_categories - 1;
	    }
	}
        else if (c == KEY_DOWN)
        {
            if (active == 2)
            {
                pf++;
                if (pf == no_platforms)
                    pf = 0;
            }
	    if (active == 3)
	    {
		cat++;
		if (cat == no_categories)
		    cat = 0;
	    }
	}
 	else if (c == BS_KEY1 || c == BS_KEY2 || c == KEY_BACKSPACE)
	{
	    if (active == 1)
	    {
		i = strlen(archive_entry.name);
		if (i > 0)
		    archive_entry.name[--i] = '\0';
	    }
	    if (active == 4)
	    {
		i = strlen(archive_entry.description);
		if (i > 0)
		    archive_entry.description[--i] = '\0';
	    }
	}
	else if (strchr(VALIDCHARSTR,c))
	{
	    if (active == 1)
	    {
		i = strlen(archive_entry.name);
		if (i < ARCHIVENAMELENGTH-1)
		{
		    archive_entry.name[i] = c;
		    archive_entry.name[i+1] = '\0';
		}
	    }
	    else if (active == 4)
	    {
	cr: i = strlen(archive_entry.description);
		/* getyx(infobox,y,x); */
		if (x < 56 || c == '@')
		{
		    archive_entry.description[i] = c;
		    archive_entry.description[i+1] = '\0';
		}
	    }
	}
    }
    if (changed)
	init_list();
    delwin(infobox);
    touchwin(tapestatus);
    wrefresh(tapestatus);
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     search_list                                                 */
/* Purpose:  Searches a list entry that matches the substring            */
/*************************************************************************/
void search_list(char *substring,int pos)
{
    int i;
    list_el *p;
    char msg[90];
    
    print_footer("");
    if (!strlen(substring))
	return;
    
    for (i = 0; substring[i] != '\0'; i++)
	substring[i] = toupper(substring[i]);
    
    if (archive_head == NULL)
	goto not_found;
    p = archive_head;
    for (i = 0; i <= focus_item; i++)
	p = p->next;

    while (p != NULL)
    {
	if ((strmatch(p->name,substring)) && ((pos < 0) || (pos == p->position))) 
	    break;
	i++;
	p = p->next;
    }
    
    if (p != NULL)
	goto found;
    
    print_footer("Wrapped...");
    p = archive_head;
    for (i = 0; i <= focus_item; i++)
    {
	if ((strmatch(p->name,substring)) && ((pos < 0) || (pos == p-> position)))
	    break;
	p = p->next;
    }
    
    if (i <= focus_item)
	goto found;

not_found:
    sprintf(msg,"No archive found that matches substring '%s'",substring);
    print_footer(msg);
    
    return;
    
found:
    focus_item = i;
    i -= first_item;
    if ( i < 0 || i >= LINES - 3)
    {
	first_item = focus_item - (LINES - 3) / 2;
	if (first_item + LINES - 3 > no_listitems)
	    first_item = no_listitems - LINES + 3;
	if (first_item < 0)
	    first_item = 0;
    }
    werase(listbox);
    show_list();
    highlight_list_element(-1,focus_item);
    print_statusline(focus_item + 1, no_listitems);
}


/*************************************************************************/
/* Name:     comment_focussed_archive                                    */
/* Purpose:  Searches a list entry that matches the substring            */
/*************************************************************************/
void comment_focussed_archive()
{
    list_el *le;
    int i = 0;
    char msg[80];
    
    le = archive_head;
    while (i < focus_item)
    {
        if (le == NULL)
        {
            print_footer("Sorry! Internal error 102! Read the fucking manual!");
            return;
        }
        le = le->next;
        i++;
    }

    sprintf(msg,"Comment archive %s ?",le->name);
    
    if (!MessageBox(msg,2,YESNO,1))
	return;
    
    change_info(CHANGE_ADDINFO_COMMENT,le->name,le->tapeno,le->position,"","","");
    init_list();
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
