
/*
 *	Copyright (c) 1993 by California Institute of Technology.
 *	Written by William Deich.  Not derived from licensed software.

 *    You may distribute under the terms of either the GNU General Public
 *    License or the Artistic License, as specified in the README file.
    
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>

#ifdef USE_SYSLOG
#include <syslog.h>
#endif

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef SUNOS5
#include <sys/time.h>
#include <sys/systeminfo.h>
extern int sysinfo();
#define gethostname(buf, lbuf) (sysinfo(SI_HOSTNAME, (buf), (lbuf)))
#undef NSIG
#define NSIG _sys_nsig
#else
extern int gethostname();
#include <time.h>
#endif

FILE *error_logfile = NULL;
char *error_prog = NULL;
char *error_log_prog = NULL;
char *error_command = NULL;
char *error_user = NULL;

int error_stderr = 1;

int error_syslog = 0;

#ifdef USE_SYSLOG

/* Default error priority */
#ifndef SYSLOG_PRIORITY
#define SYSLOG_PRIORITY LOG_ERR
#endif

/* Default error facility */
#ifndef SYSLOG_FACILITY
#define SYSLOG_FACILITY LOG_USER
#endif

#ifdef RLOG_MACHINE
#define OpenLog(prog, opt, fac) ropenlog((prog), (opt), (fac), RLOG_MACHINE)
#define SysLog(pri, buf) rsyslog((pri), (buf))
#else
#define OpenLog(prog, opt, fac) openlog((prog), (opt), (fac))
#define SysLog(pri, buf) syslog((pri), (buf))
#endif

int error_priority = SYSLOG_PRIORITY;
int error_facility = SYSLOG_FACILITY;
int openlog_done = 0;

#endif

static int uid = -1;
static char user[128] = "";
static char hostname[1024] = "";

#define MAXPRINT 1300

/* Error -- print error message, then optionally die.

 * Usage:	Error(show_perror, die, format, args... );
 *	Print error message according to format & args.
 *	If show_perror != 0 && errno != 0, follow error message with perror("").
 *	(If show_perror !=0, but errno==0, follow error msg with "\n").
 *	If die != 0, exit with exit(die).

 * There are several external variables that a calling program can modify:
 *	error_stderr: controls whether messages go to stderr.  Default enabled.
 *	error_logfile: controls whether messages go to a logfile. Default off.
 *	error_command: controls whether msgs go to a popen'd command. Def off.
 *	error_syslog: controls whether messages go to syslog.  Default: off.
 *	error_user: Controls username used when printing messages.
 * In detail:

 * If error_prog != NULL, then the message to stderr is preceded
 *	with "<error_prog>: ".

 * If error_stderr == 0, then the message is NOT directed to stderr.
 *	By default error_stderr == 1.

 * If error_logfile != NULL, then the message is also directed to
 *	that file, preceded with
 *		error_log_prog: user@hostname timestamp
 * 	If error_log_prog is NULL, it isn't printed.  We keep separate
 *	error_prog and error_log_prog so that the program name can be
 *	printed on stderr (where there might otherwise be confusion to the
 *	user about which program it is) but optionally not printed in the
 *	logfile, which is typically unique to the program, so that the
 *	program name is redundant.

 * If error_user != NULL, then the username used in messages is error_user,
 *	instead of the default name found by looking at the user's password
 *	entry.

 * If error_command != NULL and *error_command != '\0', then error_command
 *	is popen'd and the message is piped in.  If the popen fails, Error()
 *	is silent about the problem.  NOTE: the command is executed separately
 *	for each call to this routine.

 * If compiled with USE_SYSLOG defined, and the caller sets error_syslog != 0,
 *	then the message is passed to syslog(), at priority error_priority,
 *	using facility error_facility.
 *	In this case, the fmt string and the printf output must be less
 *	than MAXPRINT characters each. This is because syslog() accepts a
 *	printf-style variadic argument list, but it doesn't have a va_list
 *	version.  Therefore we print into a string and pass that onto syslog().
 *	As a side effect, you can't use syslog-specific "%m" in the fmt.
 *	If the error_prog string is non-null, then just before the first
 *	call to syslog, openlog is called with an ident string = error_prog.
 *	Note that this is done just once: you can't change error_prog
 *	between messages.
 * If compiled with USE_SYSLOG and RLOG_MACHINE defined, then we use the 
 *	rsyslog() routines (see rsyslog.c) to provide networked syslog
 *	messages.  Otherwise we compile with standard syslog routines.

 * Return code is -1, so you can print error messages and return an error
 *	code with   return Error(...);
 */

#ifdef __STDC__
/* VARARGS3 */
int
Error(
    int show_perror,	/* If errno != 0, follow msg with perror("") */
    int die,		/* If !0, exit with exit(die) */
    char *fmt,		/* Print rest of args with fprintf(stderr, fmt, ...) */
    ... )
{
    va_list ap;
    int error = errno;
    FILE *error_cmd = NULL;

    /* Program name */
    if (error_stderr && error_prog)
	(void) fprintf(stderr, "%s: ", error_prog);

    if (error_command && *error_command)
	error_cmd = popen(error_command, "w");

    if (error_log_prog) {
	if (error_logfile)
	    (void) fprintf(error_logfile, "%s: ", error_log_prog);
	if (error_cmd)
	    (void) fprintf(error_cmd, "%s: ", error_log_prog);
    }

    if (error_logfile || error_syslog || error_command) {
	if (getuid() != uid || *user == '\0') {
	    struct passwd *pw;
	    pw = getpwuid((uid=getuid()));
	    if (pw) (void) strcpy(user, pw->pw_name);
	}
    }

    if (error_logfile || error_cmd) {
	/* user@hostname & timestamp */
	char *s;
	time_t tptr;
	if (error_log_prog) {
	    if (error_logfile)
		fprintf(error_logfile, "%s: ", error_log_prog);
	    if (error_cmd)
		fprintf(error_cmd, "%s: ", error_log_prog);
	}
	if (*hostname == '\0')
	    (void) gethostname(hostname, sizeof(hostname));
	(void) time(&tptr);
	s = ctime(&tptr);
	s[strlen(s) - 1] = '\0';
	if (error_logfile)
	    (void) fprintf(error_logfile, "%s@%s %s\t",
			    error_user ? error_user : user,
			    hostname, s);
	if (error_cmd)
	    (void) fprintf(error_cmd, "%s@%s %s\t",
			    error_user ? error_user : user,
			    hostname, s);
    }

    if (error_stderr) {
	/* User's msg */
	va_start(ap, fmt);
	(void) vfprintf(stderr, fmt, ap);
	va_end(ap);
    }

    if (error_logfile) {
	/* User's msg */
	va_start(ap, fmt);
	(void) vfprintf(error_logfile, fmt, ap);
	va_end(ap);
    }

    if (error_cmd) {
	/* User's msg */
	va_start(ap, fmt);
	(void) vfprintf(error_cmd, fmt, ap);
	va_end(ap);
    }

    if (show_perror) {
	if (error) {
	    errno = error;
	    if (error_stderr)
		perror("");
	    if (error_logfile) {
		(void) fprintf(error_logfile, "Error %d\n", error);
		/* (void) fprintf(error_logfile, "%s\n", strerror(error)); */
	    }
	    if (error_cmd) {
		(void) fprintf(error_cmd, "Error %d\n", error);
		/* (void) fprintf(error_cmd, "%s\n", strerror(error)); */
	    }
	} else {
	    if (error_stderr)
		(void) fputc('\n', stderr);
	    if (error_logfile)
		(void) fputc('\n', error_logfile);
	    if (error_cmd)
		(void) fputc('\n', error_cmd);
	}
    }

#ifdef USE_SYSLOG
    if (error_syslog) {
	char newfmt[MAXPRINT], buf[MAXPRINT];
	if (!openlog_done && error_prog) {
	    OpenLog(error_prog, 0, error_facility);
	    openlog_done = 1;
	}
	sprintf(newfmt, "(%s) ", error_user ? error_user : user);
	strcat(newfmt, fmt);
	va_start(ap, fmt);
	(void) vsprintf(buf, newfmt, ap);
	va_end(ap);
	SysLog(SYSLOG_PRIORITY, buf);
    }
#endif

    if (die)
	(void) exit(die);

    if (error_cmd)
	pclose(error_cmd);

    return -1;

}
#else

/* VARARGS3 */
int
Error( va_alist )
va_dcl
{
    va_list ap;
    int die, show_perror;
    char *fmt;
    int error = errno;
    FILE *error_cmd = NULL;

    /* Program name */
    if (error_stderr && error_prog)
	(void) fprintf(stderr, "%s: ", error_prog);

    if (error_command && *error_command)
	error_cmd = popen(error_command, "w");

    if (error_log_prog) {
	if (error_logfile)
	    (void) fprintf(error_logfile, "%s: ", error_log_prog);
	if (error_cmd)
	    (void) fprintf(error_cmd, "%s: ", error_log_prog);
    }

    if (error_logfile || error_syslog || error_command) {
	if (getuid() != uid || *user == '\0') {
	    struct passwd *pw;
	    pw = getpwuid((uid=getuid()));
	    if (pw) (void) strcpy(user, pw->pw_name);
	}
    }

    if (error_logfile || error_cmd) {
	/* user@hostname & timestamp */
	char *s;
	time_t tptr;
	if (error_log_prog) {
	    if (error_logfile)
		fprintf(error_logfile, "%s: ", error_log_prog);
	    if (error_cmd)
		fprintf(error_cmd, "%s: ", error_log_prog);
	}
	if (*hostname == '\0')
	    (void) gethostname(hostname, sizeof(hostname));
	(void) time(&tptr);
	s = ctime(&tptr);
	s[strlen(s) - 1] = '\0';
	if (error_logfile)
	    (void) fprintf(error_logfile, "%s@%s %s\t",
				error_user ? error_user : user,
				hostname, s);
	if (error_cmd)
	    (void) fprintf(error_cmd, "%s@%s %s\t",
				error_user ? error_user : user,
				hostname, s);
    }

    if (error_stderr) {
	/* User's msg */
	va_start(ap);
	show_perror = va_arg(ap, int);
	die = va_arg(ap, int);
	fmt = va_arg(ap, char *);
	(void) vfprintf(stderr, fmt, ap);
	va_end(ap);
    }

    if (error_logfile) {
	/* User's msg */
	va_start(ap);
	show_perror = va_arg(ap, int);
	die = va_arg(ap, int);
	fmt = va_arg(ap, char *);
	(void) vfprintf(error_logfile, fmt, ap);
	va_end(ap);
    }

    if (error_cmd) {
	/* User's msg */
	va_start(ap);
	show_perror = va_arg(ap, int);
	die = va_arg(ap, int);
	fmt = va_arg(ap, char *);
	(void) vfprintf(error_cmd, fmt, ap);
	va_end(ap);
    }

    /* User's msg */
    va_start(ap);
    show_perror = va_arg(ap, int);
    die = va_arg(ap, int);
    va_end(ap);

    if (show_perror) {
	if (error) {
	    errno = error;
	    if (error_stderr)
		perror("");
	    if (error_logfile)
		(void) fprintf(error_logfile, "Error %d\n", error);
	    if (error_cmd)
		(void) fprintf(error_cmd, "Error %d\n", error);
	} else {
	    if (error_stderr)
		(void) fputc('\n', stderr);
	    if (error_logfile)
		(void) fputc('\n', error_logfile);
	    if (error_cmd)
		(void) fputc('\n', error_cmd);
	}
    }

#ifdef USE_SYSLOG
    if (error_syslog) {
	char newfmt[MAXPRINT], buf[MAXPRINT];
	va_start(ap);
	show_perror = va_arg(ap, int);
	die = va_arg(ap, int);
	fmt = va_arg(ap, char *);
	if (!openlog_done && error_prog) {
	    OpenLog(error_prog, 0, error_facility);
	    openlog_done = 1;
	}
	sprintf(newfmt, "(%s) ", error_user ? error_user : user);
	strcat(newfmt, fmt);
	(void) vsprintf(buf, newfmt, ap);
	va_end(ap);
	SysLog(SYSLOG_PRIORITY, buf);
    }
#endif

    if (die)
	(void) exit(die);

    if (error_cmd)
	pclose(error_cmd);

    return -1;

}
#endif
