/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate variables
 */

#include <ctype.h>

#include <error.h>
#include <symtab.h>
#include <trace.h>
#include <vargram.h>
#include <variable.h>
#include <vargram.gen.h> /* must be last */


static blob_ty		*source;
static char		*lexpos;
static wlist		*reference;
static symtab_ty	*symtab;
static symtab_ty	*special;
static wlist		result;


static void init _((void));

static void
init()
{
	typedef struct table_ty table_ty;
	struct table_ty
	{
		char	*name;
		char	*value;
	};

	static table_ty table[] =
	{
		{ "%",	"",					},
		{ "%D",	"",					},
		{ "%F",	"",					},
		{ "*",	"%",					},
		{ "*D",	"[dirname %]",				},
		{ "*F",	"[entryname %]",			},
		{ "<",	"[resolve [head [need]]]",		},
		{ "<D",	"[dirname [resolve [head [need]]]]",	},
		{ "<F",	"[entryname [resolve [head [need]]]]",	},
		{ "?",	"[resolve [younger]]",			},
		{ "?D",	"[dirname [resolve [younger]]]",	},
		{ "?F",	"[entryname [resolve [younger]]]",	},
		{ "@",	"[target]",				},
		{ "@D",	"[dirname [target]]",			},
		{ "@F",	"[entryname [target]]",			},
		{ "^",	"[resolve [need]]",			},
		{ "^D",	"[dirname [resolve [need]]]",		},
		{ "^F",	"[entryname [resolve [need]]]",		},
		{ "MAKE",	"[self]",			},
		{ "MAKEFLAGS",	"[getenv [upcase [self]]]",	},
		{ "VERSION",	"[version]",			},
		{ "VPATH",	"[search_list]",		},
	};

	table_ty	*tp;
	string_ty	*name;
	string_ty	*value;

	if (symtab)
		return;
	trace(("init()\n{\n"/*}*/));
	symtab = symtab_alloc(SIZEOF(table));
	for (tp = table; tp < ENDOF(table); ++tp)
	{
		name = str_from_c(tp->name);
		value = str_from_c(tp->value);
		symtab_assign(symtab, name, value);
		str_free(name);
	}
	trace((/*{*/"}\n"));
}


string_ty *
vargram_lookup(name)
	string_ty	*name;
{
	string_ty	*data;
	string_ty	*result;

	trace(("vargram_lookup(\"%s\")\n{\n"/*}*/, name->str_text));
	assert(symtab);
	data = 0;
	if (special)
		data = symtab_query(special, name);
	if (!data)
		data = symtab_query(symtab, name);
	if (data)
		result = str_copy(data);
	else
	{
		wl_append_unique(reference, name);
		result = str_format("[%S]", name);
	}
	trace(("return \"%s\";\n", result->str_text));
	trace((/*{*/"}\n"));
	return result;
}


#ifdef DEBUG

static char *unctrl _((int));

static char *
unctrl(c)
	int		c;
{
	static char	buf[5];

	if (c == '\n')
		return "\\n";
	if (c == '^' || c == '\\' || c == '\'')
	{
		buf[0] = '\\';
		buf[1] = c;
		buf[2] = 0;
		return buf;
	}
	if (isprint(c))
	{
		buf[0] = c;
		buf[1] = 0;
		return buf;
	}
	c ^= 0x40;
	if (isprint(c))
	{
		buf[0] = '^';
		buf[1] = c;
		buf[2] = 0;
		return buf;
	}
	c ^= 0x40;
	buf[0] = '\\';
	buf[1] = '0' + ((c >> 6) & 3);
	buf[2] = '0' + ((c >> 3) & 7);
	buf[3] = '0' + (c & 7);
	buf[4] = 0;
	return buf;
}

#endif


int
vargram_lex()
{
	int		c;
	int		len;
	char		buf[5];
	int		token;

	trace(("vargram_lex()\n{\n"/*}*/));
	len = 0;
	c = (unsigned char)*lexpos++;
	trace(("c = '%s';\n", unctrl(c)));
	switch (c)
	{
	case 0:
		--lexpos;
		token = 0;
		goto done;

	case '$':
		token = DOLLAR;
		goto done;

	case ':':
		token = COLON;
		goto done;

	case '=':
		token = EQU;
		goto done;

	case ',':
		token = COMMA;
		goto done;

	case ' ':
	case '\f':
	case '\n':
	case '\t':
#if __STDC__ >= 1
	case '\v':
#endif
		token = SPACE;
		goto done;

	case '(':
		token = LP;
		goto done;

	case ')':
		token = RP;
		goto done;

	case '{':
		token = LB;
		goto done;

	case '}':
		token = RB;
		goto done;

	case '\\':
		c = (unsigned char)*lexpos++;
		trace(("c = '%s';\n", unctrl(c)));
		if (!c)
		{
			--lexpos;
			c = '\\';
		}
		break;
	}

	len = 0;
	switch (c)
	{
	case '\b':
		buf[len++] = '\\';
		buf[len++] = 'b';
		break;

	case '\f':
		buf[len++] = '\\';
		buf[len++] = 'f';
		break;

	case '\n':
		buf[len++] = '\\';
		buf[len++] = 'n';
		break;

	case '\t':
		buf[len++] = '\\';
		buf[len++] = 't';
		break;

#if __STDC__ >= 1
	case '\v':
		buf[len++] = '\\';
		buf[len++] = 'v';
		break;
#endif

	case ' ':
	case ';':
	case ':':
	case '=':
	case '"':
	case '\'':
	case '\\':
	case '{'/*}*/:
	case /*{*/'}':
		buf[len++] = '\\';
		buf[len++] = c;
		break;

	default:
		if (!isprint(c))
		{
			buf[len++] = '\\';
			if (isdigit(*lexpos) || (c & 0300))
				buf[len++] = '0' + ((c >> 6) & 3);
			if (isdigit(*lexpos) || (c & 0370))
				buf[len++] = '0' + ((c >> 3) & 7);
			buf[len++] = '0' + (c & 7);
		}
		else
			buf[len++] = c;
		break;
	}
	vargram_lval.lv_string = str_n_from_c(buf, len);
	token = PLAIN;

	done:
	trace(("return %d;\n", token));
	trace((/*{*/"}\n"));
	return token;
}


void
vargram_error(fmt sva_last)
	char		*fmt;
	sva_last_decl
{
	va_list		ap;
	string_ty	*s;
	int		len;

	sva_init(ap, fmt);
	s = str_vformat(fmt, ap);
	va_end(ap);

	len = s->str_length;
	if (len > 0 && s->str_text[len - 1] == '\n')
		--len;
	blob_error(source, "variable reference %.*S", len, s);
	str_free(s);
}


void
vargram_result(s)
	string_ty	*s;
{
	static string_ty *t1;
	static string_ty *t2;
	string_ty	*tmp;

	trace(("vargram_result(\"%s\")\n{\n"/*}*/, s->str_text));
	if (!s->str_length)
	{
		str_free(s);
		s = str_from_c("\"\"");
	}
	if (!t1)
	{
		t1 = str_from_c("/*");
		t2 = str_from_c("/\\*");
	}
	tmp = str_substitute(t1, t2, s);
	str_free(s);
	wl_append(&result, tmp);
	str_free(tmp);
	trace((/*{*/"}\n"));
}


void
variable_rename(in, out, ref)
	blob_ty		*in;
	blob_list_ty	*out;
	wlist		*ref;
{
	size_t		j;

	trace(("variable_rename(in = %08lX, out = %08lX, ref = %08lX)\n{\n"/*}*/, (long)in, (long)out, (long)ref));
	trace_string(in->text->str_text);
	init();
	source = in;
	lexpos = in->text->str_text;
	reference = ref;
	vargram_parse();
	for (j = 0; j < result.wl_nwords; ++j)
	{
		blob_list_append
		(
			out,
			blob_alloc
			(
				str_copy(result.wl_word[j]),
				in->file_name,
				in->line_number
			)
		);
	}
	wl_free(&result);
	if (special)
	{
		symtab_free(special);
		special = 0;
	}
	trace((/*{*/"}\n"));
}


static void reap _((void *));

static void
reap(p)
	void		*p;
{
	string_ty	*s;

	s = p;
	str_free(s);
}


void
variable_archive(target, member)
	string_ty	*target;
	string_ty	*member;
{
	string_ty	*name;
	string_ty	*value;

	assert(!special);
	special = symtab_alloc(6);
	special->reap = reap;

	name = str_from_c("%");
	value = str_copy(member);
	symtab_assign(special, name, value);
	str_free(name);

	name = str_from_c("%D");
	value = str_format("[dirname %S]", member);
	symtab_assign(special, name, value);
	str_free(name);

	name = str_from_c("%F");
	value = str_format("[entryname %S]", member);
	symtab_assign(special, name, value);
	str_free(name);

	name = str_from_c("@");
	value = str_copy(target);
	symtab_assign(special, name, value);
	str_free(name);

	name = str_from_c("@D");
	value = str_format("[dirname %S]", target);
	symtab_assign(special, name, value);
	str_free(name);

	name = str_from_c("@F");
	value = str_format("[entryname %S]", target);
	symtab_assign(special, name, value);
	str_free(name);
}
