/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to parse variable references
 */

%{

#include <ac/string.h>

#include <trace.h>
#include <vargram.h>
#include <variable.h>

#ifdef	DEBUG
#define YYDEBUG 1
extern int yydebug;
#endif


static string_ty *patvar _((string_ty *, string_ty *, string_ty *));

static string_ty *
patvar(name, from, to)
	string_ty	*name;
	string_ty	*from;
	string_ty	*to;
{
	string_ty	*tmp;
	string_ty	*result;

	if (!strchr(from->str_text, '%'))
	{
		tmp = from;
		from = str_format("%%%S", tmp);
		str_free(tmp);
	}
	if (!strchr(to->str_text, '%'))
	{
		tmp = to;
		to = str_format("%%%S", tmp);
		str_free(tmp);
	}
	tmp = vargram_lookup(name);
	str_free(name);
	result = str_format("[patsubst %S %S %S]", from, to, tmp);
	str_free(tmp);
	str_free(from);
	str_free(to);
	return result;
}

%}

%token	COLON
%token	COMMA
%token	DOLLAR
%token	EQU
%token	LB
%token	LP
%token	PLAIN
%token	RB
%token	RP
%token	SPACE

%union
{
	string_ty	*lv_string;
	wlist		lv_list;
}

%type	<lv_string>	PLAIN var name csl gizzards plain ossl
%type	<lv_string>	namec arg argc
%type	<lv_list>	ssl

%%

strings
	: dbg string
	| strings SPACE string
	;

dbg
	: /* empty */
		{
#if YYDEBUG
			yydebug = trace_pretest_;
#endif
		}
	;

string
	: gizzards
		{ vargram_result($1); }
	| gizzards LP ssl RP
		{
			long		j;

			for (j = 0; j < $3.wl_nwords; ++j)
			{
				vargram_result
				(
					str_format("%S(%S)", $1, $3.wl_word[j])
				);
			}
			str_free($1);
			wl_free(&$3);
		}
	;

gizzards
	: var
		{ $$ = $1; }
	| plain
		{ $$ = $1; }
	| gizzards var
		{
			$$ = str_catenate($1, $2);
			str_free($1);
			str_free($2);
		}
	| gizzards plain
		{
			$$ = str_catenate($1, $2);
			str_free($1);
			str_free($2);
		}
	;

plain
	: PLAIN
		{ $$ = $1; }
	| COLON
		{ $$ = str_from_c("\\:"); }
	| COMMA
		{ $$ = str_from_c(","); }
	| EQU
		{ $$ = str_from_c("\\="); }
	| LB
		{ $$ = str_from_c("\\{"/*}*/); }
	| RB
		{ $$ = str_from_c(/*{*/"\\}"); }
	| RP
		{ $$ = str_from_c(/*(*/")"); }
	;

var
	: DOLLAR DOLLAR
		{ $$ = str_from_c("$"); }
	| DOLLAR PLAIN
		{
			$$ = vargram_lookup($2);
			str_free($2);
		}
	| DOLLAR LP name RP
		{
			$$ = vargram_lookup($3);
			str_free($3);
		}
	| DOLLAR LB name RB
		{
			$$ = vargram_lookup($3);
			str_free($3);
		}
	| DOLLAR LP name COLON name EQU name RP
		{ $$ = patvar($3, $5, $7); }
	| DOLLAR LB name COLON name EQU name RB
		{ $$ = patvar($3, $5, $7); }
	| DOLLAR LP name SPACE csl RP
		{
			$$ = str_format("[%S %S]", $3, $5);
			str_free($3);
			str_free($5);
		}
	| DOLLAR LB name SPACE csl RB
		{
			$$ = str_format("[%S %S]", $3, $5);
			str_free($3);
			str_free($5);
		}
	;

name
	: namec
		{ $$ = $1; }
	| name namec
		{
			$$ = str_catenate($1, $2);
			str_free($1);
			str_free($2);
		}
	;

namec
	: PLAIN
		{ $$ = $1; }
	| COMMA
		{ $$ = str_from_c(","); }
	| var
		{ $$ = $1; }
	;

csl
	: ossl
		{ $$ = $1; }
	| csl comma ossl
		{
			$$ = str_format("%S %S", $1, $3);
			str_free($1);
			str_free($3);
		}
	;

ossl
	: /* empty */
		{ $$ = str_from_c(""); }
	| ssl
		{
			$$ = wl2str(&$1, 0, $1.wl_nwords - 1, (char *)0);
			wl_free(&$1);
		}
	;

ssl
	: arg
		{
			wl_zero(&$$);
			wl_append(&$$, $1);
			str_free($1);
		}
	| ssl SPACE arg
		{
			$$ = $1;
			wl_append(&$$, $3);
			str_free($3);
		}
	;

arg
	: argc
		{ $$ = $1; }
	| arg argc
		{
			$$ = str_catenate($1, $2);
			str_free($1);
			str_free($2);
		}
	;

argc
	: PLAIN
		{ $$ = $1; }
	| EQU
		{ $$ = str_from_c("\\="); }
	| COLON
		{ $$ = str_from_c("\\:"); }
	| var
		{ $$ = $1; }
	;

comma
	: COMMA
	| comma SPACE
	;
