/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate compound statements
 */

#include <mem.h>
#include <stmt/compound.h>

typedef struct stmt_compound_ty stmt_compound_ty;
struct stmt_compound_ty
{
	STMT
	size_t		nlines;
	size_t		nlines_max;
	stmt_ty		**line;
};


static void constructor _((stmt_ty *));

static void
constructor(that)
	stmt_ty		*that;
{
	stmt_compound_ty *this;

	this = (stmt_compound_ty *)that;
	this->nlines = 0;
	this->nlines_max = 0;
	this->line = 0;
}


static void destructor _((stmt_ty *));

static void
destructor(that)
	stmt_ty		*that;
{
	stmt_compound_ty	*this;
	size_t		j;

	this = (stmt_compound_ty *)that;
	for (j = 0; j < this->nlines; ++j)
		stmt_free(this->line[j]);
	if (this->line)
		mem_free(this->line);
	this->nlines = 0;
	this->nlines_max = 0;
	this->line = 0;
}


static void emit _((stmt_ty *));

static void
emit(that)
	stmt_ty		*that;
{
	stmt_compound_ty *this;
	size_t		j;

	this = (stmt_compound_ty *)that;
	if (this->nlines < 1)
		return;
	for (j = 0; j < this->nlines; ++j)
		stmt_emit(this->line[j]);
}


static void regroup _((stmt_ty *));

static void
regroup(that)
	stmt_ty		*that;
{
	stmt_compound_ty *this;
	stmt_ty		**line;
	size_t		nlines;
	size_t		j;

	this = (stmt_compound_ty *)that;
	line = this->line;
	nlines = this->nlines;
	this->line = 0;
	this->nlines = 0;
	this->nlines_max = 0;
	for (j = 0; j < nlines; ++j)
		stmt_regroup(line[j]);

	j = 0;
	while (j < nlines)
	{
		if (line[j]->white_space)
		{
			stmt_ty		*sp;

			sp = stmt_compound_alloc();
			stmt_compound_append(that, sp);
			for (;;)
			{
				stmt_compound_append(sp, line[j]);
				++j;
				if (!line[j - 1]->white_space)
					break;
				if (j >= nlines)
					break;
			}
		}
		else
		{
			stmt_compound_append(that, line[j]);
			++j;
		}
	}
	if (line)
		mem_free(line);
}


static int intersect _((wlist *, wlist *));

static int
intersect(a, b)
	wlist		*a;
	wlist		*b;
{
	size_t		j;

	for (j = 0; j < a->wl_nwords; ++j)
		if (wl_member(b, a->wl_word[j]))
			return 1;
	return 0;
}


static void sort _((stmt_ty *));

static void
sort(that)
	stmt_ty		*that;
{
	stmt_compound_ty *this;
	size_t		j;

	this = (stmt_compound_ty *)that;

	for (j = 0; j < this->nlines; ++j)
		stmt_sort(this->line[j]);

	j = 0;
	while (j < this->nlines)
	{
		stmt_ty		*sp;
		size_t		k, m;

		sp = this->line[j];
		if (!sp->mdef.wl_nwords)
		{
			++j;
			continue;
		}
		for
		(
			k = 0;
			k < j && !intersect(&this->line[k]->ref, &sp->mdef);
			++k
		)
			;
		if (k >= j)
		{
			++j;
			continue;
		}
		for (m = j; m > k; --m)
			this->line[m] = this->line[m - 1];
		this->line[k] = sp;
		j = 0;

		/*
		 * infinite loop possible:
		 * assume this is a valid Makefile
		 */
	}
}


static stmt_method_ty method =
{
	sizeof(stmt_compound_ty),
	"compound",
	constructor,
	destructor,
	emit,
	regroup,
	sort,
};


stmt_ty *
stmt_compound_alloc()
{
	return stmt_alloc(&method);
}


void
stmt_compound_prepend(that, lp)
	stmt_ty		*that;
	stmt_ty		*lp;
{
	stmt_compound_ty *this;
	size_t		j;

	this = (stmt_compound_ty *)that;
	if (this->nlines >= this->nlines_max)
	{
		size_t		nbytes;

		this->nlines_max = this->nlines_max * 2 + 4;
		nbytes = this->nlines_max * sizeof(stmt_ty *);
		this->line = mem_change_size(this->line, nbytes);
	}
	for (j = this->nlines; j > 0; --j)
		this->line[j] = this->line[j - 1];
	this->nlines++;
	this->line[0] = lp;
	stmt_variable_merge(that, lp);
}


void
stmt_compound_append(that, lp)
	stmt_ty		*that;
	stmt_ty		*lp;
{
	stmt_compound_ty *this;

	this = (stmt_compound_ty *)that;
	if (this->nlines >= this->nlines_max)
	{
		size_t		nbytes;

		this->nlines_max = this->nlines_max * 2 + 4;
		nbytes = this->nlines_max * sizeof(stmt_ty *);
		this->line = mem_change_size(this->line, nbytes);
	}
	this->line[this->nlines++] = lp;
	stmt_variable_merge(that, lp);
}
