/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate blobs
 */

#include <blob.h>
#include <emit.h>
#include <error.h>
#include <mem.h>
#include <s-v-arg.h>

static blob_efunc notify;


blob_ty *
blob_alloc(text, file_name, line_number)
	string_ty	*text;
	string_ty	*file_name;
	long		line_number;
{
	blob_ty		*this;

	this = mem_alloc(sizeof(blob_ty));
	this->reference_count = 1;
	this->text = text;
	this->file_name = str_copy(file_name);
	this->line_number = line_number;
	return this;
}


void
blob_free(this)
	blob_ty		*this;
{
	this->reference_count--;
	if (this->reference_count > 0)
		return;
	assert(this->reference_count == 0);
	str_free(this->text);
	str_free(this->file_name);
	mem_free(this);
}


blob_ty *
blob_copy(this)
	blob_ty		*this;
{
	this->reference_count++;
	return this;
}


void
blob_error_notify(func)
	blob_efunc	func;
{
	notify = func;
}


void
blob_error(bp, fmt sva_last)
	blob_ty		*bp;
	char		*fmt;
	sva_last_decl
{
	string_ty	*buffer;
	va_list		ap;

	sva_init(ap, fmt);
	buffer = str_vformat(fmt, ap);
	va_end(ap);
	error("%S: %ld: %S", bp->file_name, bp->line_number, buffer);
	str_free(buffer);
	if (notify)
		notify();
}


void
blob_warning(bp, fmt sva_last)
	blob_ty		*bp;
	char		*fmt;
	sva_last_decl
{
	string_ty	*buffer;
	va_list		ap;

	sva_init(ap, fmt);
	buffer = str_vformat(fmt, ap);
	va_end(ap);
	error("%S: %ld: warning: %S", bp->file_name, bp->line_number, buffer);
	str_free(buffer);
}


blob_list_ty *
blob_list_alloc()
{
	blob_list_ty	*lllp;

	lllp = mem_alloc(sizeof(blob_list_ty));
	lllp->length = 0;
	lllp->maximum = 0;
	lllp->list = 0;
	return lllp;
}


void
blob_list_free(lllp)
	blob_list_ty	*lllp;
{
	long		j;

	for (j = 0; j < lllp->length; ++j)
		blob_free(lllp->list[j]);
	if (lllp->list)
		mem_free(lllp->list);
	mem_free(lllp);
}


void
blob_list_append(lllp, llp)
	blob_list_ty	*lllp;
	blob_ty		*llp;
{
	if (lllp->length >= lllp->maximum)
	{
		size_t		nbytes;

		lllp->maximum = lllp->maximum * 2 + 8;
		nbytes = lllp->maximum * sizeof(blob_ty *);
		lllp->list = mem_change_size(lllp->list, nbytes);
	}
	lllp->list[lllp->length++] = llp;
}


void
blob_list_prepend(lllp, llp)
	blob_list_ty	*lllp;
	blob_ty		*llp;
{
	size_t		j;

	if (lllp->length >= lllp->maximum)
	{
		size_t		nbytes;

		lllp->maximum = lllp->maximum * 2 + 8;
		nbytes = lllp->maximum * sizeof(blob_ty *);
		lllp->list = mem_change_size(lllp->list, nbytes);
	}
	for (j = lllp->length; j > 0; --j)
		lllp->list[j] = lllp->list[j - 1];
	lllp->length++;
	lllp->list[0] = llp;
}


void
blob_emit(bp)
	blob_ty		*bp;
{
	emit_line_number(bp->line_number, bp->file_name);
	emit_string(bp->text);
}


void
blob_list_delete(blp, bp)
	blob_list_ty	*blp;
	blob_ty		*bp;
{
	size_t		j, k;

	for (j = 0; j < blp->length; ++j)
		if (blp->list[j] == bp)
			break;
	if (j >= blp->length)
		return;
	blob_free(blp->list[j]);
	for (k = j + 1; k < blp->length; ++k)
		blp->list[k - 1] = blp->list[k];
	blp->length--;
}
