/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: operating system start point, and command line argument parsing
 */

#include <ac/stddef.h>
#include <ac/string.h>
#include <stdio.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <builtin.h>
#include <cook.h>
#include <env.h>
#include <error.h>
#include <help.h>
#include <id.h>
#include <lex.h>
#include <listing.h>
#include <main.h>
#include <option.h>
#include <parse.h>
#include <trace.h>
#include <version.h>


enum
{
	arglex_token_action,
	arglex_token_no_action,
	arglex_token_book,
	arglex_token_no_book,
	arglex_token_errok,
	arglex_token_no_errok,
	arglex_token_fingerprint,
	arglex_token_no_fingerprint,
	arglex_token_force,
	arglex_token_no_force,
	arglex_token_include,
	arglex_token_log,
	arglex_token_no_log,
	arglex_token_metering,
	arglex_token_no_metering,
	arglex_token_persevere,
	arglex_token_no_persevere,
	arglex_token_precious,
	arglex_token_no_precious,
	arglex_token_silent,
	arglex_token_no_silent,
	arglex_token_star,
	arglex_token_no_star,
	arglex_token_strip_dot,
	arglex_token_no_strip_dot,
	arglex_token_touch,
	arglex_token_no_touch,
	arglex_token_trace,
	arglex_token_no_trace,
	arglex_token_tty,
	arglex_token_no_tty,
	arglex_token_update,
	arglex_token_no_update
};

static arglex_table_ty argtab[] =
{
	{ "-Action",		(arglex_token_ty)arglex_token_action,	},
	{ "-No_Action",		(arglex_token_ty)arglex_token_no_action, },
	{ "-Book",		(arglex_token_ty)arglex_token_book,	},
	{ "-No_Book",		(arglex_token_ty)arglex_token_no_book,	},
	{ "-Continue",		(arglex_token_ty)arglex_token_persevere, },
	{ "-No_Continue",	(arglex_token_ty)arglex_token_no_persevere, },
	{ "-Errok",		(arglex_token_ty)arglex_token_errok,	},
	{ "-No_Errok",		(arglex_token_ty)arglex_token_no_errok,	},
	{ "-FingerPrint",	(arglex_token_ty)arglex_token_fingerprint, },
	{ "-No_FingerPrint",	(arglex_token_ty)arglex_token_no_fingerprint, },
	{ "-Forced",		(arglex_token_ty)arglex_token_force,	},
	{ "-No_Forced",		(arglex_token_ty)arglex_token_no_force,	},
	{ "-Include",		(arglex_token_ty)arglex_token_include,	},
	{ "-\\I*",		(arglex_token_ty)arglex_token_include,	},
	{ "-LOg",		(arglex_token_ty)arglex_token_log,	},
	{ "-List",		(arglex_token_ty)arglex_token_log,	},
	{ "-No_LOg",		(arglex_token_ty)arglex_token_no_log,	},
	{ "-No_List",		(arglex_token_ty)arglex_token_no_log,	},
	{ "-Meter",		(arglex_token_ty)arglex_token_metering,	},
	{ "-No_Meter",		(arglex_token_ty)arglex_token_no_metering, },
	{ "-Precious",		(arglex_token_ty)arglex_token_precious,	},
	{ "-No_Precious",	(arglex_token_ty)arglex_token_no_precious, },
	{ "-Silent",		(arglex_token_ty)arglex_token_silent,	},
	{ "-No_Silent",		(arglex_token_ty)arglex_token_no_silent, },
	{ "-STar",		(arglex_token_ty)arglex_token_star,	},
	{ "-No_STar",		(arglex_token_ty)arglex_token_no_star,	},
	{ "-Strip_Dot",		(arglex_token_ty)arglex_token_strip_dot, },
	{ "-No_Strip_Dot",	(arglex_token_ty)arglex_token_no_strip_dot, },
	{ "-TErminal",		(arglex_token_ty)arglex_token_tty,	},
	{ "-No_TErminal",	(arglex_token_ty)arglex_token_no_tty,	},
	{ "-Touch",		(arglex_token_ty)arglex_token_touch,	},
	{ "-No_Touch",		(arglex_token_ty)arglex_token_no_touch,	},
	{ "-TRace",		(arglex_token_ty)arglex_token_trace,	},
	{ "-No_TRace",		(arglex_token_ty)arglex_token_no_trace,	},
	{ "-Update",		(arglex_token_ty)arglex_token_update,	},
	{ "-No_Update",		(arglex_token_ty)arglex_token_no_update, },
	{ 0, (arglex_token_ty)0, }, /* end marker */
};


/*
 *  NAME
 *	usage - options diagnostic
 *
 *  SYNOPSIS
 *	void usage(void);
 *
 *  DESCRIPTION
 *	Usage is called when the user has made a syntactic or semantic error
 *	on the command line.
 *
 *  CAVEAT
 *	This function does NOT return.
 */

static void usage _((void));

static void
usage()
{
	fprintf(stderr, "usage: %s [ <option>... ][ <filename>... ]\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERSion\n", progname);
	quit(1);
}


/*
 * NAME
 *	help - give some
 *
 * SYNOPSIS
 *	void help(void);
 *
 * DESCRIPTION
 *	Help is used to provide the use with some assistance.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Assumes os_get_search_rules has been called already.
 */

static void cook_help _((void));

static void
cook_help()
{
	static char *text[] =
	{
#include <../man1/cook.h>
	};

	help(text, SIZEOF(text), usage);
}


/*
 * NAME
 *	argparse - parse command line
 *
 * SYNOPSIS
 *	void argparse(option_level_ty);
 *
 * DESCRIPTION
 *	The argparse function is used to parse command lines.
 *
 * RETURNS
 *	void
 */

static void argparse _((option_level_ty));

static void
argparse(level)
	option_level_ty	level;
{
	option_number_ty	type = -1;

	switch (arglex())
	{
	case arglex_token_help:
		if (level != OPTION_LEVEL_COMMAND_LINE)
		{
			fatal
			(
				"may not use %s in environment variable",
				arglex_value.alv_string
			);
		}
		cook_help();
		quit(0);
	
	case arglex_token_version:
		if (level != OPTION_LEVEL_COMMAND_LINE)
		{
			fatal
			(
				"may not use %s in environment variable",
				arglex_value.alv_string
			);
		}
		version();
		quit(0);

	default:
		break;
	}
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			error
			(
				"misplaced \"%s\" command line argument",
				arglex_value.alv_string
			);
			usage();

#ifdef DEBUG
		case arglex_token_tracing:
			if (arglex() != arglex_token_string)
				fatal("-TRACIng requires one or more string arguments");
			for (;;)
			{
				trace_enable(arglex_value.alv_string);
				if (arglex() != arglex_token_string)
					break;
			}
			continue;
#endif

		case arglex_token_include:
			{
				string_ty *s;

				if (arglex() != arglex_token_string)
					fatal("-Include requires a string argument");
				s = str_from_c(arglex_value.alv_string);
				wl_append_unique(&option.o_search_path, s);
				str_free(s);
			}
			break;

		case arglex_token_trace:
			type = OPTION_TRACE;
			normal_on:
			if (option_already(type, level))
			{
				too_many:
				fatal
				(
					"too many \"%s\" options",
					arglex_value.alv_string
				);
			}
			option_set(type, level, 1);
			break;

		case arglex_token_no_trace:
			type = OPTION_TRACE;
			normal_off:
			if (option_already(type, level))
				goto too_many;
			option_set(type, level, 0);
			break;

		case arglex_token_tty:
			type = OPTION_TERMINAL;
			goto normal_on;

		case arglex_token_no_tty:
			type = OPTION_TERMINAL;
			goto normal_off;

		case arglex_token_precious:
			type = OPTION_PRECIOUS;
			goto normal_on;

		case arglex_token_no_precious:
			type = OPTION_PRECIOUS;
			goto normal_off;

		case arglex_token_log:
			if (option_already(OPTION_LOGGING, level))
				goto too_many;
			option_set(OPTION_LOGGING, level, 1);
			if (arglex() != arglex_token_string)
				continue;
			if (option.o_logfile)
				str_free(option.o_logfile);
			option.o_logfile = str_from_c(arglex_value.alv_string);
			break;

		case arglex_token_no_log:
			type = OPTION_LOGGING;
			goto normal_off;

		case arglex_token_book:
			if (option_already(OPTION_BOOK, level))
				goto too_many;
			option_set(OPTION_BOOK, level, 1);
			if (arglex() != arglex_token_string)
				continue;
			if (option.o_book)
				str_free(option.o_book);
			option.o_book = str_from_c(arglex_value.alv_string);
			break;

		case arglex_token_no_book:
			type = OPTION_BOOK;
			goto normal_off;

		case arglex_token_silent:
			type = OPTION_SILENT;
			goto normal_on;

		case arglex_token_no_silent:
			type = OPTION_SILENT;
			goto normal_off;

		case arglex_token_metering:
			type = OPTION_METER;
			goto normal_on;

		case arglex_token_no_metering:
			type = OPTION_METER;
			goto normal_off;

		case arglex_token_touch:
			type = OPTION_TOUCH;
			goto normal_on;

		case arglex_token_no_touch:
			type = OPTION_TOUCH;
			goto normal_off;

		case arglex_token_action:
			type = OPTION_ACTION;
			goto normal_on;

		case arglex_token_no_action:
			type = OPTION_ACTION;
			goto normal_off;

		case arglex_token_persevere:
			type = OPTION_PERSEVERE;
			goto normal_on;

		case arglex_token_no_persevere:
			type = OPTION_PERSEVERE;
			goto normal_off;

		case arglex_token_errok:
			type = OPTION_ERROK;
			goto normal_on;

		case arglex_token_no_errok:
			type = OPTION_ERROK;
			goto normal_off;

		case arglex_token_force:
			type = OPTION_FORCE;
			goto normal_on;

		case arglex_token_no_force:
			type = OPTION_FORCE;
			goto normal_off;

		case arglex_token_fingerprint:
			type = OPTION_FINGERPRINT;
			goto normal_on;

		case arglex_token_no_fingerprint:
			type = OPTION_FINGERPRINT;
			goto normal_off;

		case arglex_token_string:
			if (level != OPTION_LEVEL_COMMAND_LINE)
			{
				if (strchr(arglex_value.alv_string, '='))
				{
					fatal
					(
			      "may not assign variables in environment variable"
					);
				}
				else
				{
					fatal
					(
				  "may not name targets in environment variable"
					);
				}
			}
			else
			{
				char		*cp;
				string_ty	*s;

				cp = strchr(arglex_value.alv_string, '=');
				if (!cp)
				{
					s = str_from_c(arglex_value.alv_string);
					wl_append(&option.o_target, s);
					str_free(s);
				}
				else
				{
					s = str_from_c(arglex_value.alv_string);
					wl_append(&option.o_vardef, s);
					str_free(s);
				}
			}
			break;

		case arglex_token_star:
			type = OPTION_STAR;
			goto normal_on;

		case arglex_token_no_star:
			type = OPTION_STAR;
			goto normal_off;

		case arglex_token_strip_dot:
			type = OPTION_STRIP_DOT;
			goto normal_on;

		case arglex_token_no_strip_dot:
			type = OPTION_STRIP_DOT;
			goto normal_off;

		case arglex_token_update:
			type = OPTION_UPDATE;
			goto normal_on;

		case arglex_token_no_update:
			type = OPTION_UPDATE;
			goto normal_off;
		}
		arglex();
	}
}


/*
 * NAME
 *	main - initial entry point for cook
 *
 * SYNOPSIS
 *	void main(int argc, char **argv);
 *
 * DESCRIPTION
 *	Main is the initial entry point for cook.
 *
 * RETURNS
 *	Exit is always through exit().
 *	The exit code will be 0 for success, or 1 for some error.
 */

int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	int		retval;

	/*
	 * initialize things
	 * (order is critical here)
	 */
#ifdef TIMING
	os_totals_start();
#endif
	arglex_set_progname(argv[0]);
	env_initialize();
	str_initialize();
	id_initialize(); 
	builtin_initialize();
	lex_initialize();
	parse_initialize();

	/*
	 * parse the command line
	 */
	arglex_init_from_env(argv[0], argtab);
	argparse(OPTION_LEVEL_ENVIRONMENT);

	/*
	 * parse the command line
	 */
	arglex_init(argc, argv, argtab);
	argparse(OPTION_LEVEL_COMMAND_LINE);

	option_tidy_up();

	log_open();

	/*
	 * turn on progress stars if they asked for them
	 */
	if (option_test(OPTION_STAR))
		star_enable();

	/*
	 * read in the cook book
	 *
	 * If there are #include-cooked directives,
	 * we may need to do it more than once.
	 */
	if (!option.o_book)
		fatal("no book found");
	for (;;)
	{
		int	status;
		long	j;

		for (j = 0; j < option.o_vardef.wl_nwords; ++j)
		{
			char		*s;
			char		*cp;
			string_ty	*name;
			string_ty	*value;
			wlist		wl;
	
			s = option.o_vardef.wl_word[j]->str_text;
			cp = strchr(s, '=');
			assert(cp);
			if (!cp)
				continue;
			name = str_n_from_c(s, cp - s); 
			value = str_from_c(cp + 1);
			str2wl(&wl, value, (char *)0, 0);
			str_free(value);
			id_assign(name, &wl);
			str_free(name);
			wl_free(&wl);
		}

		parse(option.o_book);
		status = cook_auto_required();
		if (status < 0)
			quit(1);
		if (!status)
			break;
		id_reset();
		cook_reset();
	}

	/*
	 * work out what to cook.
	 * If no targets have been given, use the first explicit recipe.
	 */
	if (!option.o_target.wl_nwords)
		cook_find_default(&option.o_target);
	assert(option.o_target.wl_nwords);

	/*
	 * cook the target
	 */
	retval = cook(&option.o_target);

#ifdef TIMING
	os_totals();
#endif
	quit(retval);
	/*NOTREACHED*/
	return 0;
}
