/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate expression trees
 *
 * This file contains the functions for manipulating expression
 * trees; building, interpreting and freeing them.
 */

#include <ac/stddef.h>
#include <stdio.h>

#include <builtin.h>
#include <cook.h>
#include <error.h>
#include <expr.h>
#include <id.h>
#include <lex.h>
#include <match.h>
#include <mem.h>
#include <option.h>
#include <stmt.h>
#include <s-v-arg.h>
#include <trace.h>
#include <word.h>


static  position     *expr_context;


/*
 * NAME
 *	expr_alloc - allocate a pointer structure
 *
 * SYNOPSIS
 *	expr *expr_alloc(void);
 *
 * DESCRIPTION
 *	The expr_alloc function is used to allocate an expression node
 *	structure in dynamic memory.  It will initially be filled with zeros.
 *	The e_op field is defined as being non-zero to allow detection
 *	of some classes of missuse of the expression nodes.
 *
 * RETURNS
 *	A pointer to a "expr" in dynamic memory.
 *
 * CAVEAT
 *	The expression node is allocated in dynamic memory,
 *	it is the callers responsibility to ensure that it is freed
 *	when it is finished with, by a call to expr_free().
 */

expr *
expr_alloc()
{
	expr	*ep;

	trace(("expr_alloc()\n{\n"/*}*/));
	ep = mem_alloc_clear(sizeof(expr));
	ep->e_references = 1;
	ep->e_position.pos_name = str_copy(lex_cur_file());
	ep->e_position.pos_line = lex_cur_line();
	trace(("return %08lX;\n", ep));
	trace((/*{*/"}\n"));
	return ep;
}


/*
 * NAME
 *	expr_copy - copy and expression
 *
 * SYNOPSIS
 *	expr *expr_copy(expr *);
 *
 * DESCRIPTION
 *	The expr_copy function is used to make a copy of an expression tree.
 *
 * RETURNS
 *	The expr_copy function returns a pointer to the root of the copied
 *	expression tree.
 *
 * CAVEAT
 *	The result is in dynamic memory, used expr_free to dispose of it when
 *	finished with.
 */

expr *
expr_copy(ep)
	expr	*ep;
{
	trace(("expr_copy(ep = %08X)\n{\n"/*}*/, ep));
	ep->e_references++;
	trace(("return %08lX;\n", ep));
	trace((/*{*/"}\n"));
	return ep;
}


/*
 * NAME
 *	expr_free - free expression tree
 *
 * SYNOPSIS
 *	void expr_free(expr *ep);
 *
 * DESCRIPTION
 *	The expr_free function is used to free expression trees.
 *
 * CAVEAT
 *	It is assumed that the expression trees are all
 *	dynamically allocated.  Use expr_alloc() to allocate them.
 */

void
expr_free(ep)
	expr	*ep;
{
	trace(("expr_free(ep = %08X)\n{\n"/*}*/, ep));
	assert(ep);
	ep->e_references--;
	if (ep->e_references > 0)
		goto ret;
	str_free(ep->e_position.pos_name);
	switch ((int)ep->e_op)
	{
	default:
		fatal("illegal expression opcode %d (bug)", ep->e_op);

	case OP_CAT:
		expr_free(ep->e_left);
		expr_free(ep->e_right);
		break;

	case OP_FUNC:
		el_free(&ep->e_list);
		break;

	case OP_WORD:
		str_free(ep->e_word);
		break;
	}
	mem_free(ep);
ret:
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	el_free - free expression lists
 *
 * SYNOPSIS
 *	void el_free(elist *elp);
 *
 * DESCRIPTION
 *	The el_free function is used to free expression lists,
 *	it calls expr_free for each expression in the list.
 *
 * CAVEAT
 *	It is assumed that the expressions are dynamically allocated,
 *	and that the expression list was grown using el_append().
 *	The actual structure pointed to is NOT assumed to be in dynamic memory
 *	and should not be passed to free().
 */

void
el_free(elp)
	elist	*elp;
{
	int	j;

	trace(("el_free(elp = %08X)\n{\n"/*}*/, elp));
	for (j = 0; j < elp->el_nexprs; j++)
		expr_free(elp->el_expr[j]);
	if (elp->el_nexprs)
		mem_free(elp->el_expr);
	elp->el_nexprs = 0;
	elp->el_expr = 0;
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	el_copy - copy expression list
 *
 * SYNOPSIS
 *	void el_copy(elist *to, elist *from);
 *
 * DESCRIPTION
 *	The el_copy function is used to copy the list of expression trees
 *	pointed to by `from' into the expression list pointed to by `to'.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	The el_free function must be used to dispose of the list when
 *	finished with.
 */

void
el_copy(to, from)
	elist	*to;
	elist	*from;
{
	int	j;

	trace(("el_copy(to = %08X, from = %08X)\n{\n"/*}*/, to, from));
	el_zero(to);
	if (!from->el_nexprs)
		return;
	for (j = 0; j < from->el_nexprs; j++)
		el_append(to, from->el_expr[j]);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	expr_eval - evaluate an expression
 *
 * SYNOPSIS
 *	int expr_eval(wlist *result, expr *ep);
 *
 * DESCRIPTION
 *	The expr_eval function is used to evaluate an expression.
 *
 * RETURNS
 *	The results of the expression evaluation are appended to
 *	the word list pointed to by 'results' argument using wl_append.
 *
 *	The functiuon result is -1 for evaluation errors, 0 for no error.
 *
 * CAVEAT
 *	It is assumed that the results wordlist has been initialised
 *	before it was passed to expr_eval().
 *
 *	The result returned from this function are allocated in dynamic memory.
 *	It is the responsibility of the caller to ensure that they are freed
 *	when they are finished with, using wl_free().
 */

int
expr_eval(result, ep)
	wlist	*result;
	expr	*ep;
{
	int	retval;

	trace(("expr_eval(result = %08X, ep = %08X)\n{\n"/*}*/, result, ep));
	assert(ep);
	switch ((int)ep->e_op)
	{
	default:
		error
		(
			"%s: %d: illegal expression opcode %d (bug)",
			ep->e_position.pos_name->str_text,
			ep->e_position.pos_line,
			ep->e_op
		);
		expr_eval_fails:
		option_set_errors();
		retval = -1;
		goto ret;

	case OP_WORD:
		{
			match_ty *field;

			/*
			 * If a wildcard mapping is in force (we are performing
			 * actions bound to an implicit recipe) the word will be
			 * mapped before it is returned.
			 */
			field = match_top();
			if (field)
			{
				string_ty *s;

				s = reconstruct(ep->e_word, field);
				wl_append(result, s);
				str_free(s);
			}
			else
				wl_append(result, ep->e_word);
		}
		break;

	case OP_FUNC:
		{
			wlist	 wl;

			if (el2wl(&wl, &ep->e_list))
			{
				wl_free(&wl);
				goto expr_eval_fails;
			}
			switch (wl.wl_nwords)
			{
			case 0:
				break;

			case 1:
				{
					wlist	value;

					if (id_search(wl.wl_word[0], &value))
					{
						int		j;

						for (j = 0; j < value.wl_nwords; j++)
							wl_append(result, value.wl_word[j]);
						wl_free(&value);
						break;
					}

					/*
					 * If the variable is not found,
					 * fall through into the function case
					 */
				}

			default:
				{
					bifp	code;

					code = builtin_search(wl.wl_word[0]);
					if (!code)
					{
						error
						(
							"%s: %d: undefined %s \"%s\"",
							ep->e_position.pos_name->str_text,
							ep->e_position.pos_line,
							wl.wl_nwords >= 2 ? "function" : "variable",
							wl.wl_word[0]->str_text
						);
						goto expr_eval_fails;
					}
					expr_context = &ep->e_position;
					if (code(result, &wl))
						goto expr_eval_fails;
				}
				break;
			}
			wl_free(&wl);
		}
		break;

	case OP_CAT:
		{
			wlist	left;
			wlist	right;
			int	j;

			/*
			 * Form the two word lists.
			 * Tack the last word of the left list
			 * onto the first word of the right list.
			 *
			 * There are other conceivable ways to do this,
			 * but this definition gives the fewest surprises.
			 */
			wl_zero(&left);
			if (expr_eval(&left, ep->e_left))
			{
				wl_free(&left);
				goto expr_eval_fails;
			}
			wl_zero(&right);
			if (expr_eval(&right, ep->e_right))
			{
				wl_free(&left);
				wl_free(&right);
				goto expr_eval_fails;
			}
			switch ((left.wl_nwords ? 1 : 0) | (right.wl_nwords ? 2 : 0))
			{
			case 0:
				/* both lists empty */
				break;

			case 1:
				/* right list empty */
				for (j = 0; j < left.wl_nwords; j++)
					wl_append(result, left.wl_word[j]);
				break;

			case 2:
				/* left list empty */
				for (j = 0; j < right.wl_nwords; j++)
					wl_append(result, right.wl_word[j]);
				break;

			case 3:
				{
					string_ty *s;

					/* at least one word in each list */
					for (j = 0; j < left.wl_nwords - 1; j++)
						wl_append(result, left.wl_word[j]);
					s = str_catenate(left.wl_word[j], right.wl_word[0]);
					wl_append(result, s);
					str_free(s);
					for (j = 1; j < right.wl_nwords; j++)
						wl_append(result, right.wl_word[j]);
				}
				break;
			}
			wl_free(&left);
			wl_free(&right);
		}
		break;
	}
	retval = 0;
ret:
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	el_append - append to an expression list
 *
 * SYNOPSIS
 *	void el_append(elist *el, expr *e);
 *
 * DESCRIPTION
 *	The el_append function is used to append an expression to an expression
 *	list.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	The expression has not been copied, so do not hand it
 *	to expr_free after you append it.
 *
 *	It is assumed that the elist has been previously initialised by a
 *	    elist el;
 *	    el_zero(&el);
 *	statement (or similar) before this function is called.
 */

void
el_append(el, e)
	elist	*el;
	expr	*e;
{
	size_t	nbytes;

	trace(("el_append(el = %08X, e = %08X)\n{\n"/*}*/, el, e));
	assert(el);
	assert(e);
	assert(!el->el_nexprs || !!el->el_expr);
	nbytes = (el->el_nexprs + 1 ) * sizeof(expr *);
	el->el_expr = mem_change_size(el->el_expr, nbytes);
	el->el_expr[el->el_nexprs++] = expr_copy(e);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	el2wl - expression list to word list
 *
 * SYNOPSIS
 *	int el2wl(wlist *wl, elist *el);
 *
 * DESCRIPTION
 *	The el2wl function is used to turn an expression list into a word list.
 *
 * RETURNS
 *	The word list is initialised before it is used to
 *	store the results of evaluating the expressions.
 *
 *	The function return value is -1 for errors and 0 for success.
 *
 * CAVEAT
 *	The results returned by this function are allocated in dynamic memory.
 *	It is the responsibility of the caller to free them when finished with,
 *	by a call to wl_free().
 */

int
el2wl(wlp, elp)
	wlist	*wlp;
	elist	*elp;
{
	int	j;
	int	retval;

	trace(("el2wl(wlp = %08X, elp = %08X)\n{\n"/*}*/, wlp, elp));
	retval = 0;
	wl_zero(wlp);
	for (j = 0; j < elp->el_nexprs; j++)
	{
		if (expr_eval(wlp, elp->el_expr[j]))
		{
			retval = -1;
			break;
		}
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	expr_eveal_condition - evaluate condition
 *
 * SYNOPSIS
 *	int expr_eval_condition(expr *);
 *
 * DESCRIPTION
 *	The expr_eval_condition function is used to evaluate an expression to
 *	yeild a true/false result.  The expression is evaluated into a word
 *	list.  A false result is if all of the resulting strings are empty or
 *	0, true otherwise.
 *
 * RETURNS
 *	The expr_eval_condition function returns 0 if the condition is false,
 *	and nonzero if it is true.  The value -1 is returned on error.
 *
 * CAVEAT
 *	The str_bool function is used to test the booean value of a string;
 *	changeing the behaviour of that function will change the behaviour of
 *	this one.
 */

int
expr_eval_condition(ep)
	expr	*ep;
{
	wlist	wl;
	int	j;
	int	result;

	trace(("expr_eval_condition(ep = %08X)\n{\n"/*}*/, ep));
	assert(ep);
	wl_zero(&wl);
	if (expr_eval(&wl, ep))
	{
		result = -1;
		goto ret;
	}
	for (j = 0; j < wl.wl_nwords; ++j)
	{
		if (str_bool(wl.wl_word[j]))
		{
			wl_free(&wl);
			result = 1;
			goto ret;
		}
	}
	wl_free(&wl);
	result = 0;
ret:
	trace(("return %d;\n", result));
	trace((/*{*/"}\n"));
	return result;
}


/*VARARGS1*/
void
expr_error(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	string_ty	*buffer;

	sva_init(ap, s);
	buffer = str_vformat(s, ap);
	va_end(ap);
	error
	(
		"%S: %d: %S",
		expr_context->pos_name,
		expr_context->pos_line,
		buffer
	);
	str_free(buffer);
}


void
el_zero(elp)
	elist	*elp;
{
	elp->el_nexprs = 0;
	elp->el_expr = 0;
}
