/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin pathname functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/pathname.h>
#include <error.h>
#include <expr.h>
#include <os.h>


/*
 * NAME
 *	builtin_dir - dir part
 *
 * SYNOPSIS
 *	int builtin_dir(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	"dir" is a built-in function of cook, described as follows:
 *	This function requires one or more arguments,
 *	the name of a files of which to get the dir parts.
 *
 * RETURNS
 *	It returns a string containing the directory parts
 *	of the named files.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_dir(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty	*s;

		s = os_dirname(args->wl_word[j]);
		if (!s)
			return -1;
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_entryname - entryname part
 *
 * SYNOPSIS
 *	int builtin_entryname(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires one or more arguments,
 *	the name of a files of which to get the entryname parts.
 *
 * RETURNS
 *	It returns a string containing the entryname parts
 *	of the named files.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_entryname(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty	*s;

		s = os_entryname(args->wl_word[j]);
		if (!s)
			return -1;
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_pathname - pathname part
 *
 * SYNOPSIS
 *	int builtin_pathname(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires one or more arguments,
 *	the name of a files of which to get the pathname parts.
 *
 * RETURNS
 *	It returns a string containing the pathname parts
 *	of the named files.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_pathname(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty	*s;

		s = os_pathname(args->wl_word[j]);
		if (!s)
			return -1;
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}
