/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of this are candidates for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * This explicitly and forever excluded from being a builtin function
 * is anything which known or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <builtin/match.h>
#include <error.h>
#include <expr.h>
#include <match.h>


/*
 * NAME
 *	builtin_match - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_match(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_match(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	match_ty	*field;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 2; j < args->wl_nwords; j++)
	{
		field = match(args->wl_word[1], args->wl_word[j]);
		if (field)
		{
			wl_append(result, str_true);
			match_free(field);
		}
		else
			wl_append(result, str_false);
	}
	return 0;
}


/*
 * NAME
 *	builtin_match - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_match(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_match_mask(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	match_ty	*field;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 2; j < args->wl_nwords; j++)
	{
		field = match(args->wl_word[1], args->wl_word[j]);
		if (field)
		{
			wl_append(result, args->wl_word[j]);
			match_free(field);
		}
	}
	return 0;
}


/*
 * NAME
 *	builtin_fromto - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_fromto(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_fromto(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	match_ty	*field;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 3)
	{
		expr_error
		(
			"%s: requires at least two arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 3; j < args->wl_nwords; j++)
	{
		field = match(args->wl_word[1], args->wl_word[j]);
		if (field)
		{
			string_ty	*s;

			s = reconstruct(args->wl_word[2], field);
			wl_append(result, s);
			str_free(s);
			match_free(field);
		}
		else
			wl_append(result, args->wl_word[j]);
	}
	return 0;
}
