/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin filter_out function
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/filter_out.h>
#include <error.h>
#include <expr.h>
#include <match.h>


/*
 * NAME
 *	builtin_filter_out - wilcard filtering
 *
 * SYNOPSIS
 *	int builtin_filter_out(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Filter_out is a built-in function of cook, described as follows:
 *	This function requires at least one argument.
 *	Filter_out uses the first argument as a pattern, and the result
 *	is the those of the second and subsequent arguments which do
 *	not match the patern.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_filter_out(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	match_ty	*field;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 2; j < args->wl_nwords; j++)
	{
		field = match(args->wl_word[1], args->wl_word[j]);
		if (field)
			match_free(field);
		else
			wl_append(result, args->wl_word[j]);
	}
	return 0;
}
