/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin defined function
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/defined.h>
#include <error.h>
#include <expr.h>
#include <id.h>


/*
 * NAME
 *	builtin_defined - is a variable defined
 *
 * SYNOPSIS
 *	int builtin_defined(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires a single argument,
 *	the name of a variable to be tested for existence.
 *
 * RETURNS
 *	It returns "1" (true) if the named variable is defined
 *	and "" (false) if it is not.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_defined(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		wlist		value;

		if (id_search(args->wl_word[j], &value))
		{
			wl_free(&value);
			wl_append(result, str_true);
		}
		else
			wl_append(result, str_false);
	}
	return 0;
}
