/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to access the builtin functions
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of functionality is candidate for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * Explicitly and forever excluded from being a builtin function
 * is anything which knows or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <builtin.h>
#include <builtin/addprefix.h>
#include <builtin/addsuffix.h>
#include <builtin/basename.h>
#include <builtin/boolean.h>
#include <builtin/collect.h>
#include <builtin/cook.h>
#include <builtin/defined.h>
#include <builtin/execute.h>
#include <builtin/exists.h>
#include <builtin/filter_out.h>
#include <builtin/find_command.h>
#include <builtin/findstring.h>
#include <builtin/getenv.h>
#include <builtin/glob.h>
#include <builtin/home.h>
#include <builtin/join.h>
#include <builtin/match.h>
#include <builtin/opsys.h>
#include <builtin/pathname.h>
#include <builtin/split.h>
#include <builtin/stringset.h>
#include <builtin/strip.h>
#include <builtin/subst.h>
#include <builtin/suffix.h>
#include <builtin/text.h>
#include <builtin/unsplit.h>
#include <builtin/word.h>
#include <symtab.h>


typedef struct func_ty func_ty;
struct func_ty
{
	char	*f_name;
	bifp	f_code;
};

static symtab_ty *symtab;


/*
 * NAME
 *	func - table of built-in functions
 *
 * SYNOPSIS
 *	func_ty func[];
 *
 * DESCRIPTION
 *	Func is a table of function names and pointers
 *	for the built-in functions of cook.
 */

static func_ty func[] =
{
	{ "addprefix",		builtin_addprefix,	},
	{ "addsuffix",		builtin_addsuffix,	},
	{ "and",		builtin_and,		},
	{ "basename",		builtin_basename,	},
	{ "cando",		builtin_cando,		},
	{ "catenate",		builtin_catenate,	},
	{ "collect",		builtin_collect,	},
	{ "collect_lines",	builtin_collect,	},
	{ "count",		builtin_count,		},
	{ "defined",		builtin_defined,	},
	{ "dir",		builtin_dir,		},
	{ "dirname",		builtin_dir,		},
	{ "downcase",		builtin_downcase,	},
	{ "entryname",		builtin_entryname,	},
	{ "execute",		builtin_execute,	},
	{ "exists",		builtin_exists,		},
	{ "filter",		builtin_match_mask,	},
	{ "filter-out",		builtin_filter_out,	},
	{ "filter_out",		builtin_filter_out,	},
	{ "find_command",	builtin_find_command,	},
	{ "findstring",		builtin_findstring,	},
	{ "fromto",		builtin_fromto,		},
	{ "getenv",		builtin_getenv,		},
	{ "glob",		builtin_glob,		},
	{ "head",		builtin_head,		},
	{ "home",		builtin_home,		},
	{ "if",			builtin_if,		},
	{ "in",			builtin_in,		},
	{ "join",		builtin_join,		},
	{ "match",		builtin_match,		},
	{ "match_mask",		builtin_match_mask,	},
	{ "matches",		builtin_match,		},
	{ "mtime",		builtin_mtime,		},
	{ "not",		builtin_not,		},
	{ "operating_system",	builtin_opsys,		},
	{ "or",			builtin_or,		},
	{ "os",			builtin_opsys,		},
	{ "pathname",		builtin_pathname,	},
	{ "patsubst",		builtin_fromto,		},
	{ "prepost",		builtin_prepost,	},
	{ "quote",		builtin_quote,		},
	{ "resolve",		builtin_resolve,	},
	{ "shell",		builtin_collect,	},
	{ "sort",		builtin_sort,		},
	{ "sort_newest",	builtin_sort_newest,	},
	{ "split",		builtin_split,		},
	{ "stringset",		builtin_stringset,	},
	{ "strip",		builtin_strip,		},
	{ "subst",		builtin_subst,		},
	{ "suffix",		builtin_suffix,		},
	{ "tail",		builtin_tail,		},
	{ "unsplit",		builtin_unsplit,	},
	{ "upcase",		builtin_upcase,		},
	{ "uptodate",		builtin_uptodate,	},
	{ "wildcard",		builtin_glob,		},
	{ "word",		builtin_word,		},
	{ "words",		builtin_count,		},
};


/*
 * NAME
 *	builtin_initialize - start up builtins
 *
 * SYNOPSIS
 *	void builtin_initialize(void);
 *
 * DESCRIPTION
 *	The builtin_initialize function is used to initialize the symbol table
 *	with the names and pointers to the builtin functions.
 *
 * CAVEAT
 *	This function must be called after the id_initialize function.
 */

void
builtin_initialize()
{
	func_ty		*fp;
	string_ty	*s;

	if (!symtab)
		symtab = symtab_alloc(SIZEOF(func));
	for (fp = func; fp < ENDOF(func); ++fp)
	{
		s = str_from_c(fp->f_name);
		symtab_assign(symtab, s, fp->f_code);
		str_free(s);
	}
}


bifp
builtin_search(name)
	string_ty	*name;
{
	if (!symtab)
		return 0;
	return symtab_query(symtab, name);
}
