/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: operating system entry point, and command line argument parsing
 */

#include <ac/stddef.h>
#include <stdio.h>
#include <ac/string.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <cache.h>
#include <error.h>
#include <help.h>
#include <main.h>
#include <sniff.h>
#include <str.h>
#include <trace.h>
#include <version.h>


static void usage _((void));

static void
usage()
{
	fprintf(stderr, "usage: %s [ <option>... ] <filename>\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERsion\n", progname);
	exit(1);
}


static void c_incl_help _((void));

static void
c_incl_help()
{
	static char *text[] =
	{
#include <../man1/c_incl.h>
	};

	trace(("c_incl_help()\n{\n"/*}*/));
	help(text, SIZEOF(text), usage);
	trace((/*{*/"}\n"));
}


enum
{
	arglex_token_absent_local_error,
	arglex_token_absent_local_ignore,
	arglex_token_absent_local_mention,
	arglex_token_absent_program_error,
	arglex_token_absent_program_ignore,
	arglex_token_absent_system_error,
	arglex_token_absent_system_ignore,
	arglex_token_absent_system_mention,
	arglex_token_include,
	arglex_token_no_cache,
	arglex_token_no_system,
	arglex_token_lang_c,
	arglex_token_lang_roff,
	arglex_token_prefix,
	arglex_token_suffix,
	arglex_token_verbose
};

static	arglex_table_ty	argtab[] =
{
	{
		"-Absent",
		(arglex_token_ty)arglex_token_absent_local_ignore,
	},
	{
		"-Absent_Local_Error",
		(arglex_token_ty)arglex_token_absent_local_error,
	},
	{
		"-Absent_Local_Ignore",
		(arglex_token_ty)arglex_token_absent_local_ignore,
	},
	{
		"-Absent_Local_Mention",
		(arglex_token_ty)arglex_token_absent_local_mention,
	},
	{
		"-Absent_Program_Error",
		(arglex_token_ty)arglex_token_absent_program_error,
	},
	{
		"-Absent_Program_Ignore",
		(arglex_token_ty)arglex_token_absent_program_ignore,
	},
	{
		"-Absent_System_Error",
		(arglex_token_ty)arglex_token_absent_system_error,
	},
	{
		"-Absent_System_Ignore",
		(arglex_token_ty)arglex_token_absent_system_ignore,
	},
	{
		"-Absent_System_Mention",
		(arglex_token_ty)arglex_token_absent_system_mention,
	},
	{
		"-C",
		(arglex_token_ty)arglex_token_lang_c,
	},
	{
		"-Empty_If_Absent",
		(arglex_token_ty)arglex_token_absent_program_ignore,
	},
	{
		"-\\I*",
		(arglex_token_ty)arglex_token_include,
	},
	{
		"-Include",
		(arglex_token_ty)arglex_token_include,
	},
	{
		"-No_Cache",
		(arglex_token_ty)arglex_token_no_cache,
	},
	{
		"-No_System",
		(arglex_token_ty)arglex_token_no_system,
	},
	{
		"-PREfix",
		(arglex_token_ty)arglex_token_prefix,
	},
	{
		"-Roff",
		(arglex_token_ty)arglex_token_lang_roff,
	},
	{
		"-System_absent",
		(arglex_token_ty)arglex_token_absent_system_ignore,
	},
	{
		"-SUFfix",
		(arglex_token_ty)arglex_token_suffix,
	},
	{
		"-Verbose",
		(arglex_token_ty)arglex_token_verbose,
	},

	/* end marker */
	{ 0, (arglex_token_ty)0, },
};


int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	char		*source;
	int		no_system;
	int		no_cache;
	sniff_ty	*language;

	arglex_init(argc, argv, argtab);
	str_initialize();
	cache_initialize();
	switch (arglex())
	{
	case arglex_token_help:
		c_incl_help();
		exit(0);

	case arglex_token_version:
		version();
		exit(0);

	default:
		break;
	}

	source = 0;
	no_system = 0;
	no_cache = 0;
	language = 0;
	option.o_absent_local = -1;
	option.o_absent_system = -1;
	option.o_absent_program = -1;
	while (arglex_token != arglex_token_eoln)
	{
		switch(arglex_token)
		{
		default:
			error
			(
				"misplaced \"%s\" command line argument",
				arglex_value.alv_string
			);
			usage();

		case arglex_token_string:
			if (source)
				fatal("too many filenames specified");
			source = arglex_value.alv_string;
			break;

		case arglex_token_verbose:
			if (option.o_verbose)
			{
				duplicate:
				fatal
				(
					"duplicate \"%s\" option",
					arglex_value.alv_string
				);
			}
			++option.o_verbose;
			break;

		case arglex_token_absent_local_ignore:
			if (option.o_absent_local != -1)
				goto duplicate;
			option.o_absent_local = absent_ignore;
			break;

		case arglex_token_absent_local_mention:
			if (option.o_absent_local != -1)
				goto duplicate;
			option.o_absent_local = absent_mention;
			break;

		case arglex_token_absent_local_error:
			if (option.o_absent_local != -1)
				goto duplicate;
			option.o_absent_local = absent_error;
			break;

		case arglex_token_absent_system_ignore:
			if (option.o_absent_system != -1)
				goto duplicate;
			option.o_absent_system = absent_ignore;
			break;

		case arglex_token_absent_system_mention:
			if (option.o_absent_system != -1)
				goto duplicate;
			option.o_absent_system = absent_mention;
			break;

		case arglex_token_absent_system_error:
			if (option.o_absent_system != -1)
				goto duplicate;
			option.o_absent_system = absent_error;
			break;

		case arglex_token_absent_program_ignore:
			if (option.o_absent_program != -1)
				goto duplicate;
			option.o_absent_program = absent_ignore;
			break;

		case arglex_token_absent_program_error:
			if (option.o_absent_program != -1)
				goto duplicate;
			option.o_absent_program = absent_error;
			break;

		case arglex_token_include:
			if (arglex() != arglex_token_string)
				fatal("-Include requires a string argument");
			sniff_include(arglex_value.alv_string);
			break;

		case arglex_token_no_system:
			if (no_system)
				goto duplicate;
			no_system++;
			break;

		case arglex_token_no_cache:
			if (no_cache)
				goto duplicate;
			no_cache++;
			break;

		case arglex_token_lang_c:
			if (language)
				goto duplicate;
			language = &lang_c;
			break;

		case arglex_token_lang_roff:
			if (language)
				goto duplicate;
			language = &lang_roff;
			break;
		
		case arglex_token_prefix:
			if (prefix)
				goto duplicate;
			if (arglex() != arglex_token_string)
				usage();
			prefix = arglex_value.alv_string;
			break;
		
		case arglex_token_suffix:
			if (suffix)
				goto duplicate;
			if (arglex() != arglex_token_string)
				usage();
			suffix = arglex_value.alv_string;
			break;

#ifdef DEBUG
		case arglex_token_tracing:
			if (arglex() != arglex_token_string)
				fatal("-TRACIng requires one or more string arguments");
			for (;;)
			{
				trace_enable(arglex_value.alv_string);
				if (arglex() != arglex_token_string)
					break;
			}
			continue;
#endif
		}
		arglex();
	}
	if (option.o_absent_local == -1)
		option.o_absent_local = absent_mention;
	if (option.o_absent_system == -1)
		option.o_absent_system = absent_ignore;
	if (option.o_absent_program == -1)
		option.o_absent_program = absent_error;
	if (!source)
		fatal("no input file specified");

	/*
	 * set the language to be used
	 */
	if (!language)
		language = &lang_c;
	sniff_language(language);

	/*
	 * apply any default or suffix search rules
	 * or anything else defined by the language
	 */
	if (!no_system)
		sniff_prepare();

	/*
	 * read and analyze the file
	 */
	if (!no_cache)
		cache_read();
	sniff(source);
	if (!no_cache)
		cache_write();
	exit(0);
	return 0;
}
