/* icntotif.c: convert RIP .ICN file (EGA) to Tag Image File Format
 *
 * Eric P. Scott, San Francisco State University, March 1994
 * Copyright 1994 by Eric P. Scott.  All rights reserved.
 *
 * This software is "freeware" but is not in the public domain.
 *
 * The author grants you a nonexclusive royalty-free license to copy,
 * redistribute, and use this software for any purpose provided that
 * source distributions retain this entire copyright and license.
 * You may not sublicense or distribute modified versions to anyone else
 * except as part of another product or program, and modified versions
 * must be so labeled.
 *
 * You are encouraged to give proper credit to the author in binary
 * distributions and associated documentation.
 *
 * This software is provided "as is" with no warranties of any kind,
 * including the warranties of merchantability and fitness for a
 * particular purpose.
 *
 * TIFF is a trademark of Aldus Corporation.
 *
 * RIPscrip is a trademark of TeleGrafix Communications, Inc.
 *
 * This software is not supported or endorsed by TeleGrafix.
 *
 */

/*
 * This program processes the Icon File format described in _RIPscrip
 * Graphics Protocol Specification_, Revision 1.54, by TeleGrafix
 * Communications, Inc.
 *
 * Output conforms to both TIFF 5.0 (1988) and TIFF 6.0 (1992).
 */

/* Revision History
 *
 * 1.0  4/13/94  EPS  Initial Release
 *
 */

#ifndef lint
static char sccsid[]="@(#)icntotif.c	1.0 (SFSU) 4/13/94";
#endif

#ifdef NX_COMPILER_RELEASE_3_0
#include <ansi/stdio.h>
#include <ansi/stdlib.h>
#else
#include <stdio.h>
#include <stdlib.h>
#endif

void putushort(unsigned short v, FILE *stream)
{
	putc((int)(v>>8)&0xff, stream);
	putc((int)v&0xff, stream);
}

void putulong(unsigned long v, FILE *stream)
{
	putc((int)(v>>24)&0xff, stream);
	putc((int)(v>>16)&0xff, stream);
	putc((int)(v>>8)&0xff, stream);
	putc((int)v&0xff, stream);
}

#define TIFF_TYPE_SHORT 3
#define TIFF_TYPE_LONG 4
#define TIFF_TYPE_RATIONAL 5

void puttag(unsigned short tag, unsigned short type,
	unsigned long count, FILE *stream)
{
	putc((int)(tag>>8)&0xff, stream);
	putc((int)tag&0xff, stream);
	putc((int)(type>>8)&0xff, stream);
	putc((int)type&0xff, stream);
	putc((int)(count>>24)&0xff, stream);
	putc((int)(count>>16)&0xff, stream);
	putc((int)(count>>8)&0xff, stream);
	putc((int)count&0xff, stream);
}

main(int argc, char *argv[])
{
	register unsigned char *p;
	register int c;
	register unsigned int l, r;
	unsigned int width, height;
	unsigned char *ibuf, *lbuf;
	unsigned int nstrips, bpr, rps;
	unsigned long doff;

	if (argc!=3) {
		(void)fprintf(stderr, "Usage: %s iconfile tifffile\n", *argv);
		exit(1);
	}
	if (!freopen(argv[1], "rb", stdin)) {
		perror(argv[1]);
		exit(1);
	}
	if ((c=getchar())==EOF) {
	runt:
		(void)fprintf(stderr, "%s: short file?\n", argv[1]);
		exit(1);
	}
	width=c&0xff;
	if ((c=getchar())==EOF) goto runt;
	width+=(c&0xff)<<8;
	if ((c=getchar())==EOF) goto runt;
	height=c&0xff;
	if ((c=getchar())==EOF) goto runt;
	height+=(c&0xff)<<8;
	if (height>=12696||width>=3300) {
		(void)fprintf(stderr, "%s: absurd image dimensions %u x %u?\n",
		*argv, width+1, height+1);
		exit(1);
	}
	width++;
	height++;
	if (!(ibuf=(unsigned char *)malloc((size_t)((width+1)/2)*height+4))) {
		(void)fprintf(stderr, "%s: malloc failed for %u x %u?\n",
			*argv, width, height);
		exit(1);
	}
	/* convert planar to meshed */
	lbuf=ibuf;
	r=height; do {
		p=lbuf;
		for (l=0;l<width;l+=8) {
			if ((c=getchar())==EOF) goto runt;
			*p=0;
			if (c&128) *p=0x80;
			if (c&64) *p|=8;
			*++p=0;
			if (c&32) *p=0x80;
			if (c&16) *p|=8;
			*++p=0;
			if (c&8) *p=0x80;
			if (c&4) *p|=8;
			*++p=0;
			if (c&2) *p=0x80;
			if (c&1) *p|=8;
			p++;
		}
		p=lbuf;
		for (l=0;l<width;l+=8) {
			if ((c=getchar())==EOF) goto runt;
			if (c&128) *p|=0x40;
			if (c&64) *p|=4;
			p++;
			if (c&32) *p|=0x40;
			if (c&16) *p|=4;
			p++;
			if (c&8) *p|=0x40;
			if (c&4) *p|=4;
			p++;
			if (c&2) *p|=0x40;
			if (c&1) *p|=4;
			p++;
		}
		p=lbuf;
		for (l=0;l<width;l+=8) {
			if ((c=getchar())==EOF) goto runt;
			if (c&128) *p|=0x20;
			if (c&64) *p|=2;
			p++;
			if (c&32) *p|=0x20;
			if (c&16) *p|=2;
			p++;
			if (c&8) *p|=0x20;
			if (c&4) *p|=2;
			p++;
			if (c&2) *p|=0x20;
			if (c&1) *p|=2;
			p++;
		}
		p=lbuf;
		for (l=0;l<width;l+=8) {
			if ((c=getchar())==EOF) goto runt;
			if (c&128) *p|=0x10;
			if (c&64) *p|=1;
			p++;
			if (c&32) *p|=0x10;
			if (c&16) *p|=1;
			p++;
			if (c&8) *p|=0x10;
			if (c&4) *p|=1;
			p++;
			if (c&2) *p|=0x10;
			if (c&1) *p|=1;
			p++;
		}
		lbuf+=(width+1)/2;
	} while (--r>0);
#ifdef DEBUG
	while ((c=getchar())!=EOF)
		(void)fprintf(stderr, "garbage byte: %02x\n", c&0xff);
#endif
	bpr=(width+1)/2;
	rps=8192/bpr;
	if (rps<=0) rps=1;
	else if (rps>height) rps=height;
	nstrips=(height+rps-1)/rps;
	doff=182L;	/* 8+2+14*12+4 */
	if (!freopen(argv[2], "wb", stdout)) {
		perror(argv[2]);
		exit(1);
	}
	putc('M', stdout); putc('M', stdout);	/* Big-endian TIFF */
	putushort(42, stdout);		/* TIFF "version number" */
	putulong(8L, stdout);		/* offset to IFDs */
	putushort(14, stdout);		/* count of entries that follow */

	/* NewSubfileType */
	puttag(254, TIFF_TYPE_LONG, 1L, stdout);
	putulong(0L, stdout);

	/* ImageWidth */
	puttag(256, TIFF_TYPE_LONG, 1L, stdout);
	putulong((unsigned long)width, stdout);

	/* ImageLength */
	puttag(257, TIFF_TYPE_LONG, 1L, stdout);
	putulong((unsigned long)height, stdout);
	
	/* BitsPerSample */
	puttag(258, TIFF_TYPE_SHORT, 1L, stdout);
	putushort(4, stdout);
	putc('\0', stdout); putc('\0', stdout);
	
	/* Compression */
	puttag(259, TIFF_TYPE_SHORT, 1L, stdout);
	putushort(1, stdout);		/* (No compression) */
	putc('\0', stdout); putc('\0', stdout);
	
	/* PhotometricInterpretation */
	puttag(262, TIFF_TYPE_SHORT, 1L, stdout);
	putushort(3, stdout);		/* (Palette Color) */
	putc('\0', stdout); putc('\0', stdout);

	/* StripOffsets */
	puttag(273, TIFF_TYPE_LONG, (unsigned long)nstrips, stdout);
	if (nstrips>1) {
	    putulong(doff, stdout);
	    doff+=nstrips*4L;
	}
	else putulong(doff+112L, stdout);	/* 8+8+96 */

	/* SamplesPerPixel */
	puttag(277, TIFF_TYPE_SHORT, 1L, stdout);
	putushort(1, stdout);
	putc('\0', stdout); putc('\0', stdout);

	/* RowsPerStrip */
	puttag(278, TIFF_TYPE_LONG, 1L, stdout);
	putulong((unsigned long)rps, stdout);

	/* StripByteCounts */
	puttag(279, TIFF_TYPE_SHORT, (unsigned long)nstrips, stdout);
	if (nstrips>2) {
	    putulong(doff, stdout);
	    doff+=nstrips*2L;
	}
	else {
	    putushort(bpr*rps, stdout);
	    putushort(bpr*(height-rps), stdout);
	}

	/* XResolution */
	puttag(282, TIFF_TYPE_RATIONAL, 1L, stdout);
	putulong(doff, stdout);
	doff+=8L;

	/* YResolution */
	puttag(283, TIFF_TYPE_RATIONAL, 1L, stdout);
	putulong(doff, stdout);
	doff+=8L;

	/* ResolutionUnit */
	puttag(296, TIFF_TYPE_SHORT, 1L, stdout);
	putushort(2, stdout);		/* (Inch) */
	putc('\0', stdout); putc('\0', stdout);

	/* ColorMap */
	puttag(320, TIFF_TYPE_SHORT, 48L, stdout);
	putulong(doff, stdout);	/* offset (from BOF) to subcurves */
	doff+=96L;	/* 48*2 */

	/* end of IFD */
	putc('\0', stdout); putc('\0', stdout);
	putc('\0', stdout); putc('\0', stdout);

	if (nstrips>1) {	
		/* StripOffsets (ulong)*/
		l=nstrips; do {
			putulong(doff, stdout);
			doff+=(unsigned long)(bpr*rps);
		} while (--l>0);
		if (nstrips>2) {
			/* StripByteCounts (ushort) */
			l=nstrips-1; do {
				putushort(bpr*rps, stdout);
			} while (--l>0);
			putushort(bpr*(height-rps*(nstrips-1)), stdout);
		}
	}

	/* XResolution */
	putulong(72L, stdout);
	putulong(1L, stdout);

	/* YResolution */
	putulong(279L, stdout);
	putulong(5L, stdout);

	/* ColorMap subcurves */
	putushort(0x0000, stdout);	/* R 0 (black) */
	putushort(0x0000, stdout);	/* R 1 (blue) */
	putushort(0x0000, stdout);	/* R 2 (green) */
	putushort(0x0000, stdout);	/* R 3 (cyan) */
	putushort(0xaaaa, stdout);	/* R 4 (red) */
	putushort(0xaaaa, stdout);	/* R 5 (magenta) */
	putushort(0xaaaa, stdout);	/* R 6 (brown) */
	putushort(0xaaaa, stdout);	/* R 7 (lightgray) */
	putushort(0x5555, stdout);	/* R 8 (darkgray) */
	putushort(0x5555, stdout);	/* R 9 (lightblue) */
	putushort(0x5555, stdout);	/* R10 (lightgreen) */
	putushort(0x5555, stdout);	/* R11 (lightcyan) */
	putushort(0xffff, stdout);	/* R12 (lightred) */
	putushort(0xffff, stdout);	/* R13 (lightmagenta) */
	putushort(0xffff, stdout);	/* R14 (yellow) */
	putushort(0xffff, stdout);	/* R15 (white) */
	putushort(0x0000, stdout);	/* G 0 (black) */
	putushort(0x0000, stdout);	/* G 1 (blue) */
	putushort(0xaaaa, stdout);	/* G 2 (green) */
	putushort(0xaaaa, stdout);	/* G 3 (cyan) */
	putushort(0x0000, stdout);	/* G 4 (red) */
	putushort(0x0000, stdout);	/* G 5 (magenta) */
	putushort(0x5555, stdout);	/* G 6 (brown) */
	putushort(0xaaaa, stdout);	/* G 7 (lightgray) */
	putushort(0x5555, stdout);	/* G 8 (darkgray) */
	putushort(0x5555, stdout);	/* G 9 (lightblue) */
	putushort(0xffff, stdout);	/* G10 (lightgreen) */
	putushort(0xffff, stdout);	/* G11 (lightcyan) */
	putushort(0x5555, stdout);	/* G12 (lightred) */
	putushort(0x5555, stdout);	/* G13 (lightmagenta) */
	putushort(0xffff, stdout);	/* G14 (yellow) */
	putushort(0xffff, stdout);	/* G15 (white) */
	putushort(0x0000, stdout);	/* B 0 (black) */
	putushort(0xaaaa, stdout);	/* B 1 (blue) */
	putushort(0x0000, stdout);	/* B 2 (green) */
	putushort(0xaaaa, stdout);	/* B 3 (cyan) */
	putushort(0x0000, stdout);	/* B 4 (red) */
	putushort(0xaaaa, stdout);	/* B 5 (magenta) */
	putushort(0x0000, stdout);	/* B 6 (brown) */
	putushort(0xaaaa, stdout);	/* B 7 (lightgray) */
	putushort(0x5555, stdout);	/* B 8 (darkgray) */
	putushort(0xffff, stdout);	/* B 9 (lightblue) */
	putushort(0x5555, stdout);	/* B10 (lightgreen) */
	putushort(0xffff, stdout);	/* B11 (lightcyan) */
	putushort(0x5555, stdout);	/* B12 (lightred) */
	putushort(0xffff, stdout);	/* B13 (lightmagenta) */
	putushort(0x5555, stdout);	/* B14 (yellow) */
	putushort(0xffff, stdout);	/* B15 (white) */
	
	/* finally, the strips themselves */
	(void)fwrite((void *)ibuf, (size_t)((width+1)/2), (size_t)height,
		stdout);
	/* free(ibuf) */
	exit(0);
}
