static char rcsid[] = "$Id: signs.c,v 1.3 1992/01/29 14:33:52 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *  Copyright 1993 Philippe-Andre Prindeville, Telecom Paris.
 *
 *****************************************************************************/

/*  signs.c
 *
 *  This file contains all the routines for the creation and manipulation of 
 *  symbols used in mxgdb.  There are 3 different signs:
 *    arrow  - a solid right arrow to indicate the current execution point.
 *    updown - an outlined right arrow to indicate position in stack trace.
 *    stop   - a stop hand symbol to indicate a breakpoint is set.
 *    bomb  - a bomb symbol to indicate the point of segmentation fault.
 *
 *  To display a sign on a given line in the source window, it is first
 *  created and mapped.  To undisplay it, the sign is unmapped.  It can
 *  be mapped again when the sign is needed.  Note that the sign is never
 *  moved, so that there can be as many signs created (but not mapped) as
 *  the number of lines in the source window.
 *  For arrow and updown, there can be at most one of each mapped at a time.
 *  For stop, there can be more than one mapped at the same time.
 *  
 *  signs_init():    Initialize signs data structs.
 *  PutSign():       Create a symbol, calculate position, display it (static).
 *  UpdateStops():   Update all stops and display only the viewable ones.
 *  RemoveStops():   Remove the stop at a line num.
 *  ClearStops():    Remove all stops.
 *  UpdateArrow():   Unmap current arrow and redisplay.
 *  UpdateUpdown():  Unamap current updown and redisplay.
 *  UpdateBomb():    Unamap current bomb and redisplay.
 */

#include "global.h"
#include "bitmaps.h"
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Label.h>

#include <assert.h>

XImage		*ArrowImage, *UpDownImage, *StopImage, *BombImage;

Arrow 		arrow;
Updown 		updown;
Stops		stops[MAXSTOPS];	/* array of stops */
Bomb 		bomb;
Cardinal	nstops;			/* number of stops */

/* Initialize data structures */
void signs_init()
{
    int i;

    bzero(&arrow, sizeof(arrow));
    bzero(&updown, sizeof(updown));
    nstops = 0;
    bzero(&stops, sizeof(stops));
    bzero(&bomb, sizeof(bomb));
}

/* clear the window, thus forcing it to be redrawn... */
void UpdateSigns(file)
    FileRec *file;
{
    XClearArea(XtDisplay(sourceDrawingArea), XtWindow(sourceDrawingArea), 0, 0, 0, 0, True);
}  

/*  Create an arrow, updown, stop, or bomb XImage  */
static XImage *GetImage(parent, sign)
     Widget	parent;
     int        sign;
{
    Display	*display = XtDisplay(parent);
    Screen	*screen = XtScreen(parent);

    switch(sign) {
      case _ARROW_SIGN:
	if (! ArrowImage) {
	    ArrowImage = XCreateImage(display, DefaultVisualOfScreen(screen),
					1, XYBitmap, 0, INFO(arrow));
	    ArrowImage->bitmap_bit_order = LSBFirst;
	    ArrowImage->byte_order = LSBFirst;
	}
	return ArrowImage;
      case _UPDOWN_SIGN:
	if (! UpDownImage) {
	    UpDownImage = XCreateImage(display, DefaultVisualOfScreen(screen),
					1, XYBitmap, 0, INFO(updown));
	    UpDownImage->bitmap_bit_order = LSBFirst;
	    UpDownImage->byte_order = LSBFirst;
	}
	return UpDownImage;
      case _STOP_SIGN:
	if (! StopImage) {
	    StopImage = XCreateImage(display, DefaultVisualOfScreen(screen),
					1, XYBitmap, 0, INFO(stop));
	    StopImage->bitmap_bit_order = LSBFirst;
	    StopImage->byte_order = LSBFirst;
	}
	return StopImage;
      case _BOMB_SIGN:
	if (! BombImage) {
	    BombImage = XCreateImage(display, DefaultVisualOfScreen(screen),
					1, XYBitmap, 0, INFO(bomb));
	    BombImage->bitmap_bit_order = LSBFirst;
	    BombImage->byte_order = LSBFirst;
	}
	return BombImage;
      default:
	abort();
      }
}

/* Calculate the base line of text line [i] */
static Position CalculateImagePos(wid, line)
    Widget wid;
    int line;
{
    Position x = -1, y = -1;
  
    (void) XmTextPosToXY(wid, displayedFile->linepos[line], &x, &y);
    return y;
}

static void PutSign(wid, sign, file, line)
    Widget wid;
    int sign;
    char *file;
  int line; 
{
    static GC gcs[_NUM_SIGNS] = { 0 };
    GC gc;
    XImage *image;
    Position x, y;
    Dimension width;
    int offset;
  
    if (! gcs[0] ) {
        XGCValues values;
  
	XtVaGetValues(wid,
		      XmNbackground, &values.background,
		      NULL);

        values.foreground = app_resources.stop_color;
	gcs[_STOP_SIGN-1] = XtGetGC(wid, GCForeground|GCBackground, &values);
        values.foreground = app_resources.arrow_color;
	gcs[_ARROW_SIGN-1] = XtGetGC(wid, GCForeground|GCBackground, &values);
        values.foreground = app_resources.updown_color;
	gcs[_UPDOWN_SIGN-1] = XtGetGC(wid, GCForeground|GCBackground, &values);
        values.foreground = app_resources.bomb_color;
	gcs[_BOMB_SIGN-1] = XtGetGC(wid, GCForeground|GCBackground, &values);
      }
  
    switch (sign) {
    case _ARROW_SIGN:
        offset = 1; break;
    case _UPDOWN_SIGN:
	offset = 1; break;
    case _STOP_SIGN:
	offset = 0; break;
    case _BOMB_SIGN:
	offset = 1; break;
    }

    gc = gcs[sign-1];
    XtGetValue(wid, XmNwidth, &width);

    if (!strcmp(file, displayedFile->pathname)
     && line >= displayedFile->topline && line <= displayedFile->bottomline) {
        image = GetImage(wid, sign);
	y = CalculateImagePos(sourceWindow, line) - image->height + 3;
	x = width - ((offset + 1) * image->width) + 2;
        XPutImage(XtDisplay(wid), XtWindow(wid), gc, image, 0, 0,
		  x, y, image->width, image->height);
    }
}

/* someone might have clicked on a sign... in which case, we should set
 * the current line in the source file to the corresponding place.  if
 * we wanted to be fancy, we could also figure out if a symbol had been
 * selected (for moving or deleting).
 */
void ClickOnDrawingAreaCB(wid, client_data, call_data)
    Widget wid;
    caddr_t client_data;
    XmDrawingAreaCallbackStruct *call_data;
{
    Position x, y;
    XmTextPosition pos;
    int line;

    assert(call_data->reason == XmCR_INPUT);

    switch (call_data->event->type) {
    case ButtonPress:
      break;
    case ButtonRelease:
      x = ((XButtonEvent *) (call_data->event))->x;
      y = ((XButtonEvent *) (call_data->event))->y;
      /* we won't bother verifying that a sign was indeed under the mouse */
      if (displayedFile) {
	pos = XmTextXYToPos(sourceWindow, 0, y);
	    
	XmTextSetInsertionPosition(sourceWindow, pos);
      }
      break;
   default:
      break;
    }
}

/* Redraw signs (callback for sourceDrawingArea widget) */
void DrawSigns(wid, client_data, call_data)
    Widget wid;
    caddr_t client_data;
    XmDrawingAreaCallbackStruct *call_data;
{
    int line, i;

    if (! displayedFile) return;

    PutSign(wid, _ARROW_SIGN, arrow.file, arrow.line);

    for (i = 1; i <= nstops; i++)
      if (stops[i].file != NULL)
	PutSign(wid, _STOP_SIGN, stops[i].file, stops[i].line);

    PutSign(wid, _BOMB_SIGN, bomb.file, bomb.line);

    PutSign(wid, _UPDOWN_SIGN, updown.file, updown.line);
}

/*
 * Given a line number, unmap the stop sign associated with that line.
 */
void RemoveStop(line)
int line;
{
    if (displayedFile
	&& line >= displayedFile->topline && line <= displayedFile->bottomline) 
	  UpdateSigns(displayedFile);
}

void ClearStops()
{
    bzero(&stops, sizeof(stops));
    nstops = 0;				/* necessary? */
}
