/* $Id: gdb_handler.c,v 1.2 1992/01/17 23:02:08 jtsillas Exp $ */

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  gdb_handler.c
 *
 *	WARNING : gdb_handler.c is included by handler.c for GDB.
 *
 *    Contain action handlers for the parser to invoke upon a gdb command.
 *
 *    updown_handler():		Update file, line label, updown arrow position.
 *    debug_handler():		Check directory use list, display main 
 *                              source file.
 *    pwd_handler():		Update current working directory.
 *    search_handler():		Adjust source file to display matched line.
 *    display_info_handler():   Update display window.
 *    break_handler():		Place stop sign on line or function 
 *                              or address specified.
 *    info_dir_handler():	Update search directory list. 
 *    directory_handler():	Update search directory list. 
 *    list_handler():		Adjust source file to display result. 
 *    info_line_handler():	Update current file. 
 *    info_source_handler():    Update source file information.
 *    clear_handler():		Remove stop sign.
 *    display_handler():	Update display window.
 *    info_break_handler():	Update stop signs.
 *    cd_handler():			Record current working directory.
 *    frame_curr_handler():	Update current function name.
 *    exec_handler():		Update file, line label, arrow position.
 *    done_handler():		Program execution completed, clear breakpoints.
 *    source_handler():		Exec commands of source file specified.
 *    query_gdb_echo():		Send command with echo on.
 *    HelpButtonActivate():     Help selection callback (static).
 */

#include <ctype.h>
#include "global.h"
#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>

#ifdef SYSV 
#   include <signal.h>
#endif

static void HelpButtonActivate(Widget, XtPointer, XtPointer);

void info_source_handler(char *);

/*  
 *  Display an outlined arrow to locate the calling routine in a stack
 *  frame. The appropriate file with the calling routine is displayed 
 *  and the file variable is set accordingly.
 */
void updown_handler()
{
    char command[LINESIZ], *func, *file;
    int	 line;

    line = Token.line;
    func = XtNewString(Token.func);
    if (line <= 0) line = 1;
    LoadCurrentFile();
    if (displayedFile)
	file = displayedFile->pathname;
    if (line <= 0 || func == NULL || file == NULL)
    	{
    	XtFree(func);
		return;
		}
		
    if (displayedFile && strcmp(file, displayedFile->pathname)) {
	LoadFile(file);
    }
    updown.line = line;
    strcpy(updown.func, func);
    if (displayedFile)
    	strcpy(updown.file, displayedFile->pathname);
    AdjustText(line);
    XtFree(func);
}

void debug_handler()
{
  /* debug_handler is executed at 
     start-up and with 'symbol-file' 
     command */

  info_source_handler(NULL);    /* say no current compilation directory */

  query_gdb("set height 0\n");
  query_gdb("set print pretty\n");
  query_gdb("show directories\n");

  displayedFile = NULL;		/* force reloading of source file
				   here we use query_gdb_echo instead 
				   of query_gdb so that any
				   error message will be displayed ! */
  
  query_gdb_echo("list ,main\n");	/* tell gdb to use main file 
 					   and get line number of main(). 
					   (,main will end at main) */
  
  if (LoadCurrentFile() == 0)
    {
      arrow.line = 0;			/* clear arrow sign */
      updown.line = 0;		/* clear updown sign */
      bomb.line = 0;			/* clear bomb sign */
      ClearStops();
      UpdateSigns(displayedFile);
    }
  
  UpdateMessageWindow("Ready for execution", NULL);
  query_gdb("display\n");		        /* clear display window */
}

void pwd_handler(s)
     char *s;
{
  strcpy(cwd, (char *)strtok(s, "\n"));
}

void search_handler()
{
  AdjustText(Token.line);
}

/*  Show output on the display window. If output is null but the 
 *  display window is managed, replace contents of the display window 
 *  with the null string.
 */
void display_info_handler(popup_req)
     int popup_req;
{
  Arg args[MAXARGS];

  if (!Token.display || !strcmp(Token.display, "")) {
    if (!XtIsManaged(displayFD))
      return;
    else {
      XtFree(Token.display);
      Token.display = XtNewString("");
    }
  }
  if (!XtIsManaged(displayFD) && popup_req)
    XtManageChild(displayFD);
  
  XtSetArg(args[0], XmNvalue, (XtArgVal) Token.display);
  XtSetValues(displayWindow, args, 1);
  XtFree(Token.display);
  Token.display = 0;
}

/*  Place a stop sign next to the line specified on the source file window 
 *  if it is to be viewable.
 */
void break_handler()
{
  char string[LINESIZ];
  char file[MAXNAME];
  int line;
  int stop;
  
  if (Token.stop == 0 || Token.line == 0 || Token.file == 0)
    return;

  strcpy(file, Token.file);
  line = Token.line;
  stop = Token.stop;
  
  if (Token.stop >= MAXSTOPS)	/* see MAXSTOPS in signs.c */
    {
      fprintf(stderr,"Too many breakpoints\n");
      return;
    }
  

  /* make sure the file we want is currently setting $cdir */
  
  sprintf(string, "list %s:%d,%d\n", file, line, line);
  query_gdb(string);
  
  /* load & display file if none is displayed */
	
  if ((stops[stop].file = GetPathname(file)) == NULL)
    return;
  
  if (displayedFile == NULL)
    {
      LoadFile(file);
      AdjustText(line);
    }
  
  stops[stop].line = line;
  stops[stop].tag = 0;
  nstops = stop;
  
  /* display breakpoint sign if file is displayed */
  
  if (displayedFile)
    {
      if (!strcmp(stops[stop].file, displayedFile->pathname))
	UpdateSigns(displayedFile);
    }
}

/*  info directories 
 */
void info_dir_handler()
{
  if (Token.file)
    MakeDirList(Token.file);
}

void directory_handler()
{
  query_gdb("show directories\n");
}

void list_handler()
{
  int	 line;
  
  line = Token.line;
  
  if (line)
    {
      /* We will display the last line listed. 
	 Since we used 'list ,main' we will effectively 
	 display main in that case. */
      
      LoadCurrentFile();
      AdjustText(line);
    }
  else
    {
      AppendDialogText("Error list command\n");
      bell(0);
    }
}

void info_line_handler() 	/* Command was 'info line' */
{
  if (Token.file)
    strcpy(CurrentFile, Token.file);
  else
    strcpy(CurrentFile, "");
}

/*
 *  Clear handler remove the stop specified and undisplayed the stopsign
 *  if it's visible. It calls the gdb status command to find out what 
 *  stops are left, and then update the array of stops accordingly.
 */
void clear_handler()
{
  query_gdb("info break\n");	/* update breakpoints */
}

void display_handler()	/* display or undisplay */
{
  query_gdb("display\n");	/* update display */
}

void info_break_handler(output_string)
     char *output_string;
{
  int  i; 
  int	 line;
  char c;
  char type[20],disp[20],enb[20];
  
  if (!output_string)
    return;
  
  if (strncmp(output_string,"Num Type",8)) { /* Pre 4.5 */
    while(*output_string) {
      if (*(output_string++) == '#') {
	if (sscanf(output_string, "%d %c", &i,&c) == 2)
	  if (i > 0 && i <= nstops && stops[i].line > 0 && c == 'y') 
	    stops[i].tag = 1;
      }
    }
  } else { /* for 4.5 Display (Ken Mandelberg, km@mathcs.emory.edu) */
    while(*output_string) {
      if (*(output_string++) == '\n') {
	if (sscanf(output_string, "%d%s%s%s", &i, type, disp, enb) == 4)
	  if (i > 0 && i <= nstops && stops[i].line > 0 &&
	      !strcmp(type,"breakpoint") && !strcmp(enb,"y")) 
	    stops[i].tag = 1;
      }
    }
  }
  
  for (i=1; i<=nstops; i++)
    if (stops[i].line > 0) {
      if (stops[i].tag)
	stops[i].tag = 0;
      else {
	line = stops[i].line;
	stops[i].line = 0;
	stops[i].file = NULL;
	if (LineToStop_no(line) == 0)
	  RemoveStop(line);
      }
    }
}

void cd_handler(s)
     char *s;
{
  strcpy(cwd,s);
}

/* this handler justs update the function name.
   Because the function name is not always displayed
   after next,step ... */
static char* funcname = 0;
void frame_curr_handler()
{
  if (Token.func == NULL)
    return;
  
  if (funcname)
    {
      XtFree(funcname);
      funcname = 0;
    }
  
  funcname = XtNewString(Token.func);
}

/*  Handle gdb output of run, cont, next, step, return commands.
 *  Result of output parsing is returned in a set of tokens.
 *
 *  If message is not 0, this is an important message and should
 *  be displayed instead of Token.mesg.
 *  This message will hold the Bus error and segmentation violation errors.
 *  signal is the signal number received (if any).
 */
void exec_handler(message, signal)
     char *message;
     int signal;
{
  int line, status;
  char *func;
  
  /* Print "stopped in ..." line in message window 
   * Adjust text displayed
   */
  if (Token.line == 0) 
    return; 
  
  if (message)
    UpdateMessageWindow(message, NULL);
  else
    UpdateMessageWindow(Token.mesg, NULL);
  
  line = Token.line;
  func = (Token.func) ? XtNewString(Token.func) : 0;
    
  if (Token.file)
    status = LoadFile(Token.file);
  
  display_info_handler(FALSE);		/* uses Token.display ! */
  
  /* because of tbreak, we have to call info break here */
  
  query_gdb("info break\n");	/* update breakpoints */
  
  if (func == NULL)
    {
      query_gdb("frame\n"); /* this will just update funcname 
			       (see frame_curr_handler) */
      func = funcname;
      if (func == NULL)
	return;
      funcname = 0;	/* tell frame_curr_handler WE 
			   are going to XtFree it */
    }
		
  arrow.line = line;			/* update arrow sign position */
  strcpy(arrow.func, func);
  
  updown.line = 0;			/* remove updown, if any */
  if (displayedFile) {
    strcpy(arrow.file, displayedFile->pathname);
  }
    
  /* Display bomb sign if segmentation fault occurs in source code */
  
  if (status != -1 && message && signal == SIGSEGV) {
    arrow.line = 0;
    bomb.line = line;
    if (func)
      strcpy(bomb.func, func);
    if (displayedFile) strcpy(bomb.file, displayedFile->pathname);
  }
  else
    bomb.line = 0;
  
  AdjustText(line);
  UpdateFunctionLabel(func);
  XtFree(func);
}

/*  Remove all the arrow and updown signs, print message, then 
 *  change the file variable to the file name displayed.
 */
void done_handler(message,signal)
     char *message;
     int signal;
{
  char command[LINESIZ];

  arrow.line = 0;
  updown.line = 0;
  UpdateSigns(displayedFile);
  UpdateMessageWindow("Ready for execution", NULL);
}

void source_handler()
{
  char *file;
  FILE *fp;
  char s[LINESIZ];
  
  if (!Token.file || !strcmp(Token.file, ""))
    {
      XtFree(Token.file);
      Token.file = XtNewString(".gdbinit");	/* default is .gdbinit */
    }
 
  if (fp = fopen(Token.file, "r")) 
    {
      while (fgets(s, LINESIZ, fp))
	{
	  /* DO NOT SEND \n and Take care of source command */
	  if ((*s != '#') && strcmp(s,"\n"))
	    {
	      if ((!gdb_source_command(s,TRUE)) 
		  && (!gdb_define_command(s,fp)))	
		{
		  write_gdb(s);
		  insert_command(s);
		  AppendDialogText(s);
		}
	      Prompt = False;
	      while (!Prompt)
		read_from_gdb();
	    }
	}
      close((int)fp);
    }
}

/*  Sends a command to gdb and read the corresponding output through
 *  read_from_gdb().
 *
 *	Same as query_gdb() in gdb.c except that Echo = True.
 */
void query_gdb_echo(command)
     char *command;
{
  write_gdb(command);
  insert_command(command);
  
  Echo = True;
  Prompt = False;
  while (!Prompt)
    read_from_gdb();
  
  Parse = True;	/* Always reset Parse and Echo to True */
  Echo = True;
}

Widget helpbuttons[NHELPBUTTONS];
char callbackargs[NHELPBUTTONS][HELP_SIZE];
Widget helplabels[NHELPLABELS];
char helpstack[HELP_LEVELS][HELP_SIZE];
int helpstackidx=0;
char help_buttons_use_flag = 1;

static void HelpButtonActivate(w, helpname, call_data)
     Widget w;
     XtPointer helpname;
     XtPointer call_data;
{
  char command[LINESIZ];
  strcpy(helpstack[helpstackidx+1], /* I put the name here so */
	 (char*) helpname);	    /* that I can return later. No name is */
  ++helpstackidx;		    /* necessary for the toplevel. */
  sprintf(command, "help %s\n", (char*) helpname);
  help_buttons_use_flag = 1;
  query_gdb(command);
  help_buttons_use_flag = 0;
}

/*
 *  I've tried to make this function as efficient as possible in terms of
 *  processing and memory. This is why it is not very readable. I use two
 *  Widget lists to track all the Widgets I create as buttons or labels.
 *  This functions will not destroy any Widgets once it creates them
 *  because it is an expensive operation to create so many and destroy them.
 *  Instead I recycle them wherever I can using a high water mark. The
 *  only thing that may break this function is if GNU decides to add many
 *  more help entries and we run out of Widget pointers or if GNU decides
 *  to throw out their current format (using "--" to separate names from
 *  description.
 */
void help_handler(command,output)
char *command;
char *output;
{
  char *p = output;
  char buttonstring[LINESIZ];
  char labelstring[LINESIZ];
  char *useforlabel;
  int genindex;
  static int nlabel=0;		/* These remain static to tell us how */
  static int nbutton=0;		/* many widgets to unmanage and reuse */

  if(!XtIsManaged(helpFD))	/* Popup the help Dialog */
    XtManageChild(helpFD);

  if(!help_buttons_use_flag)    /* Reset SP if help was from dialog box cmd */
    helpstackidx = 0;		/* or from pull-down menu */

  XtUnmanageChild(helpselectscroll); /* Do this to keep form from resizing */

  if(helpstackidx == 1)		/* The SP tells us when to draw the backup */
    {				/* button and label. */
      XtManageChild(helpupbutton);
      XtManageChild(helpupbuttonlabel);
    }
  if(helpstackidx == 0) 
    {
      XtUnmanageChild(helpupbuttonlabel);
      XtUnmanageChild(helpupbutton);
    }

  for(genindex = 0; genindex < nlabel; genindex++) /* Unmanage the used */
      XtUnmanageChild(helplabels[genindex]); /* labels and buttons */
  nlabel = 0;
  for(genindex = 0; genindex < nbutton; genindex++)
      XtUnmanageChild(helpbuttons[genindex]);
  nbutton = 0;
		
  while(*p)		/* Build the labels from the gdb help  */
    {
      int buttonindex = 0;
      int labelindex = 0;
      Boolean isbutton = False;

      while(*p != '\n')
	{
	  if(isbutton == False)
	    buttonstring[buttonindex++] = *p;
	  else
	    labelstring[labelindex++] = *p;
	  p++;
	  if((*p == '-') && (*(p+1) == '-'))
	    {
	      isbutton = True;
	      buttonstring[buttonindex-1] = '\0';
	    }
	}
      if(isbutton == True)
	{
	  static Arg buttonargs[] = {
	    { XmNleftAttachment, XmATTACH_FORM },
	    { XmNtopAttachment, XmATTACH_WIDGET },
	    { XmNhighlightOnEnter, True },
	    { 0, 0 },
	    { 0, 0 },
	  };

	  labelstring[labelindex] = '\0';
	  useforlabel = labelstring;
	  strcpy(callbackargs[nbutton], buttonstring);

	  XtSetArg(buttonargs[3], XmNlabelString, XmStrMk(buttonstring));
	  if(!nbutton)
	    XtSetArg(buttonargs[4], XmNtopWidget, helplabels[nlabel-1]);
	  else
	    XtSetArg(buttonargs[4], XmNtopWidget, helpbuttons[nbutton-1]);
	  if(!helpbuttons[nbutton])
	    {
	      helpbuttons[nbutton] = 
		XtCreateManagedWidget("helpbutton", xmPushButtonWidgetClass,
				      helpselectscroll, buttonargs, 5);
	    }
	  else
	    {
	      XtRemoveCallback(helpbuttons[nbutton], XmNactivateCallback,
			       HelpButtonActivate, callbackargs[nbutton]);
	      XtSetValues(helpbuttons[nbutton], buttonargs, 5);
	      XtManageChild(helpbuttons[nbutton]);
	    }
	  XtAddCallback(helpbuttons[nbutton], XmNactivateCallback,
			HelpButtonActivate, callbackargs[nbutton]);
	  nbutton++;
	}
      else
	{
	  buttonstring[buttonindex] = '\0';
	  useforlabel = buttonstring;
	}
      {
	Arg *labelargs;
	Cardinal nargs;

	if(isbutton == True)
	  {
	    static Arg labelargs1[] = {
	      { XmNrightAttachment, XmATTACH_FORM },
	      { XmNleftAttachment, XmATTACH_WIDGET },
	      { XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET },
	      { XmNtopAttachment, XmATTACH_NONE }, 
	      { XmNalignment, XmALIGNMENT_BEGINNING },
	      { 0, 0 },
	      { 0, 0 },
	      { 0, 0 },
	    };

	    XtSetArg(labelargs1[5], XmNbottomWidget, helpbuttons[nbutton-1]);
	    XtSetArg(labelargs1[6], XmNleftWidget, helpbuttons[nbutton-1]);
	    nargs = 7;
	    labelargs = labelargs1;
	  }
	else
	  {
	    static Arg labelargs2[] = {
	      { XmNrightAttachment, XmATTACH_FORM },
	      { XmNleftAttachment, XmATTACH_FORM },
	      { XmNbottomAttachment, XmATTACH_NONE },
	      { XmNalignment, XmALIGNMENT_CENTER },
	      { 0, 0 },
	      { 0, 0 },
	      { 0, 0 },
	    };

	    if(!nlabel) 
	      {
		XtSetArg(labelargs2[4], XmNtopAttachment, XmATTACH_FORM);
		nargs = 5;
	      }
	    else
	      {
		XtSetArg(labelargs2[4], XmNtopAttachment, XmATTACH_WIDGET);
		XtSetArg(labelargs2[5], XmNtopWidget, helplabels[nlabel-1]);
		nargs = 6;
	      }
	    labelargs = labelargs2;
	  }
	XtSetArg(labelargs[nargs], XmNlabelString, XmStrMk(useforlabel));
	nargs++;
	if(!helplabels[nlabel])
	  helplabels[nlabel] = XtCreateManagedWidget("helplabel",
						     xmLabelWidgetClass,
						     helpselectscroll,
						     labelargs, nargs);
	else
	  {
	    XtSetValues(helplabels[nlabel], labelargs, nargs);
	    XtManageChild(helplabels[nlabel]);
	  }
      }
      nlabel++;
      p++;
    }
  XtManageChild(helpselectscroll);
}

/*
 * Update source file information
 *
 * input : current compilation directory
 *         (returned from 'info source' gdb command).
 * output : none.
 */
 
char source_file[MAXPATHLEN+1];             /* The source file */

void info_source_handler(src)
     char *src;
{
  if (src == NULL) {
    source_file[0] = '\0';
  }
  else {
    strncpy(source_file, src, MAXPATHLEN);
    source_file[MAXPATHLEN] = '\0';
  }
}


