static char rcsid[] = "$Id: callgdb.c,v 1.5 1992/04/01 14:22:19 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  callgdb.c
 *
 *    Set up communication between gdb and mxgdb using pseudo terminal, and
 *    call gdb.
 *
 *    open_master():	Open the master side of pty (static).
 *    open_slave(): 	Open the slave side of pty (static).
 *    callgdb(): 	Invoke the child process.
 *    callxtty():       Invoke the slave tty for the inferior gdb process.
 */

#include "global.h"
#include <sys/types.h>

#ifdef HAVE_TERMIO
#  include <termio.h>
#else
#  include <sgtty.h>
#endif

#if defined(hpux) || (defined(sun) && defined(SVR4))
#  include <sys/file.h>
#endif

#ifdef SCO
#  include <sys/fcntl.h>
#endif

#ifdef STR_PTY
#  include <sys/stropts.h>
#  include <signal.h>
#endif

#if defined(PTC_PTY)
#  include <sys/stropts.h>
#  include <sys/stat.h>
#  include <sys/sysmacros.h>
#elif defined(sgi)
#  include <sys/stat.h>
#endif

static int open_master(void);
static int open_slave(void);

#ifdef STR_PTY
FILE *gdbfpin = NULL, *gdbfpout = NULL;
#else
FILE *gdbfp = NULL;     /* file pointer to gdb */
#endif
int gdbpid = 0;	        /* gdb process id */
int xttypid = 0;        /* xtty process id */
static int gdbInputId;  /* gdb input id */

#if defined(PTC_PTY)
static char pty[] = "/dev/ptc";         /* clone dev of pseudo-terminal */
static char tty[15];                    /* slave side of pseudo-terminal */
int pts_master;                         /* master filedes */
char gdbtty[15];
#endif

#if defined(IRIX4_PTY)
char *_getpty(int*, int, mode_t, int);
static char tty[15];
static char pty[15];
char gdbtty[15];
#endif

#ifdef STR_PTY
static char pty[] = "/dev/ptmx";
static char tty[15];                      /* slave side of pseudo-terminal */
int pts_master;                         /* master filedes */
char gdbtty[15];
#endif

#if !defined(STR_PTY) && !defined(PTC_PTY) && !defined(IRIX4_PTY)
static char 	pty[11] = "/dev/pty??";	/* master side of pseudo-terminal */
static char 	tty[11] = "/dev/tty??";	/* slave side of pseudo-terminal */
char gdbtty[11] = "/dev/tty??";
#endif

static int hold_slave = -1;
extern char	*gdbprompt;

/*
 *  mxgdb talks to gdb through a pseudo terminal which is a pair of master
 *  and slave devices: /dev/pty?? and /dev/tty??, where ?? goes from p0 to
 *  sf (system dependent).  The pty is opened for both read and write. Some 
 *  systems use SYSV STREAMS based pty's. For these define STR_PTY. Some use
 *  /dev/ptc based pty's, for those use PTC_PTY.
 */
static int open_master()
{
  int  i, master, fd;
  char c;

#ifdef IRIX4_PTY
  char *line;
  SIG_PF oldintr;
#endif
  
#if defined(PTC_PTY)
  if ((pts_master = open(pty, O_RDWR)) >= 0)
    {
      struct stat sb;
      
      if (fstat(pts_master, &sb) < 0)
	close(pts_master);
      else
	{
	  hold_slave = minor(sb.st_rdev);
	  return(pts_master);
	}
    }
#endif

#ifdef IRIX4_PTY
  if ((oldintr = signal(SIGCHLD, SIG_DFL)) == SIG_ERR)
    {
      perror("bad return from signal");
      exit(1);
    }
  line = _getpty(&master, O_RDWR | O_NDELAY, 0600, 0);
  if (signal(SIGCHLD, oldintr) == SIG_ERR)
    {
      perror("bad return from signal");
      exit(1);
    }
  if(line)
    {
      strcpy(tty, line);
      strcpy(gdbtty, line);
      return(master);
    }
#endif

#ifdef STR_PTY
  if ((pts_master = open(pty, O_RDWR)) >= 0)
    {
      grantpt(pts_master);
      unlockpt(pts_master);
      return(pts_master);
    }
#endif
  
#if !defined(STR_PTY) && !defined(PTC_PTY) && !defined(IRIX4_PTY)
  for (c='p'; c<'z'; c++) 
    {
      for (i=0; i<16; i++) 
	{
	  pty[8] = c;
	  pty[9] = "0123456789abcdef"[i];
	  tty[8] = c;
	  tty[9] = pty[9];
	  /*
	   * I need to check that tty is not the same device we are using
	   * for gdb's pseudo-terminal. Xtty has to find its own. If this
	   * it the first pseudo-tty then gdbtty[0] is "??" otherwise it
	   * keeps the last letters of gdb's tty.
	   */
	  if (strcmp(&gdbtty[8], &tty[8]) &&
	      (master = open(pty, O_RDWR)) >= 0)
	    {
	      if ((hold_slave = open(tty, O_RDWR)) >= 0)
		return (master);
	      else close(master);
	    }
	}
    }
#endif
  
  fprintf(stderr, "mxgdb: all ptys in use\n");
  return(-1);
}

static int open_slave()
{
  int           pts_slave;

#ifdef STR_PTY
  char          *ptsname();
  char          *slavename;

  slavename = ptsname (pts_master);	/* get name of slave */
  strcpy (tty, slavename);		/* copy over to save area */
  strcpy (gdbtty, slavename);
  if ((pts_slave = open(tty, O_RDWR)) >= 0)
    {
      /*XXX todo: need to make sure these succeed */
      ioctl (pts_slave, I_PUSH, "ptem");
      ioctl (pts_slave, I_PUSH, "ldterm");
      ioctl (pts_slave, I_PUSH, "ttcompat");
      return(pts_slave);
    }
#endif

#if defined(PTC_PTY)
  sprintf(tty, "/dev/ttyq%d", hold_slave);
  if ((pts_slave = open(tty, O_RDWR)) >= 0) {
    strcpy(gdbtty, tty);
    return(pts_slave);
  }
#endif

#ifdef IRIX4_PTY
  if ((pts_slave = open(tty, O_RDWR)) >= 0) {
    strcpy(gdbtty, tty);
    return(pts_slave);
  }
#endif

#if !defined(STR_PTY) && !defined(PTC_PTY) && !defined(IRIX4_PTY)
  return(hold_slave);
#else
  fprintf(stderr, "mxgdb: failed to open slave pty\n");
  return(-1);
#endif

}

/*
 * callxtty - function which finds an unused pseudo-tty pair, invokes
 * a slave xterm using -S and returns the master device for use by the
 * child gdb. This function is called from XttyCallback.
 *
 */
char *callxtty()
{
  char *argv[10];
  char arg1buf[LINESIZ];
  char arg3buf[LINESIZ];
  int master, slave;

#if defined(STR_PTY)
  fprintf(stderr, "mxgdb: Xtty not supported on this sytem in this release\n");
  return NULL;
#else

  /*
   * Opening the master and slave pty ensures it is not being used
   */
  if((master = open_master()) == -1)
    return NULL;
  if((slave = open_slave()) == -1)
    return NULL;

  xttypid = fork();
  if (xttypid == -1) {
    perror("mxgdb error: cannot fork process");
    return(NULL);
  }
  else if (xttypid) { 
    close(master);
    close(slave);
    sleep(1);
    return(tty);
  }
  else {

    /* 
     * The xterm slave process "inherits" the slave filedes so we set
     * the tty before invoking the slave xterm.
     */
#ifdef HAVE_TERMIO
    struct termio Termio;

    ioctl(slave, TCGETA, &Termio);
#ifdef SCO
    Termio.c_cflag |= PARENB|CS8|HUPCL|CREAD;
    Termio.c_iflag |= IXON;
#else
    Termio.c_cflag |= CS8;
    Termio.c_iflag |= ICRNL|IGNPAR|BRKINT;
#endif
    Termio.c_lflag |= ECHO|ISIG|ICANON|ECHONL|ECHOE|ECHOK;
    Termio.c_oflag |= ONLCR|OPOST;
    ioctl(slave, TCSETA , &Termio);
#else
    /*
     * I'm not too sure what should go in here. Is there a BSD hack out
     * in netland who can help? Basically I wish to do exactly what I
     * did for SYSV above. This may cause problems in BSD systems wishing
     * to use Xtty.
     */
    struct sgttyb tty_data;

    ioctl(slave, TIOCGETP, &tty_data);
    tty_data.sg_flags |= CRMOD|ECHO;
    ioctl(slave, TIOCSETP, &tty_data);
#endif

    argv[0] = "xterm";
    argv[1] = arg1buf;
    argv[2] = "-title";
    argv[3] = arg3buf;
    argv[4] = "-tn";
    argv[5] = "xterm";
    argv[6] = NULL;

    sprintf(arg1buf, "-S%c%c%d", tty[8], tty[9], slave);
    sprintf(arg3buf, "Mxgdb Xtty: %s", tty);
    execvp("xterm", argv);
    fprintf(stderr, "mxgdb error: cannot exec xterm");
    exit(2);
  }
#endif /* !STR_PTY && !PTC_PTY && !IRIX4_PTY */
}

void callgdb(argc, argv)
     int argc;
     char *argv[];
{
  int fd = -1;
  int master;		/* file descriptor of master pty */
  int slave = -1; 	/* file descriptor of slave pty */
  char *debugger; 	/* name of executable debugger */
  pid_t pid;
  int pgrp;
#ifdef HAVE_TERMIO
  struct termio Termio;
#else
  struct sgttyb tty_data;
#endif
  
  debugger = (char *) getenv("MXGDB_DEBUGGER");	/* first looks up env var */
  gdbprompt = (char *) getenv("GDB_PROMPT");

  if (debugger == NULL || *debugger == '\0')
    debugger = DEBUGGER;
  
  /* construct gdb prompt string based on the name of debugger invoked */
  if (gdbprompt == NULL || *gdbprompt == '\0')
    gdbprompt = GDBPROMPT;

    /*
     * Clear controlling tty.  Do this now, so that open_slave and
     * open_master will cause the selected pty to become the
     * controlling tty.
     */

#if defined(SVR4)	/* (MJH) */
    if ((tcgetsid(0) != tcgetpgrp(0)) && /* Check if fore- or back-ground  */
	(fd = open("/dev/tty", O_RDWR|O_NDELAY)) > 0) {
  	ioctl(fd, TIOCNOTTY, 0);
  	close(fd);
  	}
#else	/* not SVR4 */
	if ((fd = open("/dev/tty", O_RDWR)) > 0) {
#ifndef SYSV 
  	ioctl(fd, TIOCNOTTY, 0);
#endif /* SYSV */
  	close(fd);
  	}
#endif	/* SVR4 */

  if((master = open_master()) == -1)
    exit(2);
  
  gdbpid = fork();
  if (gdbpid == -1) {
    perror("mxgdb error: cannot fork process");
    exit(2);
  }
  else if (gdbpid) {
    /* 
     * Parent : close the slave side of pty
     *	    close stdin and stdout
     *	    set the gdb file descriptor to nonblocking mode
     *	    open file pointer with read/write access to gdb
     *	    set line buffered mode
     *	    register gdb input with X
     */
    if (slave >= 0) {
      close(slave);
    }
    close(0);
    close(1);

    fcntl(master, F_SETFL, FNDELAY);
#ifdef STR_PTY
    gdbfpin = (FILE *)fdopen(master, "r");
    gdbfpout = (FILE *)fdopen(master, "w");
#else
    gdbfp = (FILE *)fdopen(master, "r+");
#endif

#ifdef STR_PTY
#ifdef SYSV
    setvbuf(gdbfpin, NULL, _IONBF, 0);
    setvbuf(gdbfpout, NULL, _IONBF, 0);
#else /* SYSV */
    setbuf(gdbfpin, NULL);
    setbuf(gdbfpout, NULL);
#endif /* SYSV */
#else /* STR_PTY */
#ifdef SYSV
    setvbuf(gdbfp, NULL, _IONBF, 0);
#else /* SYSV */
    setbuf(gdbfp, NULL);
#endif /* SYSV */
#endif /* STR_PTY */

    gdbInputId = XtAppAddInput(app_context, master, 
			       (XtPointer) XtInputReadMask, 
			       ReadGdbCallback, NULL);

    /*
     * We make a copy of the master in case we wish to restore the pty
     * after creating an Xtty
     */
#if !defined(STR_PTY) && !defined(PTC_PTY) && !defined(IRIX4_PTY)
    strcpy(gdbtty, pty);
#endif
  }
  else {
    /*
     * Now set the process group of the terminal and of us
     * to our process id.  This clears us from the control
     * of the other process group. We also free ourselves from 
     * the controlling tty.
     */
#if (defined(SYSV) || defined(SYSV_SETPGRP)) && !defined(SVR4)
    setpgrp();
#endif

    /*
     * We open_slave after the fork() as it will become our
     * controlling tty by default since we just did a setsid()/setpgrp().
     */
    if((slave = open_slave()) == -1)
      exit(2);

    /*
     * Modify local and output mode of slave pty.
     */
#ifdef HAVE_TERMIO
    ioctl(slave, TCGETA, &Termio);
    Termio.c_lflag &= ~ECHO;        /* No echo */
    Termio.c_oflag &= ~ONLCR;	    /* Do not map NL to CR-NL on output */
    Termio.c_cc[VINTR] = 3;
    ioctl(slave, TCSETA, &Termio);
#else
    ioctl(slave, TIOCGETP, &tty_data);
    tty_data.sg_flags &= ~ECHO;     /* No echo */
    tty_data.sg_flags &= ~CRMOD;    /* Do not map NL to CR-NL on output */
    ioctl(slave, TIOCSETP, &tty_data);
#endif

    dup2(slave, 0);
    dup2(slave, 1);
    dup2(slave, 2);
    if (slave > 2)
      close(slave);
    close(master);

    fcntl(1, F_SETFL, FAPPEND);
#ifdef SYSV
    setvbuf (stdout, NULL, _IONBF, 0);
#else
    setbuf (stdout, NULL);
#endif

	/*
	 * Set our process group to that of the terminal,
	 * so we can change the group of the terminal.
	 */
#ifndef SYSV
	ioctl(0, TIOCGPGRP, &pgrp);
	setpgrp(0, pgrp);

	/*
	 * Now set the process group of the terminal and of us
	 * to our process id.  This clears us from the control
	 * of the other process group.
	 */
	pid = getpid();
	ioctl(0, TIOCSPGRP, &pid);
	setpgrp(0, pid);
#endif /* not SYSV */

#ifdef SVR4						/* (MJH) */
	tcsetpgrp(0, tcgetpgrp(0));
	tcsetpgrp(0, getpid());
#endif /* SVR4 */

    argv[0] = debugger;
    execvp(debugger, argv);
    fprintf(stderr, "mxgdb error: cannot exec %s", debugger);
    exit(2);
  }
}

