// window.c : basic source file for the Grafix package
// the "GNU Public Lincense" policy applies to all programs of this package
// (c) Wolfgang Koehler, wolf@first.gmd.de, Dec. 1994
//     Kiefernring 15
//     14478 Potsdam, Germany
//     0331-863238

#include "X11/cursorfont.h"

#include <stdlib.h>
#include <stdio.h>

#include "window.h"
#include "eventnames.h"  // from Xlib Manual

#define TABSIZE 0x200   // max 512 entrys

void * wid_ptr[TABSIZE]; // table of this pointers for CallBack indexing

// "thisW" transforms Window-ID to address in the table wid_ptr
// returns the address of this-pointer inside the table !!
// so setting is possible eg : void ** tw = thisW(Win); *tw = this;
void ** thisW(unsigned Win)  
{  return &wid_ptr[Win % TABSIZE]; };

void error(char * message) {
  printf("fatal error: %s\n",message); exit(1); }

// zB. fuer menu_bars : Platz fuer picture in den Name-string einbauen
char * ext_string(char * Name) { 
  char * nn = new char[strlen(Name) + 3]; // delete unnoetig, marginal
  sprintf(nn," %s ",Name); return nn; } // _Name_

Display * display;
int screen;
GC gc_copy, gc_xor, gc_clear, gc_rubber; // some often used gcs
XFontStruct * fixed_fn;       // and fonts
Cursor watch_cursor;
Cursor text_cursor;  // for edit_windows

unsigned depth;
unsigned black, white;

// convinience functions
GC CreateGC(unsigned long mask, XGCValues * gcv)
{ return XCreateGC(display, DefaultRootWindow(display), mask, gcv); }

// set default gc for line drawing etc.
void set_color(int color) {
   static XGCValues values; 
   values.foreground = color;
   XChangeGC(display,gc_copy, GCForeground, &values);
}

Colormap def_cmap;

int alloc_color(unsigned red, unsigned green, unsigned blue) {
 // allokieren eines cmap-entries fuer die geg. Farbe -> returnt pixel-Wert
  // -1 fuer Versagen : kein freier entry
  XColor col = { 0, red, green, blue };
  if (XAllocColor(display,def_cmap,&col) == 0) {   
    printf(" Warning : Color map full (%x,%x,%x) \n",red,green,blue); 
 /* Suche nach nahegelegener Farbe : klappt nicht (int Arithmetik!)  
    int pix, pmin; float t, min = 1.0e12;
    for (pix=0; pix <256; pix++) { 
      col.pixel = pix; XQueryColor(display,def_cmap,&col); 
      t = SQR(red-col.red) + SQR(green-col.green) + SQR(blue-col.blue);
      if (t < min) { min = t; pmin = pix; }
    }
    printf(" choose %d \n",pmin);
    */
    return(-1); }
  return col.pixel;
}

int alloc_named_color(char *colname) {
  XColor col;
  if (XAllocNamedColor(display,def_cmap,colname,&col,&col) == 0)
    { error(" Color map full"); }
  return col.pixel;
}
// die GCs fuer die Button-Darstellung
GC button_fg_gc, button_br_gc, button_lw_gc; 
 
// init global used screen values
void init_globals(char * DISP) { 
  if (DISP == NULL) DISP = getenv("DISPLAY");
  display = XOpenDisplay(DISP);   
  if (display == NULL) { 
    char s[100]; 
    sprintf(s,"cannot open display '%s'",DISP);
    error(s);
  }
  screen = DefaultScreen(display);    
  depth = XDefaultDepth(display,screen);
  def_cmap = DefaultColormap(display,screen);
  black = BlackPixel(display,screen);
  white = WhitePixel(display,screen);
  XGCValues gcv; gcv.function = GXcopy ;     
  gcv.foreground = black; gcv.background = white; 
  gc_copy = CreateGC(GCFunction | GCForeground , &gcv);
  gcv.function = GXxor;
  gc_xor = CreateGC(GCFunction | GCForeground , &gcv);
  gcv.function = GXclear; 
  gc_clear= CreateGC(GCFunction, &gcv);

//XGCValues={function,plane_mask,foreground,background,linewidth,linestyle,..}
  XGCValues val_rubber = { GXinvert, AllPlanes,0,0,2, LineOnOffDash}; 

  // rubberband for temp. coordinate lines
  gc_rubber= CreateGC(GCFunction | GCPlaneMask | GCLineWidth| GCLineStyle, 
		      &val_rubber); 

  fixed_fn = XLoadQueryFont(display,"fixed");
 
  // Definieren der gcs fuer die Button-Darstellung  
  int button_fg_pix = alloc_color(0x6180,0xa290,0xc300); // Inneres des Buttons
  int button_br_pix = alloc_color(0x8a20,0xe380,0xffff); // heller Rand
  int button_lw_pix = alloc_color(0x30c0,0x5144,0x6180); // dunkler Rand
  XGCValues values;

  values.foreground = button_fg_pix;
  button_fg_gc = CreateGC(GCForeground, &values);
  values.foreground = button_br_pix;
  button_br_gc =  CreateGC(GCForeground, &values);
  values.foreground = button_lw_pix;
  button_lw_gc =  CreateGC(GCForeground, &values);

  watch_cursor = XCreateFontCursor(display,XC_watch);
  text_cursor = XCreateFontCursor(display,XC_xterm);
}

window * pulldown_mapped = NULL; 
// global list : used to destroy an pulldown window 
// with the next button release 
// in the CallBack function; not very nice though

window::window(char * DISP) { // constructor for root window
  init_globals(DISP);
  width = DisplayWidth(display,screen);
  height = DisplayHeight(display,screen);
  Win = DefaultRootWindow(display);
  // printf(" root Window %x %d %d \n",Win, width, height);
};

int debug_create = False; // not defined in window.h (private)

window::window(window & parent, int w, int h, int x, int y, int bw) { 
  width = (w ? w : parent.width) - 2*bw;
  height = (h ? h : parent.height) - 2*bw;
  border_width = bw;
  parentw = &parent;
  hidden = False;  CB_info = False;
  mainw = parent.mainw; // simples Durchreichen 
  children = NULL; // eigene children List = leer
  win_list * pp = new win_list; 
  pp->next = parent.children;
  pp->child = this; 
  pp->x = x; pp->y = y;
  parent.children = pp;  
  parent.add_child(this);
  help = NULL;
  type = SimpleType;
  Win = XCreateSimpleWindow(display, parent.Win, x,y, width, height,
			    border_width,black,0); 
 
  if (debug_create) printf(" Window %x %d %d \n",Win,width, height);
  
  gc = gc_copy; // default gc
  text_gc = gc_copy; // gc for DrawString, PlaceText; temoprarily changed
  void ** tW = thisW(Win);
  if (*tW) error("too many windows !"); // entry already set !!
  else *tW = this;

  // die default mask, einzelne Unterklassen koennen abweichen
  selection_mask =  ButtonPressMask | ButtonReleaseMask 
	         | Button1MotionMask | PointerMotionMask
	         | EnterWindowMask | LeaveWindowMask | ExposureMask 
                 | StructureNotifyMask ;
                // vom WM ausgeloest fuer : Resize & iconify, raise (18,19,22)
};

window root_window; // definition !!!

int debug_delete = 0; // printf bei destruktoren

window::~window() {
   if (debug_delete) printf("destr %x %x\n",Win,this); 
  // das Loeschen der Sub-Windows passiert im destructor von main_window
  *thisW(Win) = NULL; // um weitere callbacks zum Window zu verhindern 
  if (this == &root_window) return;
  win_list *ch = children;
  while (children) { 
    delete children->child; ch = children; 
    children = children->next; delete ch;
  } 
}

void window::Map() { 
  if (hidden) return; 
  draw_interior();  
  XMapWindow(display, Win);
}

/* Bemerkung zum ResizeRequest : 
   wenn das Bit in der SelectMask gesetzt ist, nimmt der WM keine 
   XResizeRequestEvents mehr an, da sie vom application Window abgefangen
   werden !
*/
void window::Realize() {  
  // printf("Realize %x %d %d\n",Win,width,height);
  XSelectInput(display, Win, selection_mask);
  Map(); // hier ist die Reihenfolge (Select, Map) entscheidend !
};

void window::Unmap() { XUnmapWindow(display,Win); };

// Realize the whole Window tree from this root
void window::RealizeChildren() { 
  Realize();
  win_list * cc = children;
  while (cc) { 
    window * ch = cc->child;
    ch->RealizeChildren();
    cc = cc->next;
  }; 
}

void window::set_backing_store() {
  XSetWindowAttributes attr; attr.backing_store = WhenMapped;
  XChangeWindowAttributes(display,Win,CWBackingStore,&attr);
}
void window::set_save_under() {
  XSetWindowAttributes attr; attr.save_under = TRUE;
  XChangeWindowAttributes(display,Win,CWSaveUnder,&attr);
}
	   
void window::clear()
  { XFillRectangle(display, Win, gc_clear, 0,0,width,height); }

void window::DrawString(int x, int y, char * string) { 
  XDrawString(display,Win,text_gc,x,y, string, strlen(string)); }

void window::PlaceText(char * string, int x, int y,
		   XFontStruct * fn) { 
    XSetFont(display, gc, fn->fid);
    int tw = XTextWidth(fn, string, strlen(string));   
    int th = fn->ascent; // + fn->descent;
    if (x == 0) x = (eff_width() - tw) / 2;  // horizontal zentriert
    if (y == 0) y = (height + th) / 2; // vertikal zentriert
    DrawString(x,y,string); 
  }

void window::line(int x0, int y0, int x1, int y1) {
    XDrawLine(display, Win, gc_copy, x0, y0, x1, y1); 
  }

void window::DrawPoint(int x, int y) {
    XDrawPoint(display, Win, gc_copy, x, y); 
  }

void window::BPress_1_CB(XButtonEvent ev) { }; 
void window::BPress_3_CB(XButtonEvent ev) 
{ if (help) help->do_popup(ev.x_root,ev.y_root + 20);
  //  popup below cursor pos. else use:   help->RealizeChildren();
}

void window::Expose_CB(XExposeEvent ev) { 
/*  if (CB_info) 
    printf("expose %d %d %d %d %d\n",ev.count,ev.x,ev.y,ev.width,ev.height);
*/
  if (ev.count == 0) redraw();
}

/* definitions aus /usr/include/X11/X.h : 
   siehe -> eventnames.h 
   KeyPress		2   KeyRelease		3
   ButtonPress		4   ButtonRelease	5
   MotionNotify		6   EnterNotify		7
   LeaveNotify		8   FocusIn		9
   FocusOut		10  KeymapNotify	11
   Expose		12  GraphicsExpose	13
   NoExpose		14  VisibilityNotify	15
   CreateNotify		16  DestroyNotify	17
   UnmapNotify		18  MapNotify		19
   MapRequest		20  ReparentNotify	21
   ConfigureNotify	22  ConfigureRequest	23
   GravityNotify	24  ResizeRequest	25
   CirculateNotify	26  CirculateRequest	27
   PropertyNotify	28  SelectionClear	29
   SelectionRequest	30  SelectionNotify	31
   ColormapNotify	32  ClientMessage	33
   MappingNotify	34  LASTEvent		35
*/

// the main callback function
pulldown_button * active_button = NULL;

void window::CallBack(XEvent &event) 
{ if (CB_info) printf("Event %s (%d) in Win %x\n",event_names[event.type],
		       event.type,event.xany.window);
  if (hidden) return;
  if ((event.type == ButtonRelease))
    { if (pulldown_mapped) // hat ein pulldown menu gemapped -> unmap it
	{ pulldown_mapped->Unmap(); pulldown_mapped = NULL; }  
      
      if (active_button) // restore invoker button
	{ active_button->default_frame(); active_button = NULL;}
    }
  switch (event.type) 
    { 
    case ButtonPress:    
        BPress_CB(event.xbutton); // allgemeiner BPress-handler, zB. buttons   
	switch (event.xbutton.button) {
	  case 1: BPress_1_CB(event.xbutton); break;
	  case 3: BPress_3_CB(event.xbutton); break;
	}
        break;
    case ButtonRelease:
        BRelease_CB(event.xbutton);  break;
    case EnterNotify: 
	Enter_CB(event.xcrossing); break;
    case LeaveNotify:
	Leave_CB(event.xcrossing); break;
    case Expose:
	Expose_CB(event.xexpose); break;
    case KeyPress:
	KeyPress_CB(event.xkey);  break;
    case MotionNotify: 
	Motion_CB(event.xmotion); break;
    case ConfigureNotify:
        Configure_CB(event.xconfigure); break;
    default: break;
    }; 
}

window::add_help(char * WMName,char * text[])
{ int lines, cols;
  compute_text_size(text,cols,lines);
  help = new text_popup(WMName,6*cols + 10, 20*lines + 30,text);
}

static main_window *watch_main = NULL; 

void window::WatchCursor() { // temporaeres Setzen, in event-loop rueckgesetzt
  XDefineCursor(display,mainw->Win,watch_cursor); 
  XFlush(display);
  watch_main = mainw;
}

// static float xf_res,yf_res; // resize-factors: either global (in CB) or 
// computed locally inside resize itself

void window::resize(int w, int h) {
  if (width == w && height == h) return; // evtl. nur move Events
  // printf("resize %x: %d %d (%d %d)\n",Win,w,h,width,height); 
  float xf_res = float(w)/width, yf_res = float(h)/height;
  width = w; height = h;  
  XResizeWindow(display,Win,w,h);
  win_list *ch = children; 
  while (ch) { 
    window * cw = ch->child;
    int xn = int(xf_res * ch->x + .5), yn = int(yf_res * ch->y + .5);
    ch->x = xn; ch->y = yn;  // neue x,y - Koordinaten des child-Windows
    XMoveWindow(display,cw->Win, xn, yn);
    cw->resize(int(xf_res * cw->width + .5), int(yf_res * cw->height + .5)); 
    ch = ch->next; 
  } 
} 

// ##################     main_window class    ###########################
#include "icon.h"

main_window::main_window(char * WMName, int w, int h) 
  : window(root_window, w, h ) { 
  mainw = this; // ich bins selbst
  XStoreName(display, Win, WMName);  // set name in WM-frame
  Cursor main_cursor = XCreateFontCursor(display,XC_left_ptr);

  XDefineCursor(display, Win, main_cursor);
  set_icon(icon_bits,icon_width,icon_height);
}

main_window::~main_window() {
  XDestroyWindow(display,Win); 
  if (debug_delete) printf(" destructor main_window : %x\n",Win);
  // window::~window(); 
  // expliziter destr. fuer Basisklasse nicht noetig, passiert automatisch !
}

// only for popup windows : show them at given x,y : 
// 1. make it small before moving it, 2. realize, 3. move to x,y-position
// nicht sehr beeindruckend
void main_window::do_popup(int x, int y)
{ // XResizeWindow(display, Win, 1, 1); 
  RealizeChildren(); 
  // XMoveResizeWindow(display, Win, x, y, width, height);
}
	
void main_window::Configure_CB(XConfigureEvent ev) {  
  // printf("config %d %d %d \n",ev.type,ev.width,ev.height);
  // xf_res = float(ev.width)/width; yf_res = float(ev.height)/height;
  resize(ev.width, ev.height); 
}

// die Event loop :
// aus dem Window-ID wird der this-Pointer des Windows errechnet (Tabelle)
// dann wird die CallBack function aufgerufen, die ueber den Mechanismus
// virtueller Funktionen die spezifischen CallBacks der abgeleiteteten 
// Unterklassen aufruft !

void handle_event(XEvent &event) {
  unsigned wid = event.xany.window;
  void ** tW = thisW(wid);        // ***** das ist der Knueller **** !!
  if (*tW) ((window *) *tW)->CallBack(event); 
  else { /* das Window ist geloescht ! 
	    Aber fuer mit delete geloeschte Windows (auch mit XDestroyWindow..)
	    werden noch LeaveNotify-Events (type = 8) ausgeloest !!!
	    */
	}
}

// for event loop in polling_mode
Bool predicate(Display *, XEvent *, char *) { return TRUE; } 
Bool polling_mode = False;     // default : use XNextEvent in main_loop
void (*poll_handler)() = NULL;

void main_window::main_loop() {
  exit_flag = False;
  RealizeChildren(); 
  while (1)
    { XEvent event;   
      if (polling_mode) { // polling : used only for special applications
        if (! XCheckIfEvent(display,&event,predicate,"loop")) {
	  if (poll_handler) (*poll_handler)(); 
          continue;  // no event handling needed
        }
      } else XNextEvent(display, &event);
      handle_event(event);
      // in case of the queue has filled during handling -> flush events 
      while (XCheckMaskEvent(display, KeyPressMask | KeyReleaseMask 
                                      | PointerMotionMask,&event));
      if (exit_flag) break; // is set by quit buttons
    if (watch_main) 
      { XUndefineCursor(display,watch_main->Win); watch_main = NULL; }
    }
  Unmap();
  // printf("main_loop exited\n");
}

#include "X11/Xutil.h"

void main_window::set_icon(char ibits[], int iwidth, int iheight) {
  Pixmap icon_pixmap = XCreateBitmapFromData(display,Win,ibits,iwidth,iheight);
  XWMHints *wm_hints = XAllocWMHints();
  wm_hints->icon_pixmap = icon_pixmap;
  wm_hints->flags = IconPixmapHint;

  XSetWMHints(display,Win,wm_hints);
  XFree(wm_hints);
}

// ####################      pixmap_window    ############################

pixmap_window::pixmap_window(window & parent,int w,int h,int x,int y,int bw) : 
window(parent, w, h,x, y, bw) {
  pix = XCreatePixmap(display, Win, width, height, depth);
  clear();
}
pixmap_window::~pixmap_window() {
  XFreePixmap(display,pix);
  // this->window::~window(); // fuer Basisklasse, sollte automatisch erfolgen
}  

void pixmap_window::clear() { 
  XFillRectangle(display, pix, gc_clear, 0,0, width, height); 
}
 
void pixmap_window::Map() { 
  draw_interior(); 
  XCopyArea(display,pix, Win, gc_copy, 0, 0, width, height, 0, 0);
  XMapWindow(display, Win); 
}

void pixmap_window::DrawString(int x, int y, char * string) {
  XDrawString(display,pix,gc_copy,x,y, string, strlen(string)); }

void pixmap_window::line(int x0, int y0, int x1, int y1) {
  XDrawLine(display, pix, gc_copy, x0, y0, x1, y1); }

void pixmap_window::DrawPoint(int x, int y) {
    XDrawPoint(display, pix, gc_copy, x, y); 
  }

void pixmap_window::Expose_CB(XExposeEvent ev) { 
  // restore the exposed area from pixmap onto screen
  XCopyArea(display, pix, Win,gc_copy,ev.x,ev.y,ev.width,ev.height,ev.x,ev.y);
}

void pixmap_window::resize(int w, int h) {
  // printf("resize %dx%d -> %dx%d\n",width,height,w,h);
  if (width == w && height == h) return; // evtl. nur move Events
  XFreePixmap(display,pix);
  pix = XCreatePixmap(display,Win,w,h,depth);
  window::resize(w,h);
  clear(); // erst hier (braucht width, height !)
  Map();
}

// ****************** 3d-shapes ********  
// zeichnet Rechteck im 3d-look mit zwei verschiedenen GCs
// mode = up3d, flat3d (background pix), down3d
void rect3d(Window Win, int mode, short x, short y, short w, short h) {
  int i, thickness = 2;  
  GC left_top,right_bot; // Farben fuer beide Rahmen-Teile : 
  switch (mode) { 
  case up3d:   left_top = button_br_gc; right_bot = button_lw_gc; break;
  case down3d: left_top = button_lw_gc; right_bot = button_br_gc; break;
  case flat3d: left_top = button_fg_gc; right_bot = button_fg_gc; break;
  }
  for (i = 0; i < thickness; i++) {
    short xa = x+i, ya = y+i, xe = x+w-i, ye = y+h-i ; 
    XPoint xp[5] = { {xa,ye}, {xa,ya}, {xe,ya}, {xe,ye}, {xa,ye}}; // short x,y
    XDrawLines(display,Win, left_top, xp, 3, CoordModeOrigin);
    XDrawLines(display,Win, right_bot, xp+2, 3, CoordModeOrigin);
  }
}

// analog Dreieck, Spitze nach unten (Sued), x,y = linke Ecke
// mode = up3d, flat3d (background pix), down3d
void tri3d_s(Window Win, int mode, short x, short y, short w, short h) {
  int i, thickness = 2;  
  GC left_top,right_bot; // Farben fuer beide Rahmen-Teile : 
  switch (mode) { 
  case up3d:   left_top = button_br_gc; right_bot = button_lw_gc; break;
  case down3d: left_top = button_lw_gc; right_bot = button_br_gc; break;
  case flat3d: left_top = button_fg_gc; right_bot = button_fg_gc; break;
  }
  for (i = 0; i < thickness; i++) {
    short xa = x+i, ya = y+i, xe = x+w-i, ye = y+h-i, xm = x+w/2; 
    XPoint xp[4] = { {xm,ye}, {xa,ya}, {xe,ya}, {xm,ye}}; 
    XDrawLines(display,Win, left_top, xp, 3, CoordModeOrigin); 
    XDrawLines(display,Win, right_bot, xp+2, 2, CoordModeOrigin);
  }
}
// plates are pseudo-3d-windows
void plate::redraw() {
    XFillRectangle(display,Win,button_fg_gc,0,0,width,height); 
    default_frame(); // virtual function !
   }

// --------------------------- BUTTONS  -------------------------------
// ##################         button class      ###########################

void button::init_button(window *parent) {  
  in_pulldown = (parent->type == PulldownType);
  selection_mask &= ~ (PointerMotionMask);
} 
void button::redraw() {
  plate::redraw(); 
  PlaceText(Name); 
}

button::~button() {
  if (debug_delete) printf("button %s -> ",Name); 
}

void button::add_help(char **help_text) {
    char * WMName = new char[strlen(Name) + 12]; 
    sprintf(WMName,"Help : '%s'",Name);
    window::add_help(WMName,help_text); }

// ##################     help_button class    ############################
void help_button::make_popup(char *text[])
{
  // Berechnung der noetigen Windowgroesse fuer popup
 int ln, cols; 
 compute_text_size(text,cols,ln);
 help_popup = new text_popup("help", 6*cols + 10, ln*15 + 30, text); 
 // leave room for OK button (bottom)!
}

// ##################   function_button class    ############################

function_button::function_button (window & parent, char * Name,  
                                 int w, int h, int x, int y, CB cb,  ...) : 
      button (parent, Name, w, h, x, y), callback(cb) {
    int i; va_list ap; va_start(ap,cb);
    for (i=0;i<10;i++) values[i] = va_arg(ap,void*);
    va_end(ap);
  }

function_button::function_button (menu_bar & parent, char * Name, CB cb, ...) :
  button (parent, Name), callback(cb) { 
    int i; va_list ap; va_start(ap,cb);
    for (i=0;i<10;i++) values[i] = va_arg(ap,void*);
    va_end(ap);
  }
   
// ---------------------------- PULLDOWN --------------------------------

// ##################     pulldown_window class    #######################
// a window child from root_window, not yet visible, not managed from WM,
// position is determined from the button at popup time

// special cursor for pulldown menus 
Cursor pulldown_cursor = XCreateFontCursor(display,XC_right_ptr);

pulldown_window::pulldown_window (int w, int h) : main_window("",w,h) {
  XSetWindowAttributes attr; 
  attr.override_redirect = TRUE; attr.save_under = TRUE;
  attr.cursor = pulldown_cursor; 
  XChangeWindowAttributes(display, Win, 
			  CWOverrideRedirect | CWCursor | CWSaveUnder, 
			  &attr);
  type = PulldownType;
}

// ##################     pulldown_button class    ########################
// map the window (menu) on root when button is activated (BPress)
// the window is mapped to absolute co-ordinates !

pulldown_button::pulldown_button (window & parent, pulldown_window * menu, 
				  char * Name, int w, int h, int x, int y) 
: button(parent, Name, w, h, x, y) { pulldown_menu = menu; xright = 12;}

pulldown_button::pulldown_button(menu_bar & parent, pulldown_window * menu, 
				 char * Name) 
: button(parent, ext_string(Name)) { pulldown_menu = menu; xright = 12; }

pulldown_button::pulldown_button (window & parent, pulldown_window * menu, 
				  char * Name, char ** help_text,
				  int w, int h, int x, int y) 
: button(parent, Name, help_text, w, h, x, y) 
         { pulldown_menu = menu; xright = 12;}
 
void pulldown_button::picture() { 
   int offs = (height-8)/2; 
   tri3d_s(Win,up3d, width-xright,offs,8,8);
}

void pulldown_button::BPress_1_CB(XButtonEvent ev) 
{ // muss erst die Koordinaten des buttons haben um menu zu plazieren
  unsigned x, y; // *absolut*  position of pulldown menu
  /* **** old code (works too) **** 
     // komplettes Aufsteigen im Window tree bis root 
     // enthaelt auch WM-windows !!
     Window root_return, query, parent, *cc;
     query = Win; // WID of the pulldown button
     x = 0; y = height + 1;
     do {  
     unsigned xr, yr, wr, hr, br, dr, nc;
     XGetGeometry(display, query, &root_return, &xr, &yr, &wr, &hr, &br, &dr);
     XQueryTree(display, query, &root_return, &parent, &cc, &nc); 
     // printf("win = %x par = %x,  x y = %d %d\n", query, parent, xr, yr);
     x += xr; y += yr;
     query = parent;
     } while (parent != root_return);
  */
  
  // bestimmt die absoluten Koord des Maus-Pointers !
  x = ev.x_root - ev.x; // + (width-pulldown_menu->width)/2; horizontal zentr.
  y = ev.y_root - ev.y + height + 1;
  // auch nuetzlich :  XMoveResizeWindow(disp,win,x,y,w,h);
  XMoveWindow(display, pulldown_menu->Win, x, y);
 
  pulldown_menu->RealizeChildren();
  XRaiseWindow(display,pulldown_menu->Win); // Reihenfolge !!
  // 
  XGrabPointer(display,pulldown_menu->Win,TRUE,
	       Button1MotionMask | PointerMotionMask |
	       EnterWindowMask | LeaveWindowMask 
	       ,GrabModeAsync, GrabModeAsync,None, None,CurrentTime);
  pulldown_mapped = pulldown_menu; 
  active_button = this;
  // damit loest das naechste BRelease das unmappen dieses
  // pulldowns aus, und restores den default_frame (this)
  // Ueber Methode nicht zu loesen, da BRelease fuer ein anderes window
  // aktiviert wird.
  // sollte durch push/pop-Verfahren ersetzt werden, um mehrfache Popups 
  // zu erlauben
} 

// if w == 0 parent must be menu_bar -> use autoplacement
pulldown_button * make_radio_menu(window &parent, char *Name, char **list,
				  window * action_win,
				  int w, int h, int x, int y) 
{
  int wm, hm;      // size of pulldown_menu
  int nbuttons;   // number of buttons on pulldown

  char * value;

  // first parse blist to determine size of pulldown menu
  compute_text_size(list,wm,nbuttons);
  wm = wm * 6 + 10 ;        //  6 = length of char
  hm = nbuttons * 20;       // 20 = height of button
  pulldown_window * pd_wi = new pulldown_window(wm,hm);
  pulldown_button * pd_bt;
  if (w > 0) pd_bt = new pulldown_button(parent,pd_wi,Name,w,h,x,y); else
     pd_bt = new pulldown_button((menu_bar&) parent, pd_wi, Name);
  int yp = 0; 
  while (*list) { 
    value = *list++;
    button *bt = new radio_button(*pd_wi,Name,wm,20,0,yp,value, action_win);
    yp += 20;
  };
  return pd_bt;
}


// analogous : with help popup
pulldown_button * make_radio_menu(window &parent, char *Name, char **list,
				  char ** help_text, window * action_win,
				  int w, int h, int x, int y) 
{
  pulldown_button * r_menu = make_radio_menu(parent, Name, list,
					     action_win, w, h, x, y );
  r_menu->add_help(help_text);
  return r_menu;
}

// list : { {"button1", &callback1}, {"button2", &callback} }
pulldown_button * make_pulldown_menu(window &parent, char *Name, 
				     int nbuttons, struct but_cb list[],
				     int w, int h, int x, int y) {
  int wm = 0, hm;     // size of pulldown_menu
  int i;
  // first parse blist to determine size of pulldown menu
  for (i=0; i < nbuttons; i++) { 
    int ll;
    ll = strlen(list[i].bname); if (ll > wm) wm = ll;
  };
  wm = wm * 6 + 10 ;            //  6 = length of char
  hm = nbuttons * 20;           // 20 = height of button
  pulldown_window * pd = new pulldown_window(wm, hm);
  pulldown_button * pb;
  if (w > 0) pb = new pulldown_button(parent,pd,Name,w,h,x,y); else
     pb = new pulldown_button((menu_bar&) parent, pd, Name); 

  int yp = 0; 
  for (i=0; i < nbuttons; i++) { 
    button *bt= new callback_button(*pd,list[i].bname,list[i].cb,wm,20,0, yp);
    yp += 20;
  }
  return pb;
}


// ###################### text_popup #######################

text_popup::text_popup(char * WMName, int w, int h, char *text[]) :
main_window (WMName, w, h) { 
  pop_text = text; 
  button *ub = new unmap_button (*this,"OK",50,20, (width-50)/2,height-23); 
  // a button to close the popup window
}

void text_popup::Expose_CB(XExposeEvent ev) { // write the text
    int ln = 0, y = 0; 
    while (pop_text[ln]) {   
      y+= 15;
      PlaceText(pop_text[ln++], 4, y);
    }
  }

// String-Arrays: char *x[] = {"xyzuvw","XXXX",...,0} 
// max Laenge eines strings && Anzahl der strings 
void compute_text_size(char *text[], int &cols, int &lines) {
  int ll = 0;
  lines = 0; cols = 0;
  while (text[lines]) { 
    ll = strlen(text[lines++]);
    if (ll > cols) cols = ll;
    if (ll > 1000 || cols > 1000) error("text-Feld hat keinen NULL-Abschluss");
  }
}

// ##################### coord_window ##########################

coord_window::coord_window(window & parent, int w, int h, int x, int y, 
	                   int rxl, int rxr, int ryd, int ryu) :
    pixmap_window(parent,w,h,x,y), rxl(rxl), ryd(ryd) { 
    w_diff = rxr + rxl; h_diff = ryd + ryu; // die Raender-abzuege
}

void coord_window::define_coord(float x1, float y1, float x2, float y2) {
  xl = x1; yd = y1; xr = x2; yu = y2;
  x0 = rxl; y0 = height - ryd; // Window-koordinaten des Ursprungs
  w_eff = width - w_diff; h_eff = height - h_diff;
  xf = w_eff/(x2 - x1); yf = h_eff/(y2 - y1);
}

// compute total window-coordinates from world-values
int coord_window::x_window(float x) { return x0 + (int)(xf * (x - xl) + .5); }
int coord_window::y_window(float y) { return y0 - (int)(yf * (y - yd) + .5); }

XPoint coord_window::p_window(float x, float y) { // returns same as XPoint
  XPoint temp = { x_window(x), y_window(y) };
  return temp;
}
   
// back transformation : window coords to world-values
float coord_window::x_org(int xw) { return (xw - x0)/xf + xl; }
float coord_window::y_org(int yw) { return (y0 - yw)/yf + yd; }

void coord_window::x_ticks(float dx, int n) { 
  float xx; 
  if (xr < xl) dx = -dx; // failsafe
  for (xx = xl; xx < xr; xx+= dx) { 
    int x = x_window(xx); line(x,y0,x,y0+2); 
    if (n-- == 0) break;
  } 
}
void coord_window::y_ticks(float dy, int n) { 
  int i; float yy = yd;
  if (yu < yd) dy = -dy;
  for (i = 0; i < n; i++) { 
    int y = y_window(yy); line(x0,y,x0-2,y); 
    yy += dy; if (yy > yu) break;
  }
}
 
void coord_window::graph(int nx, double f[]) { 
    int i, x , y, xp, yp; 
    for (i=0; i<nx-1; i++) { 
      y = y_window(f[i]); x = x_window(i); 
      if (i > 0) line(xp, yp ,x, y);
      xp = x; yp = y; 
    }
}

//   ###########  system buttons && xwd_buttons ##########

void system_button::BPress_1_CB(XButtonEvent) { 
    printf("calling system('%s')\n",cmdline);
    system(cmdline);
  }

void xwd_button::BPress_1_CB(XButtonEvent) { 
    char cmdline[200]; 
    sprintf(cmdline,"xwd -id 0x%x %s",dumpw->Win, arg);
    printf("dump : calling system('%s')\n",cmdline);
    system(cmdline);
  }

// ############################################################
//               SCROLLBARS

void slider::redraw() { // called from Expose_CB, not invoked from move !
  // to have the scrollbar as mouse input instead slider
  // if (count++ == 0) -> doesnt work with popups (when called a second time)
  // its also no problem to call XSelectInput every time here
  XSelectInput(display,Win,ExposureMask); 
  plate::redraw(); // printf("%d \n",count); 
  line(width/2,0,width/2,height);
}

void pure_scrollbar::set_slider(int x) {
  bar = new slider(*this,sw,sh,x+2,sy); 
  xact = x;
}

void pure_scrollbar::init () { 
  sw = 19; sh = height-10; sy = 5; // start-values for slider
  xoff = sw/2+2; xmax = width-sw/2-2; xspan = xmax-xoff-1;
  set_backing_store(); // to avoid flickering of scrollbar when drawing slider
}
void pure_scrollbar::move(int x) { 
  if (x >= 0 && x <= xspan) { 
    // der reale Wert im Intervall [0..xspan]
    XMoveWindow(display,bar->Win,x + xoff - sw/2,sy); 
    callbck(x);
  }  
}

void pure_scrollbar::redraw() {   
  plate::redraw(); 
  line(xoff,height/2,xmax,height/2); // horizontale Linie
  // line(xoff,4,xoff,height-4); line(xmax,4,xmax,height-4)
}

void pure_scrollbar::resize(int w, int h) {
  plate::resize(w,h);
  sw = bar->width; sy = (height - bar->height)/2;
  xoff = sw/2+2; xmax = width-sw/2-2; xspan = xmax-xoff-1;
} 

void scrollbar::init(window &parent, int w, int h, int x, int y, 
                     int minp, int maxp, int xstart) 
{  min = minp; max = (maxp) ? maxp : xspan; 
   factor = ((double) (max - min))/xspan; 
   set_slider( (int) ((xstart-min)/factor));
   valstr(xact); // Anfangs-default
   disp_win = new display_window(parent,str,60,h,x+w-60,y,down3d);
 }

scrollbar::scrollbar(window &parent, void (*inf)(), int w, int h, 
		     int x, int y, int minp, int maxp, 
		     int xstart, char *format) :
     pure_scrollbar (parent,pwidth(w),h,x,y), format(format) {
       init(parent,w,h,x,y,minp,maxp,xstart);
       to_inform = NULL;
       inffn.empty = inf;
  }
 
// 2. Konstruktor :
scrollbar::scrollbar(window &parent, void (*inf)(void*), void * to_inf,
		     int w, int h, int x, int y, 
		     int minp, int maxp, int xstart, char *format) :
     pure_scrollbar (parent,pwidth(w),h,x,y), format(format) {
       init(parent,w,h,x,y,minp,maxp,xstart);
       to_inform = to_inf;
       inffn.vptr = inf;
  }
    
void display_window::set_text_mode(int mode) { 
  // mode 0 : Loeschen, 1 : Schreiben
  text_gc = (mode) ? gc_copy : button_fg_gc;
} 
 
void scrollbar::callbck(int pix) {   
  // schnelles Loeschen der alten Anzeige : Ueberschreiben mit background gc
  disp_win->set_text_mode(0); disp_win->draw_val();
  valstr(pix); // Neue Anzeige berechnen
  disp_win->set_text_mode(1); disp_win->draw_val();
  // reset default gc !!! , neuer Wert
  if (inffn.value) 
    { if (to_inform) (*inffn.vptr)(to_inform); else (*inffn.empty)(); }
}

void scrollbar::resize(int w, int h) {
  pure_scrollbar::resize(w,h);  
  factor = ((double) (max - min))/xspan; 
}
// ***************************************
// class "edit_window" zum Editieren von strings, max 80 Zeichen 

edit_window::edit_window(window &parent, char *str, int w,int h, int x, int y)
  :plate(parent,w,h,x,y,down3d) { 
    strncpy(value,str,200);
    XDefineCursor(display,Win,text_cursor);
    cp = strlen(value); // hinter dem string 
    selection_mask |= KeyPressMask;
  }
void edit_window::mark_cursor() { 
    XFillRectangle(display, Win, gc_xor, xs + 6*cp, 2, 6, 15);
  }
void edit_window::Enter_CB(XCrossingEvent ev) { // frame3d(flat3d); 
    mark_cursor(); 
}
void edit_window::Leave_CB(XCrossingEvent ev) { // default_frame(); 
    mark_cursor(); 
}

void edit_window::redraw() {
    plate::redraw(); 
    xs = width - 6*strlen(value) - 10; 
    DrawString(xs,16,value);
}
  
void edit_window::del_char() { // Loescht Zeichen links vom cursor
    int i; 
    for (i = cp; i <= strlen(value); i++) value[i-1] = value[i];
    cp--;
  }
void edit_window::ins_char(char x) { // Einfuegen links
    int i; 
    for (i = strlen(value); i >= cp; i--) value[i] = value[i-1];
    value[cp] = x; cp++;
  }

void edit_window::KeyPress_CB(XKeyEvent ev) {
    mark_cursor();
    KeySym keysym = XLookupKeysym(&ev,ev.state & 1);
    switch(keysym) {
    case XK_Left: while (cp <= 0) ins_char(0x20); // Auffuellen mit space
                  cp -= 1; break;
    case XK_Right: if (cp < strlen(value)) cp += 1; break;
    case XK_BackSpace: if (cp > 0) del_char(); break;
    case XK_Delete: if (cp > 0) del_char(); break;
    case XK_Escape: escape(); break;
    case XK_Return: enter(); break;               // Return = Enter
    default: 
      if ((keysym >= 0x20) && (keysym<= 0x7e)) // alle ASCII-chars
	ins_char(keysym);
    }
    redraw();
    mark_cursor(); 
}


