// eval-simple.c : image computations, generating a special adapted palette
//                creates a popup menu "eval_popup"
//  a part of PMS-grabber package
// the "GNU Public Lincense" policy applies to all programs of this package
// (c) Wolfgang Koehler, wolf@first.gmd.de, Dec. 1994
//     Kiefernring 15
//     14478 Potsdam, Germany
//     0331-863238

#include "evaluation.h"
#include "window.h"
#include "grabber.h"
#include "view.h"

unsigned char coltab555[0x8000];
unsigned char coltab332[256];
unsigned long pixtab[256]; // store allocated colors
int max_cols; // number of alloc colors
int ncols = 100; // number of colors for auto palette = 0..max_cols

void create_coltab332() {  // no more used
  unsigned r,g,b,i; 
  for (i=0; i < 256; i++) coltab332[i] = 0;
  int fact = 65535;
  for (r=0; r<8;r++) for (g=0;g<7;g++) for(b=0;b<4;b++) {
    i = b + g*4 + r*32; 
    if (i < 256) coltab332[i] = alloc_color(r*fact/7,g*fact/6,b*fact/3); 
    // printf("%d %d %d %d\n",r,g,b,i);
  }
}

// set pixel-entry in colormap 
void store_color(int pixel, unsigned red, unsigned green, unsigned blue) {
  XColor col = { pixel, red, green, blue, DoRed | DoGreen | DoBlue };
  XStoreColor(display,def_cmap,&col);
}
 
// create a standard color table for RGB555 mode :
// array pixtab[] has to be filled before with allocated color cells
// max_cols is the number of available cells
// first is a 332 - color value allocated then expand by factors 4x4x8
void std_coltab555() {
  unsigned char r,g,b;
  int i,nc;
  nc = 32*32*32; // die Zahl der entrys der ct
  printf("creating standard coltab\n");
  for (i=0; i < nc; i++) coltab555[i] = 0;
  char (*ct)[32][32] = (char (*) [32][32]) coltab555;
  int cmax = 65535;  // max color

  i = 0;
  for (r= 0; r < 8; r++) 
    for (g= 0; g < 8; g++) 
      for (b= 0; b < 4; b++) {
	unsigned char bs, gs, rs, bi, gi, ri;
	int pixel; // means : no color allocated (no free entry)
	if (i < max_cols) { // ok. free entry
	  pixel = pixtab[i++]; 
          store_color(pixel,r*cmax/7, g*cmax/7, b*cmax/3); 
	  // if no more room take the value red-1
	} else pixel = ct[4*r-1][4*g][8*b];   

	for (bs = 8*b, bi = 0; bi < 8; bs++, bi++) // blue is mapped 8-fold 
	  for (gs = 4*g, gi = 0; gi < 4; gs++, gi++) // green is mapped 4-fold
	    for (rs = 4*r, ri = 0; ri < 4; rs++, ri++) 
	      ct[rs][gs][bs] = pixel;
  }
  ncols = max_cols;
}

void grey_coltab555() {
  short i,r,g,b;
  ncols = 3*32; // 96 color cells are used
  for (i = 0; i < ncols; i++) store_color(pixtab[i],i*682,i*682,i*682);
  char (*ct)[32][32] = (char (*) [32][32]) coltab555;
  for (r= 0; r < 32; r++) for (g= 0; g < 32; g++) for (b= 0; b < 32; b++)
  ct[r][g][b] = pixtab[r+g+b];
  map_and_redraw();
}

// describe a volume element in color space 
struct volume { char low[3], high[3]; };

// compute mean value (sum) for a partial volume of 3 dim color space, 
// returns total number of points inside
unsigned mean(unsigned *colsum, volume *v, unsigned cm[3]) {
  unsigned (*csu)[32][32] =  (unsigned (*)[32][32]) colsum;
  int r,g,b,i;
  unsigned nc = 0;
  for (i = 0; i < 3; i++) cm[i] = 0;
  for (r = v->low[0]; r < v->high[0]; r++) 
    for (g = v->low[1]; g < v->high[1]; g++) 
      for (b = v->low[2]; b < v->high[2]; b++) {
        int ni = csu[r][g][b]; // color frequency for [rgb]
        nc += ni; 
	int col[3] = { r, g, b };
	for (i = 0; i < 3; i++) cm[i] += ni*col[i]; 
  }
  return (nc);
}

static double global_max; // max variance reduction
static void *part_max;  // pinter to partition to split next with do_split

class partition;
static partition *ptop = NULL; // top of partition tree

// class for non-orthogonal classification of color frequency tables
// each instance corresponds to a part volume of 3d color space
class partition {
  unsigned *cs;		// the color frequency of grabbed frame
  volume v;		// the color space volume
  double dmax, ssq;   	// best variance reduction for splitting this 
  Bool splitted;      	// True : use as brancher  
  partition *p1,*p2;  	// splitted parts
  unsigned ct[3], nt; 	// sum of colors, total points
public:
  int pind;		// sequencial index (eg. for set_palette)

  int print_vol() {
    int vol = (v.high[0]-v.low[0])*(v.high[1]-v.low[1])*(v.high[2]-v.low[2]);
    printf("rgb=(%2d,%2d)(%2d,%2d)(%2d,%2d) %4d",
	   v.low[0],v.high[0],v.low[1],v.high[1],v.low[2],v.high[2],vol);
    printf(" nt=%5d %5.1f q=(%4.1f,%4.1f,%4.1f) \n",nt, float(nt)/vol,
	   float(ct[0])/nt, float(ct[1])/nt, float(ct[2])/nt);
    return (nt);
  }
  // top constructor (for ptop)
  partition(unsigned *cols, volume vx) 
  { cs = cols; v = vx; dmax = 0; p1= p2 = NULL; splitted = False; 
    nt = mean(cs,&v,ct);  
    double ssq = 0;
    int ci; for (ci=0; ci < 3; ci++) ssq += SQR((double) ct[ci]);
    pind = 0;
    // print_vol();
  }
  // second constr. , for derived inst;  not compute mean and sum again
  partition(unsigned *cols, volume vx, int nx, unsigned cx[3], double ss) { 
    cs = cols; v = vx; dmax = 0; p1= p2 = NULL; splitted = False; 
    nt = nx; ssq = ss; 
    int ci; for (ci=0; ci<3; ci++) ct[ci] = cx[ci]; 
    // print_vol();
  }

  ~partition() { if (p1) delete p1; if (p2) delete p2; }

  // find the global max of variance reduction
  // returns false if no more possible splitting found
  Bool test_split() { 
    if (splitted) { 
      Bool x1 = p1->test_split(), x2 = p2->test_split(); 
      return (x1 || x2); // attention : Boolean shortcut !!
    } 
    else { // not yet computed && not empty
      if ((dmax == 0) && (nt > 0)) { 
	volume vp1, vp2; double sq1, sq2;
	unsigned nx, n1, n2, ci, sx1[3], sx2[3];
	volume vc = v; // volume to cut : usually v
  newcut:
	int vol = 1; // compute the numerical space of volume vc
	for (ci=0; ci<3; ci++) vol *= vc.high[ci] - vc.low[ci];
	if (vol < 2) return(False);  // vol=1 - not to cut

	for (ci=0; ci<3; ci++) { // test 3 possibil. to divide this volume 
	  if (vc.low[ci]+1 == vc.high[ci]) continue;
      	  volume vx = vc; 
 	  vx.high[ci] = (vc.low[ci] + vc.high[ci])/2;    
      	  unsigned cx[3], cy[3];
      	  int i;
          nx = mean(cs,&vx,cx);
      	  if ((nx == 0) || (nx == nt)) { // cut does not separate
	    // if all cuts fail : next time cut there where points are
	    if (nx == 0) vc.low[ci] = vx.high[ci]; // use lower part 
	    	else vc.high[ci] = vx.high[ci];  // use higher part
	    continue; // for loop
          }
      	  double var1 = 0, var2 = 0;
      	  for (i = 0; i < 3; i++) { 
	    cy[i] = ct[i] - cx[i];
            var1 += SQR ( (double) cx[i] ); var2 += SQR ( (double) cy[i] );
          }
	  // vred is the variance reduction obtained by splitting 
	  // v -> vx & v-vx
	  double vred = var1/nx + var2/(nt-nx) - ssq/nt;
          // printf(" nx = %d %d %d %d vred = %f\n",nx,cx[0],cx[1],cx[2],vred);
          if (vred > dmax) { 
            dmax = vred; 
	    n1 = nx; n2 = nt-nx;
	    vp1 = vx;  // die 2 Teilvolumina vp1 vp2
	    vp2 = vc; vp2.low[ci] = vp1.high[ci]; 
	    sx1 = cx; sx2 = cy; // die Teilsummen
	    sq1 = var1; sq2 = var2; // die Quadrate der Teilsummen
          }
        }
	if (dmax > 0) { // else no possible splitting found !
	  p1 = new partition(cs,vp1,n1,sx1,sq1);
	  p2 = new partition(cs,vp2,n2,sx2,sq2);
        } else goto newcut; // try again with volume vc
      } 
      if (dmax > global_max) { global_max = dmax; part_max = this; }
    }
    return (dmax > 0);
  }
  
  void do_split() { // really execute splitting, pnew = seq. index
    if (splitted) { p1->do_split(); p2->do_split(); } 
    else if (part_max == this)  splitted = True;
  }

  // get a new free pixtab entry and set it to mean values for this volume
  void set_palette() {
    if (splitted) { p1->set_palette(); p2->set_palette(); } 
    else {     
      pind = ptop->pind++; // own index sequentialized from ptop; 
      // printf(" p = %d %d ",pind, pixtab[pind]); print_vol();
      int pixel = pixtab[pind]; 
      int norm = (256*256 - 1)/31; // 2114
      store_color(pixel, ct[0]*norm/nt,ct[1]*norm/nt,ct[2]*norm/nt);
      char (*ctab)[32][32] = (char (*) [32][32]) coltab555;
      char r,g,b;  
      for (r = v.low[0]; r < v.high[0]; r++) 
        for (g = v.low[1]; g < v.high[1]; g++) 
          for (b = v.low[2]; b < v.high[2]; b++) {
    	    ctab[r][g][b] = pixel;
      } 
    }
  } 

  unsigned print_palette() { // print the computed color volumes & palette
    if (splitted) { return(p1->print_palette() + p2->print_palette()); } 
    else {     
      printf("%3d = %3d ",pind, pixtab[pind]);
      return (print_vol());
    }
  }
};

void set_auto_pal555();

// palette calculation from captured frame: 
// uses recursive discrimination of color space
void capture_palette() {
  if (cap_buf == NULL) return;
  int palsz = 32*32*32; // palette size of grabbed frames
  unsigned int colsum[palsz];
  memset(colsum, 0, 4*palsz);
  unsigned (*csu)[32][32] =  (unsigned (*)[32][32]) colsum;
  int i, npix =  biWidth*biHeight;

  // compute color frequency from last grabbed frame
  for (i=0; i < npix; i++) if (cap_buf[i] >= palsz) { error("6-bit"); }
  for (i=0; i < npix; i++) colsum[cap_buf[i]]++; 
  volume v0 = { { 0, 0, 0} , { 32, 32, 32 } };
  if (ptop) delete ptop;
  ptop = new partition (colsum,v0);
  for (i = 1; i < ncols ; i++) {
    global_max = 0; 
    // find the best cut, if False returned no more splitting possible
    if (! ptop->test_split() ) break;
    ptop->do_split(); // do it
    printf("."); fflush(stdout);
  } 
  printf("\nauto palette with %d colors captured\n",i);
  set_auto_pal555();
}

// set palette to computed ptop in get_auto_pal555; from captured frame
void set_auto_pal555() {
  if (ptop == NULL) capture_palette(); // first get it !
  ptop->pind = 0;
  ptop->set_palette();
  map_and_redraw();
}

// print color freq list of the last computed palette 
void freq_list() { 
  if (ptop) { 
    unsigned ntot = ptop->print_palette(); 
    printf("total = %d elements\n",ntot);
  }
}

scrollbar *wscr;

// callback for scrollbar
void wscr_cb() { ncols = wscr->value; }

main_window *eval_popup = NULL;

int color_info = False; // display color values in display line (Motion_CB)

// create a popup window to set "ncols" with scrollbar and start capture
void make_eval_popup() {
  if (eval_popup == NULL) {
    eval_popup = new main_window("evaluation settings",200,160);
    window *descr = new text_win(*eval_popup,"color number for palette",
				200,20,0,5);
    int x = 10, y = 30;
    wscr = new scrollbar(*eval_popup,wscr_cb,180,20,x,y,1,max_cols,ncols);
    y+= 25;
    menu_bar *mb = new menu_bar(*eval_popup,180,20,x,y,30,100,0); y += 30;
    button *bb[] = { 
	new callback_button(*mb,"new palette",capture_palette),
	new callback_button(*mb,"list palette",freq_list) };

    window *d2 = new text_win(*eval_popup,"display line modes",200,20,0,y); 
    y+= 25;
    menu_bar *mb2 = new menu_bar(*eval_popup,180,20,x,y,30,100,0); y += 25;
    button *b2[] = { new toggle_button(*mb2,"color",&color_info) };

    button *ub = new unmap_button(*eval_popup,"close",80,20,60,y);
  }
  eval_popup->RealizeChildren();
}
