////////////////////////////////////////////////////////////////////////////////
//  implementation of an internal BSpline utility class                       //  
//  LAST EDIT: Wed Mar  8 10:31:57 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include "bspline.h"

int _bspline::parVecDim, _bspline::matRow, _bspline::matCol, _bspline::knotVecDim, _bspline::err;

int _bspline::checkValues() {
    if (numVal < 1) {
	printf(" Bad number of values: %d, must be > 2!\n", numVal);
	return 1;
    }
    if (dim < 1) {
	printf(" Bad dimension: %d !\n", dim);
	return 1;
    }
    if (numVal < order || order < 1) {
	printf(" It must be: order <= numVal!\n");
	return 1;
    }
    if ( !(mode == _OPEN || mode == _CLOSED) ) {
	printf(" No such mode: %d !\n", (int)mode );
	return 1;
    }
    return 0;
}
	
void _bspline::setLimits() {
    // set dimensions for arrays:
    switch (mode) {
      case _OPEN:	
	parVecDim = dim;
	matRow = dim;
	matCol = numVal;
	knotVecDim = numVal+order;
	tLimit = numVal-1-order+2;
	break;
      case _CLOSED:
	parVecDim = dim;
	matRow = dim;
	matCol = numVal + order - 1;
	knotVecDim = numVal+2*order-1;
	tLimit = numVal;
    }
}

_bspline::_bspline() {
    numVal = order = dim = splChanged = 0;
    mode = _OPEN;
    tLimit = 0;
}       

_bspline::_bspline( int a, int b, int c, _SplineMode d ) {
    numVal = a;
    order = b;
    dim = c;
    mode = d;
    splChanged = 0;
    
    if( !checkValues() ) {
	setLimits();
	
	// create a new knot vector, new parameter vector and a new matrix:
	T.newGeometry( parVecDim );
	N.newGeometry( matRow, matCol );
	knotV.newGeometry( knotVecDim );
	
	createMatrix();
    } else {
	printf(" Can not compute such a b-spline!\n"); }
}

_bspline::~_bspline() {
    //delete N;
    //delete T;
}

int _bspline::update() {
    if( !splChanged ) return 0;
    if( !checkValues() ) {
	splChanged = 0;
	setLimits();
	createMatrix();
	return 0;
    } 
    return 1;
}
 
void _bspline::createMatrix() {
    int i,j;
    
    computeKnotVector(); 
   
    // reallocate parameter vector and matrix if necessary:
    if ( T.dim() != parVecDim ) T.newGeometry( parVecDim );
    
    // initialize matrix
    if( (N.Rows() != matRow) || (N.Cols() != matCol ) )
	N.newGeometry( matRow, matCol );
    
    // initialize values, from now the dimension of N is (N.getRow, N.getCol),
    // and the dimension of T is (T.getDim):
    
    // maximum parameter value, the given spline can interpolate:
    double t = 0;
    double dt = tLimit/(dim - 1);
    
    for( i = 0; i < T.dim(); i++ ) {
	T[i] = t; t = t + dt;
    }
    
    for( i = 0; i < N.Rows(); i++ ) 
	for( j = 0; j < N.Cols(); j++ ) 
	    N( i,j ) = blend( j, order, T[i] );
    
    // for closed splines, the last value = 1:
    if (mode == _OPEN)	N( N.Rows()-1, N.Cols()-1 ) = 1.0;
}

void _bspline::computeKnotVector() {
    int i;

    if( knotV.dim() != knotVecDim )
	knotV.newGeometry( knotVecDim );

    switch( mode ) {
      case _OPEN:	    
	for( i = 0; i < knotV.dim(); i++ ) {
	    if (i < order) knotV[i] = 0;  
	    if (order <= i && i < numVal) knotV[i] = i-order+1;
	    if(i >= numVal) knotV[i] = numVal - 1 - order + 2; 
	}
	break;
      case _CLOSED:
	knotV[0] = 1-order;
	for( i = 1; i < knotV.dim(); i++ ) 
	    knotV[i] = knotV[i-1] + 1;
	break;
    }
}

double _bspline::blend( int j, int f, double t) {
    double value;
       
    if( f == 1 ) {
	if (knotV[j] <= t && t < knotV[j + 1])
	    value = 1;
	else
	    value = 0;
    }
    else {  
	// check for divide by zero
	if (knotV[j+f-1] == knotV[j] && knotV[j+f] == knotV[j+1])
	    value = 0.0;
	else
	    if (knotV[j+f-1] == knotV[j]) // if denominator = 0, get deeper...
		value = (double)( (knotV[j+f]-t)/(knotV[j+f]-knotV[j+1])*blend(j+1,f-1,t) );
	    else
		if (knotV[j+f] == knotV[j+1])
			value = (double)( ( (t-knotV[j])/(knotV[j+f-1]-knotV[j]) )*blend(j,f-1,t) );
		    else
			value = (double)( (t-knotV[j])/(knotV[j+f-1]-knotV[j])*blend(j,f-1,t) +
					 (knotV[j+f]-t)/(knotV[j+f]-knotV[j+1])*blend(j+1,f-1,t) );
    }
    return value;
}

const _Vector _bspline::getParVector() const {
    return T;
}

const _Matrix _bspline::getBase() const {
    return N;
}





