////////////////////////////////////////////////////////////////////////////////
//  Source code of Classes for Scientific Visualization                       //  
//  LAST EDIT: Fri Jul  1 12:39:05 1994 by Barth(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//	        yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <scient/3d_lat.h>

// functions to read sveral kinds of parameters on a specified vertex.
// To integrate a new parameter - type, you have to clone such function.
void l3DgetDfunc( const ParamSet &pS, int _x, int _y, int _z, int dimX, int dimY, double *p ) {
    *p = pS.dData[ _z * dimX * dimY + _x * dimY + _y ];
}
void l3DgetFfunc( const ParamSet &pS, int _x, int _y, int _z, int dimX, int dimY, double *p ) {
    *p = pS.fData[ _z * dimX * dimY + _x * dimY + _y ];
}
void l3DgetIfunc( const ParamSet &pS, int _x, int _y, int _z, int dimX, int dimY, double *p ) {
    *p = pS.iData[ _z * dimX * dimY + _x * dimY + _y ];
}
void l3DgetCfunc( const ParamSet &pS, int _x, int _y, int _z, int dimX, int dimY, double *p ) {
    *p = pS.cData[ _z * dimX * dimY + _x * dimY + _y ];
}
void l3DgetDefaultfunc( const ParamSet & , int , int , int , int , double * ) {
    rt_Output->errorVar(" No parameters given. ",0 );
}

// functions to write sveral kinds of parameters on a specified vertex.
// To integrate a new parameter - type, you have to clone such function.
void RT_3DLattice::setParameter(int pnr, int _x, int _y, int _z, double _p ) {
    paramData[pnr].dData[ _z * dims[0] * dims[1] + _x * dims[1] + _y ] = _p;
}
void RT_3DLattice::setParameter(int pnr, int _x, int _y, int _z, float _p ) {
    paramData[pnr].fData[ _z * dims[0] * dims[1] + _x * dims[1] + _y ] = _p;
}
void RT_3DLattice::setParameter(int pnr, int _x, int _y, int _z, int _p ) {
    paramData[pnr].iData[ _z * dims[0] * dims[1] + _x * dims[1] + _y ] = _p;
}
void RT_3DLattice::setParameter(int pnr, int _x, int _y, int _z, char _p ) {
    paramData[pnr].cData[ _z * dims[0] * dims[1] + _x * dims[1] + _y ] = _p;
}

void RT_3DLattice::create() { 
// for further implementations
// until now 3DLattice only distributes functionality to it\'s
// 2Dlattices.
// In the future there will be own visualisation methods, for 
// example the famous "Marching cube"
}

// A plane (XY, XZ or YZ) will be positioned in the volume grid.
// The position is specified by a number of steps at the grid in 
// the corresponding direction
void RT_3DLattice::walkto( RT_PlaneType _pt, int _wpnr ) {
    double pm, *pvec, vtx[4];
    int pp, indices[3];
    double pmin, pmax;
    int pnr, i, j;
    switch( _pt ) {
      case XY: 
	pvec = new double[ dims[0] * dims[1] ];
	for ( pnr=0; pnr<MAX_PARAM; pnr++ ) if (paramData[pnr].loaded) {
	    pp = 0;
	    for ( i=0; i<dims[0]; i++) {
		for ( j=0; j<dims[1]; j++ ) {
		    getParameter( pnr, i, j, _wpnr, &pm );
		    pvec[pp++] = pm;
		}
	    }
	    walkPlane[ XY ]->loadParameters(pnr, RTSC_DOUBLE, (double*)pvec );
	    getParExtremes( pnr, &pmin, &pmax );
	    walkPlane[ XY ]->setParExtremes( pnr, pmin, pmax );
	}
	walkPlane[ XY ]->translate( RT_Vector( 0, 0, -walkerShiftXY));
	indices[0] = indices[1] = 0; indices[2] = _wpnr;
	getVertex( indices, vtx );
	walkerShiftXY = vtx[2];
	walkPlane[ XY ]->translate( RT_Vector( 0, 0, walkerShiftXY ));
	walkPlane[ XY ]->geomChanged();
	break;
      case XZ:
	pvec = new double[ dims[2] * dims[0] ];
	for ( pnr=0; pnr<MAX_PARAM; pnr++ ) if (paramData[pnr].loaded) {
	    pp = 0;
	    for ( i=0; i<dims[0]; i++) {
		for ( j=0; j<dims[2]; j++ ) {
		    getParameter( pnr, i, _wpnr, j, &pm );
		    pvec[pp++] = pm;
		}
	    }
	    walkPlane[ XZ ]->loadParameters(pnr, RTSC_DOUBLE, (double*)pvec );
	    getParExtremes( pnr, &pmin, &pmax );
	    walkPlane[ XZ ]->setParExtremes( pnr, pmin, pmax );
	}
	walkPlane[ XZ ]->translate( RT_Vector( 0, 0, walkerShiftXZ ));
	indices[0] = indices[2] = 0; indices[1] = _wpnr;
	getVertex( indices, vtx );
	walkerShiftXZ = vtx[1];
	walkPlane[ XZ ]->translate( RT_Vector( 0, 0, -walkerShiftXZ ));
	walkPlane[ XZ ]->geomChanged();
	break;
      case YZ:
	pvec = new double[ dims[2] * dims[1] ];
	for ( pnr=0; pnr<MAX_PARAM; pnr++ ) if (paramData[pnr].loaded) {
	    pp = 0;
	    for ( i=0; i<dims[1]; i++) {
		for ( j=0; j<dims[2]; j++ ) {
		    getParameter( pnr, _wpnr, i, j, &pm );
		    pvec[pp++] = pm;
		}
	    }
	    walkPlane[ YZ ]->loadParameters(pnr, RTSC_DOUBLE, (double*)pvec );
	    getParExtremes( pnr, &pmin, &pmax );
	    walkPlane[ YZ ]->setParExtremes( pnr, pmin, pmax );
	}
	walkPlane[ YZ ]->translate( RT_Vector( 0, 0, -walkerShiftYZ ));
	indices[1] = indices[2] = 0; indices[0] = _wpnr;
	getVertex( indices, vtx );
	walkerShiftYZ = vtx[0];
	walkPlane[ YZ ]->translate( RT_Vector( 0, 0, walkerShiftYZ ));
	walkPlane[ YZ ]->geomChanged();
	break;
    }
    if (pvec) delete pvec;
}

void RT_3DLattice::initPlanes() {
// The 3DLattice is represented by 6 2DLattices, the borderplanes.
// Additionally, 3 2DLattices ( the walkplanes ) can be sent through
// the volume. The geometry of all these planes is initialized here
    double *xvec = new double[dims[0]];
    double *yvec = new double[dims[1]];
    double *zvec = new double[dims[2]];
    double vtx[4], minC[4], maxC[4], minQ[4];
    int indices[3];
    indices[1] = indices[2] = 0;
    for ( int i=0; i<dims[0]; i++ ) {
	indices[0] = i;
	getVertex( indices, vtx );
	xvec[i] = vtx[0];
    }
    indices[0] = indices[2] = 0;
    for ( i=0; i<dims[1]; i++ ) {
	indices[1] = i;
	getVertex( indices, vtx );
	yvec[i] = vtx[1];
    }
    indices[0] = indices[1] = 0;
    for ( i=0; i<dims[2]; i++ ) {
	indices[2] = i;
	getVertex( indices, vtx );
	zvec[i] = vtx[2];
    }
    pl2d[ BACK_PLANE ]->loadGeometry( dims[0], dims[1], xvec, yvec );
    indices[0] = indices[1] = indices[2] = 0;
    getVertex( indices, minC );
    indices[0] = dims[0]-1; indices[1] = dims[1]-1;
    indices[2] = dims[2]-1;
    getVertex( indices, maxC );

    pl2d[ BACK_PLANE ]->translate( RT_Vector( 0, 0, minC[2] ));
    pl2d[ FRONT_PLANE ]->loadGeometry( dims[0], dims[1], xvec, yvec );
    pl2d[ FRONT_PLANE ]->translate( RT_Vector( 0, 0, maxC[2] ));
    walkPlane[ XY ]->loadGeometry( dims[0], dims[1], xvec, yvec );

    pl2d[ BOTTOM_PLANE ]->loadGeometry( dims[0], dims[2], xvec, zvec );
    pl2d[ TOP_PLANE ]->loadGeometry( dims[0], dims[2], xvec, zvec );
    indices[0] = indices[1] = indices[2] = 0;
    pl2d[ TOP_PLANE ]->getVertex( indices, minQ );
    indices[0] = dims[0]-1; indices[1] = dims[1]-1; indices[2] = dims[2]-1;
    walkPlane[ XZ ]->loadGeometry( dims[0], dims[2], xvec, zvec );
    walkPlane[ XZ ]->rotate( RT_Vector( M_PI_2, 0, 0 ));

    pl2d[ TOP_PLANE ]->translate( RT_Vector( 0.0, maxC[1], -minQ[2]));
    pl2d[ TOP_PLANE ]->rotate( RT_Vector( M_PI_2, 0, 0 ));
    pl2d[ BOTTOM_PLANE ]->translate( RT_Vector( 0.0, minC[1], -minQ[2]));
    pl2d[ BOTTOM_PLANE ]->rotate( RT_Vector( M_PI_2, 0, 0 ));

    pl2d[ LEFT_PLANE ]->loadGeometry( dims[1], dims[2], yvec, zvec );
    pl2d[ RIGHT_PLANE ]->loadGeometry( dims[1], dims[2], yvec, zvec );
    indices[0] = indices[1] = indices[2] = 0;
    pl2d[ LEFT_PLANE ]->getVertex( indices, minQ );
    indices[0] = dims[0]-1; indices[1] = dims[1]-1; indices[2] = dims[2]-1;
    pl2d[ LEFT_PLANE ]->translate( RT_Vector( minC[0], 0.0, -minQ[2]));
    pl2d[ LEFT_PLANE ]->rotate( RT_Vector( 0, M_PI_2, M_PI_2 ));
    pl2d[ RIGHT_PLANE ]->translate( RT_Vector( maxC[0], 0.0, -minQ[2]));
    pl2d[ RIGHT_PLANE ]->rotate( RT_Vector( 0, M_PI_2, M_PI_2 ));

    walkPlane[ YZ ]->loadGeometry( dims[1], dims[2], yvec, zvec );
    walkPlane[ YZ ]->rotate( RT_Vector( 0, M_PI_2, M_PI_2 ));
}

void RT_3DLattice::createGeometry() {
}
 
void RT_3DLattice::createParameters() {
}

void RT_3DLattice::setVertex(int *co, double *vt) {
    if (!geometry) {
        rt_Output->errorVar( get_name(), ": Bad vertex access! No geometry field initialized.", 0 );
        return;
    }
    int offset = 0; 
    switch (get_geomDim()) {
      case 3: offset += co[2] * dims[0] * dims[1];   // tdims[1];
      case 2: offset += co[1] * dims[0];             // tdims[0];
      case 1: offset += co[0];
    break;
      default:
      rt_Output->errorVar( get_name(), ": Bad geometrical dimensions specified!", 0 );
    };
    double *tmp = &geometry[offset * coord];
    for (int i = 0; i < coord; i++) *tmp++ = *vt++;
}

void RT_3DLattice::getVertex(int *co, double *vt)const {
    if (!geometry) {
        rt_Output->errorVar( get_name(), ": Bad vertex access! No geometry field initialized.", 0 );
        return;
    }
    int offset = 0; 
    switch (get_geomDim()) {
      case 3: offset += co[2] * dims[0] * dims[1];    // tdims[1];
      case 2: offset += co[1] * dims[0];              // tdims[0];
      case 1: offset += co[0];
    break;
      default:
      rt_Output->errorVar( get_name(), ": Bad geometrical dimensions specified!", 0 );
    };
    double *tmp = &geometry[offset * coord];
    for (int i = 0; i < coord; i++) *vt++ = *tmp++;
}

int  RT_3DLattice::walkF, RT_3DLattice::parF;
char *RT_3DLattice::walkV, *RT_3DLattice::parV;

RT_ParseEntry RT_3DLattice::table[] = {
    { "-walkto", RTP_STRING, (char*)&walkV, &walkF, "Specify {ARG 1 PlaneType} {ARG 2 Step} for the plane to display.",RTPS_STRING },
    { "-loadParameters", RTP_STRING, (char*)&parV, &parF, "Specify the {ARG 1 ParameterIdentifier} {ARG 2 ParameterList} for the lattice.",RTPS_STRING },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_3DLattice::objectCMD(char *argv[]) {
    int t = RT_Scientific::objectCMD( argv );
    RT_parseTable( argv, table );
    if (walkF) {                  
	char **xargv; int xargc;
	int _ptype, _wpnr;
	if ( Tcl_SplitList( rt_Ip, walkV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 2 ) {
		RT_string2int( xargv[0], _ptype );
		RT_string2int( xargv[1], _wpnr );
		walkto( (RT_PlaneType)_ptype, _wpnr );
		t++;
	    }
	    free((char*)xargv);
	}
    }
    if (parF) { 
	char **xargv; int xargc;
	int _parId; double *_p = 0;
	if ( Tcl_SplitList( rt_Ip, parV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 2 ) {
		RT_string2int( xargv[0], _parId );
		char **xxargv; int xxargc;
		if ( Tcl_SplitList( rt_Ip, xargv[1], &xxargc, &xxargv) == TCL_OK) {
		    _p = new double[ xxargc ];
		    for ( int i=0; i<xxargc; i++ ) RT_string2double( xxargv[i], _p[i] );
		    loadParameters( _parId, RTSC_DOUBLE, _p );
		    t++;
		    free((char*)xxargv);
		}
	    }
	    free((char*)xargv);
	    if (_p) delete _p;
	}
    }
    return t;
}

void RT_3DLattice::getParameter(int pnr, int _x, int _y, int _z, double *p )const {
    l3Dgetfunc[pnr]( paramData[pnr], _x, _y, _z, dims[0], dims[1], p );
}

void RT_3DLattice::setParamType( int _np, RT_ScType _pt ) {
    paramData[ _np ].paramType = _pt;
    switch ( _pt ) {
      case RTSC_DOUBLE: l3Dgetfunc[_np] = l3DgetDfunc; break;
      case RTSC_FLOAT: l3Dgetfunc[_np] = l3DgetFfunc; break;
      case RTSC_INTEGER: l3Dgetfunc[_np] = l3DgetIfunc; break;
      case RTSC_CHAR: l3Dgetfunc[_np] = l3DgetCfunc; break;
    }
}

void RT_3DLattice::loadParameters( int _np, RT_ScType _pt, double *_p ) {
    if ( _pt != RTSC_DOUBLE ) {
	rt_Output->warningVar(" RTSC_DOUBLE is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar(" Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar(" Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    for ( int k=0; k < dims[2]; k++ ) {
		setParameter( _np, i, j, k, _p[ dims[0] * dims[1] * k + i * dims[1] + j] );
	    }
	}
    }
    setPlaneParameters(_np);
}


void RT_3DLattice::loadParameters( int _np, RT_ScType _pt, float *_p ) {
    if ( _pt != RTSC_FLOAT ) {
	rt_Output->warningVar(" RTSC_FLOAT is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar(" Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar(" Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    for ( int k=0; k < dims[2]; k++ ) {
		setParameter( _np, i, j, k, _p[ dims[0] * dims[1] * k + i * dims[1] + j] );
	    }
	}
    }
    setPlaneParameters(_np);
}

void RT_3DLattice::loadParameters( int _np, RT_ScType _pt, int *_p ) {
    if ( _pt != RTSC_INTEGER ) {
	rt_Output->warningVar(" RTSC_INTEGER is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar(" Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar(" Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    for ( int k=0; k < dims[2]; k++ ) {
		setParameter( _np, i, j, k, _p[ dims[0] * dims[1] * k + i * dims[1] + j] );
	    }
	}
    }
    setPlaneParameters(_np);
}

void RT_3DLattice::loadParameters( int _np, RT_ScType _pt, char *_p ) {
    if ( _pt != RTSC_CHAR ) {
	rt_Output->warningVar(" RTSC_CHAR is not the type of the corresponding parameter-array. This may cause problems later. ", 0 );
	return;
    }
    if ( _np > MAX_PARAM ) {
	char tmp[20];
	sprintf( tmp, "%d", MAX_PARAM );
	rt_Output->errorVar(" Sorry, only ", tmp, " parameters per vertex allowed.", 0 );
	return;
    }
    if ( paramData[ _np ].loaded ) {
	char tmp[20];
	sprintf( tmp, "%d", _np );
	rt_Output->warningVar(" Parameter number ", tmp, " was in use. Now it is overloaded", 0 );
    }
    setParamType( _np, _pt );
    currParNr = _np;
    paramData[ currParNr ].loaded = 1;
    newParameters();
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    for ( int k=0; k < dims[2]; k++ ) {
		setParameter( _np, i, j, k, _p[ dims[0] * dims[1] * k + i * dims[1] + j] );
	    }
	}
    }
    setPlaneParameters(_np);
}

void RT_3DLattice::setPlaneParameters(int _pnr) {
// the corresponding parameters from the volume must be distributed 
// to the borderplanes
    double *tmp, *tmp2;
    tmp = new double[ dims[0] * dims[1] ];
    tmp2 = new double[ dims[0] * dims[1] ];
    int pp=0; double p;
    for ( int i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[1]; j++ ) {
	    getParameter(_pnr, i, j, 0, &p ); tmp[pp] = p;
	    getParameter(_pnr, i, j, dims[2]-1, &p ); tmp2[pp] = p;
	    pp++;
	}
    }
    pl2d[FRONT_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp2 );
    pl2d[BACK_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp );
    delete tmp; delete tmp2;
    tmp = new double[ dims[2] * dims[1] ];
    tmp2 = new double[ dims[2] * dims[1] ];
    pp=0;
    for ( i=0; i < dims[1]; i++ ) {
	for ( int j=0; j < dims[2]; j++ ) {
	    getParameter(_pnr, 0, i, j, &p ); tmp[pp] = p;
	    getParameter(_pnr, dims[0]-1, i, j, &p ); tmp2[pp] = p;
	    pp++;
	}
    }
    pl2d[LEFT_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp );
    pl2d[RIGHT_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp2 );
    delete tmp; delete tmp2;
    tmp = new double[ dims[2] * dims[0] ];
    tmp2 = new double[ dims[2] * dims[0] ];
    pp=0;
    for ( i=0; i < dims[0]; i++ ) {
	for ( int j=0; j < dims[2]; j++ ) {
	    getParameter(_pnr, i, 0, j, &p ); tmp[pp] = p;
	    getParameter(_pnr, i, dims[1]-1, j, &p ); tmp2[pp] = p;
	    pp++;
	}
    }
    pl2d[BOTTOM_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp );
    pl2d[TOP_PLANE]->loadParameters(_pnr, RTSC_DOUBLE, tmp2 );
    delete tmp; delete tmp2;
    double pmin, pmax;
    getParExtremes( _pnr, &pmin, &pmax );
    for ( i=0; i<6; i++ ) pl2d[i]->setParExtremes( _pnr, pmin, pmax );
}

void RT_3DLattice::getParExtremes( int _pnr, double *pmin, double *pmax ) {
    double p;
    getParameter( _pnr, 0, 0, 0, pmin );
    getParameter( _pnr, 0, 0, 0, pmax );
    for ( int i=0; i<dims[0]; i++ ) {
	for ( int j=0; j<dims[1]; j++ ) {
	    for ( int k=0; k<dims[2]; k++ ) {
		getParameter( _pnr, i, j, k, &p );
		*pmin = ( *pmin < p ) ? *pmin : p;
		*pmax = ( *pmax > p ) ? *pmax : p;
	    }
	}
    }
}

///////////////////////////////////////////////////////////////////////////////
//     Implementation of a 3D uniform lattice
///////////////////////////////////////////////////////////////////////////////

void RT_3DUniformLattice::createGeometry() {
}
 
void RT_3DUniformLattice::createParameters() {
}

void RT_3DUniformLattice::loadGeometry( int _nx, int _ny, int _nz, double _x0, double _y0, double _z0, double _dx, double _dy, double _dz ) {
    if ((_nx<=0) || (_ny<=0) || (_nz<=0)) {
	rt_Output->errorVar( get_name(), ": All dimensions must be greater than 0. ", 0 );
	return;
    }
    dims[0] = _nx;
    dims[1] = _ny;
    dims[2] = _nz;
    newGeometry();
    newDims = 1;
    int indices[3];
    double vtx[4];
    double x = x0 = _x0; 
    double y = y0 = _y0; 
    double z = z0 = _z0;
    dx = _dx; dy = _dy; dz = _dz;
    for ( int i=0; i < _nx; i++ ) {
	for ( int j=0; j < _ny; j++ ) {
	    for ( int k=0; k<_nz; k++ ) {
		indices[0] = i; indices[1] = j; indices[2] = k;
		vtx[0] = x; 
		vtx[1] = y;
		vtx[2] = z;
		vtx[3] = 0.0;
		setVertex( indices, vtx );
		z += _dz;
	    }
	    y += _dy;
	    z = _z0;
	}
	x += _dx;
	y = _y0;
    }
    initPlanes();
}

int RT_3DUniformLattice::ugeoF;
char *RT_3DUniformLattice::ugeoV;

RT_ParseEntry RT_3DUniformLattice::table[] = {
    { "-loadGeometry", RTP_STRING, (char*)&ugeoV, &ugeoF, "Specify {ARG 1 DimX}, {ARG 2 DimY}, {ARG 3 DimZ}, {ARG 4 X0}, {ARG 5 Y0}, {ARG 6 Z0}, {ARG 7 DX}, {ARG 8 DY}, {ARG 9 DZ} for the geometry of the lattice.",RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_3DUniformLattice::objectCMD(char *argv[]) {
    int t = RT_3DLattice::objectCMD( argv );
    RT_parseTable( argv, table );
    if (ugeoF) { 
	char **xargv; int xargc;
	int nx, ny, nz;
	double x0, y0, z0, dx, dy, dz;
	if ( Tcl_SplitList( rt_Ip, ugeoV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 9 ) {
		RT_string2int( xargv[0], nx );
		RT_string2int( xargv[1], ny );
		RT_string2int( xargv[2], nz );
		RT_string2double( xargv[3], x0 );
		RT_string2double( xargv[4], y0 );
		RT_string2double( xargv[5], z0 );
		RT_string2double( xargv[6], dx );
		RT_string2double( xargv[7], dy );
		RT_string2double( xargv[8], dz );
		loadGeometry( nx, ny, nz, x0, y0, z0, dx, dy, dz );
		t++;
	    }
	    free((char*)xargv);
	}
    }
    return t;
}

int RT_3DUniformLattice::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", RTN_3D_U_LATTICE, " { String } {  Creates a 3D uniform lattice. {ARG 1 Name} of the new object.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 2) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_3D_U_LATTICE, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	new RT_3DUniformLattice( argv[1] );
	return TCL_OK;
    }
    else return TCL_ERROR;
}


int RT_3DUniformLattice::isA(const char *_c)const {
    if (RTM_isA(_c, RTN_3D_U_LATTICE )) return 1;
    return RT_Scientific::isA( _c );
}

///////////////////////////////////////////////////////////////////////////////
//     Implementation of a 3D perimetric lattice
///////////////////////////////////////////////////////////////////////////////

void RT_3DPerimetricLattice::createGeometry() {
}
 
void RT_3DPerimetricLattice::createParameters() {
}

void RT_3DPerimetricLattice::loadGeometry( int _nx, int _ny, int _nz, double *_x, double *_y, double *_z ) {
    if ((_nx<=0) || (_ny<=0) || (_nz<=0)) {
	rt_Output->errorVar( get_name(), ": All dimensions must be greater than 0. ", 0 );
	return;
    }
    dims[0] = _nx;
    dims[1] = _ny;
    dims[2] = _nz;
    newGeometry();
    newDims = 1;
    int indices[3];
    double vtx[4];
    for ( int i=0; i < _nx; i++ ) {
	for ( int j=0; j < _ny; j++ ) {
	    for ( int k=0; k<_nz; k++ ) {
		indices[0] = i; indices[1] = j; indices[2] = k;
		vtx[0] = _x[i]; 
		vtx[1] = _y[j];
		vtx[2] = _z[k];
		vtx[3] = 0.0;
		setVertex( indices, vtx );
	    }
	}
    }
    initPlanes();
}

int RT_3DPerimetricLattice::pgeoF;
char *RT_3DPerimetricLattice::pgeoV;

RT_ParseEntry RT_3DPerimetricLattice::table[] = {
    { "-loadGeometry", RTP_STRING, (char*)&pgeoV, &pgeoF, "Specify {ARG 1 DimX}, {ARG 2 DimY}, {ARG 3 DimZ}, {ARG 4 X0}, {ARG 5 Y0}, {ARG 6 Z0}, {ARG 7 DX}, {ARG 8 DY}, {ARG 9 DZ} for the geometry of the lattice.", RTPS_STRING },
	{ 0, RTP_END, 0, 0, 0, 0 }
};

int RT_3DPerimetricLattice::objectCMD(char *argv[]) {
    int t = RT_3DLattice::objectCMD( argv );
    RT_parseTable( argv, table );
    if (pgeoF) { 
	char **xargv; int xargc;
	int nx, ny, nz;
	double *xvec, *yvec, *zvec;
	if ( Tcl_SplitList( rt_Ip, pgeoV, &xargc, &xargv) == TCL_OK) {
	    if ( xargc == 6 ) {
		RT_string2int( xargv[0], nx );
		RT_string2int( xargv[1], ny );
		RT_string2int( xargv[2], nz );
		xvec = new double[ nx ]; yvec = new double[ ny ]; zvec = new double[ nz ];
		char **xxargv; int xxargc;
		if ( Tcl_SplitList( rt_Ip, xargv[3], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == nx ) for ( int i=0; i<nx; i++ ) RT_string2double( xxargv[i], xvec[i] );
		    free((char*)xxargv);
		}
		if ( Tcl_SplitList( rt_Ip, xargv[4], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == ny ) for ( int i=0; i<ny; i++ ) RT_string2double( xxargv[i], yvec[i] );
		    free((char*)xxargv);
		}
		if ( Tcl_SplitList( rt_Ip, xargv[5], &xxargc, &xxargv) == TCL_OK) {
		    if ( xxargc == nz ) for ( int i=0; i<nz; i++ ) RT_string2double( xxargv[i], zvec[i] );
		    free((char*)xxargv);
		}
		loadGeometry( nx, ny, nz, xvec, yvec, zvec );
		if (xvec) delete xvec;
		if (yvec) delete yvec;
		if (zvec) delete zvec;
		t++;
	    }
	    free((char*)xargv);
	}
    }
    return t;
}

int RT_3DPerimetricLattice::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", RTN_3D_P_LATTICE, " { String } {  Creates a 3D perimetric lattice. {ARG 1 Name} of the new object.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 2) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_3D_P_LATTICE, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	new RT_3DPerimetricLattice( argv[1] );
	return TCL_OK;
    }
    else return TCL_ERROR;
}

int RT_3DPerimetricLattice::isA(const char *_c)const {
    if (RTM_isA(_c, RTN_3D_P_LATTICE )) return 1;
    return RT_Scientific::isA( _c );
}
