/*-
 * Copyright (c) 1992, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 * Copyright (c) 1994, 1995
 *	Keith Bostic.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char sccsid[] = "@(#)svi_get.c	9.6 (Berkeley) 1/30/95";
#endif /* not lint */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/time.h>

#include <bitstring.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>

#include "compat.h"
#include <db.h>
#include <regex.h>

#include "vi.h"
#include "../vi/vcmd.h"
#include "svi_screen.h"

/*
 * svi_get --
 *	Fill a buffer from the terminal for vi.
 */
input_t
svi_get(sp, tiqh, prompt, flags)
	SCR *sp;
	TEXTH *tiqh;
	ARG_CHAR_T prompt;
	u_int flags;
{
	SMAP *esmp;
	SVI_PRIVATE *svp;
	TEXT *tp;
	recno_t sv_lno, sv_tm_lno;
	size_t cnt, sv_cno, sv_t_maxrows, sv_t_minrows, sv_t_rows, sv_tm_off;
	int eval;

	svp = SVP(sp);

	/* Save current cursor. */
	sv_lno = sp->lno;
	sv_cno = sp->cno;

	if (!IS_ONELINE(sp)) {
		/*
		 * Fake like the user is doing input on the last line of the
		 * screen.  This makes all of the scrolling work correctly,
		 * and allows us the use of the vi text editing routines, not
		 * to mention practically infinite length ex commands.
		 *
		 * Save the current location.
		 */
		sv_tm_lno = TMAP->lno;
		sv_tm_off = TMAP->off;
		sv_t_rows = sp->t_rows;
		sv_t_minrows = sp->t_minrows;
		sv_t_maxrows = sp->t_maxrows;

		/*
		 * If it's a small screen, TMAP may be small for the screen.
		 * Fix it, filling in fake lines as we go.
		 */
		if (IS_SMALL(sp))
			for (esmp =
			    HMAP + (sp->t_maxrows - 1); TMAP < esmp; ++TMAP) {
				TMAP[1].lno = TMAP[0].lno + 1;
				TMAP[1].off = 1;
			}

		/* Build the fake entry. */
		TMAP[1].lno = TMAP[0].lno + 1;
		TMAP[1].off = 1;
		SMAP_FLUSH(&TMAP[1]);
		++TMAP;

		/* Reset the screen information. */
		sp->t_rows = sp->t_minrows = ++sp->t_maxrows;
	}

	/* Move to the last line. */
	sp->lno = TMAP[0].lno;
	sp->cno = 0;

	/* Don't update the modeline for now. */
	F_SET(svp, SVI_INFOLINE);

	if (O_ISSET(sp, O_ALTWERASE))
		LF_SET(TXT_ALTWERASE);
	if (O_ISSET(sp, O_TTYWERASE))
		LF_SET(TXT_TTYWERASE);
	LF_SET(TXT_APPENDEOL |
	    TXT_CR | TXT_ESCAPE | TXT_INFOLINE | TXT_MAPINPUT);
	eval = v_ntext(sp, tiqh, NULL, NULL, 0, NULL, prompt, 0, flags);

	F_CLR(svp, SVI_INFOLINE);

	if (!IS_ONELINE(sp)) {
		/* Restore the screen information. */
		sp->t_rows = sv_t_rows;
		sp->t_minrows = sv_t_minrows;
		sp->t_maxrows = sv_t_maxrows;

		/*
		 * If it's a small screen, TMAP may be wrong.  Clear any
		 * lines that might have been overwritten.
		 */
		if (IS_SMALL(sp)) {
			for (cnt = sp->t_rows; cnt <= sp->t_maxrows; ++cnt) {
				(void)svp->scr_move(sp, RLNO(sp, cnt), 0);
				(void)svp->scr_clrtoeol(sp);
			}
			TMAP = HMAP + (sp->t_rows - 1);
		} else
			--TMAP;

		/*
		 * The map may be wrong if the user entered more than one
		 * (logical) line.  Fix it.  If the user entered a whole
		 * screen, this will be slow, but we probably don't care.
		 */
		while (sv_tm_lno != TMAP->lno || sv_tm_off != TMAP->off)
			if (svi_sm_1down(sp))
				return (INP_ERR);
	} else
		F_SET(sp, S_SCR_REDRAW);

	/*
	 * Invalidate the cursor and the line size cache, the line never
	 * really existed.  This fixes bugs where the user searches for
	 * the last line on the screen + 1 and the refresh routine thinks
	 * that's where we just were.
	 */
	F_SET(svp, SVI_CUR_INVALID);
	SVI_SCR_CFLUSH(svp);

	/* Restore current cursor. */
	sp->lno = sv_lno;
	sp->cno = sv_cno;

	/* If an error occurred, return now. */
	if (eval)
		return (INP_ERR);

	/*
	 * If data was entered, or nothing was entered, and we terminated
	 * on a <backspace> or <carriage-return>, return now.
	 */
	tp = sp->tiqp->cqh_first;
	if (tp->term == TERM_OK) {
		;
	}
	if (tp->term != TERM_ESC) {
		;
	}
	return (INP_OK);
}
