
;;; Continued Fraction Crud
;;; by David Svoboda


;;; Stuff for delayed evaluation
;;; Delayed evaluation lists are stored as normal lists, except
;;; that at some point, a function known as the delay function
;;; appears in the list, followed by its arguments.
;;; An example would be the continued fraction representation of
;;; (sqrt 2) = '(1 2 2 #'cycle '(1 2))

;; Returns T if LIST has exactly one number and a delay function.
(defun delayed (list)
    (and (listp list)
	 (listp (cdr list))
	 (functionp (cadr list))))

;; If a list is delayed, returns the delay function.
;; Otherwise returns NIL. 
(defun delay-function (list)
    (if (delayed list)
	(cadr list)))

;; If a list is delayed, returns the delay function's arguments.
;; Otherwise returns NIL. 
(defun delay-args (list)
    (if (delayed list)
	(cddr list)))

;; If normal list, acts just like CDR.
;; If delayed, forces evaluation of next term, so there's always
;; one term availible.
;; Alters the list to include the new term as a side effect UNLESS
;; the delay function is in DONT-CHANGE-LIST.
;; See comments on DONT-CHANGE-LIST. 
(defun fcdr (list)
    (if (delayed list)
	(if (member (delay-function list) dont-change-list)
	    (apply (delay-function list) (delay-args list))
	    (progn (setf (cdr list) (apply (delay-function list)
					   (delay-args list)))
		   (cdr list)))
	(cdr list)))

;; Returns the first n elements of list, forcing terms if necessary.
(defun first-n-elements (list n)
    (cond ((null list) nil)
	  ((= n 1) (list (car list)))
	  (t (cons (car list) (first-n-elements (fcdr list) (- n 1))))))


;;; Cons-Tree manipulation.
;;; These functions are like mapcar, but will work on any setup
;;; possible with cons cells.

;; F is some unary function.
;; Takes a tree and performs F on each element of the tree.
(defun map-tree (f bi-tree)
    (labels ((map (tree)
		  (if (atom (car tree))	; Then (CDR TREE) is also atomic.
		      (list* (funcall f (car tree)) (funcall f (cdr tree)))
		      (list* (map (car tree)) (map (cdr tree))))))
	    (map bi-tree)))

;; F is some binary function.
;; Takes a tree and returns one value of F on the tree.
;; i.e. if F was #'+, would sum the elements.
(defun tree-acc (f bi-tree)
    (labels ((acc (tree)
		  (If (atom (car tree))	; Then (CDR TREE) is also atomic.
		      (funcall f (car tree) (cdr tree))
		      (funcall f (acc (car tree)) (acc (cdr tree))))))
	    (acc bi-tree)))


;;; Matrix manipulation
;;; Matrices are stored as balanced binary trees.
;;; i.e. [ 3 4 ]  <=> ((3 . 1) . (4 . 6))
;;;      [ 1 6 ]
;;; and  [ 1 -6  ]
;;;      [  5  9 ]
;;;      [       ] <=> (((1 . 5) . (-6 . 9)) . ((3 . 7) . (2 . 4)))
;;;      [ 3  2  ]      
;;;      [  7  4 ]
;;; Keep in mind that each cell represents a ratio.
;;; i.e. (1 . 5) <=> 1/5
 
(setq identity-matrix '((1 . 0) . (0 . 1)))

;; Returns the dimensions of a matrix
(defun dimensions (matrix)
    (If (listp matrix)
	(+ 1 (dimensions (car matrix)))
	0))

;; Reduces the elements of a matrix to lowest terms.
;; (divides the elements by their GCD.)
(defun lowest (matrix)
    (let ((factor (tree-acc #'gcd matrix)))
	 (if (not (= 1 factor))
	     (map-tree #'(lambda (x) (/ x factor)) matrix)
	     matrix)))

;; Return a list of ratios of a matrix.
;; Computationally:		[ a  b  ]
;;				[  e  f ]
;; [ a b ]	    	  and	[       ] -> (a/e b/f c/g d/h)
;; [ c d ] -> (a/c b/d)		[ c  d  ]  
;;				[  g  h ]
(defun ratios (matrix)
    (if (= 1 (dimensions matrix))
	(list (if (zerop (cdr matrix))
		  infinity
		  (/ (car matrix) (cdr matrix))))
	(append (ratios (car matrix)) (ratios (cdr matrix)))))

;; Given a matrix, returns the min and max ratios.
(defun min-max-ratios (matrix)
    (let ((ratio-list (ratios matrix)))
	 (If (member infinity ratio-list)
	     (let ((finite-list (remove infinity ratio-list)))
		  (if finite-list
		      (list* (apply #'min finite-list) infinity)
		      (list* infinity infinity)))
	     (list* (apply #'min ratio-list) (apply #'max ratio-list)))))

;; (These shifts are used in homographic function computations.)
;; Shift matrix 1 to the left. Matrix must be at least 2D.
;; This is, in computation:	[ a  b  ]    [ xa+b     a   ]
;;				[  e  f ]    [     xe+f   e ] 
;; [ a b ]    [ xa+b a ]   and	[       ] -> [		    ]	
;; [ c d ] -> [ xc+d c ]	[ c  d  ]    [ xc+d     c   ]
;;				[  g  h ]    [     xg+h   g ]
(defun left-shift (matrix x)
    (left-shift-2 matrix 1 x))

;; Shift matrix 1 to the left, using 2 numbers instead of 1.
;; This is, in computation:	[ a  b  ]    [ ya+xb     a   ]
;;				[  e  f ]    [     ye+xf   e ]
;; [ a b ]    [ ya+xb a ]  and	[       ] -> [		     ]
;; [ c d ] -> [ yc+xd c ]	[ c  d  ]    [ yc+xd     c   ]
;;				[  g  h ]    [     yg+xh   g ]
(defun left-shift-2 (matrix x y)
    (labels ((s (m)
		(if (= 2 (dimensions m))
		    (list* (list* (+ (* y (caar m)) (* x (cadr m)))
				  (+ (* y (cdar m)) (* x (cddr m))))
			   (list* (caar m) (cdar m)))
		    (list* (s (car m)) (s (cdr m))))))
	    (s matrix)))

;; Shift matrix 1 down. The matrix must be at least 3D.
;; This is, in computation:
;; [ a  b  ]    [  c        d       ]
;; [  e  f ]    [      g        h   ]
;; [       ] -> [	            ]
;; [ c  d  ]    [ xc+a     xf+b     ]
;; [  g  h ]    [     xg+e     xh+f ]
(defun down-shift (matrix x)
    (labels ((s (m)
		(if (= 3 (dimensions m))
		    (list* (list* (list* (caadr m) (cdadr m))
				  (list* (caddr m) (cdddr m)))
			   (list* (list* (+ (* x (caadr m)) (caaar m))
					 (+ (* x (cdadr m)) (cdaar m)))
				  (list* (+ (* x (caddr m)) (cadar m))
					 (+ (* x (cdddr m)) (cddar m)))))
		    (list* (s (car m)) (s (cdr m))))))
	    (s matrix)))

;; Does a Euclidian reduction of a matrix.
;; Computationally:			[ a  b  ]    [ e        f        ]
;;					[  e  f ]    [  a-(e*x)  b-(f*x) ]
;; [ a b ]    [    c       d    ]  and	[       ] -> [		    ]
;; [ c d ] -> [ a-(c*x) b-(d*x) ]	[ c  d  ]    [ g        h        ]
;;					[  g  h ]    [  c-(g*x)  d-(h*x) ]
(defun euclid-reduce (matrix x)
    (labels ((r (m)
		(if (= 1 (dimensions m))
		    (list* (cdr m) (- (car m) (* x (cdr m))))
		    (list* (r (car m)) (r (cdr m))))))
	    (r matrix)))

;; Does a decimal reduction of a matrix. Used for decimal approximation.
;; Computationally:
;; [ a b ]    [ base*(a-(c*x)) base*(b-(d*x)) ]
;; [ c d ] -> [       c              d        ]
;;  and
;; [ a  b  ]    [ base*(a-(e*x))   base*(b-(f*x))   ]
;; [  e  f ]    [               e                f  ]
;; [       ] -> [		                    ]
;; [ c  d  ]    [ base*(c-(g*x))   base*(d-(h*x))   ]
;; [  g  h ]    [		g	         h  ]
(defun decimal-reduce (matrix x)
    (labels ((r (m)
		(if (= 1 (dimensions m))
		    (list* (* base (- (car m) (* x (cdr m)))) (cdr m))
		    (list* (r (car m)) (r (cdr m))))))
	    (r matrix)))

;; Given a 3d matrix, returns T if the left side ratios are
;; closer together than the bottom side ratios.
;; (This is used to determine whether to shift left or down.)
(defun left-smaller (matrix)
    (let* ((ratios (ratios matrix))
	   (a (car ratios))
	   (c (caddr ratios))
	   (d (cadddr ratios)))
	  (cond ((infinite c)
		 (cond ((infinite a) t)
		       ((infinite d) nil)
		       (t (> a d))))
		((infinite a) nil)
		((infinite d) t)
		(t (< (abs (- c a)) (abs (- c d)))))))

;; Given a matrix of rational or floating numbers, multiplies
;; them all by some value so they are all integers.
(defun raise-to-int (matrix)
    (let* ((rat-matrix (map-tree #'rationalize matrix))
	   (denom-matrix (map-tree #'denominator rat-matrix))
	   (value (tree-acc #'lcm denom-matrix))
	   (new-matrix (map-tree #'(lambda (x) (* x value)) rat-matrix)))
	  (lowest new-matrix)))


;;; Homographic and bihomographic functions are represented as matrices.
;;; i.e.  3x + 4
;;;      -------- <=> [ 3 4 ]  
;;;        x + 6      [ 1 6 ]
;;; and                        [ 1 -6  ]
;;;     3xy +  x + 2y - 6      [  5  9 ]
;;;    ------------------- <=> [       ] 
;;;     7xy + 5x + 4y + 9      [ 3  2  ] 
;;;                            [  7  4 ]

;; Applies homographic function to number.
(defun apply-homograph (matrix number)
    (/ (+ (* (caar matrix) number) (cadr matrix))
       (+ (* (cdar matrix) number) (cddr matrix))))

;; As functions, matrices may have a fixed point. 
;; Returns the integer value of the fixed point of the function.
(defun fixed-point (matrix)
; Given a guess at the fixed point, a better guess can be
; calcalated by averaging guess and f(guess) where f is our
; homographic function.
    (labels ((fp (guess)
		 (let* ((result (apply-homograph matrix guess))
			(interval (list* (min result guess) 
					 (max result guess)))
			(remainder (small interval)))
		       (if remainder
			   (floor guess)
			   (fp (/ (+ guess result) 2))))))
; We need a large enough initial guess to find the positive f.p.
; If the denominator has a zero, we want a value greater than that
; value. Otherwise the function is linear and has only 1 or 0 fps.
	    (let ((initial-guess (if (not (zerop (cdar matrix)))
				     (+ 1 (abs (/ (cddr matrix)
						  (cdar matrix))))
				     1)))
		 (fp initial-guess))))

;; Composes two homographic functions (which is another homographic
;; function. Computationally this is just matrix multiplication
;; [ a b ] x [ e f ] = [ ae+bg af+bh ]
;; [ c d ]   [ g h ]   [ ce+dg cf+dh ]
(defun mat-mul (matrix1 matrix2)
    (let ((a (caar matrix1)) 	  (e (caar matrix2))
	  (b (cadr matrix1))	  (f (cadr matrix2))
	  (c (cdar matrix1))	  (g (cdar matrix2))
	  (d (cddr matrix1))	  (h (cddr matrix2)))
	 (list* (list* (+ (* a e) (* b g)) (+ (* c e) (* d g)))
		(list* (+ (* a f) (* b h)) (+ (* c f) (* d h))))))

;; Reduces bihomograph to homograph given x
;; [ a  b  ]
;; [  e  f ]    [ cx+d  ax+b ]
;; [	   ] -> [            ]
;; [ c  d  ]    [ gx+h  ex+f ]
;; [  g  h ]
(defun apply-bihomograph-x (matrix x)
    (list* (list* (+ (* (caadr matrix) x) (caddr matrix))
		  (+ (* (cdadr matrix) x) (cdddr matrix)))
	   (list* (+ (* (caaar matrix) x) (cadar matrix))
		  (+ (* (cdaar matrix) x) (cddar matrix)))))

;; Reduces bihomograph to homograph given y
;; [ a  b  ]
;; [  e  f ]    [ cy+a  dy+b ]
;; [	   ] -> [	     ]
;; [ c  d  ]    [ gy+e  hy+f ]
;; [  g  h ]
(defun apply-bihomograph-y (matrix y)
    (list* (list* (+ (* (caadr matrix) y) (caaar matrix))
		  (+ (* (cdadr matrix) y) (cdaar matrix)))
	   (list* (+ (* (caddr matrix) y) (cadar matrix))
		  (+ (* (cdddr matrix) y) (cddar matrix)))))


;;; Math type stuff

;; Used for decimal approximations
(setq base 10)

;; Used in just about everything. 
(setq infinity 'infinity)

(defun infinite (exp)
    (equal exp infinity))

;; Takes reciprical of a rational.
(defun rat-recip (rat)
    (cond ((zerop rat) infinity)
	  ((infinite rat) 0)
	  (t (/ (denominator rat) (numerator rat)))))

;; Takes the summation of a list.
(defun sum (list)
    (if (null list)
	0
	(+ (car list) (sum (cdr list)))))

;; Takes the product of a list.
(defun prd (list)
    (if (null list)
	1
	(* (car list) (prd (cdr list)))))

;; If a and b are real close together, but there is an integer
;; between them, normally SMALL will return NIL. But if SMALL-FLAG
;; is set, and (b - a) < SMALL-ENOUGH, then SMALL will return the
;; integer in the interval and the difference.
(setq small-flag t)
(setq small-enough 1/2000)

;; If |a| and |b| are rather big, it is assumed they will grow
;; without bound, so if BIG-FLAG is set and BIG-ENOUGH > |a|,|b|
;; then SMALL will return INFINITY.
(setq big-flag t)
(setq big-enough 100000)

;; Given an interval (represent by an improper list of 2 elements)
;; returns: the value of the interval if it contains a single value.
;; (integer difference) if the interval is within [integer, integer+1],
;; or NIL otherwise. 
(defun small (interval)
    (let ((a (car interval))
	  (b (cdr interval)))
	 (cond ((equal a b)
		a)
	       ((or (infinite a)
		    (infinite b))
		nil)
	       ((or (= (floor a) (floor b))
		    (= (+ 1 (floor a)) b))
		(list (floor a) (- b a)))
	       ((and small-flag (< (- b a) small-enough))
		(list (floor b) (- b a)))
	       ((and big-flag (> (abs a) big-enough)
		     (> (abs b) big-enough))
		infinity)
	       (t nil))))

;; Given result and base, figures out the lowest power base can be
;; raised to and be less than result.
;; NOTE: Only works for log-base > 1. 
(defun sup-power (log-base result)
    (labels ((aux (n)
		  (let ((log-base-power (expt log-base n)))
		       (if (> log-base-power result)
			   (bi-search (/ n 2) n)
			   (if (> (* log-base log-base-power) result)
			       n
			       (aux (* 2 n))))))
; AUX keeps squaring log-base until its greater than result.
	     (bi-search (a b)
		 (let* ((c (round (/ (+ a b) 2)))
			(log-base-power (expt log-base c)))
		       (if (> log-base-power result)
			   (bi-search a c)
			   (if (> (* log-base log-base-power) result)
			       c
			       (bi-search c b))))))
; BI-SEARCH does a binary search within the interval [a,b]
; for the integer power.
	    (cond ((> log-base result) 0)
		  ((or (and (<= log-base 1) (>= result 1))
		       (and (>= log-base 1) (<= result 1)))
		   infinity)	 ; LOG-BASE is in [0,1].
		  (t (aux 1)))))

;; If rat-value is a cf-approximation to some cf, returns the
;; error bound.
;; Horwitz's Theorem states that if p/q is a cf-approximation to a then
;; | p     |         1
;; | - - a | <= ------------
;; | q     |    (sqrt 5)*q*q
;; We are using 682/305 as a rational approximation to (sqrt 5).
(defun accuracy (rat-value)
    (rat-recip (* 682/305 (denominator rat-value) 
		  (denominator rat-value))))


;;; Basic type stuff to do to CFs

;; Returns the CF of a rational or decimal.
(defun mk-cf (number)
    (let* ((rat-number (rationalize number))
	   (first (floor rat-number))
	   (rest (- rat-number first)))
	  (if (= rest 0)
	      (list first)
	      (list first #'mk-cf (rat-recip rest)))))

;; Adds an integer k to a cf
(defun int-add (k cf)
    (cons (+ k (car cf)) (cdr cf)))

;; Takes recripical of a cf. 
(defun cf-recip (cf)
    (if (zerop (car cf))
	(fcdr cf)
	(cons 0 cf)))

;; Returns T iff cf1 > cf2 
(defun greater (cf1 cf2)
    (cond ((null cf2) nil)
	  ((null cf1) t)
	  ((> (car cf1) (car cf2)) t)
	  ((< (car cf1) (car cf2)) nil)
	  (t (greater (fcdr cf2) (fcdr cf1)))))

;; Returns the smallest cf (in list length) betseen cf1 and cf2.
(defun smallest-rat (cf1 cf2)
    (cond ((or (null cf1) (null cf2)) nil)
	  ((< (car cf1) (car cf2))
	   (list (floor (/ (+ (car cf1) (car cf2)) 2))))
	  (t (cons (car cf1) (smallest-rat (fcdr cf1) (fcdr cf2))))))

;; Returns the already-computed values of the cf. 
(defun known (cf)
    (if (delayed cf)
	(list (car cf))
	(cons (car cf) (known (fcdr cf)))))

;; Returns the rational convergent corresponding to known values of cf. 
(defun known-approx (cf)
    (rat-value (known cf)))


;;; Fun stuff to do to CFs.

;; Applies matrix to the cf.
;; i.e. if x is a cf. and A = [ a b ]
;;	    ax + b	      [ c d ]
;; computes --------
;;	    cx + d 
;; NOTE: If cf is delayed and the delay function is also
;; HOMOGRAPH, concatenates the 2 homographic functions on the cf
;; into one. This saves computation time. 
(defun homograph (cf matrix)
    (let* ((interval (min-max-ratios matrix))
	   (remainder (small interval)))
	  (cond ((null (cdr cf)) (mk-cf (apply-homograph matrix (car cf))))
		((integerp remainder) remainder)
		((numberp remainder) (mk-cf remainder))
		((infinite remainder) nil)
		(remainder (list (car remainder) #'homograph cf 
				 (euclid-reduce matrix (car remainder))))
		((and (delayed cf) (equal (delay-function cf) #'homograph))
		 (let* ((our-matrix (left-shift matrix (car cf)))
			(their-matrix (cadr (delay-args cf)))
			(new-matrix (mat-mul our-matrix their-matrix)))
		       (homograph (car (delay-args cf) new-matrix))))
		(t (homograph (fcdr cf) (left-shift matrix (car cf)))))))

; Computes rational * cf
(defun rat-mul (cf rational)
    (let* ((rat (rationalize rational))
	   (p (numerator rat))
	   (q (denominator rat)))
	  (homograph cf (list* (list* p 0) (list* 0 q)))))

; Computes rational + cf
; Uses the formula:  c + p/q = (qc + p)/q
(defun rat-add (cf rational)
    (let* ((rat (rationalize rational))
	   (p (numerator rat))
	   (q (denominator rat)))
	  (homograph cf (list* (list* q 0) (list* p q)))))

; Given a cf, returns interval of rational numbers CF may be in
; Uses n elements or all known elements, whichever is greater.
; Note: If CF is finite, returns single (rational) value of CF.
(defun cf-value (cf n)
    (first-n-elements cf n)
    (do ((matrix identity-matrix (left-shift matrix (car cf-part)))
	 (cf-part cf (fcdr cf-part)))
	((or (delayed cf-part) (null (cdr cf-part))) 
	 (if (delayed cf-part) 
	     (min-max-ratios matrix)
	     (apply-homograph matrix (car cf-part))))))

;; Given a finite (completely known) cf, returns its value as a rational.
(defun rat-value (cf)
    (cf-value cf 1))


;;; The real important type stuff

;; Applies 3D matrix to two cfs.
;; i.e. if x and y are the cfs and matrix is:
;; [ a  c  ] returns: 
;; [  b  d ]		exy + ax + gy + c
;; [	   ]	       -------------------
;; [ e  g  ]		fxy + bx + hy + d
;; [  f  h ]
(defun bihomograph (cf1 cf2 matrix)
    (let* ((interval (min-max-ratios matrix))
	   (remainder (small interval)))
	  (cond ((null (cdr cf1))
		 (homograph cf2 (apply-bihomograph-x matrix (car cf1))))
		((null (cdr cf2))
		 (homograph cf1 (apply-bihomograph-y matrix (car cf2))))
		((integerp remainder) remainder)
		((numberp remainder) (mk-cf remainder))
		((infinite remainder) nil)
		(remainder (list (car remainder) #'bihomograph cf1 cf2 
				 (euclid-reduce matrix (car remainder))))
		((left-smaller matrix)
		 (bihomograph (fcdr cf1) cf2 (left-shift matrix (car cf1))))
		(t (bihomograph cf1 (fcdr cf2)
		       (down-shift matrix (car cf2)))))))

;; The four basic operations:
(setq add-matrix '(((1 . 0) . (0 . 1)) . ((0 . 0) . (1 . 0))))
(setq sub-matrix '(((1 . 0) . (0 . 1)) . ((0 . 0) . (-1 . 0))))
(setq mul-matrix '(((0 . 0) . (0 . 1)) . ((1 . 0) . (0 . 0))))
(setq div-matrix '(((1 . 0) . (0 . 0)) . ((0 . 0) . (0 . 1))))

(defun cf-add (cf1 cf2)
    (bihomograph cf1 cf2 add-matrix))

(defun cf-sub (cf1 cf2)
    (bihomograph cf1 cf2 sub-matrix))

(defun cf-mul (cf1 cf2)
    (bihomograph cf1 cf2 mul-matrix))

(defun cf-div (cf1 cf2)
    (bihomograph cf1 cf2 div-matrix))


;;; Square roots of numbers. 

;; Generates a surd CF given matrix. [ a  b ]
;; NOTE: Matrix must be of the form: [ c -a ]
(defun find-fp (matrix)
    (let* ((interval (min-max-ratios matrix))
	   (remainder (small interval)))
	  (cond ((integerp remainder) remainder)
		((numberp remainder) (mk-cf remainder))
		((infinite remainder) nil)
		(remainder
		    (list (car remainder) #'find-fp
			  (euclid-reduce (left-shift matrix (car remainder))
			      (car remainder))))
		(t (let ((fp (fixed-point matrix)))
			(list fp #'find-fp 
			      (euclid-reduce (left-shift matrix fp) fp)))))))

;; Returns cf-value of square root of a rational.
(defun cf-sqrt (rational)
    (find-fp (list* (list* 0 (denominator rational))
		    (list* (numerator rational) 0))))

;; Returns cf-value for positive solution of
;;   2
;; ax  + bx + c = 0
;; where a,b,c are integers.
(defun cf-quadratic (a b c)
    (find-fp (list* (list* (* -1 b) (* 2 a))
		    (list* (* -2 c) b))))


;;; Logarithms of rationals.

;; Calculates log when result < log-base. Result & log-base are rational.
(defun cf-log (log-base result)
    (if (= result 1) (list 0)
	(let ((n (sup-power result log-base)))
	     (if (infinite n) nil
		 (let ((b2 (rationalize (/ log-base (expt result n)))))
		      (list n #'cf-log result b2))))))

;; Returns a CF of log        result.  Result may be greater than base.
;;		      log-base	     
(defun mk-cf-log (log-base result)
    (let* ((rat-result (rationalize result))
	   (rat-base (rationalize log-base))
	   (first (sup-power rat-base rat-result))
	   (next-result (/ rat-result (expt rat-base first))))
	  (cons first (cf-log rat-base next-result))))


;;; Non-regular continued fractions can be stored as two streams
;;; one of numerators, one of denominators.

;; Given a non-regular CF (composed of a numerator stream and a 
;; denominator stream, returns a regular CF
;; The elements of the streams may be rationals or decimals. 
(defun regulate (denoms nums matrix)
    (let* ((interval (min-max-ratios matrix))
	   (remainder (small interval)))
	  (cond ((integerp remainder) remainder)
		((numberp remainder) (mk-cf remainder))
		((infinite remainder) nil)
		(remainder (list (car remainder) #'regulate denoms nums
				 (euclid-reduce matrix (car remainder))))
		((not (cdr denoms))
		 (let ((rest (+ (car denoms) (if (car nums) (car nums) 0))))
		      (mk-cf (apply-homograph matrix (rationalize rest)))))
		(t (regulate (fcdr denoms) (fcdr nums)
		       (lowest (raise-to-int (left-shift-2 matrix (car nums)
						 (car denoms)))))))))

;; Given a denominator and numerator stream, produces a CF. 
(defun mk-regular (denoms nums)
    (regulate denoms (cons 1 nums) identity-matrix))

;; Given an almost-regular CF (where the elements are not all integers)
;; returns a stream of integers (a regular CF)
(defun mk-int-stream (denoms)
    (mk-regular denoms (constant 1)))


;;; A particular type of numeric sequence is a decimal stream, which
;;; represents an irrational number, whence the first element is the
;;; integer part and each next element is the next corresponding
;;; decimal. i.e. pi <=> (3 1 4 1 5 9 ...)

;; Given a rational, returns a decimal stream. 
(defun decimalize-rat (rational)
    (list (floor rational) #'decimalize-rat 
	  (* base (- rational (floor rational)))))

;; Given a CF and homograph matrix, returns a decimal stream.
(defun decimalize (cf matrix)
    (let* ((interval (min-max-ratios matrix))
	   (remainder (small interval)))
	  (cond ((integerp remainder)
		 remainder)
		((null (cdr cf))
		 (decimalize-rat (apply-homograph matrix (car cf))))
		((numberp remainder)
		 (mk-cf remainder))
		(remainder
		    (list (car remainder) #'decimalize cf 
			  (decimal-reduce matrix (car remainder))))
		(t (decimalize (fcdr cf) (left-shift matrix (car cf)))))))

;; Given a cf, returns a decimal-stream
(defun mk-decimal (cf)
    (decimalize cf identity-matrix))

;; Given a decimal stream, returns the stream evaluated at
;; PLACES digits of precision 
(defun decimal-places (decimalize places)
    (first-n-elements decimalize (+ 1 places)))

;; Given a CF, returns all significant digits in the decimal
;; approximation of the CF. 
(defun acc-dec-approx (cf)
    (let* ((approx-cf (known cf))
	   (dec-value (mk-decimal approx-cf))
	   (accuracy (accuracy (rat-value approx-cf)))
	   (places (sup-power base (rat-recip (* 2 accuracy)))))
	  (decimal-places dec-value places)))


;;; More delay functions

;; Produces (n n n ...)
(defun constant (n)
    (list n #'constant n))

;; F is a function mapping from naturals to naturals.
;; Returns a stream of (F(0) F(1) F(2) ...)
(defun natural-function (f n)
    (list (funcall f n) #'natural-function f (+ 1 n)))

;; All surds will use this function.
(defun cycle (list)
    (append list (list #'cycle list)))

;; If stream-list is a list of streams, returns each element, one at a time.
;; Example of an infinite sequence.
(defun merge-streams (stream-list)
    (append (mapcar #'car stream-list)
	    (list #'merge-streams (mapcar #'fcdr stream-list))))

;; If stream-list is a list of streams, returns the sum of each
;; set of elements
(defun add-streams (stream-list)
    (cons (sum (mapcar #'car stream-list))
	  (list #'add-streams (mapcar #'fcdr stream-list))))

;; If stream-list is a list of streams, returns the product of each
;; set of elements
(defun mul-streams (stream-list)
    (cons (prd (mapcar #'car stream-list))
	  (list #'mul-streams (mapcar #'fcdr stream-list))))

;; FCDR, when it must evaluate more numbers, changes the list as a
;; side effect. This avoids re-evaluating these numbers. However, sometimes
;; this wastes more space than is worthwhile. So, for certain delay
;; functions, FCDR merely evaluates more numbers, without changing the list
;; The list of delay functions for which FCDR does not store new numbers
;; is called DONT-CHANGE-LIST
(setq dont-change-list (list #'constant #'cycle #'natural-function
			     #'merge-streams #'add-streams #'mul-streams))

;; Some derivitaves of NATURAL-FUNCTION.
(defun sequential (n0)
    (natural-function #'identity n0))
(defun arithmetic (a r)
    (natural-function #'(lambda (n) (+ a (* r n))) 0)) 
(defun geometric (a r)
    (natural-function #'(lambda (n) (* a (expt r n))) 0))
(defun sequence-power (p)
    (natural-function #'(lambda (n) (expt n p)) 1))


;; Some sequences

(setq naturals (sequential 0))
(setq odds (arithmetic 1 2))
(setq evens (arithmetic 2 2))
(setq squares (sequence-power 2))


;; Some CF's 
(setq sqrt2 (cons 1 (constant 2)))
(setq sqrt3 (cons 1 (cycle '(1 2))))
(setq golden-mean (constant 1))

;; PI = 4*(1 + 1/(3 + 4/(5 + 9/(7 + ....))))
(setq cf-pi (regulate odds (cons 1 squares) '((0 . 1) . (4 . 0))))

;; e = 2 (1 2k+2 1) = 2 1 2 1 1 4 1 1 6 1 1 8 1 ...
(setq cf-e (cons 2 (merge-streams (list (constant 1) evens (constant 1)))))


;;; Some other things CFs can approximate.
;;; In each case x is a rational or float and the function returns
;;; a cf approximation. 

(defun cf-tanh (x)
    (mk-regular (cons 0 odds) (cons x (constant (* x x)))))

(defun cf-exp (x)
    (regulate (cons 0 (mul-streams (list (arithmetic 2 4)
					 (constant (/ 1 x)))))
	(constant 1) '((-1 . 1) . (-1 . -1))))

(defun cf-tan (x)
    (mk-regular (cons 0 odds)
	(cons x (constant (* -1 x x)))))

(defun cf-sin (x)
    (mk-regular (append '(0 1) (add-streams (list (mul-streams (list evens
	                              (fcdr odds))) (constant (* -1 x x)))))
	(cons x (cons (* x x) (mul-streams (list (constant (* x x)) evens
						 (fcdr odds)))))))
