# This module enables text to be 'collapsed', that is replaced with a one-line
# representation of the former text.
#
# Collapsing text can work on either selections or regions.
# One can collapse any selected area, or expand every collapsed area. 
# Regions have the additional advantage that one can collapse every region, or
# every 'filler' between region, or both regions and fillers, producing some
# sort of 'outline' of the original text.

# First the non-region stuff


load_library_module preventbind.tcl

# Given text between start and end, returns an adequate short description.
proc selection_indicate {t start end} {
	set minchars [expr [lindex [$t configure -width] 4] / 2]
	if {[$t compare "$start linestart" >= "$end linestart"]} {
		scan [$t index "$start lineend"] "%d.%d" dummy row
		if {$row < $minchars} {return ""}}		

	set newnum [string range [gensym] 3 end]
	incr minchars -15
	if {[$t compare "$end linestart" > "$end - $minchars chars"]} {
		set end_set [$t get "$end linestart" $end]
	} else {set end_set [$t get "$end - $minchars chars wordend" $end]}
	if {[$t compare "$start lineend" < "$start + $minchars chars"]} {
		set start_set [$t get $start "$start linestart"]
	} else {set start_set [$t get $start "$start + $minchars chars wordstart"]}
	return "$start_set\...\[$newnum\]...$end_set"
}

# Replaces text between start and end with a collapsed version.
# saving original text. Can be undone with expand_text.
# If collapse_fn returns "", does not collapse text.
proc collapse_text {t start end {collapse_fn selection_indicate}} {
	set start [$t index $start]
	set end [$t index $end]
	set collapsed_indicator [$collapse_fn $t $start $end]
	if {$collapsed_indicator == ""} {return ""}

# For efficiency, we don't put the expanded text at the end of the killbuffer
# we rather put it before the first expanded text already there, or at the
# end if no expanded text exists yet.
	if {[catch ".t_kill index was_collapsed.first" kill_index]} {
		set kill_index [.t_kill index end]
	}
	.t_kill insert $kill_index \n
	.t_kill tag remove was_collapsed $kill_index
	copy_region $t .t_kill $start $end $kill_index
	.t_kill insert $kill_index $collapsed_indicator
	set offset [string length $collapsed_indicator]
	.t_kill tag add was_collapsed $kill_index "$kill_index +$offset chars"
	$t delete $start $end
	$t insert $start $collapsed_indicator
	$t tag add collapsed $start "$start +$offset chars"
	return $start
}

# If begin and end indicate a collapsed range in t, returns the original
# range in .t_kill. Returns 3 items:
# Start of collapsed header, Start of collapsed text, end of collapsed text.
proc collapsed_range {t begin end} {
	set result ""
	set ranges [.t_kill tag ranges was_collapsed]
	set lr [llength $ranges]
	set text [$t get $begin $end]
	for {set i 0} {$i < $lr} {incr i 2} {if {[eval .t_kill get \
			[lrange $ranges $i [expr $i+1]]] == $text} {
		lappend result [lindex $ranges $i] [lindex $ranges [expr $i +1]]
		if {[expr $i + 2] == $lr} {lappend result "end-1c"
		} else {lappend result "[lindex $ranges [expr $i +2]] -1c"}
		break
	}}
	return $result
}

proc expand_text {t index} {
	set thisrange [tag_thisrange $t collapsed $index]
	set kill_info [eval collapsed_range $t $thisrange]
	eval $t delete $thisrange
	copy_region .t_kill $t [lindex $kill_info 1] [lindex $kill_info 2] \
						 [lindex $thisrange 0]
	.t_kill delete "[lindex $kill_info 0] -1c" [lindex $kill_info 2]
	return [lindex $thisrange 0]
}

proc toggle_collapse_selection {t} {
	if {![catch "$t index sel.first"]} {
		set start [$t index sel.first]
		set end [$t index sel.last]
		$t tag remove sel sel.first sel.last
		if {[catch "$t mark set insert [collapse_text $t $start $end]"]} {beep ; return}
	} elseif {[lsearch [$t tag names insert] collapsed] != -1} {
		$t mark set insert [expand_text $t insert]
	} else {beep ; return}
}

proc expand_text_all {t} {
	while {![catch "$t index collapsed.first" index]} {
		expand_text $t $index
}}

# Writes contents of t out to f (via puts), including contents of collapsed
# text
proc text_write {t f start end} {
	set ranges [tag_ranges $t collapsed $start $end]
	if {$ranges == ""} {puts $f [$t get $start $end] nonewline ; return}
	puts $f [$t get $start [lindex $ranges 0]] nonewline
	set kill_info [collapsed_range $t [lindex $ranges 0] [lindex $ranges 1]]
	text_write .t_kill $f [lindex $kill_info 1] [lindex $kill_info 2]

	set l [llength $ranges]
	for {set i 2} {$i < $l} {incr i 2} {
		puts $f [$t get [lindex $ranges [expr $i - 1]] [lindex $ranges $i]] nonewline
		set kill_info [collapsed_range $t [lindex $ranges $i] [lindex $ranges [expr $i + 1]]]
		text_write .t_kill $f [lindex $kill_info 1] [lindex $kill_info 2]
	}
	puts $f [$t get [lindex $ranges [expr $l - 1]] $end] nonewline
}

# From filebind.tcl, except uses new text_get to get collapsed regions.
proc write_file {t f path name} {
	global use_pipe
	if {([string match \|* $name])} {	set file [open $name w]
	} elseif $use_pipe {			set file [open "| $name" w]
	} else {				set file [open $path/$name w]}
	beth_busy $t text_write $t $file 1.0 end
	close $file
}


# Collapse bindings
proc collapse_bind {f m} {
	global Keys
	parse_bindings Text \
M-O		{toggle_collapse_selection %W} \
Double-Button-2	{%W mark set insert @%x,%y ; toggle_collapse_selection %W} \
M-C-o		{expand_text_all %W}

	if {[winfo exists $m]} {
		parse_menuentries $m.extras.m {
{Collapse 0 "" 				{Selection 0 M-O}
					{"Expand All" 0 M-C-o}
		}}
}}


# Collapse bindings for regions
# Modules that define regions may like to have a function to display
# some text when the region is collapsed. It goes like this...
#
# region_collapse_indicate t start end -- given the start and end of the region
#				to be collapsed, returns some appropriate text
#				describing the region.
# filler_collapse_indicate t start end --- given the start and end of the filler
#				to be collapsed, returns some appropriate text
#				describing the filler.
#
# If either function is not defined, the collapsing indicator for collapsing
# selections is used.
#
# There are several warnings to note when using collapsing regions.
#
# --- Collapsing every region  or filler or all can take a long time since the
# 	functions region_prev and region_end (not region_next) have to be called
# 	once for every region. Therefore these functions should be as fast as
#	possible.
#
# --- The text used to indicate a collapsed region must still satisfy the
#	region functions. In other words the region functions prev,end,next
#	should work whether a region or filler is collapsed or not.
#
# --- The text used to indicate a collapsed area must be unique. This is because
#	said text is used to locate the collapsed text in the killbuffer, so
#	duplicate entries will cause incorrect text to be expanded.
#	(The selection_indicate function does this by generating a new number
#	for each region it collapses)
#
# --- If some region or filler should not be collapsed (ie if the representative
#	text would be bigger than the collapsed text itself,) then the function
#	should return "", which will cancel the collapse function.

# Figure out if collapse_fn is for region, filler, or selection value.
proc figure_collapse_fn {flag} {
	if $flag {
		if {[info procs filler_collapse_indicate] != ""} {
			return filler_collapse_indicate
	}} else {if {[info procs region_collapse_indicate] != ""} {
			return region_collapse_indicate
	}}
	return selection_indicate
}

proc toggle_collapse_region {t {collapse_fn ""}	{prev_fn region_prev}
				{next_fn region_next} {end_fn region_end}} {
	if {([info procs $prev_fn] == "") || ([info procs $end_fn] == "") ||
		([info procs $next_fn] == "")} {beep ; return}

# Figure out boundaries of region or filler.
	set start [$prev_fn $t insert]
	if {$start == ""} {
		set filler_flag 1
		set start 1.0
		set end [$next_fn $t insert]
	} else {set end [$end_fn $t $start]
		if {[$t compare insert < $end]} {
			set filler_flag 0
		} else {set filler_flag 1
			set start $end
			set end [$next_fn $t insert]
	}}
	if {$filler_flag && ($end == "")} {set end end}

	if $filler_flag {set start "$start +1c" ; set end "$end -1c"}

# If region or filler is one collapsed region, expand it.
	set start [$t index $start]	; 	set end [$t index $end]
	if {[$t tag nextrange collapsed $start] == [list $start $end]} {
		move_insert $t "$end +1c"
		expand_text $t $start
		return
	}

	if {$collapse_fn == ""} {set collapse_fn [figure_collapse_fn $filler_flag]}
	move_insert $t "$end +1c"
	if {[collapse_text $t $start $end $collapse_fn] == ""} {beep ; return}
}

# Which should be regions, fillers, or all
proc collapse_text_all {t {which all} {collapse_region_fn ""} 
	{collapse_filler_fn ""}	{prev_fn region_prev} {end_fn region_end}} {
	if {$collapse_region_fn == ""} {
		set collapse_region_fn [figure_collapse_fn 0]}
	if {$collapse_filler_fn == ""} {
		set collapse_filler_fn [figure_collapse_fn 1]}

	set next end
	set index [$prev_fn $t end]
	while {($index != "") && [$t compare $index != 1.0]} {
		set end [$end_fn $t $index]
		if {($which != "regions") && [$t compare "$end +1c" < "$next -1c"]} {
			collapse_text $t "$end +1c" "$next -1c" $collapse_filler_fn}
		if {$which != "fillers"} {
			collapse_text $t $index $end $collapse_region_fn}
		set next $index
		set index [$prev_fn $t [$t index "$index -1c"]]
	}
	if {$which != "regions"} {
		collapse_text $t "1.0 +1c" "$next -1c" $collapse_filler_fn
}}


proc collapse_regions_bind {f m args} {
	global Keys
	parse_bindings Text \
C-O		{toggle_collapse_region %W} \
Double-Button-2	{%W mark set insert @%x,%y ; toggle_collapse_region %W} \
C-Y		{beth_busy %W collapse_text_all %W regions} \
M-Y		{beth_busy %W collapse_text_all %W fillers} \
M-C-y		{beth_busy %W collapse_text_all %W}

	if {[winfo exists $m]} {
		global region_name
		parse_menuentries $m.extras.m.collapse [list \
	[list $region_name 0 C-O] \
	[list "Collapse All" 9 ""	[list $region_name 0 C-Y] \
					[list "Stuff between $region_name\s" 0 M-Y] \
					{Both 0 M-C-y}]]
}}


collapse_bind $frame $menu
if $edit_flag {lappend prevent_edit_tags collapsed
} else {catch {text .t_kill
	.t_kill delete 1.0 end
	.t_kill insert 1.0 "\n"
}}
lappend prevent_select_tags collapsed
if {[info exists regions_defined]} {
	collapse_regions_bind $frame $menu
} else {trace variable regions_defined w "collapse_regions_bind $frame $menu"
}
