#
# Code for adding keybindings and menus to remote widgets
#

source "[file dirname [info script]]/../aux/teach.tcl"
source "$TH_Dir/aux/bindings.$TH_Bindings_Set/bindings.tcl"
catch {source "$TH_Dir/aux/bindings.$TH_Bindings_Set/$Bind_Keyword.tcl"}
catch {source "$XF_Dir/templates/Procedures/KeysymBox.t"}

# Help text.
set TH_Bind_Help "" ; append TH_Bind_Help {
Numbered Menus

These menus contains all the functions that this tool provides. Notice that each
menu entry has a key associated with it, as well as an implicit function. When
you teach a remote widget the stuff in this tool, the remote widget gets the
keybindings associated with the functions, and it may also get the same menu
shown here. (though the menu will probably have a name other than "Functions").

This menu is mainly to show you what functions are available, and their
associated keys. However, you can use this menu just to execute the function
without teaching bindings or menus, or to change the function's menu or key.

If the 'Change' button is 'on', and you select a menu function, a 'Binding
Window' is brought up which allows you to change the binding. See below on
information about the Binding Window.

If the 'Change' button is 'off', and you select a menu function, the cursor
changes to a cross, and you must select a widget in any remote application. Once
you select that widget. the menu's function gets executed in that widget. This
may require this tool to teach the remote tool some code, but it adds no
keybindings or menus. This is useful for performing functions on remote
applications without teaching them any keys or menus.


The Keys Label (that's the area that says "Press Keys Here")

This label is similar to the Functions menu in that it demonstrates the keys
that can be taught to other applications. When you press a key here, if that key
has a binding, then the program behaves the same as if you had selected the
corresponding entry in the Functions menu. That is, if the Change button is on,
that binding is brought up in a bindings window for you to change, and if the
Change button is off, you can select a remote widget to perform the binding's
function.


The Change Button

This is a simple checkbutton that can be turned on and off, and its main duty is
to change the behavior of keys in the Keys label, and of entries in the
Functions menu. When this button is 'off', all keys invoked in the Keys label,
and all functions invoked in the Functions menu get sent to a remote widget.
When this button is 'on', the first key or function to get invoked is brought up
in the Binding Window for you to change. The Change button gets turned off now,
so you can only change one binding at a time.


The Binding Window

This window gets brought up and allows you to change the keys and menus
associated with each binding. You can hit 'OK' to confirm your changes, or
'Cancel' to forget the changes you've made.

The keys entry contains the keys that perform the associated binding function.
More than one key can be bound, and all keys listed will perform the function.
The first key is shown as an 'accelerator' on the corresponding menu entry.
Also, you can leave this entry blank, whence no keys get bound to the function.
The "Keys" button, if active, brings up the Keysyms window from XF (if it is
available), which is helpful for selecting unusual keys on the keyboard.

The menus entry contains the menu hierarchy where the menu entry that performs
this function gets placed. Each element in the menu hierarchy consists of a word
or phrase and a number. The phrase is the text that gets placed in the menu
hierarchy, and the number indicates which character in the phrase gets
underlined, with 0 being the leftmost character, 1 the next one, etc. Leaving
out the number, or using -1 causes no character to be underlined. The first
phrase is the name of the menubutton to store this function on. The last phrase
is the name of the menuentry to bind the function to. Any phrases in the middle
indicate cascade menuentries between the top menubutton and the command
menuentry.

This entry can be left blank, in which case no menu entry gets created for this
function. Otherwise, it must contain at least two items, the menubutton item and
the command item. It can also contain any number of cascade items in between.

Beware that while it is possible to leave blank both the menu entry and the keys
entry, allowing a function to have no menu or key essientially 'locks out' that
function, since there would then be no way to refer to that function. You would
have to restart the tool to reuse that function.

} $TH_Help


# For a binding, returns the code necessary to teach that widget (or class) the
# keybinding. Prefix is "uplevel #0" or "send $app" for the app that contains
# the widget.
proc keybind_code {widget binding cmd {prefix "uplevel #0"}} {
# What binding should we really use?
  if {$prefix == "uplevel #0"} {
    set command "execute_or_change_binding $binding \{$cmd\}"
  } else {set command $cmd}

  global All_Bindings
  set bindings [lindex $All_Bindings($binding) 0]
  set needed_bindings ""
  foreach binding $bindings {
    if {[eval "$prefix bind $widget $binding"] != $command} {
      lappend needed_bindings $binding
  }}
  switch [llength $needed_bindings] {
    0 {return ""
  } 1 {return "bind $widget [lindex $needed_bindings 0] \{$command\}\n"
  } default {return "foreach th_binding \{$needed_bindings\} \{\n\
  bind $widget \$th_binding \{$command\}\}\n"
}}}

# Returns code necessary to pack something immdeiately on side $side
# of existing widget w.
proc pack_side_parms {w {side top} {prefix "uplevel #0"}} {
  set info [eval $prefix pack newinfo $w]
  set i [lsearch $info "-side"] ; incr i
  set w_side [lindex $info $i]
  switch [list $w_side $side] {
    {top bottom} - {left right} - {bottom top} - {right left} {
      set code "-after $w -side $w_side"
    } default {set code "-before $w -side $side"
  }}
  set sides {left right top botto} ; set anchor {w e n s}
  return "$code -anchor [lindex {n e w s} [lsearch {top right left bottom} \
		$w_side]]"
}

# Returns code necessary to generate the menubar.
proc menubar_code {widget {prefix "uplevel #0"}} {
  set menubar $widget ; append menubar "_mb"
  set code ""
  set parent [eval "$prefix winfo parent $widget"]
  if {![catch "$prefix pack slaves $parent" result]} {
    if {[lsearch $result $menubar] >= 0} {return $code}}
  set code "pack $menubar -f x [pack_side_parms $widget top $prefix]\n$code"

  if {[eval "$prefix winfo exists $menubar"]} {return $code}
  return "frame $menubar\n$code"
}

# For a binding, returns the code necessary to teach that widget (or class) the
# menu. Prefix is "uplevel #0" or "send $app" for the app that contains the
# widget.
proc menu_code {widget binding cmd {prefix "uplevel #0"}} {
  global All_Bindings
  set code ""

# Important variables
  set menu_sets [lindex $All_Bindings($binding) 1]
  if {[llength $All_Bindings($binding)] <= 2} {set variable ""
  } else {
    set stuff [regexp_replace [lindex $All_Bindings($binding) 2] %W $widget]
    set variable "TH([lindex $stuff 0])"
    set on [lindex $stuff 1]
    set off [lindex $stuff 2]
    if {$on == ""} {set on 1}
    if {$off == ""} {set off 0}
   }
  if {$menu_sets == ""} {return}
  set ml [expr [llength $menu_sets] - 1]
  set first_menu_set [lindex $menu_sets 0]
  set first_menu_name [lindex $first_menu_set 0]
  set last_menu_set [lindex $menu_sets $ml]
  set last_menu_name [lindex $last_menu_set 0]
  set cascade_menu_sets [lrange $menu_sets 1 [expr $ml - 1]]
  set menubar $widget ; append menubar "_mb"
  set menubutton "$menubar.[string tolower $first_menu_name]"
  set first_menu_widget "$menubutton.m"
  set last_menu_widget $first_menu_widget
  foreach cascade_menu_set $cascade_menu_sets {
    append last_menu_widget ".[string tolower [lindex $cascade_menu_set 0]]"
  }

# What binding should we really use?
  if {$prefix == "uplevel #0"} {
    set command "execute_or_change_binding \{$binding\} \{$cmd\}"
  } else {
    set command "if \{\[winfo exists $widget\]\} \{\n  $cmd\n\} else \{\n  destroy $menubar\}"
  }

# Check existance of command menuentry.
  if {[llength $last_menu_set] > 0} {set underline [lindex $last_menu_set 1]
  } else {set underline -1}

  set accel [lindex [lindex $All_Bindings($binding) 0] 0]
  if {$prefix != "uplevel #0"} {
    if {[catch "$prefix bind $widget $accel" result]} {return}
    if {$result == ""} {
      catch "$prefix bind [eval $prefix winfo class $widget] $accel" result}
      set result [regexp_replace $result %W $widget]
    if {$result != $cmd} {set accel ""}}

  set configure_code "$last_menu_widget entryconfigure \{$last_menu_name\} -u $underline  -acc \{$accel\} -co \{$command\}"
  if {$variable == ""} {
    set create_code "$last_menu_widget add command"
  } else {
    set create_code "$last_menu_widget add checkbutton -variable $variable\\\n  -onvalue $on -offvalue $off"
  }
  append create_code " -l \{$last_menu_name\} -u $underline -acc \{$accel\} -co \{$command\}"
  if {![catch "$prefix $last_menu_widget entryconfigure \{\{$last_menu_name\}\} -co" result]} {
    if {([lindex $result 4] == $command) && ([eval $prefix $last_menu_widget \
       entryconfigure \{\{$last_menu_name\}\} -acc] == $accel)} {return $code
    } else {set code "$configure_code\n$code"
  }} else {set code "$create_code\n$code"}

# Check existance of cascade menus and menuentries.
  set menu_widget $first_menu_widget
  set cascade_code ""
  for {set i 0} {$i < $ml} {incr i} {
    set name [lindex [lindex $menu_sets $i] 0]

# Check if this menu exists.
    if {$i < $ml} {
      if {![eval "$prefix winfo exists $menu_widget"]} {
        append cascade_code "menu $menu_widget\n"
    }}

# Check to see if this menu has a cascade entry to the next menu.
    if {$i < [expr $ml - 1]} {
      set next_set [lindex $menu_sets [expr $i+1]]
      set next_name [lindex $next_set 0]
      set next_widget "$menu_widget.[string tolower $next_name]"
      if {[llength $next_set] > 0} {set next_underline [lindex $next_set 1]
      } else {set underline -1}
      if {[catch "$prefix $menu_widget entryconfigure $next_name"]} {
        append cascade_code "$menu_widget add cascade -l $next_name -u $next_underline -m $next_widget\n"
      }
      set menu_widget $next_widget
   }}
   set code "$cascade_code$code"

# Check to see if menubutton exists.
  if {![catch "$prefix pack slaves $menubar" result]} {
    if {[lsearch $result $menubutton] >= 0} {return $code}}
  set code "pack $menubutton -in $menubar -side left\n$code"
  if {[eval "$prefix winfo exists $menubutton"]} {return $code}
  if {[llength $first_menu_set] > 0} {set underline [lindex $first_menu_set 1]
  } else {set underline -1}
  set code "menubutton $menubutton -m $first_menu_widget -text $first_menu_name -u $underline\n$code"

  return "[menubar_code $widget $prefix]\n$code"
}

# Teaches keybindings to widget or class w, in app. (If not given, defaults
# to local)
proc teach_keybindings {w bindings {app ""}} {
  foreach binding $bindings {
    set b [lindex $binding 0]
    if {[llength $binding] < 2} {set cmd [lindex $binding 0]
    } else {set cmd [lindex $binding 1]}
    if {$app != ""} {
      set prefix "send \"$app\""
      do_cmd $app [keybind_code $w $b $cmd $prefix] 0
    } else {
      uplevel #0 [keybind_code $w $b $cmd]
}}}

proc regexp_replace {string old_exp new_exp} {
  if {[regsub -all $old_exp $string $new_exp new_string]} {
    return $new_string
  } else {return $string
}}

# Teaches menubindings to widget w in app. (If not given, defaults to local)
proc teach_menubindings {widget bindings {app ""}} {
  foreach binding $bindings {
    set b [lindex $binding 0]
    if {[llength $binding] < 2} {set cmd [lindex $binding 0]
    } else {set cmd [lindex $binding 1]}
    if {$app != ""} {
      set cmd [regexp_replace $cmd %W $widget]
      set prefix "send \"$app\""
      do_cmd $app [menu_code $widget $b $cmd $prefix] 0
    } else {
      uplevel #0 [menu_code $widget $b $cmd]
}}}

# Creates a label to send remote keybindings
proc create_keybind_label {} {
  if {![winfo exists .bind.key]} {
    label .bind.key -text "Press keys here"
    bind .bind.key <Enter> "focus .bind.key"
    pack .bind.key -side right -expand yes -f x
  }
  global Local_Bindings
  teach_keybindings .bind.key $Local_Bindings
}

# Creates a menu to send remote functions.
proc create_menubind_frame {} {
  global Local_Bindings
  teach_menubindings .bind.key $Local_Bindings
  pack .bind.key_mb -side left
  set i 1
  foreach child [winfo children .bind.key_mb] {
    $child configure -text "#$i" -u -1
    incr i
}}

# Given a binding, allows the user to change it.
proc change_binding {binding} {
  global All_Bindings
  set result [eval change_binding_dialog $binding]
  if {$result != ""} {
    set All_Bindings($binding) $result
    create_menubind_frame
    create_keybind_label
}}

# Pops up a window with the binding specs for the user to change.
proc change_binding_dialog {binding} {
  global All_Bindings
  toplevel .cb
  wm title .cb "Change: $binding"
  wm iconname .cb "Change Binding"
  grab .cb
  frame .cb.keys
  pack .cb.keys -side top -f x -expand no
  entry .cb.keys.e
  .cb.keys.e insert 0 [lindex $All_Bindings($binding) 0]
  pack .cb.keys.e -f x -expand yes -side right
  button .cb.keys.l -text "Keys" -co {.cb.keys.e insert insert [KeysymBox $XF_Dir/lib/Keysyms] ; grab .cb}
  if {[info procs KeysymBox] == ""} {.cb.keys.l configure -sidetate disabled}
  pack .cb.keys.l
  frame .cb.menus
  pack .cb.menus -side top -f x -expand no
  entry .cb.menus.e
  .cb.menus.e insert 0 [lindex $All_Bindings($binding) 1]
  label .cb.menus.l -text "Menus:"
  pack .cb.menus.e -f x -expand yes -side right
  pack .cb.menus.l
  button .cb.ok -text "OK" -co "set CB_Done 1"
  pack .cb.ok -side left -expand yes -f x
  button .cb.cancel -text "Cancel" -co "set CB_Done 2"
  pack .cb.cancel -side left -expand yes -f x
  global CB_Done
  set CB_Done 0
  tkwait variable CB_Done
  if {$CB_Done == 2} {set result ""} else {
    set result [list [.cb.keys.e get] [.cb.menus.e get]]
  }
  grab release .cb
  destroy .cb
  return $result
}

# Given a set of bindings, choose the one matching choose, and execute it.
proc execute_chosen_binding {app widget choose bindings} {
  clear_output
  foreach binding $bindings {
    if {[lindex $binding 0] == $choose} {
      teach_code $app
      do_cmd $app [regexp_replace [lindex $binding 1] %W $widget] 0
      return
  }}
  th_beep
}

# Executed upon invoking a local menuentry or key in the Key label. Either
# change the binding or execute it remotely.
proc execute_or_change_binding {binding cmd} {
  global Change_Binding
  if $Change_Binding {
    set Change_Binding 0
    change_binding $binding
  } else {
    bind all <Button>	"execute_remote_binding %X %Y %b \{$cmd\}"
    toggle_grab
}}

# Prepare to execute a command remotely.
proc execute_remote_binding {x y b cmd} {
  bind all <Button> 	""
  toggle_grab
  if {![which_widget $x $y 1 app widget]} {th_beep ; return}
  set cmd \
   [regexp_replace [regexp_replace $cmd {.bind.key} $widget] {.buttons} $widget]
  execute_remote_command $app $widget $cmd
}

# Execute command remotely.
proc execute_remote_command {app widget cmd} {
  clear_output
  teach_code $app
  do_cmd $app $cmd 0
}


if {[info globals Local_Bindings] == ""} {
  set Local_Bindings $Bindings($Local_Bindword)
}
if {![catch "frame .bind"]} {
  pack .bind -before .output -side bottom -f x
  checkbutton .bind.change -text "Change" \
            -variable Change_Binding -onvalue 1 -offvalue 0
  pack .bind.change -side right
}
create_keybind_label
create_menubind_frame
clear_output

