
/* Convert.c - Convert an entry from/to external, internal and
 * various presentation/interface formats.
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	Qddb_Convert(Schema *, int, void *, int)
 *	Qddb_FindAttributeNumber(Schema *, int *, int)
 */

/* Entry conversions 
 */
static InCoreEntry	*ConvertReadableToInternal _ANSI_ARGS_((Schema *, char *));
static char		*ConvertInternalToReadable _ANSI_ARGS_((Schema *, InCoreEntry *));
static InCoreEntry	*ConvertExternalToInternal _ANSI_ARGS_((Schema *, Entry));
static void		ConvertInternalToExternal _ANSI_ARGS_((Schema *, InCoreEntry *, Entry *));
static InCoreEntry	*ConvertTCLvalueToInternal _ANSI_ARGS_((Schema *, char *));
static char		*ConvertInternalToTCLvalue _ANSI_ARGS_((Schema *, InCoreEntry *));
static InCoreEntry	*ConvertTCLExternalToInternal _ANSI_ARGS_((Schema *, TCLEntry));
static char		*ConvertInternalToTCLExternal _ANSI_ARGS_((Schema *, InCoreEntry *));
static InCoreEntry	*ConvertDataTreeToInternal _ANSI_ARGS_((Schema *, DataTree **));
static DataTree		**ConvertInternalToDataTree _ANSI_ARGS_((Schema *, InCoreEntry *));
static void 		*ConvertXXXToInternal _ANSI_ARGS_((Schema *, int, void *));
static void 		*ConvertInternalToXXX _ANSI_ARGS_((Schema *, void *, int));

/* From the parsers...
 */
void Qddb_EntryParse_Set _ANSI_ARGS_((Schema *));
InCoreEntry *Qddb_EntryParse_Get();

void Qddb_TCLParse_Set _ANSI_ARGS_((Schema *));
InCoreEntry *Qddb_TCLParse_Get();

void ResetTCLParser _ANSI_ARGS_((void));
int TCLParse _ANSI_ARGS_((void));

/* Qddb_Convert -- Convert between various QDDB internal structure types.
 * Any type within a class of types (ENTRY, SCHEMA) can be converted to any other.
 */
void *Qddb_Convert(schema, FromType, FromPtr, ToType)
    Schema			*schema;
    int				FromType;
    void 			*FromPtr;
    int				ToType;
{
    void			*retval, *retval2;

#if defined(DIAGNOSTIC)
    if ((FromType & QDDB_ENTRYTYPES) == 0) {
	fprintf(stderr, "Qddb_Convert: Bad FromType %x", FromType);
	exit(1);
    }
#endif
    if (FromPtr == NULL)
	return NULL;
    /* An ENTRY type */
    if (FromType == QDDB_ENTRYTYPE_INTERNAL)
	return ConvertInternalToXXX(schema, FromPtr, ToType);
    /* XXX -> Internal */
    retval = ConvertXXXToInternal(schema, FromType, FromPtr);
#if defined(DIAGNOSTIC)
    Qddb_SortInCoreEntries(schema, retval);
#endif
    if (ToType == QDDB_ENTRYTYPE_INTERNAL || retval == NULL)
	return retval;
    /* XXX -> Internal -> YYY */
    retval2 = ConvertInternalToXXX(schema, retval, ToType);
    /* Free up the Internal intermediate form we created */
    Qddb_Free(QDDB_TYPE_INCOREENTRY, retval);
    return retval2;
}

int Qddb_FindAttributeNumber(schema, Attributes, HowMany)
    Schema		*schema;
    size_t		Attributes[];
    int			HowMany;
{
    size_t		i, j;

    for (i = 1; i <= schema->NumberOfAttributes; i++) {
	for (j = 1; j < HowMany; j++) {
	    if (schema->Entries[i].AncestorNumber[j] != Attributes[j-1])
		break;
	}
	if (j == HowMany) {
	    if (schema->Entries[i].Level != HowMany)
		continue;
	    if (Attributes[j-1] == schema->Entries[i].Number)
		return i;
	}
    }
    return -1;
}

static void *ConvertInternalToXXX(schema, FromPtr, ToType)
    Schema			*schema;
    void 			*FromPtr;
    int				ToType;
{
    void			*retval = NULL;

    switch (ToType) {
    /* Internal -> XXX */
    case QDDB_ENTRYTYPE_EXTERNAL:
	ConvertInternalToExternal(schema, (InCoreEntry *)FromPtr, (Entry *)&retval);
	break;
    case QDDB_ENTRYTYPE_READABLE:
	retval = (void *)ConvertInternalToReadable(schema, (InCoreEntry *)FromPtr);
	break;
    case QDDB_ENTRYTYPE_TCLVALUE:
	retval = (void *)ConvertInternalToTCLvalue(schema, (InCoreEntry *)FromPtr);
	break;
    case QDDB_ENTRYTYPE_TCLEXTERNAL:
	retval = (void *)ConvertInternalToTCLExternal(schema, (InCoreEntry *)FromPtr);
	break;
    case QDDB_ENTRYTYPE_DATATREE:
	retval = (void *)ConvertInternalToDataTree(schema, (InCoreEntry *)FromPtr);
	break;
    default:
	retval = NULL;
    }
    return retval;
}

static void *ConvertXXXToInternal(schema, FromType, FromPtr)
    Schema			*schema;
    int				FromType;
    void 			*FromPtr;
{
    void 			*retval;

    /* XXX -> Internal */
    switch (FromType) {
    case QDDB_ENTRYTYPE_EXTERNAL:
	retval = (void *)ConvertExternalToInternal(schema, (Entry)FromPtr);
	break;
    case QDDB_ENTRYTYPE_READABLE:
	retval = (void *)ConvertReadableToInternal(schema, (char *)FromPtr);
	break;
    case QDDB_ENTRYTYPE_TCLVALUE:
	retval = (void *)ConvertTCLvalueToInternal(schema, (char *)FromPtr);
	break;
    case QDDB_ENTRYTYPE_TCLEXTERNAL:
	{
	    TCLEntry		entry = NULL;

	    Qddb_SplitBufferIntoLines(&entry, (char *)FromPtr, strlen((char *)FromPtr));
	    retval = (void *)ConvertTCLExternalToInternal(schema, (TCLEntry)entry);
            Qddb_Free(QDDB_TYPE_ENTRY, entry);
	}
	break;
    case QDDB_ENTRYTYPE_DATATREE:
	retval = (void *)ConvertDataTreeToInternal(schema, (DataTree **)FromPtr);
	break;
    default:
	retval = NULL;
    }
    return retval;
}

static InCoreEntry *ConvertReadableToInternal(schema, input)
    Schema			*schema;
    char			*input; /* buffer full of a readable format entry */
{
    InCoreEntry			*retval;
    int				ResetEntryParser(), EntryParse();

    qddb_errno = 0;
    if (qddb_errmsg != NULL)
	Free(qddb_errmsg);
    qddb_errmsg = NULL;
    ResetEntryParser();
    Qddb_EntryParse_Set(schema);
    Qddb_SetBuffer(input);
    if (EntryParse() != 0)
	retval = NULL;
    else
	retval = Qddb_EntryParse_Get();
    return retval;
}

static char *TranslateDataToReadable(data)
    char		*data;
{
    char		*ch = Malloc(BUFSIZ);
    int			bufsiz = BUFSIZ, i;

    for (i = 0; *data; i++, data++) {
	if (bufsiz-3 <= i)
	    ch = Realloc(ch, (size_t)(bufsiz += BUFSIZ));
	if (*data == '"') {
	    ch[i] = '\\';
	    ch[++i] = '"';
	} else
	    ch[i] = *data;
    }
    ch[i] = '\0';
    ch = Realloc(ch, (size_t)(i+1));
    return ch;
}

static char *ConvertInternalToReadable(schema, ThisEntry)
    Schema		*schema;
    InCoreEntry		*ThisEntry;
{
    int			i;
    int			level = 1;
    int			LastAttr = -1;
    char		tabs[BUFSIZ], buf[BUFSIZ], *ch;
    InCoreEntry		*LastInCore = NULL;
    SchemaNode		*LastEntry = NULL;

    if (ThisEntry == NULL) {
    	return NULL;
    }
    Qddb_InitBuffer();
    sprintf(buf, "$NUMBER$ = \"%d\";\n", ThisEntry->SequenceNumber);
    Qddb_ConcatBuffer(buf);
    tabs[0] = '\0';
    while (ThisEntry->SequenceNumber != 0) {
	SchemaNode		*TmpEntry;
	int			flag;

	TmpEntry = schema->Entries+ThisEntry->AttributeNumber;
	if (LastAttr != -1)
	    LastEntry = schema->Entries+LastAttr;
	if (TmpEntry->Level == 1) {
	    if (LastAttr != -1) {
		for (i = 1; i < LastEntry->Level; i++) {
                    /* add )'s and remove all tabs */
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s)\n", tabs);
		    Qddb_ConcatBuffer(buf);
		}
	    }
            LastAttr = ThisEntry->AttributeNumber;
	    LastInCore = ThisEntry;
	    level = 1;
	    ch = TranslateDataToReadable(ThisEntry->Data);
	    sprintf(buf, "%s = \"", TmpEntry->Name);
	    Qddb_ConcatBuffer(buf);
	    Qddb_ConcatBuffer(ch);
	    Free(ch);
	    Qddb_ConcatBuffer("\"\n");
	} else { /* Not a level 1 leaf */
	    if (LastAttr == -1) { /* 1st time through */
		LastAttr = ThisEntry->AttributeNumber;
		LastInCore = ThisEntry;
		/* Now do appropriate indentation */
		for (i = 1; i < TmpEntry->Level; i++) {
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1, i);
		    sprintf(buf, "%s%s (\n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		}
		ch = TranslateDataToReadable(ThisEntry->Data);
		sprintf(buf, "%s%s = \"", tabs, TmpEntry->Name);
		Qddb_ConcatBuffer(buf);
		Qddb_ConcatBuffer(ch);
		Free(ch);
		Qddb_ConcatBuffer("\"\n");
		level = TmpEntry->Level;
	    } else if (LastEntry->Level == TmpEntry->Level) {
		/* STEPS:
		 * 	1. Check prefix match
		 *	2. Close any finished subattributes
		 *	3. Open any new subattributes
		 *	4. Print the leaf
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = TmpEntry->Level;
		for (i = TmpEntry->Level-1; i >= 1; i--) { /* close old */
		    if (TmpEntry->AncestorNumber[i] != LastEntry->AncestorNumber[i]) {
			flag = i;
		    } 
		} 
		/* Check instances, are we starting a new instance? */
		/* Instances start at 0 */
		for (i = TmpEntry->Level-1; i >= 0; i--) {
		    if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			if (flag > i+1)
			    flag = i+1;
		    }
		}
		for (i = flag; i < TmpEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s)\n", tabs);
		    Qddb_ConcatBuffer(buf);		    
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s%s (\n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		    level++;
		}
		ch = TranslateDataToReadable(ThisEntry->Data);
		sprintf(buf, "%s%s = \"", tabs, TmpEntry->Name);
		Qddb_ConcatBuffer(buf);
		Qddb_ConcatBuffer(ch);
		Free(ch);
		Qddb_ConcatBuffer("\"\n");
		LastInCore = ThisEntry;
	    } else if (LastEntry->Level > TmpEntry->Level) {
		/* Going down in length; LastEntry has more
		 * levels, so we must start from the beginning
		 * to find where they differ.
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = TmpEntry->Level;
		for (i = 1; i < TmpEntry->Level; i++) { /* close old */
		    if (TmpEntry->AncestorNumber[i] != LastEntry->AncestorNumber[i]) {
			flag = i;
			break;
		    } 
		} 
		/* Instances start at 0 */
		for (i = TmpEntry->Level-1; i >= 0; i--) {
		    if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			if (flag > i+1)
			    flag = i+1;
		    }	
		}
		for (i = flag; i < LastEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s)\n", tabs);
		    Qddb_ConcatBuffer(buf);
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s%s (\n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		    level++;
		}
		ch = TranslateDataToReadable(ThisEntry->Data);
		sprintf(buf, "%s%s = \"", tabs, TmpEntry->Name);
		Qddb_ConcatBuffer(buf);
		Qddb_ConcatBuffer(ch);
		Free(ch);
		Qddb_ConcatBuffer("\"\n");
		LastInCore = ThisEntry;
	    } else if (LastEntry->Level < TmpEntry->Level) {
		/* Going up in length, TmpEntry has more levels
		 * this time.
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = LastEntry->Level;
		for (i = 1; i < LastEntry->Level; i++) { /* close old */
		    if (TmpEntry->AncestorNumber[i] != LastEntry->AncestorNumber[i]) {
			flag = i;
			break;
		    } 
		} 
		/* Check instances, are we starting a new instance? */
		if (LastEntry->Level > 1) {
		    for (i = TmpEntry->Level-1; i >= 0; i--) {
			if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			    if (i+1 < flag)
				flag = i+1;
			}
		    }
		}
		for (i = flag; i < LastEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s)\n", tabs);
		    Qddb_ConcatBuffer(buf);
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s%s (\n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		    level++;
		}
		ch = TranslateDataToReadable(ThisEntry->Data);
		sprintf(buf, "%s%s = \"", tabs, TmpEntry->Name);
		Qddb_ConcatBuffer(buf);
		Qddb_ConcatBuffer(ch);
		Free(ch);
		Qddb_ConcatBuffer("\"\n");
		LastInCore = ThisEntry;
	    }
	}
	ThisEntry++;
    }
    /* Terminate any unclosed ('s */
    for (i = 1; i < level; i++) {
	tabs[strlen(tabs)-1] = '\0';
	sprintf(buf, "%s)\n", tabs);
	Qddb_ConcatBuffer(buf);
    }
    sprintf(buf, "\n\n");
    Qddb_ConcatBuffer(buf);
    return Qddb_GetBuffer();
}

static InCoreEntry *ConvertExternalToInternal(schema, MyEntry)
    Schema		*schema;
    Entry		MyEntry;
{
    char		*Data;
    int			SequenceNumber, i, InRetVal, GenIndex, attr_num;
#if defined(DIAGNOSTIC)
    int			testvar;
#endif
    Entry		j;
    InCoreEntry		*RetVal;
    size_t		Instances[MAXLEVEL];
    size_t		Attributes[MAXLEVEL];

    sscanf(*MyEntry, "%%0 %*c %d", &SequenceNumber);
    /* Need to allocate the internal entries */
    for (i = 0, j = MyEntry; *j != NULL; i++, j++); /* how many do we need? */
    RetVal = (InCoreEntry *)Malloc(sizeof(InCoreEntry)*(i+2));
#if defined(DIAGNOSTIC)
    testvar = i+2;
#endif
    MyEntry++; 
    InRetVal = 0; /* Now index into RetVal */
    while (*MyEntry != NULL) {
	int 		Itsy, Bitsy;

	i = 0;
	/* Okay, now we have a particular attribute that we must parse
	 * into an internal copy.  We must first find the attribute
	 * in the Schema that corresponds to this instance and 
	 * add the attribute to the array of InCoreEntries.
	 */
	Bitsy = strlen(*MyEntry);
	if (Bitsy == 0) { /* hack, something is setting this incorrectly */
	    Free(*MyEntry);
	    *MyEntry = NULL;
	    break;
	}
	for (Itsy = 1; !isspace((*MyEntry)[Itsy]) && Itsy < Bitsy; ) {
	    char	*num;

	    num = (*MyEntry)+Itsy;
	    while ((*MyEntry)[Itsy] != '.')	/* must have 1st period */
		Itsy++;
	    (*MyEntry)[Itsy] = '\0';
	    Attributes[i] = atoi(num);
	    Itsy++;
    	    num = (*MyEntry)+Itsy;
	    while ((*MyEntry)[Itsy] != '.' && !isspace((*MyEntry)[Itsy]))
		Itsy++;
	    if (!isspace((*MyEntry)[Itsy])) {
		(*MyEntry)[Itsy] = '\0';
		Itsy++;
		Instances[i] = atoi(num);
	    } else {
		(*MyEntry)[Itsy] = '\0';
		Instances[i] = atoi(num);
		(*MyEntry)[Itsy] = ' ';
	    }
	    i++;
	}
	Itsy++;
	attr_num = Qddb_FindAttributeNumber(schema, Attributes, i);
#if defined(DIAGNOSTIC)
	if (attr_num < 0) {
            int			j;

	    for (j = 0; j < i; j++)
		fprintf(stderr, "Attributes[%d] == %d\n", j, Attributes[j]);
	    fprintf(stderr, "MyEntry: %s\n", *MyEntry);
	    PANIC("ConvertExternalToInternal: Qddb_FindAttributeNumber failed");
	}
#endif
	RetVal[InRetVal].AttributeNumber = attr_num;
	if (schema->Entries[RetVal[InRetVal].AttributeNumber].Type == Date) {
	    Data = Qddb_DateTimeToString(*MyEntry+Itsy, 
					 schema->Entries[RetVal[InRetVal].AttributeNumber].Format);
#if defined(DEBUG)
	fprintf(stderr, "Date: %x\n", Data);
#endif
	    RetVal[InRetVal].BytesOfData = strlen(Data);
	} else {
	    size_t		len = strlen(*MyEntry+Itsy);
	    char		*tmp, *dataspot;

	    tmp = *MyEntry + Itsy;
	    Data = Malloc(len+1);
#if defined(DEBUG)
	fprintf(stderr, "Data: %x\n", Data);
#endif
	    tmp = *MyEntry + Itsy;
	    dataspot = Data;
	    while (*tmp != '\0') {
		if (*tmp == '\\' && *(tmp+1) == '\n') {
		    tmp++;
		} else if (*tmp == '\\' && *(tmp+1) == '\\') {
		    tmp++;
		    *dataspot++ = *tmp++;
		} else {
		    *dataspot++ = *tmp++;
		}
	    }
	    *dataspot = '\0';
	    RetVal[InRetVal].BytesOfData = strlen(Data);
	}
	RetVal[InRetVal].SequenceNumber = SequenceNumber;
	RetVal[InRetVal].Data = Data;
	RetVal[InRetVal].Marked = Inapplicable;
	for (GenIndex = 0; GenIndex < i; GenIndex++)
	    RetVal[InRetVal].Instance[GenIndex] = Instances[GenIndex];
	InRetVal++;
	MyEntry++;
    }
#if defined(DIAGNOSTIC)
    if (InRetVal > testvar-1)
	PANIC("ConvertExternalToInternal: Too many entries");
#endif
    RetVal[InRetVal].SequenceNumber = 0; /* SequenceNumber 0 means EOL */
    return RetVal;
}

static char *BuildExternalLine(schema, internal, bytes, bytelen)
    Schema		*schema;
    InCoreEntry		*internal;
    char		*bytes;
    int			bytelen;
{
    SchemaNode		*schema_entry;
    char		*retval, buf[BUFSIZ], *tmpbuf, *tmpret;
    int			len, level, i, num_newlines;
    
    schema_entry = schema->Entries + internal->AttributeNumber;
    level = schema_entry->Level;
    if (level > 1) {
	sprintf(buf, "%%%d.%d", (int)schema_entry->AncestorNumber[1], (int)internal->Instance[0]);
	len = strlen(buf);
	for (i = 2; i < level; i++) {
	    sprintf(buf+len, ".%d.%d", (int)schema_entry->AncestorNumber[i], (int)internal->Instance[i-1]);
	    len += strlen(buf+len);
	}
	sprintf(buf+len, ".%d.%d", (int)schema_entry->Number, (int)internal->Instance[i-1]);
	len += strlen(buf+len);
    } else {
	sprintf(buf, "%%%d.%d", (int)schema_entry->Number, (int)internal->Instance[0]);
	len = strlen(buf);
    }
    len++;
    strcat(buf, " ");
#if defined(DEBUG)
    if (strlen(buf) != len) {
	fprintf(stderr, "BuildExternalLine: strlen %d != len %d\n", strlen(buf), len);
	fflush(stderr);
	exit(1);
    }
#endif
    for (num_newlines = 0, i = 0; bytes[i] != '\0'; i++) {
	if (bytes[i] == '\n' || bytes[i] == '\\')
	    num_newlines++;
    }
    tmpret = retval = Malloc((size_t)(len+bytelen+num_newlines+1));
    tmpbuf = buf;
    while (*tmpbuf != '\0') {
	*tmpret++ = *tmpbuf++;
    }
    tmpbuf = bytes;
    while (*tmpbuf != '\0') {
	/* backslash the newlines */
	if (*tmpbuf == '\n') {
	    *tmpret++ = '\\';
	    *tmpret++ = *tmpbuf++;
	} else if (*tmpbuf == '\\') {
	    *tmpret++ = '\\';
	    *tmpret++ = *tmpbuf++;	    
	} else {
	    *tmpret++ = *tmpbuf++;
	}
    }
    *tmpret = '\0';
    return retval;
}
    

static void ConvertInternalToExternal(schema, Intern, Extern)
    Schema		*schema;
    InCoreEntry		*Intern;
    Entry		*Extern;
{
    Entry		TmpExtern;
    char		*ch, buf[BUFSIZ];
    int			i;

#if defined(DIAGNOSTIC)
    if (Extern == NULL)
	PANIC("ConvertInternalToExternal: Extern == NULL");
#endif
    if (*Extern != NULL)
	Qddb_Free(QDDB_TYPE_ENTRY, *Extern);
    if (Intern == NULL || Intern->SequenceNumber == 0) {
	*Extern = NULL;
	return;
    }
    sprintf(buf, "%%0 V %d", Intern->SequenceNumber);
    ch = Malloc(strlen(buf)+1);
    strcpy(ch, buf);
    for (i = 0; Intern[i].SequenceNumber != 0; i++);
    i++;
    *Extern = (Entry)Malloc(sizeof(char *)*(i+1));
    **Extern = ch;
    for (TmpExtern = *Extern + 1; Intern->SequenceNumber != 0; Intern++, TmpExtern++) {
	char		*adate;

	if (strlen(Intern->Data) == 0) {
	    TmpExtern--;
	    continue;
	}
	/* Build the current External format line */
	if (schema->Entries[Intern->AttributeNumber].Type == Date) {
	    adate = Qddb_DateStringToTime(Intern->Data);
	    *TmpExtern = BuildExternalLine(schema, Intern, adate, strlen(adate));
	    Free(adate);
	} else {
	    *TmpExtern = BuildExternalLine(schema, Intern, Intern->Data, strlen(Intern->Data));
	}
    }
    *TmpExtern = NULL;
}

static InCoreEntry *ConvertTCLvalueToInternal(schema, input)
    Schema			*schema;
    char			*input;
{
#if !defined(USE_TCL)
    fprintf(stderr, "Error: TCLvalue type not supported.\n");
    fprintf(stderr, "Recompile Qddb with the --with-tcl or --with-tk options.\n");
    return NULL;
#else
    InCoreEntry			*retval;

    ResetTCLParser(); 
    Qddb_TCLParse_Set(schema);
    Qddb_SetBuffer(input);
    if (TCLParse() != 0)
	retval = NULL;
    else
	retval = Qddb_TCLParse_Get();
    return retval;
#endif
}

static char *ConvertInternalToTCLvalue(schema, ThisEntry)
    Schema		*schema;
    InCoreEntry		*ThisEntry;
{
#if !defined(USE_TCL)
    fprintf(stderr, "Error: TCLvalue type not supported.\n");
    fprintf(stderr, "Recompile Qddb with the --with-tcl or --with-tk options.\n");
    return NULL;
#else
    int			i;
    int			level = 1;
    int			LastAttr = -1;
    char		tabs[BUFSIZ], buf[BUFSIZ];

    if (ThisEntry == NULL) {
    	return NULL;
    }
    Qddb_InitBuffer();
    sprintf(buf, "{ $NUMBER$ \"%d\" }\n", ThisEntry->SequenceNumber);
    Qddb_ConcatBuffer(buf);
    tabs[0] = '\0';
    while (ThisEntry->SequenceNumber != 0) {
	SchemaNode		*TmpEntry, *LastEntry = NULL;
	InCoreEntry		*LastInCore = NULL;
	int			flag;

	if (ThisEntry->Data != NULL)
	    StripBackslashes(ThisEntry->Data);
	TmpEntry = schema->Entries+ThisEntry->AttributeNumber;
	if (LastAttr != -1)
	    LastEntry = schema->Entries+LastAttr;
	if (TmpEntry->Level == 1) {
	    if (LastAttr != -1) {
		for (i = 1; i < LastEntry->Level; i++) {
                    /* add }'s and remove all tabs */
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s}\n", tabs);
		    Qddb_ConcatBuffer(buf);
		}
	    }
            LastAttr = ThisEntry->AttributeNumber;
	    LastInCore = ThisEntry;
	    level = 1;
	    sprintf(buf, "{ %s \"%s\" }\n", TmpEntry->Name, ThisEntry->Data);
	    Qddb_ConcatBuffer(buf);
	} else { /* Not a level 1 leaf */
	    if (LastAttr == -1) { /* 1st time through */
		LastAttr = ThisEntry->AttributeNumber;
		LastInCore = ThisEntry;
		/* Now do appropriate indentation */
		for (i = 1; i < TmpEntry->Level; i++) {
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1, i);
		    sprintf(buf, "%s{ %s \n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		}
		sprintf(buf, "%s{ %s \"%s\" }\n", tabs, TmpEntry->Name, ThisEntry->Data);
		Qddb_ConcatBuffer(buf);
		level = TmpEntry->Level;
	    } else if (LastEntry->Level == TmpEntry->Level) {
		/* STEPS:
		 * 	1. Check prefix match
		 *	2. Close any finished subattributes
		 *	3. Open any new subattributes
		 *	4. Print the leaf
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = TmpEntry->Level;
		for (i = TmpEntry->Level-1; i >= 1; i--) { /* close old */
		    if (TmpEntry->AncestorNumber[i] !=
			LastEntry->AncestorNumber[i]) {
			flag = i;
		    } 
		} 
		/* Check instances, are we starting a new instance? */
		/* Instances start at 0 */
		if (LastEntry->Level > 1) {
		    for (i = TmpEntry->Level-1; i >= 0; i--) {
			if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			    if (flag > i+1)
				flag = i+1;
			}
		    }
		}
		for (i = flag; i < TmpEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s}\n", tabs);
		    Qddb_ConcatBuffer(buf);
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s{ %s \n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		    level++;
		}
		sprintf(buf, "%s{ %s \"%s\" }\n", tabs, TmpEntry->Name, ThisEntry->Data);
		Qddb_ConcatBuffer(buf);
		LastInCore = ThisEntry;
	    } else if (LastEntry->Level > TmpEntry->Level) {
		/* Going down in length; LastEntry has more
		 * levels, so we must start from the beginning
		 * to find where they differ.
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = TmpEntry->Level;
		for (i = 1; i < TmpEntry->Level; i++) { /* close old */
		    if (TmpEntry->AncestorNumber[i] !=
 			LastEntry->AncestorNumber[i]) {
			flag = i;
			break;
		    } 
		} 
		/* Instances start at 0 */
		for (i = TmpEntry->Level-1; i >= 0; i--) {
		    if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			if (flag > i+1)
			    flag = i+1;
		    }
		}
		for (i = flag; i < LastEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s}\n", tabs);
		    Qddb_ConcatBuffer(buf);
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s{ %s \n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);
		    strcat(tabs, "\t");
		    level++;
		}
		sprintf(buf, "%s{ %s \"%s\" }\n", tabs, TmpEntry->Name, ThisEntry->Data);
		Qddb_ConcatBuffer(buf);		
		LastInCore = ThisEntry;
	    } else if (LastEntry->Level < TmpEntry->Level) {
		/* Going up in length, TmpEntry has more levels
		 * this time.
		 */
		LastAttr = ThisEntry->AttributeNumber;
		flag = LastEntry->Level;
		for (i = 1; i < LastEntry->Level; i++) { /* close old */
		    if (TmpEntry->AncestorNumber[i] !=
			LastEntry->AncestorNumber[i]) {
			flag = i;
			break;
		    } 
		} 
		/* Check instances, are we starting a new instance? */
		for (i = TmpEntry->Level-1; i >= 0; i--) {
		    if (ThisEntry->Instance[i] != LastInCore->Instance[i]) {
			if (i+1 < flag)
			    flag = i+1;
		    }
		}
		for (i = flag; i < LastEntry->Level; i++) {
		    tabs[strlen(tabs)-1] = '\0';
		    sprintf(buf, "%s}\n", tabs);
		    Qddb_ConcatBuffer(buf);		    
		    level--;
		}
		for (i = flag; i < TmpEntry->Level; i++) { /* open new */
		    int		attr;

		    attr = Qddb_FindAttributeNumber(schema, TmpEntry->AncestorNumber+1,i);
		    sprintf(buf, "%s{ %s \n", tabs, schema->Entries[attr].Name);
		    Qddb_ConcatBuffer(buf);		    
		    strcat(tabs, "\t");
		    level++;
		}
		sprintf(buf, "%s{ %s \"%s\" }\n", tabs, TmpEntry->Name, ThisEntry->Data);
		Qddb_ConcatBuffer(buf);		
		LastInCore = ThisEntry;
	    }
	}
	ThisEntry++;
    }
    /* Terminate any unclosed ('s */
    for (i = 1; i < level; i++) {
	tabs[strlen(tabs)-1] = '\0';
	sprintf(buf, "%s}\n", tabs);
	Qddb_ConcatBuffer(buf);
    }
    sprintf(buf, "\n\n");
    Qddb_ConcatBuffer(buf);
    return Qddb_GetBuffer();
#endif
}

/* ConvertTCLExternalToInternal
 */
static InCoreEntry *ConvertTCLExternalToInternal(schema, entry)
    Schema			*schema;
    TCLEntry			entry;
{
#if !defined(USE_TCL)
    fprintf(stderr, "Error: TCLExternal type not supported.\n");
    fprintf(stderr, "Recompile Qddb with the --with-tcl or --with-tk options.\n");
    return NULL;
#else
    char			*Data, *AttributeName, *InstanceName;
    int				SequenceNumber = -1;
    InCoreEntry			*incore;
    int				incore_top = 0, i, j, k;
    int				first_line = 0;
    int				attr_num;

    if (entry == NULL) {
	return NULL;
    } else {
	int			l;
	TCLEntry		e;

	e = entry;
	l = 0;
	while (*e != NULL) {
	    l++;
	    e++;
	}
	incore = (InCoreEntry *)Malloc(sizeof(InCoreEntry)*(l+1));
	if (incore == NULL)
	    return NULL;
    }
    while (*entry != NULL) {
	char		*ch;

	ch = *entry;
	if (first_line == 0 && *ch == '$') {
	    while (*ch != '\0' && !isspace(*ch))
		ch++; /* skip $NUMBER$ */
	    SequenceNumber = atoi(ch);
	    first_line = 1;
	    entry++;
	    continue;
	}

	while (*ch != '\0' && isspace(*ch))
	    ch++; /* skip leading spaces */
	AttributeName = ch;
	while (*ch != '\0' && *ch != ',' && !isspace(*ch))
	    ch++;
	*ch++ = '\0';
	InstanceName = ch;
	while (*ch != '\0' && !isspace(*ch))
	    ch++;
	*ch++ = '\0';
	Data = ch;
	incore[incore_top].Data = (char *)Malloc(strlen(Data)+1);
	if (incore[incore_top].Data == NULL)
	    return NULL;
	incore[incore_top].Marked = Inapplicable;
	strcpy(incore[incore_top].Data, Data);
	incore[incore_top].BytesOfData = strlen(Data);
	incore[incore_top].SequenceNumber = SequenceNumber;
	attr_num = Qddb_ConvertAttributeNameToNumber(schema, AttributeName);
	if (attr_num < 0) {
	    fprintf(stderr, "Couldn't convert %s\n", AttributeName);
	    /* FIXME: free up stuff */
	    return NULL;
	}
	incore[incore_top].AttributeNumber = attr_num;
	i = j = 0;
	k = 1;
	if (StringToInstance(InstanceName, incore[incore_top].Instance) == -1) {
	    /* FIXME: free up stuff */
	    return NULL;
	}
	while (InstanceName+i < Data) {
	    while (isdigit(InstanceName[i]))
		i++;
	    InstanceName[i++] = '\0';
	    incore[incore_top].Instance[k++] = atoi(InstanceName+j);
	    j = i;
	}
	incore_top++;
	entry++;
    }
    incore[incore_top].SequenceNumber = 0;
    return incore;
#endif
}

/* ConvertInternalToTCLExternal
 */
static char *ConvertInternalToTCLExternal(schema, incore)
    Schema			*schema;
    InCoreEntry			*incore;
{
#if !defined(USE_TCL)
    fprintf(stderr, "Error: TCLExternal type not supported.\n");
    fprintf(stderr, "Recompile Qddb with the --with-tcl or --with-tk options.\n");
    return NULL;
#else
    int				i, j, len, argc;
    char			*ch, buf[BUFSIZ], *build_buf, **argv;

    for (argc = 0; incore[argc].SequenceNumber != 0; argc++);
    argc++;
    argv = (char **)Malloc(sizeof(char *)*((argc+1)*2));
    argv[argc*2+1] = NULL;
    i = 0;
    argv[i] = Malloc(9);
    strcpy(argv[i], "$NUMBER$");
    sprintf(buf, "%d", incore->SequenceNumber);
    argv[++i] = Malloc(strlen(buf)+1);
    strcpy(argv[i], buf);
    while (incore->SequenceNumber != 0) {
	char			*inst_str;
	
	build_buf = Qddb_ConvertAttributeNumberToName(schema, (int)incore->AttributeNumber);
	if (build_buf == NULL) {
	    for (j = 0; j <= i; j++)
		Free(argv[j]);
	    Free(argv);
	    return NULL;
	}
	Qddb_InitBuffer();
	Qddb_ConcatBuffer(build_buf);
	Qddb_ConcatBuffer(",");
	Free(build_buf);
	inst_str = InstanceToString(incore->Instance, (size_t)schema->Entries[incore->AttributeNumber].Level);
	Qddb_ConcatBuffer(inst_str);
	Free(inst_str);
	build_buf = Qddb_GetBuffer();
	argv[++i] = Malloc(strlen(build_buf)+1);
	strcpy(argv[i], build_buf);
	ch = incore->Data;
	len = incore->BytesOfData;
#if defined(DIAGNOSTIC)
	argv[++i] = Malloc(strlen(ch)+1);
#else
	argv[++i] = Malloc(len+1);
#endif
	strcpy(argv[i], ch);
	incore++;
    }
    ch = Tcl_Merge(i+1, argv);
    for (j = 0; j <= i; j++)
	Free(argv[j]);
    Free(argv);
    return ch;
#endif
}

static void DataTreeAttributes _ANSI_ARGS_((Schema *, InCoreEntry **, DataTree **, int *));

static void DataTreeInstances(schema, entry, tree_inst, instances)
    Schema			*schema;
    InCoreEntry			**entry;
    DataTree			*tree_inst;
    int				*instances;
{
    SchemaNode			*schema_node;
    int				level, *newinstances, i, j, num;

    schema_node = tree_inst->datatree_schema->schemanode;
    level = schema_node->Level;
    newinstances = (int *)alloca(sizeof(int)*level);
    for (i = 0; i < level-1; i++)
	newinstances[i] = instances[i];
    if (schema_node->IsLeaf == False) {
	for (i = 0; tree_inst[i].datatree_type != DATATREE_END; i++) {
	    if (tree_inst[i].datatree_type != DATATREE_CHILDREN)
		abort();
	    newinstances[level-1] = i+1;
	    DataTreeAttributes(schema, entry, tree_inst[i].datatree_children, newinstances);
	}
	return;
    }
    for (num = 0; (*entry)[num].SequenceNumber != 0; num++);
    for (i = 0; tree_inst[i].datatree_type != DATATREE_END; i++) {
	InCoreEntry		*tmp_entry;
	int			tmp;

	if (tree_inst[i].datatree_type == DATATREE_NOVAL) {
	    continue;
	}
	newinstances[level-1] = i+1;
	num++;
	*entry = (InCoreEntry *)Realloc(*entry, sizeof(InCoreEntry)*(num+1));
	(*entry)[num].SequenceNumber = 0;
	tmp = num-1;
	tmp_entry = (*entry) + tmp;
#if defined(DATATREE_DEBUG)
	fprintf(stderr, "seq: %d\n", tree_inst[i].datatree_sequence_number);
#endif
	tmp_entry->SequenceNumber = tree_inst[i].datatree_sequence_number;
	tmp_entry->Index = tmp;
	tmp_entry->Marked = Inapplicable;
	tmp_entry->Data = Qddb_DataTreeProcess(schema, NULL, tree_inst+i, QDDB_DATATREE_PROC_GETDATA, 0);

#if defined(DATATREE_DEBUG)
	fprintf(stderr, "data: %s\n", tmp_entry->Data);
#endif
	tmp_entry->BytesOfData = strlen(tmp_entry->Data); 
	tmp_entry->AttributeNumber = tree_inst[i].datatree_schema->schemanode - schema->Entries;
	for (j = 0; j < level; j++) {
	    tmp_entry->Instance[j] = newinstances[j];
	}
    }
}

static void DataTreeAttributes(schema, entry, tree, instances)
    Schema			*schema;
    InCoreEntry			**entry;
    DataTree			**tree;
    int				*instances;
{
    while (*tree != NULL) {
#if defined(DATATREE_DEBUG)
	fprintf(stderr, "intermediate node %s\n", (*tree)->datatree_schema->schemanode->Name);
#endif
	DataTreeInstances(schema, entry, (*tree), instances);
	tree++;
    }
}

static InCoreEntry *ConvertDataTreeToInternal(schema, tree)
    Schema			*schema;
    DataTree			**tree;
{
    InCoreEntry			*retval = NULL;

    if (tree == NULL || *tree == NULL)
	return NULL;
    retval = (InCoreEntry *)Malloc(sizeof(InCoreEntry));
    retval->SequenceNumber = 0;
    DataTreeAttributes(schema, &retval, tree, NULL);
    return retval;
}


static DataTree **ConvertInternalToDataTree(schema, entry)
    Schema			*schema;
    InCoreEntry			*entry;
{
    DataTree			**retval = NULL;
    int				i;

    retval = Qddb_DataTreeProcess(schema, NULL, schema->Tree, QDDB_DATATREE_PROC_NEWINSTANCE, 0);
    for (i = 0; entry[i].SequenceNumber != 0; i++) {
	(void)Qddb_DataTreeProcess(schema, retval, (void *)(entry+i), QDDB_DATATREE_PROC_ADDNODE, 0);
    }
    (void)Qddb_DataTreeProcess(schema, retval, &entry[0].SequenceNumber, QDDB_DATATREE_PROC_SETSEQ, 0);
    return retval;
}
