/*----------------------------------------------------------------------*
** File: pql_query.c
** 
** 
** Project:	PQL
** =====================================================================*
** Description:	PQL (Subset II) - pql query operation
**
**  Author:	Bjoern Lemke
**  E-Mail:	lemke@lf.net
**
** Copyright (c) 1994 Bjoern Lemke.
**----------------------------------------------------------------------*/

/* include this first, FD_SETSIZE must be defined */
#include "eng.h"

#include <string.h>
#include <stdio.h>
#include "pql_parse.h"
#include "pql_tree.h"

/***********************/
/* imported operations */
/***********************/
extern char *getvar(char *var);

/*********************/
/* public prototypes */
/*********************/
int exec_query(QueryOp *query);
int getexpr(Expr *expr, int *type, void *eptr);
int evalpredicate(Predicate *pred);
int get_table(QueryOp *query, char *attr,
	      char *table, char *talias, int *depth);

/**********************/
/* private prototypes */
/**********************/
static int prep_query(QueryOp *query);
static int new_join(JoinBuf *jbuf, Condition *cond);
static int next_join(JoinBuf *jbuf);
static Predicate *next_predicate(Condition *cond);
static int next_conjunction(Condition *cond);
static int comp_join(JoinBuf *jbuf, FromList *fromlist, int join);
static void reset_condition(Condition *cond);
static void reset_preds(Condition *cond);
static int expr_compare(char *expr1, char *expr2, int type, int comparison);
static void treat_neg(Condition *cond);
static int addtables(JoinBuf *jbuf, Predicate *pred);
static int addexptbl(JoinBuf *jbuf, Expr *expr);
static int addattrtbl(JoinBuf *jbuf, AttrSpec *attrspec);
static int addsqtbl(JoinBuf *jbuf, int depth);
static int storejoin(JoinBuf *jbuf);
static int getqueryres(QueryOp *query, int *type, void *eptr, int num);
static void order_join(QueryOp *query);
static void norm_jbuf(JoinBuf *njbuf, JoinBuf *jbuf);
static int tuple_compare(QueryOp *query, int t1, int t2);
static int tuple_swop(QueryOp *query, int t1, int t2);

/********************/
/* global variables */
/********************/

extern ATable attable[];
extern int attnum; /* number of scanned attributes */
extern int selectmode;
extern int actdepth; /* for sub query requests */


static char tpath[NAMELEN];

/***********************/
/* base table routines */
/***********************/

int get_table(QueryOp *query, char *attr, char *table, 
	      char *talias, int *depth)
{
  FromList *fromlist;
  Relation *rel;
  int i, tablegot=0;
  char *pathvar=getvar(VN_DBPATH);

  while (query != NULL) {
    fromlist = query->selcondptr->fromlistptr;
    while (fromlist != NULL) {
      if (table[0] != '\0') {
	if (!strcmp(fromlist->tablespecptr->tablealias, table)) {
	  strcpy(talias, table);
	  strcpy(table, fromlist->tablespecptr->tablename);
	  *depth = query->depth;
	  tablegot=1;
	} else if (!strcmp(fromlist->tablespecptr->tablename, table)) {
	  /* set up an appropriate alias */
	  strcpy(talias, fromlist->tablespecptr->tablealias);
	  *depth = query->depth;
	  tablegot=1;
	}
	if (tablegot) {
	  /* create path */
	  strcpy(tpath, pathvar);
	  strcat(tpath, fromlist->tablespecptr->tablename);
	  if ((rel = eng_begin(tpath, READ)) == NULL) {
	    fprintf(stderr,"PQL-ERROR: cannot access %s\n",
		    fromlist->tablespecptr->tablename);
	    return(-1);
	  }
	  
	  for (i=0; i < rel->maxidx; i++)
	    if (!strcmp(rel->name[i], attr)) {
	      strcpy(table, fromlist->tablespecptr->tablename);
	      strcpy(talias, fromlist->tablespecptr->tablealias);		  
	      eng_commit(rel);
	      *depth = query->depth;
	      return(0);
	    }
	  eng_commit(rel);
	}
      } else {
	/* create path */
	strcpy(tpath, pathvar);
	strcat(tpath, fromlist->tablespecptr->tablename);
	if ((rel = eng_begin(tpath, READ)) == NULL) {
	  fprintf(stderr,"PQL-ERROR: cannot access %s\n",
		  fromlist->tablespecptr->tablename);
	  return(-1);
	}
	
	for (i=0; i < rel->maxidx; i++)
	  if (!strcmp(rel->name[i], attr)) {
	    strcpy(table, fromlist->tablespecptr->tablename);
	    strcpy(talias, fromlist->tablespecptr->tablealias);		  
	    eng_commit(rel);
	    *depth = query->depth;
	    return(0);
	  }
	eng_commit(rel);
      }
      fromlist = fromlist->next;
    }
    query = query->prev;
  }
  return(-1);
}

/* Adding required tablehandles to the specified jbuf. This includes opening
   of the tables and make them available in jbuf */

static int addtables(JoinBuf *jbuf, Predicate *pred)
{
  ExprList *exprlist;

  switch (pred->rule) {
  case PRED_EXPR:
    if (addexptbl(jbuf, pred->expr1ptr) == -1)
      return(-1);
    if(addexptbl(jbuf, pred->expr2ptr) == -1)
      return(-1);
    return(0);
  case PRED_LIKE:
  case PRED_N_LIKE:
    if (addattrtbl(jbuf, pred->attrspecptr) == -1)
      return(-1);
    return(0);
  case PRED_QUERY:
  case PRED_IN_QUERY:
  case PRED_ALL_QUERY:
  case PRED_ANY_QUERY:
    if (addexptbl(jbuf, pred->expr1ptr) == -1)
      return(-1);
    /* add the tables required by the subquery */
    if (addsqtbl(jbuf, pred->queryopptr->depth - 1) == -1)
      return(-1);
    return(0);
  case PRED_EXISTS:
  case PRED_N_EXISTS:
    if (addsqtbl(jbuf, pred->queryopptr->depth - 1) == -1)
      return(-1);
    /* we need to add at least one  table if there are still no tables added */
    return(0);
  case PRED_N_IN_ELIST:
  case PRED_IN_ELIST:
    if (addexptbl(jbuf, pred->expr1ptr) == -1)
      return(-1);
    exprlist=pred->exprlistptr;
    while (exprlist != NULL) {
      if (addexptbl(jbuf, exprlist->exprptr) == -1)
	return(-1);
      exprlist=exprlist->next;
    }
    return(0);
  default:
    fprintf(stderr, "PQL-ERROR: rule not supported yet\n");
    return(-1);
  }
}

static int addexptbl(JoinBuf *jbuf, Expr *expr) 
{
  switch (expr->rule) {
  case EXPR_ATTSPEC:
    if (addattrtbl(jbuf, expr->attrspecptr) == -1)
      return(-1);
    return(0);
  case EXPR_PLUS:
  case EXPR_MINUS:
  case EXPR_MUL:
  case EXPR_DIV:
    if (addexptbl(jbuf, expr->expr1ptr) == -1)
      return(-1);
    if (addexptbl(jbuf, expr->expr2ptr) == -1)
      return(-1);
    return(0);
  }
}

static int addattrtbl(JoinBuf *jbuf, AttrSpec *attrspec)
{
  int i, j;
  AddTable *tableptr;
  char *pathvar=getvar(VN_DBPATH);

  /* check if attribute is referenced to higher level */
  for (i=0; i<attnum; i++) {
    if (attable[i].phase == CONDITION
	&& !strcmp(attrspec->tablename, attable[i].attrspec->tablename)
	&& !strcmp(attrspec->talias, attable[i].attrspec->talias)
	&& !strcmp(attrspec->attrname, attable[i].attrspec->attrname)
	&& attable[i].depth < actdepth) {
/*    attrspec->slot = attable[i].attrspec->slot;
      attrspec->type = attable[i].attrspec->type; */
      return(0); /* table is referenced on higher level */
    }
  }

  for (i=0; i<jbuf->tnum; i++) {
    if (!strcmp(attrspec->tablename, jbuf->tables[i])
	&& !strcmp(attrspec->talias, jbuf->talias[i]))
      return(0);
  }
  if (jbuf->atab == NULL) {
    if ((jbuf->atab=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
      perror("malloc");
      return(-1);
    }
    tableptr=jbuf->atab;
  } else {
    tableptr=jbuf->atab;
    while (tableptr->next != NULL) {
      if (!strcmp(attrspec->tablename, tableptr->rel->relname) 
	  && !strcmp(attrspec->talias, tableptr->talias))
	return(0);
      tableptr=tableptr->next;
    }
    if (!strcmp(attrspec->tablename, tableptr->rel->relname) 
	&& !strcmp(attrspec->talias, tableptr->talias))
      return(0);
    if ((tableptr->next=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
      perror("malloc");
      return(-1);
    }
    tableptr=tableptr->next;
  }
  
  strcpy(tpath, pathvar);
  strcat(tpath, attrspec->tablename);
  if ((tableptr->rel = eng_begin(tpath, READ)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access %s\n", 
	   attrspec->tablename);
    return(-1);
  }
  
  /* add alias */
  strcpy(tableptr->talias, attrspec->talias);
  
  /* set up slot pointer */
  attrspec->access = BASETABLE;
  attrspec->slot = NULL;
  
  /* search entry in relation handle */
  for (i=0; i<tableptr->rel->maxidx; i++) {
    /* get types and slots */
    for (j=0; j < attnum; j++)
      if (attable[j].depth == actdepth) {
	if (!strcmp(attable[j].attrspec->tablename, 
		    tableptr->rel->relname)
	    && !strcmp(attable[j].attrspec->talias,
		       tableptr->talias)
	    && !strcmp(attable[j].attrspec->attrname, 
		       tableptr->rel->name[i])) {
	  attable[j].attrspec->type = tableptr->rel->type[i];
	  attable[j].attrspec->atype = tableptr->rel->atype[i];
	  
	  if (tableptr->rel->atype[i] == PKEY) {
	    attable[j].attrspec->slot = 
	      tableptr->pktuplebuf + tableptr->rel->start[i];
	    attable[j].attrspec->bmbase = -1;
	  } else { 
	    attable[j].attrspec->slot = 
	      tableptr->attuplebuf + tableptr->rel->start[i];
	    /* bitmap starts after the primary */
	    attable[j].attrspec->bmbase = tableptr->rel->pksize;
	    attable[j].attrspec->bmidx = i - tableptr->rel->num_pk - 1;
	    /* set up bitmap */
	    attable[j].attrspec->bm = tableptr->attuplebuf;
	      /* = (void*)(attable[j].attrspec->bmbase + tableptr->attuplebuf); */
	  }
	}
      }
  }
  if (attrspec->slot == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access attribute %s.%s\n",
	    attrspec->tablename, attrspec->attrname);
    return(-1);
  }
  
  /* add all other involved attributes to the BASETABLE group */
  for (i=0; i < attnum; i++)
    if (attable[i].depth == actdepth)
      if (attrspec != attable[i].attrspec) /* no self compare */
	if (!strcmp(attable[i].attrspec->tablename, 
		    attrspec->tablename)
	    && !strcmp(attable[i].attrspec->talias,
		       attrspec->talias)) {
	  attable[i].attrspec->access = BASETABLE;
	}
  
  
  /* fetch first tuple */
  if (eng_fetch(tableptr->rel, tableptr->pktuplebuf, 
		tableptr->attuplebuf, FIRST)
      != MORETUP) {
    fprintf(stderr, "PQL-ERROR: cannot fetch from %s\n", tableptr->rel->relname);
    return(-1);
  }
  
  /* set up relation parameters */
  tableptr->nextrec=NEXT; /* next record will be stored */
  /* tableptr->recnum=mb_num(tableptr->rel); */
  tableptr->next=NULL;
  return(0);
}

static int addsqtbl(JoinBuf *jbuf, int depth)
{
  int i, j, k;
  AddTable *tableptr;
  char *pathvar=getvar(VN_DBPATH);

  for (i=0; i < attnum; i++) {
    /* this is the condition for adding a subquery table */
    if (attable[i].phase == CONDITION 
	&& attable[i].depth == depth 
	&& attable[i].used > depth) {

      for (j=0; j < jbuf->tnum; j++)
	if (!strcmp(attable[i].attrspec->tablename, jbuf->tables[j])
	    && !strcmp(attable[i].attrspec->talias, jbuf->talias[j]))
	    return(0); /* table already joined */

      if (jbuf->atab == NULL) {
	if ((jbuf->atab=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
	  perror("malloc");
	  return(-1);
	}
	tableptr=jbuf->atab;
      } else {
	tableptr=jbuf->atab;
	while (tableptr->next != NULL)
	  tableptr=tableptr->next;
	if ((tableptr->next=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
	  perror("malloc");
	  return(-1);
	}
	tableptr=tableptr->next;
      }

      strcpy(tpath, pathvar);
      strcat(tpath, attable[i].attrspec->tablename);      
      if ((tableptr->rel = eng_begin(tpath, READ)) == NULL) {
	fprintf(stderr, "PQL-ERROR: cannot access %s\n", 
		attable[i].attrspec->tablename);
	return(-1);
      }
  
      /* add alias */
      strcpy(tableptr->talias, attable[i].attrspec->talias);
  
      /* set up slot pointer */
      attable[i].attrspec->access = BASETABLE;
      attable[i].attrspec->slot = NULL;
  
      /* search entry in relation handle */
      for (k=0; k<tableptr->rel->maxidx; k++) {
	
	/* get types for other attributes */
	for (j=0; j < attnum; j++) 
	  if (attable[j].depth <= actdepth)
	    if (!strcmp(attable[j].attrspec->tablename, 
			tableptr->rel->relname)
		&& !strcmp(attable[j].attrspec->talias,
			   tableptr->talias)
		&& !strcmp(attable[j].attrspec->attrname, 
			   tableptr->rel->name[k])) {
		attable[j].attrspec->type = tableptr->rel->type[k];
		attable[j].attrspec->atype = tableptr->rel->atype[k];
		attable[j].attrspec->bmbase = tableptr->rel->pksize;
		attable[j].attrspec->bmidx = k - tableptr->rel->num_pk -1;
	      }
		
	        
	if(!strcmp(tableptr->rel->name[k], attable[i].attrspec->attrname)) {
	  if (tableptr->rel->atype[k] == PKEY) {
	    attable[i].attrspec->slot = tableptr->pktuplebuf +  
	      tableptr->rel->start[k];
	    attable[i].attrspec->bmbase = -1;
	  } else { 
	    attable[i].attrspec->slot = tableptr->attuplebuf +  
	      tableptr->rel->start[k];
	    /* bitmap starts after primary */
	    attable[i].attrspec->bmbase = tableptr->rel->pksize;
	    attable[i].attrspec->bmidx = k - tableptr->rel->num_pk - 1;
	    /* set up bitmap */
	    attable[i].attrspec->bm = tableptr->attuplebuf;
	      /* = (void*)(attable[i].attrspec->bmbase + tableptr->attuplebuf); */
	  }
	  /* set up type */
	  attable[i].attrspec->type = tableptr->rel->type[k];
	  attable[i].attrspec->atype = tableptr->rel->atype[k];
	}
      }
      if (attable[i].attrspec->slot == NULL) {
	fprintf(stderr, "PQL-ERROR: cannot access attribute %s\n",
		attable[i].attrspec->attrname);
	return(-1);
      }
  
      /* add all other involved attributes to the BASETABLE group */
      for (j=0; j < attnum; j++)
	if (attable[j].depth == actdepth)
	  if (attable[i].attrspec != attable[j].attrspec) /* no self compare */
	    if (!strcmp(attable[i].attrspec->tablename, 
			attable[j].attrspec->tablename)
		&& !strcmp(attable[i].attrspec->talias,
			   attable[j].attrspec->talias)) {
	      attable[j].attrspec->access = BASETABLE;
	    }
      
      /* fetch first tuple */
      if (eng_fetch(tableptr->rel, tableptr->pktuplebuf, 
		    tableptr->attuplebuf, FIRST) != MORETUP) {
	fprintf(stderr, "PQL-ERROR: cannot fetch from %s\n", 
		tableptr->rel->relname);
	return(-1);
      }
      
      /* set up relation parameters */
      tableptr->nextrec=NEXT; /* next record will be stored */
      /* tableptr->recnum=mb_num(tableptr->rel); */
      tableptr->next=NULL;
      return(0);
    }
  }
}


/*****************/
/* join routines */
/*****************/

static int next_join(JoinBuf *jbuf)
{
  AddTable *base=jbuf->atab;
  AddTable *tlist=jbuf->atab;
  int engret;

  while (tlist!=NULL) {
    if (tlist->nextrec == NEXT) {
      engret = eng_fetch(tlist->rel, tlist->pktuplebuf, 
			 tlist->attuplebuf, NEXT);
      switch (engret) {
      case -1:
	fprintf(stderr, "PQL-ERROR: cannot fetch from %s\n", 
		tlist->rel->relname); 
	return(-1);
      case ENDOFTUP:
	tlist->nextrec = FIRST;
	break;
      case MORETUP:
	tlist->nextrec = NEXT;
	return(1);
      }
    } else if (tlist->nextrec == FIRST) {
      /* reset relation to first record */
      engret = eng_fetch(tlist->rel, tlist->pktuplebuf,
			 tlist->attuplebuf, FIRST);
      switch (engret) {
      case -1:
	fprintf(stderr, "PQL-ERROR: cannot fetch from %s\n", 
		tlist->rel->relname);
	return(-1);
      case ENDOFTUP:
	return(0);
      case MORETUP:
	tlist->nextrec=NEXT;
	tlist=tlist->next;
	break;
      }
    }
  }
  return(0);
}

static int next_conjunction(Condition *cond)
{
  switch (cond->rule) {
  case COND_OR:
    if (cond->choice == 1) {
      if (!next_conjunction(cond->cond1ptr))
	cond->choice = 2;
      return(1);
    }
    /* must be choice 2 now */
    return(next_conjunction(cond->cond2ptr));
  case COND_AND:
    if (next_conjunction(cond->cond1ptr))
      return(1);
    if (next_conjunction(cond->cond2ptr)) {
      reset_condition(cond->cond1ptr);
      return(1);
    }
    return(0);
  case COND_PREDICATE:
    return(0);
  case COND_NOT:
    return(next_conjunction(cond->cond1ptr));
  }
} 

static Predicate *next_predicate(Condition *cond)
{
  Predicate *pred;

  switch (cond->rule) {
  case COND_OR:
    if (cond->choice == 1)
      return(next_predicate(cond->cond1ptr));
    /* choice must be 2 */
    return(next_predicate(cond->cond2ptr));
  case COND_PREDICATE:
    if (cond->predicateptr->marked == MARKED) {
      return(NULL);
    }
    cond->predicateptr->marked=MARKED;
    return(cond->predicateptr);
  case COND_NOT:
    return(next_predicate(cond->cond1ptr));
  case COND_AND:
    if ((pred = next_predicate(cond->cond1ptr)) == NULL)
      return(next_predicate(cond->cond2ptr));
    return(pred);
  }    
}

static int new_join(JoinBuf *jbuf, Condition *cond)
{
  AddTable *tableptr, *deltable;
  int newtupsize, i, j, k;
  int attslot;
  int ret;
  int jstored;
  /* for debugging */
  long lbuf;
  int matched;
  Predicate *pred;
  int first;

  /* loop over all predicates */
  while ((pred = next_predicate(cond)) != NULL) {

    /* set up slots of joined attributes */
    for (j=0; j < attnum; j++)
      if (attable[j].depth == actdepth)
	if (attable[j].attrspec->access == JOINBUF) {
	  attable[j].attrspec->slot=
	    (void*)(jbuf->jbuf[jbuf->actbuf]  
		    + (int)attable[j].attrspec->base);
	  if (attable[j].attrspec->bm != NULL)
	    attable[j].attrspec->bm= 
	      (void*)(jbuf->jbuf[jbuf->actbuf] + attable[j].attrspec->bmbase);
	}
    
    /* reset buf pointers */
    jbuf->bufptr[jbuf->actbuf]=0;
    jbuf->bufptr[(jbuf->actbuf + 1) % 2]=0;
    jstored = jbuf->stored;
    jbuf->stored = 0;
    
    
    /* add the required tables */
    if (addtables(jbuf, pred) == -1) {
      fprintf(stderr, "PQL-ERROR: cannot add required tables\n");
      return(-1);
    }

    if (jbuf->tnum == 0) { /* first initial join, don't join over jbuf */
      do {
	matched=1;
	switch (ret=evalpredicate(pred)) {
	case 0 :
	  matched=0;
	  break;
	case 1 :
	  break;
	case -1 :
	  return(-1);
	}
	if (matched) 
	  storejoin(jbuf);
      } while ((ret=next_join(jbuf)) == 1);
      /* error occured ? */
      if (ret == -1)
	return(-1);
  
    } else {
      for (i=0; i < jstored; i++) { /* loop through all tuples in jbuf */
	do { 
	  matched=1; 
	  switch (ret=evalpredicate(pred)) {
	  case 0 : 
	    matched=0;
	    break;
	  case 1 :
	    break;
	  case -1 :
	    return(-1);
	  }

	  if (matched)
	    storejoin(jbuf);
	}  while ((ret=next_join(jbuf)) == 1);
	/* error occured ? */
	if (ret == -1)
	  return(-1);
      
	/* skip forward in jbuf */
	for (j=0; j < attnum; j++)
	  if (attable[j].depth == actdepth)
	    if (attable[j].attrspec->access == JOINBUF) {
	    attable[j].attrspec->slot+=jbuf->tupsize;
	  }
	jbuf->bufptr[jbuf->actbuf] += jbuf->tupsize;
      }
    }
    
    
    /* store new tables after join */
    tableptr=jbuf->atab;
    newtupsize=0;    
    while (tableptr != NULL) {
      strcpy(jbuf->tables[jbuf->tnum], tableptr->rel->relname); 
      strcpy(jbuf->talias[jbuf->tnum], tableptr->talias); 
      for (i=0; i < tableptr->rel->maxidx; i++) {
	strcpy(jbuf->entry[jbuf->anum].attribute, tableptr->rel->name[i]);
	jbuf->entry[jbuf->anum].tableindex=jbuf->tnum;
	if (tableptr->rel->type[i] == NULLMAP) {
	  jbuf->entry[jbuf->anum].size=sizeof(fd_set);
	} else {
	  jbuf->entry[jbuf->anum].size=
	    tableptr->rel->start[i+1] - tableptr->rel->start[i];
	}
	jbuf->entry[jbuf->anum].type=tableptr->rel->type[i];
	jbuf->entry[jbuf->anum].atype=tableptr->rel->atype[i];
	jbuf->anum++;
	
      }

      /* set up bmbase for the joined tuples */
      for (i=0; i<attnum; i++) {
	if (!strcmp(tableptr->rel->relname, attable[i].attrspec->tablename)
	    && !strcmp(tableptr->talias, attable[i].attrspec->talias)
	    && attable[i].attrspec->access == BASETABLE)
	  attable[i].attrspec->bmbase = newtupsize + tableptr->rel->pksize;
      }

      newtupsize+=
	tableptr->rel->pksize + sizeof(fd_set) + tableptr->rel->attsize;
      jbuf->tnum++; 

      /* close tables and free memory */
      eng_commit(tableptr->rel);
      deltable=tableptr;
      tableptr=tableptr->next;
      free(deltable);
    }
    /* increase the tuple size */
    jbuf->tupsize+=newtupsize;
    
    /* all adding tables are added */
    jbuf->atab=NULL;
    
    /* all joined attributes are in jbuf now */
    for (j=0; j < attnum; j++) {
      if (attable[j].depth == actdepth && attable[j].attrspec->access == BASETABLE) {
	attable[j].attrspec->access = JOINBUF;
	       
	/* set up attrspec.base */
	k=0;
	attslot=0;
	while (k < jbuf->anum &&
	       (strcmp(attable[j].attrspec->attrname, jbuf->entry[k].attribute)
		|| strcmp(attable[j].attrspec->tablename, 
			  jbuf->tables[jbuf->entry[k].tableindex])
		|| strcmp(attable[j].attrspec->talias,
			  jbuf->talias[jbuf->entry[k].tableindex]))) {
	  attslot = attslot + jbuf->entry[k].size;
	  k++;
	}
	attable[j].attrspec->base = attslot;      
      }
    }

    /* switch jbuf */
    jbuf->actbuf=(jbuf->actbuf + 1) % 2;
  }
  reset_preds(cond);
}

/* complete the join with all other tables in fromlist */

static int comp_join(JoinBuf *jbuf, FromList *fromlist, int joined)
{
  int i, j, newtupsize, jstored, engret;
  TableSpec *tablespec;
  AddTable *tableptr, *deltable;
  char *pathvar=getvar(VN_DBPATH);

  /* open required tables */

  while (fromlist != NULL) {
    tablespec = fromlist->tablespecptr;
    
    i=0;  
    while (i < jbuf->tnum 
	   && (strcmp(tablespec->tablename, jbuf->tables[i])
	       || strcmp(tablespec->tablealias, jbuf->talias[i]))) i++;
    
    if (i == jbuf->tnum) { /* table not joined yet */
      if (jbuf->atab == NULL) {
	if ((jbuf->atab=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
	  perror("malloc");
	  return(-1);
	}
	tableptr=jbuf->atab;
      } else {
	tableptr=jbuf->atab;
	while (tableptr->next != NULL)
	  tableptr=tableptr->next;
	if ((tableptr->next=(AddTable*)malloc(sizeof(AddTable))) == NULL) {
	  perror("malloc");
	  return(-1);
	}
	tableptr=tableptr->next;
      }
      strcpy(tpath, pathvar);
      strcat(tpath, tablespec->tablename);
      if ((tableptr->rel = eng_begin(tpath, READ)) == NULL) {
	fprintf(stderr,"PQL-ERROR: cannot access %s\n", 
		tablespec->tablename);
	return(-1);
      }
      
      /* get types and bm-stuff for attributes */
      for (i=0; i < tableptr->rel->maxidx; i++) 
	for (j=0; j < attnum; j++)
	  if (attable[j].depth == actdepth)
	    if (!strcmp(attable[j].attrspec->tablename, 
			tableptr->rel->relname)) 
	      if (!strcmp(attable[j].attrspec->attrname, 
			  tableptr->rel->name[i])) {
		if (tableptr->rel->atype[i] == PKEY) {
		  /* indicate primary */

		  attable[j].attrspec->bmbase = -1;
		} else {
		  attable[j].attrspec->bmbase = tableptr->rel->pksize;
		  attable[j].attrspec->bmidx = i - tableptr->rel->num_pk - 1; 
		}
		attable[j].attrspec->type = tableptr->rel->type[i];
		attable[j].attrspec->atype = tableptr->rel->atype[i];
	      }
      
      /* search entry in relation handle */
      /* fetch first tuple */
      engret = eng_fetch(tableptr->rel, tableptr->pktuplebuf, 
			 tableptr->attuplebuf, FIRST);
      switch (engret) {
      case -1:
      case ENDOFTUP:
	fprintf(stderr,"PQL-ERROR: cannot fetch from %s\n", 
		tableptr->rel->relname);
	eng_commit(tableptr->rel);
	return(-1);
      case MORETUP:
	break;
      }
      /* set up relation parameters */
      strcpy(tableptr->talias, tablespec->tablealias);
      tableptr->nextrec=NEXT; /* second record will be stored */
      tableptr->next=NULL;
    }
    fromlist= fromlist->next;
  }
  
  if (jbuf->atab == NULL)  
    return(0);  /* nothing to join */
  
  /* now join the opened tables */
  
  /* reset buf pointers */
  jbuf->bufptr[jbuf->actbuf]=0;
  jbuf->bufptr[(jbuf->actbuf + 1) % 2]=0;
  jstored = jbuf->stored;
  jbuf->stored = 0;
  
  if (!joined && jbuf->tnum == 0)
    jstored=1; /* force one loop if no previous join has been performed*/
  
  for (i=0; i < jstored; i++) { /* loop through all tuples in jbuf */
    
    tableptr = jbuf->atab;
    
    /* store the first tuple */
    if (storejoin(jbuf)  == -1) {
      fprintf(stderr,"PQL-ERROR: cannot complete join\n");
      return(-1);
    }

    while (tableptr!=NULL) {
      if (tableptr->nextrec == NEXT) {
	engret = eng_fetch(tableptr->rel, tableptr->pktuplebuf, 
			   tableptr->attuplebuf, NEXT);
	switch (engret) {
	case -1:
	  fprintf(stderr,"PQL-ERROR: cannot fetch from %s\n", 
		  tableptr->rel->relname);
	  eng_commit(tableptr->rel);
	  return(-1);
	case ENDOFTUP:
	  tableptr->nextrec=FIRST;
	  break;
	case MORETUP:
	  /* store the tuple */
	  if (storejoin(jbuf)  == -1) {
	    fprintf(stderr,"PQL-ERROR: cannot complete join\n");
	    return(-1);
	  }
	  /* reset tableptr for next join record */
	  tableptr = jbuf->atab;
	  break;
	}
      } else {
	/* reset relation to first record */
	engret = eng_fetch(tableptr->rel, tableptr->pktuplebuf, 
		    tableptr->attuplebuf, FIRST);
	switch (engret) {
	case -1:
	case ENDOFTUP:
	  fprintf(stderr,"PQL-ERROR: cannot fetch from %s\n", 
		  tableptr->rel->relname);
	  return(-1);
	}
	tableptr->nextrec=NEXT;
	tableptr=tableptr->next; /* skip to next level */
      }
    }
    
    /* skip forward in jbuf */
    for (j=0; j < attnum; j++) {
      if (attable[j].depth == actdepth && attable[j].attrspec->access == JOINBUF)
	attable[j].attrspec->slot+=jbuf->tupsize;
    }
    jbuf->bufptr[jbuf->actbuf] += jbuf->tupsize;
  }
  
  /* update Join format */
  
  tableptr=jbuf->atab;
  newtupsize=0;    
  while (tableptr != NULL) {
    strcpy(jbuf->tables[jbuf->tnum], tableptr->rel->relname); 
    strcpy(jbuf->talias[jbuf->tnum], tableptr->talias); 
    for (i=0; i < tableptr->rel->maxidx; i++) {
      strcpy(jbuf->entry[jbuf->anum].attribute, tableptr->rel->name[i]);
      jbuf->entry[jbuf->anum].tableindex=jbuf->tnum;
      if (tableptr->rel->type[i] == NULLMAP) {
	jbuf->entry[jbuf->anum].size=sizeof(fd_set);
      } else {
	jbuf->entry[jbuf->anum].size=
	  tableptr->rel->start[i+1] - tableptr->rel->start[i];
      }
      jbuf->entry[jbuf->anum].type=tableptr->rel->type[i];
      jbuf->entry[jbuf->anum].atype=tableptr->rel->atype[i];
      jbuf->anum++;
    }

    /* set up bmbase for the joined tuples */
    for (i=0; i<attnum; i++) {
      if (!strcmp(tableptr->rel->relname, attable[i].attrspec->tablename)
	  && !strcmp(tableptr->talias, attable[i].attrspec->talias)
	  && attable[i].attrspec->access == BASETABLE)
	attable[i].attrspec->bmbase = newtupsize + tableptr->rel->pksize;
    }

    newtupsize+=
      tableptr->rel->pksize + sizeof(fd_set) + tableptr->rel->attsize;
    jbuf->tnum++;
    
    /* close tables and free memory */
    eng_commit(tableptr->rel);
    deltable=tableptr;
    tableptr=tableptr->next;
    free(deltable);
  }
  jbuf->tupsize+=newtupsize;
  
  /* all adding tables are added */
  jbuf->atab=NULL;
  
  /* switch jbuf */
  jbuf->actbuf=(jbuf->actbuf + 1) % 2;
}


/****************************/
/* additional join routines */
/****************************/

static void reset_condition(Condition *cond)
{
  switch (cond->rule) {
  case COND_OR:
    cond->choice=1;
    reset_condition(cond->cond1ptr);
    reset_condition(cond->cond2ptr);
    break;
  case COND_PREDICATE:
  case COND_NOT:
    /* do nothing */
    break;
  case COND_AND:
    reset_condition(cond->cond1ptr);
    reset_condition(cond->cond2ptr);
    break;
  }
} 

/* use De Morgan rule to treat negations */
static void treat_neg(Condition *cond)
{
  Predicate *pred;
  switch (cond->rule) {
  case COND_PREDICATE:
    if (cond->negate) {
      pred = cond->predicateptr;
      switch (pred->rule) {
      case PRED_EXPR:
      case PRED_QUERY:
      case PRED_ALL_QUERY:
      case PRED_ANY_QUERY:
	switch(pred->comparison) {
	case EQU:
	  pred->comparison = NEQUAL;
	  break;
	case NEQUAL:
	  pred->comparison = EQU;
	  break;
	case LEQUAL:
	  pred->comparison = MORE;
	  break;
	case LESS: 
	  pred->comparison = MEQUAL;
	  break;
	case MEQUAL:
	  pred->comparison = LESS;
	  break;
	case MORE:
	  pred->comparison = LEQUAL;
	  break;
	}
	break;
      case PRED_IN_QUERY:
	pred->rule = PRED_N_IN_QUERY;
	break;
      case PRED_N_IN_QUERY:
	pred->rule = PRED_IN_QUERY;
	break;
      case PRED_IN_ELIST:
	pred->rule = PRED_N_IN_ELIST;
	break;
      case PRED_N_IN_ELIST:
	pred->rule = PRED_IN_ELIST;
	break;
      case PRED_EXISTS:
	pred->rule = PRED_N_EXISTS;
	break;
      case PRED_N_EXISTS:
	pred->rule = PRED_EXISTS;
	break;
      case PRED_LIKE:
	pred->rule = PRED_N_LIKE;
	break;
      case PRED_N_LIKE:
	pred->rule = PRED_LIKE;
	break;
      }
    }
    break;
  case COND_AND:
    if (cond->negate) {
      cond->rule = COND_OR;
      cond->cond1ptr->negate=1;
      cond->cond2ptr->negate=1;
    }
    treat_neg(cond->cond1ptr);
    treat_neg(cond->cond2ptr);
    break;
  case COND_OR:
    if (cond->negate) {
      cond->rule = COND_AND;
      cond->cond1ptr->negate=1;
      cond->cond2ptr->negate=1;
    }
    treat_neg(cond->cond1ptr);
    treat_neg(cond->cond2ptr);
    break;
  case COND_NOT:
    if (!cond->negate) {
      cond->cond1ptr->negate = 1;
    }
    treat_neg(cond->cond1ptr);
    break;
  }
}

/***********************/
/* tuple store routine */
/***********************/

static int storejoin(JoinBuf *jbuf)
{
  AddTable *tableptr;
  int newbuf = (jbuf->actbuf + 1) % 2;
  int reclen;
  int i;
  
  /* copy old join */
  if (jbuf->tupsize > 0) {
    bcopy(jbuf->jbuf[jbuf->actbuf] + jbuf->bufptr[jbuf->actbuf],
	  jbuf->jbuf[newbuf] + jbuf->bufptr[newbuf], jbuf->tupsize);
    jbuf->bufptr[newbuf] += jbuf->tupsize;
  }
  
  tableptr=jbuf->atab;
  
  while(tableptr!=NULL) {
    reclen=tableptr->rel->pksize + sizeof(fd_set) + tableptr->rel->attsize;
    if (jbuf->bufptr[newbuf] + reclen >= TBUFSIZE) {
      fprintf(stderr, "PQL-ERROR: Join buffer exeeded\n");
      return(-1);
    }
    bcopy(tableptr->pktuplebuf, jbuf->jbuf[newbuf] + jbuf->bufptr[newbuf],
	  tableptr->rel->pksize);
    
    bcopy(tableptr->attuplebuf, jbuf->jbuf[newbuf] + jbuf->bufptr[newbuf] 
	  + tableptr->rel->pksize, tableptr->rel->attsize + sizeof(fd_set));

    jbuf->bufptr[newbuf] += reclen;
    tableptr=tableptr->next;
  }
  jbuf->stored++;
  return(0);
}

/***********************/
/* evaluation routines */
/***********************/

static int expr_compare(char *expr1, char *expr2, int type, int comparison)
{
  long l1, l2;
  float f1, f2;
  double d1, d2;

  switch(comparison) {
  case EQU:
    if (type == T_CHAR) {
      return(strcmp(expr1, expr2) == 0);
    }
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return (l1 == l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 == f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 == d2);
    }
  case LEQUAL:
    if (type == T_CHAR)
      return(strcmp(expr1, expr2) <= 0);
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return(l1 <= l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 <= f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 <= d2);
    }
  case MEQUAL:
    if (type == T_CHAR)
      return(strcmp(expr1, expr2) >= 0);
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return(l1 >= l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 >= f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 >= d2);
    }
  case LESS:
    if (type == T_CHAR)
      return(strcmp(expr1, expr2) < 0);
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return(l1 < l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 < f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 < d2);
    }
    case MORE:
    if (type == T_CHAR)
      return(strcmp(expr1, expr2) > 0);
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return(l1 > l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 > f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 > d2);
    }
  case NEQUAL:
    if (type == T_CHAR)
      return(strcmp(expr1, expr2) != 0);
    if (type == T_LONG) {
      bcopy(expr1, &l1, sizeof(long));
      bcopy(expr2, &l2, sizeof(long));
      return(l1 != l2);
    }
    if (type == T_FLOAT) {
      bcopy(expr1, &f1, sizeof(float));
      bcopy(expr2, &f2, sizeof(float));
      return(f1 != f2);
    }
    if (type == T_DOUBLE) {
      bcopy(expr1, &d1, sizeof(double));
      bcopy(expr2, &d2, sizeof(double));
      return(d1 != d2);
    }
  default:
    fprintf(stderr, "PQL-ERROR: invalid comparision\n");
    return(-1);;
  }
}

int evalpredicate(Predicate *pred)
{
  char exprbuf1[STRINGLEN], exprbuf2[STRINGLEN];
  ExprList *exprlist;
  int i, type, comptype;
  long l1, l2;
  float f1, f2;
  double d1, d2;

  switch(pred->rule) {
  case PRED_EXPR:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);;
    if (getexpr(pred->expr2ptr, &comptype, exprbuf2) == -1)
      return(-1);

    /* treat case of NULL value comparison */
    if (type == NVAL && comptype == NVAL)
      return(1);
    if (type == NVAL || comptype == NVAL)
      return(0);

    if (type != comptype) {
      fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
      return(-1);
    }
    return (expr_compare(exprbuf1, exprbuf2, type, pred->comparison));
  case PRED_QUERY:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    if (getqueryres(pred->queryopptr, &comptype, exprbuf2, 1)) {
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      return (expr_compare(exprbuf1, exprbuf2, type, pred->comparison));
    }
    return(0);
  case PRED_LIKE:
    if (pred->attrspecptr->type != T_CHAR ||
	pred->literalptr->rule != T_CHAR) {
      fprintf(stderr, "PQL-ERROR: like operator just on type T_CHAR\n");
      return(-1);
    }
    if(wildmat(pred->attrspecptr->slot, pred->literalptr->strval))
      return(1);
    return(0);
  case PRED_N_LIKE:
    if (pred->attrspecptr->type != T_CHAR ||
	pred->literalptr->rule != T_CHAR) {
      fprintf(stderr, "PQL-ERROR: like operator just on type T_CHAR\n");
      return(-1);
    }
    if(wildmat(pred->attrspecptr->slot, pred->literalptr->strval))
      return(0);
    return(1);
  case PRED_IN_QUERY:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    i=1;
    while (getqueryres(pred->queryopptr, &comptype, exprbuf2, i)) {
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (expr_compare(exprbuf1, exprbuf2, type, EQU)) {
	return(1);
      }
      i++;
    }
    return(0);
  case PRED_N_IN_QUERY:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    i=1;
    while (getqueryres(pred->queryopptr, &comptype, exprbuf2, i)) {
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (expr_compare(exprbuf1, exprbuf2, type, EQU)) {
	return(0);
      }
      i++;
    }
    return(1);
  case PRED_EXISTS:
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    return(pred->queryopptr->sqbuf->stored);
  case PRED_N_EXISTS:
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    return(!pred->queryopptr->sqbuf->stored);
  case PRED_ALL_QUERY:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    i=1;
    while (getqueryres(pred->queryopptr, &comptype, exprbuf2, i)) {
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (!expr_compare(exprbuf1, exprbuf2, type, pred->comparison)) {
	return(0);
      }
      i++;
    }
    return(1);
  case PRED_ANY_QUERY:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    actdepth++;
    exec_query(pred->queryopptr);
    actdepth--;
    i=1;
    while (getqueryres(pred->queryopptr, &comptype, exprbuf2, i)) {
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (expr_compare(exprbuf1, exprbuf2, type, pred->comparison)) {
	return(1);
      }
      i++;
    }
    return(0);
  case PRED_IN_ELIST:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    exprlist=pred->exprlistptr;
    while (exprlist != NULL) {
      if (getexpr(exprlist->exprptr, &comptype, exprbuf2) == -1)
	return(-1);
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (expr_compare(exprbuf1, exprbuf2, type, EQU))
	return(1);
      exprlist = exprlist->next;
    }
    return(0);
  case PRED_N_IN_ELIST:
    if (getexpr(pred->expr1ptr, &type, exprbuf1) == -1)
      return(-1);
    exprlist=pred->exprlistptr;
    while (exprlist != NULL) {
      if (getexpr(exprlist->exprptr, &comptype, exprbuf2) == -1)
	return(-1);
      if (type != comptype) {
	fprintf(stderr, "PQL-ERROR: type mismatch (%d, %d)\n", type, comptype);
	return(-1);
      }
      if (expr_compare(exprbuf1, exprbuf2, type, EQU))
	return(0);
      exprlist = exprlist->next;
    }
    return(1);
  default:
    fprintf(stderr,"PQL-ERROR: unknown predicate\n");
    return(-1);
  }
}

int getexpr(Expr *expr, int *type, void *eptr)
{
  char ebuf1[STRINGLEN], ebuf2[STRINGLEN];
  int newtype, newcomptype;
  long l1, l2, l3;
  float f1, f2, f3;
  double d1, d2, d3;
  fd_set mask;
  
  *type=-1; /* initialize to non valid type */

  switch (expr->rule) {
  case EXPR_LIT:
    switch (expr->literalptr->rule) {
    case T_LONG:
      *type=T_LONG;
      bcopy(&expr->literalptr->intval, eptr, sizeof(long));
      break;
    case T_FLOAT:
      *type=T_FLOAT;
      bcopy(&expr->literalptr->floatval, eptr, sizeof(float));
      break;
    case T_DOUBLE:
      *type=T_DOUBLE;
      bcopy(&expr->literalptr->doubleval, eptr, sizeof(double));
      break;
    case T_CHAR:
      *type=T_CHAR;
      strcpy(eptr, expr->literalptr->strval);
      break;
    case NVAL:
      /* fprintf(stderr,"PQL-ERROR: illegal NULL value in expression\n"); 
      return(-1);     */
      *type=NVAL;
      break;
    }
    break;
  case EXPR_ATTSPEC:
    /* check for NULL value here */
    if (expr->attrspecptr->atype == ORDY) {
      bcopy(expr->attrspecptr->bm, mask.fds_bits, sizeof(fd_set));
      if (FD_ISSET (expr->attrspecptr->bmidx + BMOFFSET, &mask)) {
	*type=NVAL;
	break;
      }
    } else if (expr->attrspecptr->atype == DROP) {
      fprintf(stderr, "PQL-ERROR: not allowed to retrieve dropped attribute\n");
      return(-1);
    }
    if (*type != NVAL) {
      switch (expr->attrspecptr->type) {
      case T_CHAR:
	*type=T_CHAR;
	strcpy(eptr, expr->attrspecptr->slot);
	break;
      case T_LONG:
	/* cast testing */
	switch (expr->attrspecptr->cast) {
	case T_NONE:
	  *type=T_LONG;
	  bcopy(expr->attrspecptr->slot, eptr, sizeof(long));
	  break;
	case T_FLOAT:
	  *type=T_FLOAT;
	  bcopy(expr->attrspecptr->slot, &l1, sizeof(long));
	  f1=(float)l1;
	  bcopy(&f1, eptr, sizeof(float));
	  break;
	}
	break;
      case T_FLOAT:
	*type=T_FLOAT;
	bcopy(expr->attrspecptr->slot, eptr, sizeof(float));
	break;
      case T_DOUBLE:
	*type=T_DOUBLE;
	bcopy(expr->attrspecptr->slot, eptr, sizeof(double));
	break;
      }
    }
    break;
  case EXPR_PLUS: /* + */
  case EXPR_MINUS: /* - */
  case EXPR_MUL: /* * */
  case EXPR_DIV: /* / */
    if (getexpr(expr->expr1ptr, &newtype, ebuf1) == -1)
      return(-1);
    if (getexpr(expr->expr2ptr, &newcomptype, ebuf2) == -1)
      return(-1);
    if (newtype != newcomptype)  {
      fprintf(stderr,"PQL-ERROR: mismatched types\n");
      return(-1);
    }
    if (newtype == T_CHAR) {
      fprintf(stderr,"PQL-ERROR: arithmetic operation not allowed on type T_CHAR\n");
      return(-1);
    }

    if (newtype == T_LONG) {
      *type=T_LONG;
      bcopy(ebuf1, &l1, sizeof(long));
      bcopy(ebuf2, &l2, sizeof(long));
      switch(expr->rule) {
      case EXPR_PLUS:  
	l3=l1+l2;
	break;
      case EXPR_MINUS:
	l3=l1-l2;
	break;
      case EXPR_MUL:
	l3=l1*l2;
	break;
      case EXPR_DIV:
	l3=l1/l2;
	break;
      }
      bcopy(&l3, eptr, sizeof(long));
      return(0);
    }
    if (newtype == T_FLOAT) {
      *type=T_FLOAT;
      bcopy(ebuf1, &f1, sizeof(float));
      bcopy(ebuf2, &f2, sizeof(float));
      switch(expr->rule) {
      case EXPR_PLUS:  
	f3=f1+f2;
	break;
      case EXPR_MINUS:
	f3=f1-f2;
	break;
      case EXPR_MUL:
	f3=f1*f2;
	break;
      case EXPR_DIV:
	f3=f1/f2;
	break;
      }
      bcopy(&f3, eptr, sizeof(float));
      return(0);
    }
    if (newtype == T_DOUBLE) {
      *type=T_DOUBLE;
      bcopy(ebuf1, &d1, sizeof(double));
      bcopy(ebuf2, &d2, sizeof(double));
      switch(expr->rule) {
      case EXPR_PLUS:  
	d3=d1+d2;
	break;
      case EXPR_MINUS:
	d3=d1-d2;
	break;
      case EXPR_MUL:
	d3=d1*d2;
	break;
      case EXPR_DIV:
	d3=d1/d2;
	break;
      }
      bcopy(&d3, eptr, sizeof(double));
      return(0);
    }
    break;
  case EXPR_FUNC:
    fprintf(stderr,"PQL-ERROR: function not implemented yet\n");
    return(-1);
  }
}

static int getqueryres(QueryOp *query, int *type, void *eptr, int num)
{
  SubQueryBuf *sqbuf;

  /* the term condition */
  if (num > query->sqbuf->stored) return(0);

  /* if getqueryres called, the subquery must fit some requirements */  
  if (query->sqbuf->anum > 1) {
    fprintf(stderr, "PQL-ERROR: invalid subquery format\n");
    return(-1);
  }

  /* set up type */
  *type = query->sqbuf->type;
  bcopy(query->sqbuf->sqbuf + (num-1) * query->sqbuf->size, 
	eptr,
	query->sqbuf->size);
  return (query->sqbuf->stored);
}

/************************/
/* major query routines */
/************************/

static int prep_query(QueryOp *query)
{
  TableSpec *tablespec;
  int i;
  
  /* fill up table entries in attable */
  /* we have already done this for type CONDITION */
  for (i=0; i < attnum; i++) {
    if (attable[i].depth == actdepth && attable[i].phase == SELECTION) {
      if (get_table(query, attable[i].attrspec->attrname, 
		  attable[i].attrspec->tablename,
		    attable[i].attrspec->talias, &attable[i].depth) == -1) {
	fprintf(stderr,"PQL-ERROR: cannot get table for attribute %s\n", 
		attable[i].attrspec->attrname);
	return(-1);
      }
    }
  }
    
  /* prepare wherecondptr to DNF */
  if (query->selcondptr->wherecondptr != NULL) {
    treat_neg(query->selcondptr->wherecondptr);
    reset_condition(query->selcondptr->wherecondptr);
  }
}

/* exec_query invoked from parser */

int exec_query(QueryOp *query)
{
  Predicate *pred;
  FromList *fromlist;
  TableSpec *tablespec;
  int i, records;
  long longbuf;
  float floatbuf;

  if (prep_query(query) == -1) {
    fprintf(stderr,"PQL-ERROR: preparation for query failed. Aborting ...\n");
    return(-1);
  }
  
  /* make the join */

  /* evaluate each conjunction */
  query->cj=0;
  i=0;
  if (query->selcondptr->wherecondptr) {
    do {

      /* allocate new join buffer */

      if ((query->jbuf[i] = (JoinBuf*)malloc(sizeof(JoinBuf))) == NULL) {
	perror("malloc");
	return(-1);
      }
      query->jbuf[i]->stored=0;
      query->jbuf[i]->tupsize=0;  
      query->jbuf[i]->bufptr[0]=0;  
      query->jbuf[i]->bufptr[1]=0;
      query->jbuf[i]->actbuf=0;
      query->jbuf[i]->tnum=0;
      query->jbuf[i]->anum=0;
      query->jbuf[i]->atab=NULL;
      
      /* evaluate each prediacte */
      if (new_join(query->jbuf[i], query->selcondptr->wherecondptr) == -1)
	return(-1);

      /* join with all unused tables */
      if (comp_join(query->jbuf[i], query->selcondptr->fromlistptr, 1) == -1)
	return(-1);
      
      i++;  /* increase conjunction count */
   
    } while (next_conjunction(query->selcondptr->wherecondptr));
    query->cj=i;
  } else {
    /* allocate new join buffer */
    
    if ((query->jbuf[0] = (JoinBuf*)malloc(sizeof(JoinBuf))) == NULL) {
      perror("malloc");
      return(-1);
    }
    query->jbuf[0]->stored=0;
    query->jbuf[0]->tupsize=0;  
    query->jbuf[0]->bufptr[0]=0;  
    query->jbuf[0]->bufptr[1]=0;
    query->jbuf[0]->actbuf=0;
    query->jbuf[0]->tnum=0;
    query->jbuf[0]->anum=0;
    query->jbuf[0]->atab=NULL;
    
    if (comp_join(query->jbuf[0], query->selcondptr->fromlistptr,0) == -1)
      return(-1);
    query->cj = 1; /* case treated as one conjunction */
  }

  /* allocate buffer for subquery */
  if (actdepth != 0) 
    if((query->sqbuf = (SubQueryBuf*)malloc(sizeof(SubQueryBuf))) == NULL) {
      perror("malloc");
      return(-1);
    }

  /* treat ordering here */
  if (query->distinct || (query->selcondptr->ordercondptr != NULL)) {
    order_join(query);
  }

  /* switch to required selectmode */
  if (query->selcondptr->groupcondptr != NULL) {
    if (print_group(query) == -1) {
      fprintf(stderr,"PQL-ERROR: grouping failed\n");
      return(-1);
    }
  } else {
    /* just a projection or agregation */
    switch (query->selectmode) {
    case PROJECTION:
      /* select the specified attributes */
      print_projection(query);
      break;
    case STAR:
      /* select all available attributes */
      print_all(query);
      break;
    case AGREGATION:
      /* agregating attributes */
      print_agregate(query);
      break;
    }
  }

  /* free join buffers */
  for (i=0; i<query->cj; i++)
    free(query->jbuf[i]);

  /* reset attable in case of a subquery */
  if (actdepth != 0) {

    /* reset predicates */
    if (query->selcondptr->wherecondptr != NULL)
      reset_preds(query->selcondptr->wherecondptr);

    for (i=0; i< attnum; i++) {
      if (attable[i].depth == actdepth) {
	/* attribute is not available anymore */
	attable[i].attrspec->access = BASETABLE;
      }
    }
  }
}

/* reset routine for subqueries to initialize predicates */
static void reset_preds(Condition *cond)
{
  switch (cond->rule) {
  case COND_OR:
    reset_preds(cond->cond1ptr);
    reset_preds(cond->cond2ptr);
    break;
  case COND_PREDICATE:
    /* here comes the reset */
    cond->predicateptr->marked=UNMARKED;
    break;
  case COND_NOT:
    reset_preds(cond->cond1ptr);
    break;
  case COND_AND:
    reset_preds(cond->cond1ptr);
    reset_preds(cond->cond2ptr);
    break;
  }
} 

static void order_join(QueryOp *query)
{
  int i,j;
  int tuples;
  int offset;
  AttrSpecList *alist;

  /* first canonalize all jbufs */
  tuples=query->jbuf[0]->stored;
  for (i=1; i<query->cj; i++) {
    norm_jbuf(query->jbuf[0], query->jbuf[i]);
    tuples += query->jbuf[i]->stored;
  }
  
  /* set up attribute bases */
  if (query->selcondptr->ordercondptr != NULL)
    alist = query->selcondptr->ordercondptr->attrspeclistptr;
  else
    alist = NULL;

  /* setting up order attribute list */
  while (alist != NULL) {
    offset=0;
    for (i=0; i<query->jbuf[0]->anum; i++) {
      if (!strcmp(query->jbuf[0]->entry[i].attribute, 
		  alist->attrspecptr->attrname)
	  && !strcmp(query->jbuf[0]->tables[query->jbuf[0]->entry[i].tableindex], alist->attrspecptr->tablename)
	  && !strcmp(query->jbuf[0]->talias[query->jbuf[0]->entry[i].tableindex], alist->attrspecptr->talias)) {
	/* set up slot */
	alist->attrspecptr->base = offset;
      }
      offset += query->jbuf[0]->entry[i].size;
    }
    alist = alist->next;
  }
  /* setting up projection attribute list */
  if (query->distinct) {
    /* set up the selection list attributes */
    for (i=0; i< attnum; i++) {
      if (attable[i].depth == actdepth 
	  && attable[i].phase == SELECTION) {
	offset=0;
	for (j=0; j<query->jbuf[0]->anum; j++) {
	  if (!strcmp(query->jbuf[0]->entry[j].attribute, 
		      attable[i].attrspec->attrname)
	      && !strcmp(query->jbuf[0]->tables[query->jbuf[0]->entry[j].tableindex], attable[i].attrspec->tablename)
	      && !strcmp(query->jbuf[0]->talias[query->jbuf[0]->entry[j].tableindex], attable[i].attrspec->talias)) {
	    attable[i].attrspec->base = offset;
	  }
	  offset += query->jbuf[0]->entry[i].size;
	} 
      }
    }
  }

  /* now we can start ordering */
  /* using a simple bubble sort  */
  for (i=1; i<tuples; i++) {
    for (j=0; j<tuples-i; j++) {
      if (tuple_compare(query, j, j+1) == 0)
	tuple_swop(query, j, j+1);
    }
  }
}

static int tuple_compare(QueryOp *query, int t1, int t2)
{
  int tuples;
  int jb1, jb2;
  int offset1, offset2;
  AttrSpecList *alist;
  int order;
  long l1, l2;
  float f1, f2;
  double d1, d2;
  int scmp;
  int i=0;
  int tslot;

  /* get the jbuf for t1 and t2 */

  jb1=0,jb2=0;
  offset1=t1, offset2=t2;

  if (query->selcondptr->ordercondptr != NULL)  {
    alist=query->selcondptr->ordercondptr->attrspeclistptr;
    order=query->selcondptr->ordercondptr->order;
  } else {
    alist=NULL;
    order=ASC;
  }

  tuples=0;
  while (t1 >= query->jbuf[i]->stored + tuples
	 || t2 >= query->jbuf[i]->stored + tuples) {
    tuples += query->jbuf[i]->stored;
    if (t1 >= tuples) {
      jb1++;
      offset1 = t1 - tuples;
    }
    if (t2 >= tuples) {
      jb2++;
      offset2 = t2 - tuples;
    }
    if (i<query->cj)
      i++;
    else /* error */
      return(-1);
  }

  while(alist != NULL) { 
    switch(alist->attrspecptr->type) {
    case T_CHAR:
      scmp=strcmp(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		  + query->jbuf[jb1]->tupsize * offset1
		  + alist->attrspecptr->base, 
		  query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		  + query->jbuf[jb2]->tupsize * offset2
		  + alist->attrspecptr->base);
      
      if (scmp == 0)
	break;
      if (scmp < 0 && order == ASC
	  || scmp > 0 && order == DESC)
	return(1);
      return(0);
    case T_LONG:
      bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	    + query->jbuf[jb1]->tupsize * offset1
	    + alist->attrspecptr->base, &l1, sizeof(long));
      bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
	    + query->jbuf[jb2]->tupsize * offset2
	    + alist->attrspecptr->base, &l2, sizeof(long));
      
      if (l1 == l2)
	break;
      if (l1 < l2 && order == ASC
	  || l1 > l2 && order == DESC)
	return(1);
      return(0);
    case T_FLOAT:
      bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	    + query->jbuf[jb1]->tupsize * offset1
	    + alist->attrspecptr->base, &f1, sizeof(float));
      bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
	    + query->jbuf[jb2]->tupsize * offset2
	    + alist->attrspecptr->base, &f2, sizeof(float));
      if (f1 == f2) 
	break;
      if (f1 < f2 && order == ASC
	  || f1 > f2 && order == DESC)
	return(1);
      return(0);
    case T_DOUBLE:
      bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	    + query->jbuf[jb1]->tupsize * offset1
	    + alist->attrspecptr->base, &d1, sizeof(float));
      bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
	  + query->jbuf[jb2]->tupsize * offset2
	    + alist->attrspecptr->base, &d2, sizeof(float));
      if (d1 == d2) 
	break;
      if (d1 < d2 && order == ASC
	  || d1 > d2 && order == DESC)
	return(1);
      return(0);
    }
    alist = alist->next;
  }
  /* treat further sort conditions */
  if (query->distinct) {
    if (query->selectmode == PROJECTION) {
      for (i=0; i<attnum; i++) {
	if (attable[i].depth == actdepth 
	    && attable[i].phase == SELECTION) {
	  tslot = attable[i].attrspec->base;
	  switch(attable[i].attrspec->type) {
	  case T_CHAR:
	    scmp=strcmp(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
			+ query->jbuf[jb1]->tupsize * offset1
			+ tslot, 
			query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
			+ query->jbuf[jb2]->tupsize * offset2
			+ tslot);
	    
	    if (scmp == 0)
	      break;
	    if (scmp < 0 && order == ASC
		|| scmp > 0 && order == DESC)
	      return(1);
	    return(0);
	  case T_LONG:
	    bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		  + query->jbuf[jb1]->tupsize * offset1
		  + tslot, &l1, sizeof(long));
	    bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		  + query->jbuf[jb2]->tupsize * offset2
		  + tslot, &l2, sizeof(long));
	    
	    if (l1 == l2)
	      break;
	    if (l1 < l2 && order == ASC
		|| l1 > l2 && order == DESC)
	      return(1);
	    return(0);
	  case T_FLOAT:
	    bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		  + query->jbuf[jb1]->tupsize * offset1
		  + tslot, &f1, sizeof(float));
	    bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		  + query->jbuf[jb2]->tupsize * offset2
		  + tslot, &f2, sizeof(float));
	    if (f1 == f2) 
	      break;
	    if (f1 < f2 && order == ASC
		|| f1 > f2 && order == DESC)
	      return(1);
	    return(0);
	  case T_DOUBLE:
	    bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		  + query->jbuf[jb1]->tupsize * offset1
		  + tslot, &d1, sizeof(float));
	    bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		  + query->jbuf[jb2]->tupsize * offset2
		  + tslot, &d2, sizeof(float));
	    if (d1 == d2) 
	      break;
	    if (d1 < d2 && order == ASC
		|| d1 > d2 && order == DESC)
	      return(1);
	    return(0);
	  }
	}
      }
    } else if (query->selectmode == STAR) { 
      tslot=0;
      for (i=0; i< query->jbuf[jb1]->anum; i++) {
	switch(query->jbuf[jb1]->entry[i].type) {
	case T_CHAR:
	  scmp=strcmp(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		      + query->jbuf[jb1]->tupsize * offset1
		      + tslot, 
		      query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		      + query->jbuf[jb2]->tupsize * offset2
		      + tslot);
	  
	  if (scmp == 0)
	    break;
	  if (scmp < 0 && order == ASC
	      || scmp > 0 && order == DESC)
	    return(1);
	  return(0);
	case T_LONG:
	bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	      + query->jbuf[jb1]->tupsize * offset1
	      + tslot, &l1, sizeof(long));
	  bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		+ query->jbuf[jb2]->tupsize * offset2
		+ tslot, &l2, sizeof(long));
	  
	  if (l1 == l2)
	    break;
	  if (l1 < l2 && order == ASC
	      || l1 > l2 && order == DESC)
	    return(1);
	  return(0);
	case T_FLOAT:
	  bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		+ query->jbuf[jb1]->tupsize * offset1
		+ tslot, &f1, sizeof(float));
	  bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		+ query->jbuf[jb2]->tupsize * offset2
		+ tslot, &f2, sizeof(float));
	  if (f1 == f2) 
	    break;
	  if (f1 < f2 && order == ASC
	      || f1 > f2 && order == DESC)
	    return(1);
	  return(0);
	case T_DOUBLE:
	  bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
		+ query->jbuf[jb1]->tupsize * offset1
		+ tslot, &d1, sizeof(float));
	  bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
		+ query->jbuf[jb2]->tupsize * offset2
		+ tslot, &d2, sizeof(float));
	  if (d1 == d2) 
	    break;
	  if (d1 < d2 && order == ASC
	      || d1 > d2 && order == DESC)
	    return(1);
	  return(0);
	}
	tslot += query->jbuf[jb1]->entry[i].size;
      }
    }
  }
  return(0);
}

static int tuple_swop(QueryOp *query, int t1, int t2)
{
  int tuples;
  int jb1, jb2;
  int offset1, offset2;
  AttrSpecList *alist;
  char tbuf[MAXTUPLEN];

  int i=0;
  /* get the jbuf for t1 and t2 */

  jb1=0,jb2=0;
  offset1=t1, offset2=t2;

  tuples=0;
  while (t1 >= query->jbuf[i]->stored + tuples
	 || t2 >= query->jbuf[i]->stored + tuples) {
    tuples += query->jbuf[i]->stored;
    if (t1 >= tuples) {
      jb1++;
      offset1 = t1 - tuples;
    }
    if (t2 >= tuples) {
      jb2++;
      offset2 = t2 - tuples;
    }
    if (i<query->cj)
      i++;
    else /* error */
      return(-1);
  }

  /* swop tuple 1 and 2 */
  bcopy(query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	+ query->jbuf[jb1]->tupsize * offset1, 
	tbuf, 
	query->jbuf[jb1]->tupsize);
  
  bcopy(query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
	+ query->jbuf[jb2]->tupsize * offset2, 
	query->jbuf[jb1]->jbuf[query->jbuf[jb1]->actbuf]
	+ query->jbuf[jb1]->tupsize * offset1, 
	query->jbuf[jb1]->tupsize);
  bcopy(tbuf, 
	query->jbuf[jb2]->jbuf[query->jbuf[jb2]->actbuf]
	+ query->jbuf[jb2]->tupsize * offset2, 
	query->jbuf[jb1]->tupsize);
}

static void norm_jbuf(JoinBuf *njbuf, JoinBuf *jbuf)
{

  int i,j,k;
  int offset1, offset2;

  for (i=0; i<jbuf->stored; i++) {
    offset1=0;
    for (j=0; j<njbuf->anum; j++) {
      offset2=0;
      for(k=0; k<jbuf->anum; k++) {
	if (!strcmp(jbuf->entry[k].attribute, njbuf->entry[j].attribute)
	    && !strcmp(jbuf->tables[jbuf->entry[k].tableindex],
		       njbuf->tables[njbuf->entry[j].tableindex])
	    && !strcmp(jbuf->talias[jbuf->entry[k].tableindex],
		       njbuf->talias[njbuf->entry[j].tableindex])) {
	  bcopy(jbuf->jbuf[jbuf->actbuf] + i*jbuf->tupsize + offset2,
		jbuf->jbuf[(jbuf->actbuf+1)%2] + i*jbuf->tupsize + offset1,
		jbuf->entry[k].size);
	}
	offset2 += jbuf->entry[k].size;
      }
      offset1 += njbuf->entry[j].size;
    }
  }
  jbuf->actbuf = (jbuf->actbuf+1) % 2;
  bcopy(njbuf->entry, jbuf->entry, MAXJOINATTS * sizeof(JFormat));
  bcopy(njbuf->tables, jbuf->tables, MAXTABLES * MAXTNLEN);
  bcopy(njbuf->talias, jbuf->talias, MAXTABLES * MAXTNLEN);    
}



