/*----------------------------------------------------------------------*
** File: pql_clean.c
** 
** 
** Project:	PQL
** =====================================================================*
** Description:	PQL (Subset II) - pql cleanup operation
**
**  Author:	Bjoern Lemke
**  E-Mail:	lemke@lf.net
**
** Copyright (c) 1994 Bjoern Lemke.
**----------------------------------------------------------------------*/

/* include this first, FD_SETSIZE must be defined */
#include "eng.h"

#include <stdio.h>
#include "pql_tree.h"

/*********************/
/* public prototypes */
/*********************/
void clean_query(QueryOp *query);
void clean_selection(Selection *selection);
void clean_selcond(SelCond *selcond);
void clean_fromlist(FromList *fromlist);
void clean_attrspeclist(AttrSpecList *attrspeclist);
void clean_groupcond(GroupCond *groupcond);
void clean_ordercond(OrderCond *ordercond);
void clean_condition(Condition *condition);
void clean_predicate(Predicate *predicate);
void clean_exprlist(ExprList *exprlist);
void clean_expr(Expr *expr);
void clean_function(Function *function);
void clean_funcarg(FuncArg *funcarg);
void clean_assignment(Assignment *assignment);
void clean_insert(InsertOp *insert);
void clean_update(UpdateOp *update);
void clean_delete(DeleteOp *delete);
void clean_create(CreateOp *create);
void clean_extend(ExtendOp *extend);
void clean_shrink(ShrinkOp *shrink);
void clean_shrinklist(ShrinkList *shrinklist);
void clean_extlist(ExtList *extlist);
void clean_setlist(SetList *setlist);
void clean_insertlist(InsertList *insertlist);
void clean_constantlist(ConstantList *constantlistptr);
void clean_sqbuf(SubQueryBuf *sqbuf);

/*******************/
/* public routines */
/*******************/
void clean_query(QueryOp *query)
{
  int i;
  if (query != NULL) {
    clean_selection(query->selectionptr);
    clean_selcond(query->selcondptr);
    clean_sqbuf(query->sqbuf);
    free(query);
  }
}

void clean_insert(InsertOp *insert)
{
  if (insert != NULL) {
    clean_insertlist(insert->insertlistptr);
    free(insert);
  }
}

void clean_update(UpdateOp *update)
{
  if (update != NULL) {
    clean_setlist(update->setlistptr);
    clean_condition(update->conditionptr);
    free(update);
  }
}

void clean_delete(DeleteOp *delete)
{
  if (delete != NULL) {
    clean_condition(delete->conditionptr);
    free(delete);
  }
}

void clean_create(CreateOp *create)
{
  if (create != NULL) {
    clean_extlist(create->extlistptr);
    free(create);
  }
}

void clean_extend(ExtendOp *extend)
{
  if (extend != NULL) {
    clean_extlist(extend->extlistptr);
    free(extend);
  }
}

void clean_extlist(ExtList *extlist)
{
  if (extlist != NULL) {
    clean_extlist(extlist->next);
    free(extlist->aspecptr);
    free(extlist);
  }
}

void clean_shrink(ShrinkOp *shrink)
{
  if (shrink != NULL) {
    clean_shrinklist(shrink->shrinklistptr);
    free(shrink);
  }
}

void clean_shrinklist(ShrinkList *shrinklist)
{
  if (shrinklist != NULL) {
    clean_shrinklist(shrinklist->next);
    free(shrinklist);
  }
}

void clean_setlist(SetList *setlist)
{
  if (setlist != NULL) {
    clean_setlist(setlist->next);
    clean_assignment(setlist->assignmentptr);
    free(setlist);
  }
}

void clean_assignment(Assignment *assignment)
{
  if (assignment != NULL) {
    free(assignment->attrspecptr);
    free(assignment->literalptr);
    free(assignment);
  }
}

void clean_insertlist(InsertList *insertlist)
{
  if (insertlist != NULL) {
    clean_query(insertlist->queryopptr);
    clean_constantlist(insertlist->constantlistptr);
    free(insertlist);
  }
}

void clean_constantlist(ConstantList *constantlist)
{
  if (constantlist != NULL) {
    clean_constantlist(constantlist->next);
    free(constantlist->literalptr);
    free(constantlist);
  }
}

void clean_selection(Selection *selection)
{
  if (selection != NULL) {
    clean_exprlist(selection->exprlistptr);
    free(selection);
  }
}

void clean_selcond(SelCond *selcond)
{
  if(selcond != NULL) {
    clean_fromlist(selcond->fromlistptr);
    clean_condition(selcond->wherecondptr);
    clean_groupcond(selcond->groupcondptr);
    clean_ordercond(selcond->ordercondptr);
    free(selcond);
  }
}
  
void clean_fromlist(FromList *fromlist)
{
  if (fromlist != NULL) {
    free(fromlist->tablespecptr);
    clean_fromlist(fromlist->next);
    free(fromlist);
  }
}

void clean_attrspeclist(AttrSpecList *attrspeclist)
{
  if (attrspeclist != NULL) {
    printf("cleaning attribute %s\n", attrspeclist->attrspecptr->attrname);
    free(attrspeclist->attrspecptr);
    clean_attrspeclist(attrspeclist->next);
    free(attrspeclist);
  }
}

void clean_groupcond(GroupCond *groupcond)
{
  if (groupcond != NULL) {
    free(groupcond->attrspecptr);
    clean_condition(groupcond->havingcondptr);
    free(groupcond);
  }
}

void clean_ordercond(OrderCond *ordercond)
{
  if (ordercond != NULL) {
    clean_attrspeclist(ordercond->attrspeclistptr);
    free(ordercond);
  }
}

void clean_condition(Condition *condition)
{
  if (condition != NULL) {
    clean_predicate(condition->predicateptr);
    clean_condition(condition->cond1ptr);
    clean_condition(condition->cond2ptr);
    free(condition);
  }
}

void clean_predicate(Predicate *predicate)
{
  if (predicate != NULL) {
    switch (predicate->rule) {
    case PRED_EXPR:
      clean_expr(predicate->expr1ptr);
      clean_expr(predicate->expr2ptr);
      break;
    case PRED_QUERY:
    case PRED_ALL_QUERY:
    case PRED_ANY_QUERY:
    case PRED_IN_QUERY:
    case PRED_N_IN_QUERY:
      clean_expr(predicate->expr1ptr);
      clean_query(predicate->queryopptr);
      break;
    case PRED_IN_ELIST:
    case PRED_N_IN_ELIST:
      clean_expr(predicate->expr1ptr);
      clean_exprlist(predicate->exprlistptr);
      break;
    case PRED_LIKE:
    case PRED_N_LIKE:
      free(predicate->literalptr);
      break;
    }
    free(predicate);
  }
}
    
void clean_exprlist(ExprList *exprlist)
{
  if (exprlist != NULL) {
    clean_expr(exprlist->exprptr);
    clean_exprlist(exprlist->next);
    free(exprlist);
  }
}

void clean_expr(Expr *expr)
{
  if (expr != NULL) {
    switch(expr->rule) {
    case EXPR_LIT:
      free(expr->literalptr);
      break;
    case EXPR_ATTSPEC:
      free(expr->attrspecptr);
      break;
    case EXPR_PLUS:
    case EXPR_MINUS:
    case EXPR_MUL:
    case EXPR_DIV:
      clean_expr(expr->expr1ptr);
      clean_expr(expr->expr2ptr);
      break;
    case EXPR_FUNC:
      clean_function(expr->functionptr);
      break;
    }
    free(expr);
  }
}

void clean_function(Function *function)
{
  if (function != NULL) {
    clean_expr(function->exprptr);
    free(function);
  }
}

void clean_sqbuf(SubQueryBuf *sqbuf)
{
  if (sqbuf != NULL) {
    free (sqbuf);
  }
}
