/* -*- Mode: C -*- */
/* Terminal.h - Video terminal interface
 *		Inspired by the Terminal class described in "C++,
 *		a guide for C programmers", by Sharam Hekmatpour,
 *		(c) 1990 by Prentice Hall of  Australia Pty Ltd.
 *		This version uses termcap and is not hardwired
 *		for a particular terminal type.  Also, I only
 *		implemented a bare terminal class and did not
 *		include the higher level classes (windows,
 *		menus, and forms).
 * Created by Robert Heller on Fri Dec  6 21:10:04 1991
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Header Files
 * ------------------------------------------------------------------
 * Modification History:
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 * 
 * Copyright (c) 1991,1992 by Robert heller (D/B/A Deepwoods Software)
 *        All Rights Reserved
 * 
 */
#ifndef _TERMINAL_
#define _TERMINAL_
#include <stream.h>
#include <common.h>
#include <ctype.h>
#ifdef MESSYDOS
#include <gppconio.h>
#include <keybd.h>
#else
#include <termcap.h>
#endif

#ifndef MESSYDOS
#define PosCode(buf,r,c) CopyCode(buf,tgoto(CM,c,r))
#endif
class CommandScreen;
class EditForm;
class ScrollPrompt;
class LLScrollPrompt;
// Terminal class.  Based on termcap.
class Terminal	{
// Terminal mode magic
#ifdef OSK
	static sgbuf ttym;			// Vanila terminal modes (OSK)
	static sgbuf xttym;			// Operating terminal modes (OSK)
#endif
#ifdef unix
	static struct termios ttym;		// Vanila terminal modes (unix)
	static struct termios xttym;		// Operating terminal modes (unix)
#endif
	static ErrFun errFun;			// user error function
	const  bufSize = 512;			// buffer size
	static char termBuf[bufSize];		// scratch buffer
#ifndef MESSYDOS
	const TCapsLen = 1024;			// Termcap buffer size
	static char TermType[48];		// terminal type name
	static char TCapBuf[TCapsLen];		// termcap buffer
	static char tcapbuf[TCapsLen];		// copy
	const char* const BELL = "\007";	// bell character
#endif
	static Terminal* term;			// self reference
#ifndef MESSYDOS
	// termcap capabilities
	static char* BC;			// backspace character
	static char* UP;			// up line sequence
	static char* CL;			// clear screen
	static char* CM;			// cursor movement
	static char* CE;			// clear to eol
	static char* CD;			// clear to eos
	static char* SO;			// standout start
	static char* SE;			// standout end
	static char* HO;			// home cursor
	static char* KD;			// down arrow key
	static char* KU;			// up arrow key
	static char* KR;			// right arrow key
	static char* KL;			// left arrow key
	static char* KH;			// home key
	static char* KB;			// backspace key
	static char* TI;			// terminal init
	static char* TE;			// terminal de-init
	static short ospeed;			// baud rate (used for padding)
	static char PC;				// pad character
	// copy code to a buffer
	char*  CopyCode	(char* buf,const char* code)
			{strcpy(buf,code); return buf + strlen(code);}
	// write control codes
	void WriteCode(const char*);
	void WriteCodeLines(const char*,int);
#endif
	static short cline;			// current line
	static short ccol;			// current column
	static short lines;			// screen lines
	static short colms;			// screen columns
public:
	// do various things to the terminal
#ifdef MESSYDOS
	inline void RevsPen	()	{textbackground(LIGHTGRAY);
					 textcolor(BLACK);}
	inline void PlainPen	()	{textbackground(BLACK);
					 textcolor(WHITE);}
	inline void InitChars	() 	{};
	inline void ExitChars	()	{};
	inline void Home	()	{PenPos(0,0);}
	inline void Clear	()	{clrscr();gotoxy(ccol+1,cline+1);}
	inline void ClearEOL	()	{clreol();gotoxy(ccol+1,cline+1);}
	inline void ClearEOS	()	{clreol();
					 for (int r=cline+1;r<lines;r++)
					 {
					 	gotoxy(1,r+1);
					 	clreol();
					 }
					 gotoxy(ccol+1,cline+1);}
	void Bell	();

	Terminal &put(unsigned char ch);
	inline Terminal &put(unsigned char *s) {
		while (*s) {put(*s);s++;}
		return(*this);
	}
	inline Terminal &put(unsigned char *s,int n)
			{for (int i=0;i<n;i++) put(*s++);return(*this);}
#else
	void RevsPen	()	{WriteCode(SO);}
	void PlainPen	()	{WriteCode(SE);}
	void InitChars	()	{WriteCode(TI);}
	void ExitChars	()	{WriteCode(TE);}
	void Home	()	{WriteCode(HO); cline = 0; ccol = 0;}
	void Clear	()	{if (CL == 0) {
					WriteCode(HO);WriteCodeLines(CD,lines);
				 } else WriteCodeLines(CL,lines);
				 char buf[50];
				 PosCode(buf,cline,ccol);
				 WriteCode(buf);}
	void ClearEOL	()	{WriteCode(CE);}
	void CLearEOS	()	{WriteCodeLines(CD,lines-cline);}
	void Bell	()	{WriteCode(BELL);}
#endif
	// error handler reference
	inline ErrFun& ErrorFun ()	{return errFun;}
	// Constructor and descructor
	       Terminal ();
	      ~Terminal ();
	// set cursor position
#ifdef MESSYDOS
	inline void PenPos	 (int row,int col)
		{gotoxy(col+1,row+1);cline = row;ccol = col;}
#else
	void PenPos	(int row,int col);
#endif
	// get a key from the keyboard (no echo, no wait for eol, no editing)
	int  GetKey	();
	// key available??
	Boolean KeyAvailable ();
	// error handler
	void Error	(ErrKind err,const char* msg);
	// display a message
	void Message	(const char* msg);
	// get a text string from the keyboard.  Allows editing
	int  GetLine	(char* buffer,int bufsize,
			 const char* terminators = "\n\r");
	// Put a character on the screen someplace
	void PutCharAt	(int row,int col,int ch);
	// Put a string someplace on the screen
	void PutStrAt	(int row,int col,const char* str);
	void PutStrAt	(int row,int col,const short* str);
	char* PutStrInBox (int row,int col,int width,int height,
			   const char* str);
	// get a text string from the keyboard.  Allows editing
	inline int  PromptLine	(int row,int col,const char* prompt,
			 char* buffer,int bufsize,
			 const char* terminators = "\n\r")
		{PutStrAt(row,col,prompt);return(GetLine(buffer,bufsize,
							 terminators));}
	inline Boolean YorNp(int row,int col,const char* prompt)
		{int key;
		 PutStrAt(row,col,prompt);
		 while (true) {
		 	key = GetKey();
		 	if (key == 'y' || key == 'Y') {
		 		PutStrAt(cline,ccol,"Yes");
		 		return(true);
		 	} else if (key == 'n' || key == 'N') {
		 		PutStrAt(cline,ccol,"No");
		 		return(false);
		 	} else Bell();
		 }
		}
	// fork a sub-process (this is here to handle the terminal modes
	int  forkprog	(const char** argv);
	// handle an interrupt
	friend int Interrupt();
	friend class CommandScreen;
	friend class EditForm;
	friend class ScrollPrompt;
	friend class LLScrollPrompt;
};

#endif  _TERMINAL_

