/*+-------------------------------------------------------------------------
	ecuwinutil.c - curses window utilities
	wht@n4hgf.Mt-Park.GA.US

  Defined functions:
	clear_area(win,y,x,len)
	clear_area_char(win,y,x,len,fillchar)
	winbox(win)
	window_create(title,title_x,tly,tlx,lines,cols)
	window_setup(win,title,title_x)
	windows_end(botleft_flag)
	windows_end_signal()
	windows_start()
	winget_single(win,nondelim_list,delim_list)
	wingets(win,y,x,buf,bufsize,delim,edit,pwgpos)

--------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:09-10-1992-13:59-wht@n4hgf-ECU release 3.20 */
/*:08-22-1992-15:38-wht@n4hgf-ECU release 3.20 BETA */
/*:02-09-1992-16:08-root@n4hgf-ruling characters only on  SCO (tcap curses) */
/*:08-25-1991-14:39-wht@n4hgf-SVR4 port thanks to aega84!lh */
/*:08-01-1991-03:52-wht@n4hgf-when editing string, set cursor to end */
/*:07-25-1991-12:57-wht@n4hgf-ECU release 3.10 */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#include "ecucurses.h"
#include <errno.h>
#include "ecukey.h"
#include "ecuxkey.h"
#include "termecu.h"
#include "pc_scr.h"
#ifdef USE_PROTOS
# include "protos.h"
#endif

#if !defined(ushort)
#define ushort unsigned short
#endif
#if !defined(uchar)
#define uchar unsigned char
#endif
#if !defined(uint)
#define uint unsigned int
#endif
#if !defined(ulong)
#define ulong unsigned long
#endif

extern int tty_is_multiscreen;
extern int tcap_LINES;

#ifdef M_SYSV
unsigned char sTL = at_TL;
unsigned char sTR = at_TR;
unsigned char sBL = at_BL;
unsigned char sBR = at_BR;
unsigned char sLT = at_LT;
unsigned char sRT = at_RT;
unsigned char sVR = at_VR;
unsigned char sHR = at_HR;
#else
unsigned char sTL = vanilla_TL;
unsigned char sTR = vanilla_TR;
unsigned char sBL = vanilla_BL;
unsigned char sBR = vanilla_BR;
unsigned char sLT = vanilla_LT;
unsigned char sRT = vanilla_RT;
unsigned char sVR = vanilla_VR;
unsigned char sHR = vanilla_HR;
#endif

int windows_active = 0;

int ttymode_before_window_start;

/*+-------------------------------------------------------------------------
	clear_area_char(win,y,x,len,fillchar)
--------------------------------------------------------------------------*/
void
clear_area_char(win,y,x,len,fillchar)
WINDOW *win;
int y;
int x;
int len;
char fillchar;
{
	wmove(win,y,x);
	while(len-- > 0)
		waddch(win,fillchar & 0xFF);
	wmove(win,y,x);

}	/* end of clear_area_char */

/*+-------------------------------------------------------------------------
	clear_area(win,y,x,len)
--------------------------------------------------------------------------*/
void
clear_area(win,y,x,len)
WINDOW *win;
int y;
int x;
int len;
{
	clear_area_char(win,y,x,len,' ');
}	/* end of clear_area_char */

/*+-------------------------------------------------------------------------
	windows_start()
--------------------------------------------------------------------------*/
void
windows_start()
{
	extern int tty_not_char_special;
	static int initscr_already_performed = 0;

	if(tty_not_char_special)
	{
		errno = ENOTTY;
		fprintf(stderr,"curses features unavailable when stdin not tty\r\n");
		termecu(TERMECU_CURSES_ERROR);
	}

	ttymode_before_window_start = get_ttymode();
	ttymode(0);
	if(!initscr_already_performed && !initscr())
	{
		fprintf(stderr,"curses init failure ... check terminal type\r\n");
		termecu(TERMECU_CURSES_ERROR);
	}
	initscr_already_performed = 1;
	scrollok(stdscr,0);
	savetty(); 
	raw(); 
	noecho(); 
	nonl(); 
	clear();
#if defined(M_TERMINFO)
	typeahead(-1);
#endif
	windows_active = 1;

#if defined(M_SYSV)
	if(!tty_is_multiscreen)
	{
		sTL = vanilla_TL;
		sTR = vanilla_TR;
		sBL = vanilla_BL;
		sBR = vanilla_BR;
		sLT = vanilla_LT;
		sRT = vanilla_RT;
		sVR = vanilla_VR;
		sHR = vanilla_HR;
	}
#endif

	wclear(stdscr);
	touchwin(stdscr);
	wrefresh(stdscr);

}	/* end of windows_start */

/*+-------------------------------------------------------------------------
	windows_end(botleft_flag)
--------------------------------------------------------------------------*/
void
windows_end(botleft_flag)
int botleft_flag;
{
	if(!windows_active)
		return;
/*
	endwin();
*/
	if(botleft_flag)
		tcap_cursor(tcap_LINES - 1,0);
	ttymode(ttymode_before_window_start);
	windows_active = 0;
}	/* end of windows_end */

/*+-------------------------------------------------------------------------
	windows_end_signal() -- called by termecu()
--------------------------------------------------------------------------*/
void
windows_end_signal()
{
	windows_end(0);
}	/* end of windows_end_signal */

/*+-------------------------------------------------------------------------
	winbox(win)
--------------------------------------------------------------------------*/
void
winbox(win)
WINDOW *win;
{

#if defined(SVR4)
	box(win,(unsigned long)sVR,(unsigned long)sHR);
#else
	box(win,sVR,sHR);
	wmove(win,0,0); waddch(win,(unsigned)sTL);
	wmove(win,win->_maxy - 1,0); waddch(win,(unsigned)sBL);
	wmove(win,win->_maxy - 1,win->_maxx - 1); waddch(win,(unsigned)sBR);
	wmove(win,0,win->_maxx - 1); waddch(win,(unsigned)sTR);
#endif

}	/* end of winbox */

/*+-------------------------------------------------------------------------
	window_setup(win,title,title_x)
--------------------------------------------------------------------------*/
void
window_setup(win,title,title_x)
WINDOW *win;
char *title;
int title_x;
{
	register stand = (title_x < 0);

	if(stand)
		title_x = -title_x;

	touchwin(win);
	scrollok(win,0);		/* do not scroll */
	winbox(win);
	wmove(win,0,title_x);
	if(stand)
		wstandout(win);
	waddch(win,'[');
	wprintw(win," %s ",title);
	waddch(win,']');
	if(stand)
		wstandend(win);
}	/* end of window_setup */

/*+-------------------------------------------------------------------------
	window_create(title,title_x,tly,tlx,lines,cols)
if title_x negative, make title "stand" out
--------------------------------------------------------------------------*/
WINDOW *
window_create(title,title_x,tly,tlx,lines,cols)
char *title;
int title_x;
int tly;
int tlx;
int lines;
int cols;
{
	register WINDOW *nwin = newwin(lines,cols,tly,tlx);
	
	if(nwin)
		window_setup(nwin,title,title_x);
	else
	{
		fprintf(stderr,"\r\ncurses error: cannot create new window\r\n");
		termecu(TERMECU_CURSES_ERROR);
	}
	return(nwin);
}	/* end of window_create */

/*+-------------------------------------------------------------------------
	wingets(win,y,x,buf,bufsize,delim,edit,pwgpos)

This procedure reads a string from win and returns the number
of characters read.

If edit is non-zero and pwgpos is not null, the inital string
position is set by dereferencing the pointer.

The terminating delim is returned in 'delim'.

If pwgpos is not null, the ending string position is returned in
the integer pointed to.

-1 is returned if an ESCape is typed by the keyboard user,
otherwise the count of characters in the string.

The entire line must be contained on one line (no line wrap supported).
--------------------------------------------------------------------------*/
int
wingets(win,y,x,buf,bufsize,delim,edit,pwgpos)
WINDOW *win;
int y;
register x;
register char *buf;
int bufsize;	/* includes room for null..field is 1 less */
register uchar *delim;
int edit;
int *pwgpos;
{
	register count = 0;
	register pos = 0;
	int insert_mode = 0;
	int rtn_val = 0;

	bufsize--;
	clear_area_char(win,y,x,bufsize,'_');
	if(edit)
	{
		waddstr(win,buf);
		count = pos = strlen(buf);
		if(pwgpos)
		{
			pos = *pwgpos;
			if((pos < 0) || (pos > count))
				pos = count;
		}
	}
	else
		*buf = 0;
	wmove(win,y,x + pos);

	while(1)
	{
		wrefresh(win);
		*delim = ttygetc(1);
		if((*delim < 0x20) || (*delim >= 0x7F))
		{
			switch(*delim)
			{
				case CRET:
					*delim = NL;
				case NL:
					wrefresh(win);
					rtn_val = count;
					goto RETURN;

				case BS:
				case DEL:
					if(count)
					{
						if(count == pos)
						{
							*(buf + --count) = 0;
							wmove(win,y,x + count);
							waddch(win,'_');
							wmove(win,y,x + count);
							pos--;
						}
						else
						{
							if(!pos)
								continue;
							mem_cpy(buf + pos - 1,buf + pos,count - pos);
							*(buf + --count) = 0;
							wmove(win,y,x + --pos);
							waddstr(win,buf + pos);
							waddch(win,'_');
							wmove(win,y,x + pos);
						}
					}
					continue;

				case XFcurlf:
					if(pos)
						wmove(win,y,x + --pos);
					continue;

				case XFcurrt:
					if(pos < count)
						wmove(win,y,x + ++pos);
					continue;

				case XFins:
					insert_mode = !insert_mode;
					continue;

				case ESC:
					rtn_val = -1;
					goto RETURN;

				case CTL_U:
					clear_area_char(win,y,x,bufsize,'_');
					count = 0;
					pos = 0;
					*buf = 0;
					continue;

				default:
					*(buf + count) = 0;
					rtn_val = count;
					goto RETURN;

			}	/* end of switch(*delim) */
			/*NOTREACHED*/
		}		/* end of if read delimiter */

		if(count == bufsize)
		{
			ring_bell();
			continue;
		}

		if(insert_mode && (pos != count))
		{
			waddch(win,*delim);
			waddstr(win,buf + pos);
			mem_cpy(buf + pos + 1,buf + pos,count - pos);
			*(buf + pos++) = *delim;
			*(buf + ++count) = 0;
			wmove(win,y,x + pos);
		}
		else
		{
			waddch(win,*delim);
			*(buf + pos) = *delim;
			if(pos == count)
				*(buf + ++count) = 0;
			pos++;
		}
	}	/* end of while can get character */

RETURN:
	if(pwgpos)
		*pwgpos = pos;
	return(rtn_val);

}	/* end of wingets */

/*+-------------------------------------------------------------------------
	winget_single(win,nondelim_list,delim_list)

This procedure assumes cursor is positioned, repeats reading a non-echoing
character from the keyboard until it matches a character in nondelim_list
or delim_list.  delim_list is expected to contain printable characters
and no upper-case characters.

If no match occurs, the bell is rung and the keyboard is read again.

If the input character matches a character in delim_list, the index (0-n)
of the character in delim_list is returned.  If a match occurs, an
upper-case version of the matching character is placed in the window.

If the input character matches a character in nondelim_list, the character
is returned or'ed with 0x1000

--------------------------------------------------------------------------*/
int
winget_single(win,nondelim_list,delim_list)
WINDOW *win;
register char *nondelim_list;
register char *delim_list;
{
	register itmp;
	register nlen = strlen(nondelim_list);
	register dlen = strlen(delim_list);
	register ichar;

	wrefresh(win);

	while(1)
	{
		ichar = to_lower(ttygetc(1));
		for(itmp = 0; itmp < nlen; itmp++)
		{
			if(ichar == nondelim_list[itmp])
			{
				waddch(win,to_upper(ichar));
				wrefresh(win);
				return(itmp);
			}
		}
		for(itmp = 0; itmp < dlen; itmp++)
		{
			if(ichar == delim_list[itmp])
				return(ichar | 0x1000);
		}
		ring_bell();
	}

}	/* end of winget_single */

/* end of ecuwinutil.c */
/* vi: set tabstop=4 shiftwidth=4: */
