/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header$";

/* special polygon editing routines
 */

#include <math.h>
#include <memory.h>

#include "global.h"

#include "create.h"
#include "crosshair.h"
#include "data.h"
#include "draw.h"
#include "error.h"
#include "misc.h"
#include "move.h"
#include "polygon.h"
#include "remove.h"
#include "search.h"
#include "set.h"
#include "undo.h"

/* ---------------------------------------------------------------------------
 * insert a new point into an existing polygon
 * it is inserted at the crosshair position into the segment with the
 * lowest distance
 */
void InsertPolygonPointAtCrosshairPosition(void)
{
	PolygonTypePtr	polygon;
	LayerTypePtr	layer;

		/* lookup polygon */
	if (SearchObjectByPosition(POLYGON_TYPE,
		(void **) &layer, (void **) &polygon,
		Crosshair.X, Crosshair.Y) == POLYGON_TYPE)
	{
		float				mindistance = MAX_COORD;
		PolygonPointTypePtr	ptr1 = &polygon->Points[polygon->PointN-1],
							ptr2 = &polygon->Points[0];
		Cardinal			n,
							index = 0;

			/* get segment next to crosshair location;
			 * it is determined by the last point of it
			 */
		for (n = 0; n < polygon->PointN; n++, ptr2++)
		{
			float	length,
					distance;

			length = sqrt((ptr2->X -ptr1->X)*(ptr2->X -ptr1->X) +
				(ptr2->Y -ptr1->Y)*(ptr2->Y -ptr1->Y));
			if (length != 0.0)
			{
				distance = fabs(((ptr2->Y -ptr1->Y)*(Crosshair.X -ptr1->X)-
					(ptr2->X -ptr1->X)*(Crosshair.Y -ptr1->Y))/length);
				if (distance < mindistance)
				{
					mindistance = distance;
					index = n;
				}
			}
			ptr1 = ptr2;
		}

			/* erase the old polygon
			 * create a new point at the end of it's point list
			 * adjust point order
			 * recalculate it's bounding-box
			 * redraw the polygon
			 */
		ErasePolygon(polygon);
		InsertPointIntoPolygon(polygon, index, Crosshair.X, Crosshair.Y);
		DrawPolygon(layer, polygon);
	}
}

/* ---------------------------------------------------------------------------
 * removes a point of a polygon if the number of points is > 3
 */
void RemovePolygonPointAtCrosshairPosition(void)
{
	PolygonTypePtr		polygon;
	PolygonPointTypePtr	point;

		/* lookup polygon */
	if (SearchObjectByPosition(POLYGONPOINT_TYPE,
		(void **) &polygon, (void **) &point,
		Crosshair.X, Crosshair.Y) == POLYGONPOINT_TYPE)
	{
		if (polygon->PointN > 3)
			RemovePolygonPoint(polygon, point);
	}
}

/* ---------------------------------------------------------------------------
 * go back to the  previous point of the polygon
 */
void GoToPreviousPolygonPoint(void)
{
	switch(Crosshair.AttachedPolygon.PointN)
	{
			/* do nothing if mode has just been entered */
		case 0:
			break;

			/* reset number of points and 'LINE_MODE' state */
		case 1:
			Crosshair.AttachedPolygon.PointN = 0;
			Crosshair.AttachedLine.State = STATE_FIRST;
			break;

			/* back-up one point */
		default:
		{
			PolygonPointTypePtr	points = Crosshair.AttachedPolygon.Points;
			Cardinal			n = Crosshair.AttachedPolygon.PointN -2;
	
			Crosshair.AttachedPolygon.PointN--;
			Crosshair.AttachedLine.X1 = points[n].X;
			Crosshair.AttachedLine.Y1 = points[n].Y;
			break;
		}
	}
}

/* ---------------------------------------------------------------------------
 * close polygon if possible
 */
void ClosePolygon(void)
{
	Cardinal	n = Crosshair.AttachedPolygon.PointN;

		/* check number of points */
	if (n >= 3)
	{
			/* if 45 degree lines are what we want do a quick check
			 * if closing the polygon makes sense
			 */
		if (!TEST_FLAG(ALLDIRCETIONFLAG, PCB))
		{
			Dimension	dx, dy;

			dx = abs(Crosshair.AttachedPolygon.Points[n-1].X -
				Crosshair.AttachedPolygon.Points[0].X);
			dy = abs(Crosshair.AttachedPolygon.Points[n-1].Y -
				Crosshair.AttachedPolygon.Points[0].Y);
			if (!(dx == 0 || dy == 0 || dx == dy))
			{
				Message(False, "cannot close polygon because 45 degree lines are requested\n");
				return;
			}
		}
		CopyAttachedPolygonToLayer();
	}
	else
		Message(False, "a polygon has to have at least 3 points\n");
}

/* ---------------------------------------------------------------------------
 * inserts a point into an existing polygon
 * the requested index number and the coordinates are passed
 */
PolygonPointTypePtr InsertPointIntoPolygon(PolygonTypePtr Polygon,
	Cardinal Index, Position X, Position Y)
{
	PolygonPointType	save;
	Cardinal			n;

	save = *CreateNewPointInPolygon(Polygon, X, Y);
	for (n = Polygon->PointN-1; n > Index; n--)
		Polygon->Points[n] = Polygon->Points[n-1];
	Polygon->Points[Index] = save;
	SetPolygonBoundingBox(Polygon);
	SetChangedFlag(True);
	AddObjectToInsertPolygonPointUndoList(Polygon, &Polygon->Points[Index]);
	IncrementUndoSerialNumber();
	return(&Polygon->Points[Index]);
}

/* ---------------------------------------------------------------------------
 * removes a polygon-point from a polygon
 */
void RemovePolygonPoint(PolygonTypePtr Polygon, PolygonPointTypePtr Point)
{
	LayerTypePtr		layer;
	PolygonPointTypePtr	ptr;

		/* the layer isn't passed in so we have to lookup it's pointer
		 * to see if it is switched on.
		 * !!! works for PCB only, not available for buffers !!!
		 */
	if ((layer = GetLayerOfPolygon(PCB->Data, Polygon)) != NULL)
	{
		if (layer->On)
			ErasePolygon(Polygon);
		AddObjectToRemovePolygonPointUndoList(Polygon, Point);
		IncrementUndoSerialNumber();

			/* remove point from list, keep point order */
		for (ptr = Point+1; ptr != &Polygon->Points[Polygon->PointN]; ptr++)
		{
			*Point = *ptr;
			Point = ptr;
		}
		Polygon->PointN--;
		SetPolygonBoundingBox(Polygon);

			/* redraw polygon if necessary */
		if (layer->On)
			DrawPolygon(layer, Polygon);
		SetChangedFlag(True);
	}
}

/* ---------------------------------------------------------------------------
 * moves the data of the attached (new) polygon to the current layer
 */
void CopyAttachedPolygonToLayer(void)
{
	PolygonTypePtr	polygon;
	
		/* move data to layer and clear attached struct */
	polygon = CreateNewPolygon(CURRENT, NOFLAG);
	*polygon = Crosshair.AttachedPolygon;
	memset(&Crosshair.AttachedPolygon, 0, sizeof(PolygonType));
	SetPolygonBoundingBox(polygon);
	DrawPolygon(CURRENT, polygon);
	SetChangedFlag(True);

		/* reset state of attached line */
	Crosshair.AttachedLine.State = STATE_FIRST;
}

/* ---------------------------------------------------------------------------
 * gets the layer of a polygon,
 * used by routines which move or remove polygon points
*/
LayerTypePtr GetLayerOfPolygon(DataTypePtr Data, PolygonTypePtr Polygon)
{
	ALLPOLYGON_LOOP(Data,
		if (polygon == Polygon)
			return(layer);
	);
	return(NULL);
}

