/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/misc.c,v 2.3 1994/10/29 17:33:10 nau Exp nau $";

/* misc functions used by several modules
 */

#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <math.h>
#include <unistd.h>

#include "global.h"

#include "create.h"
#include "crosshair.h"
#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "file.h"
#include "error.h"
#include "mymem.h"
#include "misc.h"
#include "move.h"
#include "remove.h"
#include "rotate.h"

#include <X11/cursorfont.h>
#include <X11/Shell.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Simple.h>

/* ---------------------------------------------------------------------------
 * some additional defines
 */
#ifndef M_PI
#define	M_PI	3.14159265358979323846
#endif
#define	M180	(M_PI/180.0)

/* ---------------------------------------------------------------------------
 * prints copyright information
 */
void Copyright(void)
{
	printf("\n"
		"                COPYRIGHT for %s version %s\n\n"
		"    PCB, interactive printed circuit board design\n"
		"    Copyright (C) 1994,1995 Thomas Nau\n\n"
		"    This program is free software; you can redistribute it and/or modify\n"
		"    it under the terms of the GNU General Public License as published by\n"
		"    the Free Software Foundation; either version 2 of the License, or\n"
		"    (at your option) any later version.\n\n"
		"    This program is distributed in the hope that it will be useful,\n"
		"    but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		"    GNU General Public License for more details.\n\n"
		"    You should have received a copy of the GNU General Public License\n"
		"    along with this program; if not, write to the Free Software\n"
		"    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n",
		Progname, RELEASE);
	exit(0);
}

/* ---------------------------------------------------------------------------
 * prints usage message
 */
void Usage(void)
{
	fprintf(stderr, "\nUSAGE: %s [standard X options] [standard options] [layout]\n"
		"or   : %s [standard X options] <exactly one special option>\n\n"
		"standard options are:\n"
		"  -alldirections:           enable 'all-direction' lines\n"
		"  +alldirections:           force 45 degree lines\n"
		"  -backup <seconds>:        time between two backups\n"
		"  -c <number>:              characters per output-line\n"
		"  -fontfile <file>:         read default font from this file\n"
		"  -lelement <command>:      command to copy element files to stdout,\n"
		"                            %%f is replaced by the filename\n"
		"                            %%p by the seachpath.\n"
		"  -lfile <command>:         command to copy layout files to stdout,\n"
		"                            %%f is replaced by the filename\n"
		"                            %%p by the seachpath.\n"
		"  -lfont <command>:         command to copy font files to stdout,\n"
		"                            %%f is replaced by the filename\n"
		"                            %%p by the seachpath.\n"
		"  -lg <layergroups>:        set layergroups of new layouts to this\n"
		"  -loggeometry <geometry>:  set the geometry of the logging window\n"
		"  -pnl <value>:             maximum display length of pin names\n"
		"  -pz <value>:              zoom factor for pinout windows\n"
		"  -reset:                   reset connections after each element\n"
		"  +reset:                   negation of '-reset'\n"
		"  -ring:                    ring bell when connection lookup is done\n"
		"  +ring:                    negation of '-r'\n"
		"  -s:                       save the last command entered by user\n"
		"  +s:                       negation of '-s'\n"
		"  -save:                    always save data before it is lost\n"
		"  +save:                    override data if required by command\n"
		"  -sfile <command>:         command to copy stdin to layout file,\n"
		"                            %%f is replaced by the filename\n"
		"  -size <width>x<height>    size of a layout\n"
		"  -v <value>:               sets the volume of the X speaker\n"
		"special options are:\n"
		"  -copyright:               prints copyright information\n"
		"  -help:                    prints this message\n"
		"  -version:                 prints the current version number\n",
		Progname, Progname);
	exit(1);
}

/* ---------------------------------------------------------------------------
 * sets the bounding box of a polygons
 */
void SetPolygonBoundingBox(PolygonTypePtr Polygon)
{
	Position	minx, miny,
				maxx, maxy;

	minx = miny = MAX_COORD;
	maxx = maxy = 0;
	POLYGONPOINT_LOOP(Polygon,
		minx = MIN(minx, point->X);
		miny = MIN(miny, point->Y);
		maxx = MAX(maxx, point->X);
		maxy = MAX(maxy, point->Y);
	);
	Polygon->BoundingBox.X1 = minx;
	Polygon->BoundingBox.Y1 = miny;
	Polygon->BoundingBox.X2 = maxx;
	Polygon->BoundingBox.Y2 = maxy;
}

/* ---------------------------------------------------------------------------
 * sets the bounding box of an elements
 */
void SetElementBoundingBox(ElementTypePtr Element)
{
	Position	minx, miny,
				maxx, maxy;
	float		fminx, fminy,
				fmaxx, fmaxy;
	int			angle1,
				angle2,
				angle;

		/* do not include the elementnames bounding box which
		 * is handles seperatly
		 */
	minx = miny = MAX_COORD;
	maxx = maxy = 0;
	ELEMENTLINE_LOOP(Element,
		minx = MIN(minx, line->X1);
		miny = MIN(miny, line->Y1);
		minx = MIN(minx, line->X2);
		miny = MIN(miny, line->Y2);
		maxx = MAX(maxx, line->X1);
		maxy = MAX(maxy, line->Y1);
		maxx = MAX(maxx, line->X2);
		maxy = MAX(maxy, line->Y2);
	);
	PIN_LOOP(Element,
		minx = MIN(minx, pin->X);
		miny = MIN(miny, pin->Y);
		maxx = MAX(maxx, pin->X);
		maxy = MAX(maxy, pin->Y);
	);
	ARC_LOOP(Element,
			/* arc->StartAngle is in [0,360], arc->Delta in [0,360] */
		angle1 = arc->StartAngle;
		angle2 = arc->StartAngle +arc->Delta;

			/* initialize limits */
		fminx = MIN(cos(M180*(float) angle1),cos(M180*(float) angle2));
		fmaxx = MAX(cos(M180*(float) angle1),cos(M180*(float) angle2));
		fminy = MIN(sin(M180*(float) angle1),sin(M180*(float) angle2));
		fmaxy = MAX(sin(M180*(float) angle1),sin(M180*(float) angle2));

			/* loop and check all angles n*180
			 * with angle1 <= a <= angle2
			 */
		for (angle = (angle1 /180 +1)*180; angle < angle2; angle += 180)
		{
			fminx = MIN(cos(M180 *(float) angle), fminx);
			fmaxx = MAX(cos(M180 *(float) angle), fmaxx);
		}

			/* loop and check all angles n*180+90
			 * with angle1 <= a <= angle2
			 */
		for (angle = ((angle1+90)/180)*180+90; angle < angle2; angle += 180)
		{
			fminy = MIN(sin(M180 *(float) angle), fminy);
			fmaxy = MAX(sin(M180 *(float) angle), fmaxy);
		}
		minx = MIN(minx, (int) (fminx *arc->Width) +arc->X);
		miny = MIN(miny, (int) (fminy *arc->Height) +arc->Y);
		maxx = MAX(maxx, (int) (fmaxx *arc->Width) +arc->X);
		maxy = MAX(maxy, (int) (fmaxy *arc->Height) +arc->Y);
	);

	Element->BoundingBox.X1 = minx;
	Element->BoundingBox.Y1 = miny;
	Element->BoundingBox.X2 = maxx;
	Element->BoundingBox.Y2 = maxy;
}

/* ---------------------------------------------------------------------------
 * creates the bounding box of a text objects
 */
void SetTextBoundingBox(SymbolTypePtr Symbol, TextTypePtr Text)
{
	unsigned char	*s = (unsigned char *) Text->TextString;
	Dimension		width = 0,
					height = 0;

		/* calculate size of the bounding box */
	Text->BoundingBox.X1 = Text->X;
	Text->BoundingBox.Y1 = Text->Y;
	for (; s && *s; s++)
	{
		width  += Symbol[*s].Width +Symbol[*s].Delta;
		height = MAX(height, Symbol[*s].Height);
	}

		/* scale values and rotate them */
	width = width *Text->Scale /100;
	height = height *Text->Scale /100;
	Text->BoundingBox.X2 = Text->BoundingBox.X1 +width;
	Text->BoundingBox.Y2 = Text->BoundingBox.Y1 +height;
	RotateBoxLowLevel(&Text->BoundingBox, Text->X, Text->Y, Text->Direction);
}

/* ---------------------------------------------------------------------------
 * returns True if data area is empty
 */
Boolean IsDataEmpty(DataTypePtr Data)
{
	Boolean		hasNoObjects;
	Cardinal	i;

	hasNoObjects = (Data->ViaN == 0);
	hasNoObjects &= (Data->ElementN == 0);
	for (i = 0; i < MAX_LAYER; i++)
		hasNoObjects = hasNoObjects &&
			Data->Layer[i].LineN == 0 && 
			Data->Layer[i].TextN == 0 &&
			Data->Layer[i].PolygonN == 0;
	return(hasNoObjects);
}

/* ---------------------------------------------------------------------------
 * gets minimum and maximum coordinates
 * returns NULL if layout is empty
 */
BoxTypePtr GetDataBoundingBox(DataTypePtr Data)
{
	static	BoxType	box;

		/* preset identifiers with highest and lowest possible values */
	box.X1 = box.Y1 = MAX_COORD;
	box.X2 = box.Y2 = 0;

		/* now scan for the lowest/highest X and Y coodinate */
	VIA_LOOP(Data,
		box.X1 = MIN(box.X1, via->X -via->Thickness/2);
		box.Y1 = MIN(box.Y1, via->Y -via->Thickness/2);
		box.X2 = MAX(box.X2, via->X +via->Thickness/2);
		box.Y2 = MAX(box.Y2, via->Y +via->Thickness/2);
	);
	ELEMENT_LOOP(Data,
		box.X1 = MIN(box.X1, element->BoundingBox.X1);
		box.Y1 = MIN(box.Y1, element->BoundingBox.Y1);
		box.X2 = MAX(box.X2, element->BoundingBox.X2);
		box.Y2 = MAX(box.Y2, element->BoundingBox.Y2);
	);
	ALLLINE_LOOP(Data,
		box.X1 = MIN(box.X1, line->X1);
		box.Y1 = MIN(box.Y1, line->Y1);
		box.X1 = MIN(box.X1, line->X2);
		box.Y1 = MIN(box.Y1, line->Y2);
		box.X2 = MAX(box.X2, line->X1);
		box.Y2 = MAX(box.Y2, line->Y1);
		box.X2 = MAX(box.X2, line->X2);
		box.Y2 = MAX(box.Y2, line->Y2);
	);
	ALLTEXT_LOOP(Data,
		box.X1 = MIN(box.X1, text->BoundingBox.X1);
		box.Y1 = MIN(box.Y1, text->BoundingBox.Y1);
		box.X2 = MAX(box.X2, text->BoundingBox.X2);
		box.Y2 = MAX(box.Y2, text->BoundingBox.Y2);
	);
	ALLPOLYGON_LOOP(Data,
		box.X1 = MIN(box.X1, polygon->BoundingBox.X1);
		box.Y1 = MIN(box.Y1, polygon->BoundingBox.Y1);
		box.X2 = MAX(box.X2, polygon->BoundingBox.X2);
		box.Y2 = MAX(box.Y2, polygon->BoundingBox.Y2);
	);
	return(IsDataEmpty(Data) ? NULL : &box);
}

/* ---------------------------------------------------------------------------
 * centers the displayed PCB around the specified point
 * (X,Y) in screen coordinates
 */
void CenterDisplay(Position X, Position Y)
{
	Position	old_x, old_y,			/* output window in viewport */
				new_x, new_y;
	float		top;					/* top of thumb */

		/* save old values (offset of upper/left corner) */
	XtVaGetValues(Output.Output, XtNx, &old_x, XtNy, &old_y, NULL);

		/* move origin half a screen width/height to get display centered */
	X -= (Output.Width /2);
	top = (float) X /(float) TO_SCREEN(PCB->MaxWidth);
	top = MIN(MAX(top, 0.0), 1.0);
	XtCallCallbacks(Output.ScrollbarBottom, XtNjumpProc, (XtPointer) &top);

	Y -= (Output.Height /2);
	top = (float) Y /(float) TO_SCREEN(PCB->MaxHeight);
	top = MIN(MAX(top, 0.0), 1.0);
	XtCallCallbacks(Output.ScrollbarLeft, XtNjumpProc, (XtPointer) &top);

		/* now get the new values (reread cause of rounding) */
	XtVaGetValues(Output.Output, XtNx, &new_x, XtNy, &new_y, NULL);

		/* the scrollbar widget triggers an expose event if (x,y)
		 * has changed. I have to do it if it hasn't changed
		 */
	if (new_x == old_x && new_y == old_y)
		ClearAndRedrawOutput();
}

/* ---------------------------------------------------------------------------
 * transforms symbol coordinates so that the left edge of each symbol
 * is at the zero position. The y coordinates are moved so that min(y) = 0
 * 
 */
void SetFontInfo(FontTypePtr Ptr)
{
	Cardinal		i, j;
	SymbolTypePtr	symbol;
	LineTypePtr		line;
	Position		totalminy = MAX_COORD;

		/* calculate cell with and height (is at least DEFAULT_CELLSIZE)
		 * maximum cell width and height
		 * minimum x and y position of all lines
		 */
	Ptr->MaxWidth = DEFAULT_CELLSIZE;
	Ptr->MaxHeight = DEFAULT_CELLSIZE;
	for (i = 0, symbol = Ptr->Symbol; i <= MAX_FONTPOSITION; i++, symbol++)
	{
		Position		minx, miny,
						maxx, maxy;

			/* next one if the index isn't used or symbol is empty (SPACE) */
		if (!symbol->Valid || !symbol->LineN)
			continue;

		minx = miny = MAX_COORD;
		maxx = maxy = 0;
		for (line = symbol->Line, j = symbol->LineN; j; j--, line++)
		{
			minx = MIN(minx, line->X1);
			miny = MIN(miny, line->Y1);
			minx = MIN(minx, line->X2);
			miny = MIN(miny, line->Y2);
			maxx = MAX(maxx, line->X1);
			maxy = MAX(maxy, line->Y1);
			maxx = MAX(maxx, line->X2);
			maxy = MAX(maxy, line->Y2);
		}

			/* move symbol to left edge */
		for (line = symbol->Line, j = symbol->LineN; j; j--, line++)
			MOVE_LINE_LOWLEVEL(line, -minx, 0);

			/* set symbol bounding box with a minimum cell size of (1,1) */
		symbol->Width = maxx -minx +1;
		symbol->Height = maxy -miny +1;

			/* check total min/max  */
		Ptr->MaxWidth = MAX(Ptr->MaxWidth, symbol->Width);
		Ptr->MaxHeight = MAX(Ptr->MaxHeight, symbol->Height);
		totalminy = MIN(totalminy, miny);
	}

		/* move coordinate system to the upper edge (lowest y on screen) */
	for (i = 0, symbol = Ptr->Symbol; i <= MAX_FONTPOSITION; i++, symbol++)
		if (symbol->Valid)
			for (line = symbol->Line, j = symbol->LineN; j; j--, line++)
				MOVE_LINE_LOWLEVEL(line, 0, -totalminy);

		/* setup the box for the default symbol */
	Ptr->DefaultSymbol.X1 = Ptr->DefaultSymbol.Y1 = 0;
	Ptr->DefaultSymbol.X2 = Ptr->DefaultSymbol.X1 +Ptr->MaxWidth;
	Ptr->DefaultSymbol.Y2 = Ptr->DefaultSymbol.Y1 +Ptr->MaxHeight;
} 

/* ----------------------------------------------------------------------
 * parses the group definition string which is a colon seperated list of
 * comma seperated layer numbers (1,2:4,6,8)
 */
int ParseGroupString(char *s, LayerGroupTypePtr LayerGroup)
{
	int		group,
			member,
			layer;

		/* clear struct */
	memset(LayerGroup, 0, sizeof (LayerGroupType));

		/* loop over all groups */
	for (group = 0; s && *s && group < MAX_LAYER; group++)
	{
		if (group >= MAX_LAYER)
			goto error;
		while (*s && isspace(*s))
			s++;

			/* loop over all group members */
		for (member = 0; *s; s++)
		{
				/* ignore white spaces and get layernumber */
			while(*s && isspace(*s))
				s++;
			if (!isdigit(*s) ||
				(layer = atoi(s)) < 1 ||
				layer > MAX_LAYER ||
				member >= MAX_LAYER)
				goto error;
			LayerGroup->Entries[group][member++] = layer-1;
			while (*s && isdigit(*s))
				s++;

				/* ignore white spaces and check for seperator */
			while (*s && isspace(*s))
				s++;
			if (!*s || *s == ':')
				break;
			if (*s != ',')
				goto error;
		}
		LayerGroup->Number[group] = member;
		if (*s == ':')
			s++;
	}
	return(0);

	error:
			/* reset structure */
		memset(LayerGroup, 0, sizeof (LayerGroupType));
		return(1);
}

/* ---------------------------------------------------------------------------
 * does some cleanup before exiting
 */
void Cleanup(void)
{
	DestroyCrosshair();
	XtDestroyWidget(Output.Toplevel);
	XtDestroyApplicationContext(Context);
	XFreeGC(Dpy, Output.fgGC);
	XFreeGC(Dpy, Output.bgGC);
}

/* ---------------------------------------------------------------------------
 * quits application
 */
void QuitApplication(void)
{
		/* save data if necessary */
	if (PCB->Changed && Settings.SaveInTMP)
			EmergencySave();

	Cleanup();
	exit(0);
}

/* ---------------------------------------------------------------------------
 * concatenates directory and filename if directory != NULL,
 * expands them with a shell and returns the found name(s) or NULL
 */
char *ExpandFilename(char *Dirname, char *Filename)
{
	static	DynamicStringType	answer;
			char				*command;
			FILE				*pipe;
			int					c;

		/* allocate memory for commandline and build it */
	DSClearString(&answer);
	if (Dirname)
	{
		command = MyCalloc(strlen(Filename) +strlen(Dirname) +7,
			sizeof(char), "Expand()");
		sprintf(command, "echo %s/%s", Dirname, Filename);
	}
	else
	{
		command = MyCalloc(strlen(Filename) +6, sizeof(char), "Expand()");
		sprintf(command, "echo %s", Filename);
	}

		/* execute it with shell */
	if ((pipe = popen(command, "r")) != NULL)
	{
			/* discard all but the first returned line */
		for(;;)
		{
			if ((c = fgetc(pipe)) == EOF || c == '\n' || c == '\r')
				break;
			else
				DSAddCharacter(&answer, c);
		}

		SaveFree(command);
		return (pclose(pipe) ? NULL : answer.Data);
	}

		/* couldn't be expanded by the shell */
	PopenErrorMessage(command);
	SaveFree(command);
	return(NULL);	
}

/* ---------------------------------------------------------------------------
 * wait for a button of key event in output window and calls the
 * action when a button event has been detected
 * cursor key events are handled
 */
Boolean GetPosition(char *MessageText)
{
	XEvent		event;
	XAnyEvent	*any = (XAnyEvent *) &event;
	KeySym		keysym;
	char		buffer[10];
	Cursor		oldCursor;

	MessagePrompt(MessageText);
	oldCursor = SetOutputXCursor(XC_hand2);

		/* eat up all events that would cause actions to be performed */
	for(;;)
	{
		XtAppNextEvent(Context, &event);
		switch(any->type)
		{
			case KeyRelease:
			case KeyPress:
			case ButtonPress:
					/* first check if we are inside the output window */
				if (any->window != Output.OutputWindow)
					break;

					/* evaluate cursor keys and modifier keys;
					 * dispatch the event if true
					 */
				XLookupString((XKeyEvent *) &event, buffer, sizeof(buffer),
					&keysym, NULL);
				if (IsCursorKey(keysym) || IsModifierKey(keysym))
					XtDispatchEvent(&event);
				else
				{
						/* abort on any other key;
						 * restore cursor and clear message line
						 */
					SetOutputXCursor(oldCursor);
					MessagePrompt(NULL);
					return(any->type == ButtonPress);
				}
				break;

			default:
				XtDispatchEvent(&event);
		}
	}
}

/* ----------------------------------------------------------------------
 * releases tmp pixmap used to save output data
 */
void ReleaseSaveUnderPixmap(void)
{
	if (VALID_PIXMAP(PCB->SaveUnder))
		XFreePixmap(Dpy, PCB->SaveUnder);

		/* mark pixmap unusable */
	PCB->SaveUnder = BadAlloc;
}

/* ----------------------------------------------------------------------
 * saves output window to tmp pixmap to reduce redrawing
 */
void SaveOutputWindow(void)
{
		/* release old pixmap and allocate new one */
	ReleaseSaveUnderPixmap();
	PCB->SaveUnder = XCreatePixmap(Dpy, Output.OutputWindow,
		Output.Width, Output.Height,
		DefaultDepthOfScreen(XtScreen(Output.Output)));

		/* make sure that cursor is off */
	if (VALID_PIXMAP(PCB->SaveUnder))
		XCopyArea(Dpy, Output.OutputWindow, PCB->SaveUnder, Output.fgGC,
			Output.OffsetX, Output.OffsetY,
			Output.Width, Output.Height,
			0, 0);
}

/* ---------------------------------------------------------------------------
 * sets the X cursor for the output window (uses cursorfont)
 * XSync() is used to force a cursor change, old memory is released
 */
unsigned int SetOutputXCursor(unsigned int Shape)
{
	unsigned int	oldShape = Output.XCursorShape;
	Cursor			oldCursor = Output.XCursor;

		/* check if window exists to prevent from fatal errors */
	if (Output.OutputWindow)
	{
		Output.XCursorShape = Shape;
		Output.XCursor= XCreateFontCursor(Dpy, Shape);
		XDefineCursor(Dpy, Output.OutputWindow, Output.XCursor);
		XSync(Dpy, False);

			/* release old cursor and return old shape */
		if (oldCursor)
			XFreeCursor(Dpy, oldCursor);
		return(oldShape);
	}
	return(DEFAULT_CURSORSHAPE);
}

/* ---------------------------------------------------------------------------
 * returns the layer number for the passed pointer
 */
int GetLayerNumber(DataTypePtr Data, LayerTypePtr Layer)
{
	int		i;

	for (i = 0; i < MAX_LAYER; i++)
		if (Layer == &Data->Layer[i])
			break;
	return(i);
}

/* ---------------------------------------------------------------------------
 * returns a pointer to an objects bounding box;
 * data is valid until the routine is called again
 */
BoxTypePtr GetObjectBoundingBox(int Type, void *Ptr1, void *Ptr2)
{
	static	BoxType	box;

	switch(Type)
	{
		case VIA_TYPE:
		{
			PinTypePtr	via = (PinTypePtr) Ptr1;

			box.X1 = via->X -via->Thickness/2;
			box.Y1 = via->Y -via->Thickness/2;
			box.X2 = via->X +via->Thickness/2;
			box.Y2 = via->Y +via->Thickness/2;
			break;
		}

		case LINE_TYPE:
		{
			LineTypePtr	line = (LineTypePtr) Ptr2;

			box.X1 = MIN(line->X1, line->X2);
			box.Y1 = MIN(line->Y1, line->Y2);
			box.X2 = MAX(line->X1, line->X2);
			box.Y2 = MAX(line->Y1, line->Y2);
			break;
		}

		case TEXT_TYPE:
		case ELEMENTNAME_TYPE:
			box = ((TextTypePtr) Ptr2)->BoundingBox;
			break;

		case POLYGON_TYPE:
			box = ((PolygonTypePtr) Ptr2)->BoundingBox;
			break;

		case ELEMENT_TYPE:
			box = ((ElementTypePtr) Ptr1)->BoundingBox;
			break;

		case PIN_TYPE:
		{
			PinTypePtr	pin = (PinTypePtr) Ptr2;

			box.X1 = pin->X -pin->Thickness/2;
			box.Y1 = pin->Y -pin->Thickness/2;
			box.X2 = pin->X +pin->Thickness/2;
			box.Y2 = pin->Y +pin->Thickness/2;
			break;
		}

		case POLYGONPOINT_TYPE:
		{
			PolygonPointTypePtr	point = (PolygonPointTypePtr) Ptr2;

			box.X1 = box.X2 = point->X;
			box.Y1 = box.Y2 = point->Y;
			break;
		}

		default:
			return(NULL);
	}
	return(&box);
}

/* ---------------------------------------------------------------------------
 * resets the layerstack setting
 */
void ResetStackAndVisibility(void)
{
	Cardinal	i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		LayerStack[i] = i;
		PCB->Data->Layer[i].On = True;
	}
	PCB->ElementOn = True;
	PCB->PinOn = True;
	PCB->ViaOn = True;
}

/* ---------------------------------------------------------------------------
 * changes the maximum size of a layout
 * adjusts the scrollbars
 * releases the saved pixmap if necessary
 * and adjusts the cursor confinement box
 */
void ChangePCBSize(Dimension Width, Dimension Height)
{
		/* force a redraw of the forbidden area */
	ReleaseSaveUnderPixmap();
	PCB->MaxWidth = Width;
	PCB->MaxHeight = Height;
	XtVaSetValues(Output.Output,
		XtNwidth, TO_SCREEN(Width),
		XtNheight, TO_SCREEN(Height),
		NULL);

		/* crosshair range is different if pastebuffer-mode
		 * is enabled
		 */
	if (Settings.Mode == PASTEBUFFER_MODE)
		SetCrosshairRange(
			PASTEBUFFER->X -PASTEBUFFER->BoundingBox.X1,
			PASTEBUFFER->Y -PASTEBUFFER->BoundingBox.Y1,
			MAX(0, Width -(PASTEBUFFER->BoundingBox.X2-PASTEBUFFER->X)),
			MAX(0, Height -(PASTEBUFFER->BoundingBox.Y2-PASTEBUFFER->Y)));
	else
		SetCrosshairRange(0, 0, (Position) Width, (Position) Height);
}

/* ----------------------------------------------------------------------
 * returns pointer to current working directory
 */
char *GetWorkingDirectory(void)
{
	static	char	path[MAXPATHLEN+1];

#ifdef SYSV
	return(getcwd(path, MAXPATHLEN));
#else
		/* seems that some BSD releases lack of a prototype for getwd() */
	return((char *) getwd(path));
#endif
}
