/* "LOG", the circuit editing and simulation system,
   "DigLOG", the digital simulator for LOG.
   Copyright (C) 1985, 1990 David Gillespie.
   Author's address: daveg@csvax.caltech.edu; 256-80 Caltech/Pasadena CA 91125.

   "AnaLOG", the analog simulator for LOG.
   Copyright (C) 1985, 1990 John Lazzaro.
   Author's address: lazzaro@csvax.caltech.edu; 256-80 Caltech.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation (any version).

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* Output from p2c, the Pascal-to-C translator */
/* From input file "gates.text" */




#include <p2c/p2c.h>


#define PFET4_G
#include "pfet4.h"


#define Cstart          1
#define N_Cgd           10
#define N_Cgs           11
#define N_Ratio         12
#define N_Vt            13
#define N_Leakage       14
#define N_FitFactor     15
#define N_Ktqn          16
#define N_Early         17

#define GateName        "PFET"


typedef struct PfetConst {
  double V0;   /*kt/qn*/
  double Kp1;   /*Pfet diode leakage current*(W/L)*/
  double Kp2;   /*qn/kt*/
  double Kp3;   /*Tanch(Kp3*x) ~=.99 when x=1*/
  double Kp4;   /*Kp1*Expo(Kp2*Vtp)/(Vtp*Vtp)*/
  double TwoKp4;   /*2*Kp4*/
  double Kp5;   /*Expo(-Kp2*Vtp)*/
  double InvVcross;   /*Inverse Early Voltage*/
  double TauCgd, TauCgs;
} PfetConst;


typedef struct ICPfet {
  double InitRatio, InitVt, InitLeakage, InitFitFactor, InitKtqn, InitCgd,
	 InitCgs, InitCgb, InitCdb, InitCsb, InitEarly;
} ICPfet;


/****************************************************************************/
/*            Procedure To Simulate Cell In Algorithm                        */
/****************************************************************************/


Local Void Ex_PFET4(act)
log_32_action *act;
{
  double MinI2, dI2dV1, dI2dV2, dI2dV3, MinI3, dI3dV1, dI3dV2, dI3dV3, MinI1,
	 dI1dV1, dI1dV2, dI1dV3;
  Ext_rec *Pin1Ptr, *Pin2Ptr, *Pin3Ptr;
  long Pin1, Pin2, Pin3;
  double VPin1, VPin2, VPin3, VMem1, VMem2, VMem3;   /*1*/
  double ICap;
  PfetConst *PfetVar;
  instlist *A_Gate;
  double Vds, Vsg, Vdg, ExpVsg, ExpVdg, ExpVds, BlendVsg, BlendVdg, BlendVds,
	 InvBlendVsg, InvBlendVdg, InvBlendVds, dBlendVsgdG, dBlendVdgdG,
	 dBlendVdsdS, Temp1, Temp2, Temp4, OldIforward, Iforward, OldIreverse,
	 Ireverse, dIforwarddD, dIforwarddG, dIforwarddS, dIreversedD,
	 dIreversedG, dIreversedS, Finv, F, dFinvdD, dFinvdS, dFinvdG,
	 Saturate, dTanch, Itotal, Ids, dIdsdG, dIdsdS, dIdsdD, Resforward,
	 Resreverse, dSaturatedS_Iforward, dSaturatedD_Iforward,
	 dSaturatedG_Iforward, dSaturatedD_Ireverse, dSaturatedG_Ireverse,
	 dSaturatedS_Ireverse, dIforwarddG_Saturate, dIforwarddS_Saturate,
	 dIforwarddD_Saturate, dIreversedG_Saturate, dIreversedD_Saturate,
	 dIreversedS_Saturate, Vdg_dBlendVdgdG, Vdg_InvBlendVdg, Kp1_ExpVdg,
	 Kp1_ExpVdg_BlendVdg, Kp4_Vdg, Vsg_dBlendVsgdG, Vsg_InvBlendVsg,
	 Kp1_ExpVsg, Kp1_ExpVsg_BlendVsg, Kp4_Vsg, ResForward_InvBlendVds,
	 BlendVds_ResReverse, Vsg_InvBlendVsg_dBlendVdsdS,
	 InvBlendVdg_InvBlendVds, Vdg_InvBlendVds_dBlendVdgdG,
	 Vdg_InvBlendVdg_dBlendVdsdS, InvBlendVsg_BlendVds,
	 Vsg_BlendVds_dBlendVsgdG, Vds_dTanch_Temp3_Itotal, F_dTanch_Itotal,
	 dSaturatedD_Itotal, dSaturatedG_Itotal, dSaturatedS_Itotal;
/* p2c: gates.text, line 998: 
 * Note: Line breaker spent 3.0+1.00 seconds, 5000 tries on line 79 [251] */
  log_grec *WITH;
  MatRec *WITH1;

  WITH = act->inst;
  A_Gate = (instlist *)WITH->info;
  PfetVar = (PfetConst *)A_Gate->InstVar;
  Pin1Ptr = (Ext_rec *)WITH->pin[0]->info;
  Pin2Ptr = (Ext_rec *)WITH->pin[1]->info;
  Pin3Ptr = (Ext_rec *)WITH->pin[2]->info;
  Pin1 = Pin1Ptr->nodenum;
  Pin2 = Pin2Ptr->nodenum;
  Pin3 = Pin3Ptr->nodenum;
  VPin1 = Pin1Ptr->last;
  VPin2 = Pin2Ptr->last;
  VPin3 = Pin3Ptr->last;
  VMem1 = A_Gate->Pininfo[0].Vmem;   /*2*/
  VMem2 = A_Gate->Pininfo[1].Vmem;
  VMem3 = A_Gate->Pininfo[2].Vmem;
  Vds = VPin3 - VPin2;
  Vsg = VPin2 - VPin1;
  Vdg = VPin3 - VPin1;
  if (fabs(Vds) < 0.000708) {
    ExpVsg = Expo(PfetVar->Kp2 * Vsg);
    ExpVdg = Expo(PfetVar->Kp2 * Vdg);
    Temp1 = PfetVar->Kp5 * ExpVsg;
    BlendVsg = 1 / (1 + Temp1);
    InvBlendVsg = 1 - BlendVsg;
    dBlendVsgdG = Temp1 * PfetVar->Kp2 * BlendVsg * BlendVsg;
    Temp2 = PfetVar->Kp5 * ExpVdg;
    BlendVdg = 1 / (1 + Temp2);
    InvBlendVdg = 1 - BlendVdg;
    dBlendVdgdG = Temp2 * PfetVar->Kp2 * BlendVdg * BlendVdg;
    ExpVds = Expo(1000000L * Vds);
    BlendVds = 1 / (1 + ExpVds);
    InvBlendVds = 1 - BlendVds;
    dBlendVdsdS = ExpVds * BlendVds * BlendVds;
    dBlendVdsdS = 1000000L * dBlendVdsdS;
    Temp4 = Vds * PfetVar->InvVcross;
    Resforward = 1 + Temp4;
    Resreverse = 1 - Temp4;
    Kp1_ExpVdg = PfetVar->Kp1 * ExpVdg;
    Kp1_ExpVdg_BlendVdg = Kp1_ExpVdg * BlendVdg;
    Kp4_Vdg = PfetVar->Kp4 * Vdg;
    Vdg_InvBlendVdg = Vdg * InvBlendVdg;
    OldIforward = Kp1_ExpVdg_BlendVdg + Vdg_InvBlendVdg * Kp4_Vdg;
    ResForward_InvBlendVds = Resforward * InvBlendVds;
    Iforward = ResForward_InvBlendVds * OldIforward;
    dIforwarddG = ResForward_InvBlendVds *
	(Kp1_ExpVdg * dBlendVdgdG - Kp1_ExpVdg_BlendVdg * PfetVar->Kp2 -
	 PfetVar->TwoKp4 * Vdg_InvBlendVdg - Kp4_Vdg * Vdg * dBlendVdgdG);
    dIforwarddS = -(dBlendVdsdS * Resforward +
		    PfetVar->InvVcross * InvBlendVds) * OldIforward;
    dIforwarddD = -(dIforwarddG + dIforwarddS);
    Kp1_ExpVsg = PfetVar->Kp1 * ExpVsg;
    Kp1_ExpVsg_BlendVsg = Kp1_ExpVsg * BlendVsg;
    Kp4_Vsg = PfetVar->Kp4 * Vsg;
    Vsg_InvBlendVsg = Vsg * InvBlendVsg;
    OldIreverse = Kp1_ExpVsg_BlendVsg + Vsg_InvBlendVsg * Kp4_Vsg;
    BlendVds_ResReverse = BlendVds * Resreverse;
    Ireverse = BlendVds_ResReverse * OldIreverse;
    dIreversedG = BlendVds_ResReverse *
	(Kp1_ExpVsg * dBlendVsgdG - Kp1_ExpVsg_BlendVsg * PfetVar->Kp2 -
	 PfetVar->TwoKp4 * Vsg_InvBlendVsg - Kp4_Vsg * Vsg * dBlendVsgdG);
    dIreversedD = -(dBlendVdsdS * Resreverse + BlendVds * PfetVar->InvVcross) *
		   OldIreverse;
    dIreversedS = -(dIreversedG + dIreversedD);
    Finv = PfetVar->V0 + Vsg_InvBlendVsg * BlendVds + Vdg_InvBlendVdg * InvBlendVds;
    F = PfetVar->Kp3 / Finv;
    Vsg_InvBlendVsg_dBlendVdsdS = Vsg_InvBlendVsg * dBlendVdsdS;
    InvBlendVdg_InvBlendVds = InvBlendVdg * InvBlendVds;
    Vdg_InvBlendVds_dBlendVdgdG = Vdg * InvBlendVds * dBlendVdgdG;
    Vdg_InvBlendVdg_dBlendVdsdS = Vdg_InvBlendVdg * dBlendVdsdS;
    InvBlendVsg_BlendVds = InvBlendVsg * BlendVds;
    Vsg_BlendVds_dBlendVsgdG = Vsg * BlendVds * dBlendVsgdG;
    dFinvdD = InvBlendVdg_InvBlendVds - Vsg_InvBlendVsg_dBlendVdsdS +
	      Vdg_InvBlendVds_dBlendVdgdG + Vdg_InvBlendVdg_dBlendVdsdS;
    dFinvdS = InvBlendVsg_BlendVds - Vdg_InvBlendVdg_dBlendVdsdS +
	      Vsg_BlendVds_dBlendVsgdG + Vsg_InvBlendVsg_dBlendVdsdS;
    dFinvdG = -InvBlendVdg_InvBlendVds - Vdg_InvBlendVds_dBlendVdgdG -
	      InvBlendVsg_BlendVds - Vsg_BlendVds_dBlendVsgdG;
    Saturate = Tanch(Vds * F);
    Itotal = Iforward + Ireverse;
    F_dTanch_Itotal = Itotal * F * (1 - Saturate * Saturate);
    Vds_dTanch_Temp3_Itotal = -(Vds * F_dTanch_Itotal / Finv);
    dSaturatedD_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdD + F_dTanch_Itotal;
    dSaturatedG_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdG;
    dSaturatedS_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdS - F_dTanch_Itotal;
    Ids = Itotal * Saturate;
    dIdsdG = (dIforwarddG + dIreversedG) * Saturate + dSaturatedG_Itotal;
    dIdsdS = (dIforwarddS + dIreversedS) * Saturate + dSaturatedS_Itotal;
    dIdsdD = (dIforwarddD + dIreversedD) * Saturate + dSaturatedD_Itotal;
  } else {
    if (Vds > 0) {
      ExpVdg = Expo(PfetVar->Kp2 * Vdg);
      Temp2 = PfetVar->Kp5 * ExpVdg;
      BlendVdg = 1 / (1 + Temp2);
      InvBlendVdg = 1 - BlendVdg;
      dBlendVdgdG = Temp2 * PfetVar->Kp2 * BlendVdg * BlendVdg;
      Resforward = 1 + Vds * PfetVar->InvVcross;
      Vdg_dBlendVdgdG = Vdg * dBlendVdgdG;
      Vdg_InvBlendVdg = Vdg * InvBlendVdg;
      Kp1_ExpVdg = PfetVar->Kp1 * ExpVdg;
      Kp1_ExpVdg_BlendVdg = Kp1_ExpVdg * BlendVdg;
      Kp4_Vdg = PfetVar->Kp4 * Vdg;
      OldIforward = Kp1_ExpVdg_BlendVdg + Vdg_InvBlendVdg * Kp4_Vdg;
      Iforward = Resforward * OldIforward;
      Finv = 1 / (PfetVar->V0 + Vdg_InvBlendVdg);
      F = PfetVar->Kp3 * Finv;
      Saturate = Tanch(Vds * F);
      dIforwarddG_Saturate = Resforward * Saturate *
	  (Kp1_ExpVdg * dBlendVdgdG - Kp1_ExpVdg_BlendVdg * PfetVar->Kp2 -
	   PfetVar->TwoKp4 * Vdg_InvBlendVdg - Kp4_Vdg * Vdg_dBlendVdgdG);
      dIforwarddS_Saturate = -PfetVar->InvVcross * OldIforward * Saturate;
      dIforwarddD_Saturate = -(dIforwarddG_Saturate + dIforwarddS_Saturate);
      dTanch = 1 - Saturate * Saturate;
      dSaturatedS_Iforward = -Iforward * F * dTanch;
      dSaturatedG_Iforward = -Vds * dSaturatedS_Iforward * Finv *
			      (InvBlendVdg + Vdg_dBlendVdgdG);
      dSaturatedD_Iforward = -(dSaturatedG_Iforward + dSaturatedS_Iforward);
      Ids = Iforward * Saturate;
      dIdsdG = dIforwarddG_Saturate + dSaturatedG_Iforward;
      dIdsdS = dIforwarddS_Saturate + dSaturatedS_Iforward;
      dIdsdD = dIforwarddD_Saturate + dSaturatedD_Iforward;
    } else {
      ExpVsg = Expo(PfetVar->Kp2 * Vsg);
      Temp1 = PfetVar->Kp5 * ExpVsg;
      BlendVsg = 1 / (1 + Temp1);
      InvBlendVsg = 1 - BlendVsg;
      dBlendVsgdG = Temp1 * PfetVar->Kp2 * BlendVsg * BlendVsg;
      Resreverse = 1 - Vds * PfetVar->InvVcross;
      Vsg_dBlendVsgdG = Vsg * dBlendVsgdG;
      Vsg_InvBlendVsg = Vsg * InvBlendVsg;
      Kp1_ExpVsg = PfetVar->Kp1 * ExpVsg;
      Kp1_ExpVsg_BlendVsg = Kp1_ExpVsg * BlendVsg;
      Kp4_Vsg = PfetVar->Kp4 * Vsg;
      OldIreverse = Kp1_ExpVsg_BlendVsg + Vsg_InvBlendVsg * Kp4_Vsg;
      Ireverse = Resreverse * OldIreverse;
      Finv = 1 / (PfetVar->V0 + Vsg_InvBlendVsg);
      F = PfetVar->Kp3 * Finv;
      Saturate = Tanch(Vds * F);
      dIreversedG_Saturate = Resreverse * Saturate *
	  (Kp1_ExpVsg * dBlendVsgdG - Kp1_ExpVsg_BlendVsg * PfetVar->Kp2 -
	   PfetVar->TwoKp4 * Vsg_InvBlendVsg - Kp4_Vsg * Vsg_dBlendVsgdG);
      dIreversedD_Saturate = -PfetVar->InvVcross * OldIreverse * Saturate;
      dIreversedS_Saturate = -(dIreversedG_Saturate + dIreversedD_Saturate);
      dTanch = 1 - Saturate * Saturate;
      dSaturatedD_Ireverse = F * dTanch * Ireverse;
      dSaturatedG_Ireverse = Vds * dSaturatedD_Ireverse * Finv *
			     (InvBlendVsg + Vsg_dBlendVsgdG);
      dSaturatedS_Ireverse = -(dSaturatedG_Ireverse + dSaturatedD_Ireverse);
      Ids = Ireverse * Saturate;
      dIdsdG = dIreversedG_Saturate + dSaturatedG_Ireverse;
      dIdsdS = dIreversedS_Saturate + dSaturatedS_Ireverse;
      dIdsdD = dIreversedD_Saturate + dSaturatedD_Ireverse;
    }
  }
  MinI3 = -Ids;
  MinI2 = Ids;
  dI2dV1 = -dIdsdG;
  dI3dV1 = dIdsdG;
  dI2dV3 = -dIdsdD;
  dI3dV3 = dIdsdD;
  dI2dV2 = -dIdsdS;
  dI3dV2 = dIdsdS;
  MinI1 = 0.0;
  dI1dV1 = 0.0;
  dI1dV2 = 0.0;
  dI1dV3 = 0.0;
  if (WITH->attr[N_Cgs - 1].UU.r != 0) {   /*C gate-source*/
    if (ChangedDt)
      PfetVar->TauCgs = WITH->attr[N_Cgs - 1].UU.r / Dt;
    ICap = PfetVar->TauCgs * (Vsg - VMem2 + VMem1);   /*4*/
    MinI2 -= ICap;
    dI2dV1 -= PfetVar->TauCgs;
    dI2dV2 += PfetVar->TauCgs;
    MinI1 += ICap;
    dI1dV1 += PfetVar->TauCgs;
    dI1dV2 -= PfetVar->TauCgs;
  }
  if (WITH->attr[N_Cgd - 1].UU.r != 0) {   /*C gate-drain*/
    if (ChangedDt)
      PfetVar->TauCgd = WITH->attr[N_Cgd - 1].UU.r / Dt;
    ICap = PfetVar->TauCgd * (Vdg - VMem3 + VMem1);
    MinI3 -= ICap;
    dI3dV1 -= PfetVar->TauCgd;
    dI3dV3 += PfetVar->TauCgd;
    MinI1 += ICap;
    dI1dV1 += PfetVar->TauCgd;
    dI1dV3 -= PfetVar->TauCgd;
  }
  WITH1 = System;
  if (Pin1 != 0) {
    WITH1->Mat[Pin1][SysCol] += MinI1;
    WITH1->Mat[Pin1][Pin1] += dI1dV1;
    WITH1->Mat[Pin1][Pin2] += dI1dV2;
    WITH1->Mat[Pin1][Pin3] += dI1dV3;
    WITH1->MatNZ[Pin1][SysCol] = true;
    WITH1->MatNZ[Pin1][Pin1] = true;
    WITH1->MatNZ[Pin1][Pin2] = true;
    WITH1->MatNZ[Pin1][Pin3] = true;
  }
  if (Pin2 != 0) {
    WITH1->Mat[Pin2][SysCol] += MinI2;
    WITH1->Mat[Pin2][Pin1] += dI2dV1;
    WITH1->Mat[Pin2][Pin2] += dI2dV2;
    WITH1->Mat[Pin2][Pin3] += dI2dV3;
    WITH1->MatNZ[Pin2][SysCol] = true;
    WITH1->MatNZ[Pin2][Pin1] = true;
    WITH1->MatNZ[Pin2][Pin2] = true;
    WITH1->MatNZ[Pin2][Pin3] = true;
  }
  if (Pin3 != 0) {
    WITH1->Mat[Pin3][SysCol] += MinI3;
    WITH1->Mat[Pin3][Pin1] += dI3dV1;
    WITH1->Mat[Pin3][Pin2] += dI3dV2;
    WITH1->Mat[Pin3][Pin3] += dI3dV3;
    WITH1->MatNZ[Pin3][SysCol] = true;
    WITH1->MatNZ[Pin3][Pin1] = true;
    WITH1->MatNZ[Pin3][Pin2] = true;
    WITH1->MatNZ[Pin3][Pin3] = true;
  }
  Capex(act->inst);   /*5*/
}  /*Ex_PFET4*/

/****************************************************************************/
/*          Procedure To Supply Current  Out of a Cell To Current Meters    */
/****************************************************************************/


Local Void Iin_PFET4(act)
log_32_action *act;
{
  Ext_rec *Pin1Ptr, *Pin2Ptr, *Pin3Ptr;
  double VPin1, VPin2, VPin3, VMem1, VMem2, VMem3;
  PfetConst *PfetVar;
  instlist *A_Gate;
  double Vds, Vsg, Vdg, ExpVsg, ExpVdg, BlendVsg, BlendVdg, BlendVds,
	 InvBlendVsg, InvBlendVdg, InvBlendVds, Temp1, Tempf, Tempb, Iforward,
	 Ireverse, Ids;
  log_grec *WITH1;

  switch (act->pin) {

  case 1:
    WITH1 = act->inst;
    Pin1Ptr = (Ext_rec *)WITH1->pin[0]->info;
    Pin2Ptr = (Ext_rec *)WITH1->pin[1]->info;
    Pin3Ptr = (Ext_rec *)WITH1->pin[2]->info;
    A_Gate = (instlist *)WITH1->info;
    PfetVar = (PfetConst *)A_Gate->InstVar;
    VMem1 = A_Gate->Pininfo[0].Vmem;
    VMem2 = A_Gate->Pininfo[1].Vmem;
    VMem3 = A_Gate->Pininfo[2].Vmem;
    VPin1 = Pin1Ptr->now;
    VPin2 = Pin2Ptr->now;
    VPin3 = Pin3Ptr->now;
    if (WITH1->attr[N_Cgs - 1].UU.r != 0)   /*C gate-source*/
      act->Iin = PfetVar->TauCgs * (VPin1 - VPin2 - VMem1 + VMem2) + act->Iin;
    if (WITH1->attr[N_Cgd - 1].UU.r != 0)   /*C gate-drain*/
      act->Iin = PfetVar->TauCgd * (VPin1 - VPin3 - VMem1 + VMem3) + act->Iin;
    CapIin(act);
    break;

  case 2:
  case 3:
    WITH1 = act->inst;
    Pin1Ptr = (Ext_rec *)WITH1->pin[0]->info;
    Pin2Ptr = (Ext_rec *)WITH1->pin[1]->info;
    Pin3Ptr = (Ext_rec *)WITH1->pin[2]->info;
    A_Gate = (instlist *)WITH1->info;
    PfetVar = (PfetConst *)A_Gate->InstVar;
    VMem1 = A_Gate->Pininfo[0].Vmem;
    VPin1 = Pin1Ptr->now;
    VPin2 = Pin2Ptr->now;
    VPin3 = Pin3Ptr->now;
    switch (act->pin) {

    case 2:
      if (WITH1->attr[N_Cgs - 1].UU.r != 0) {
	VMem2 = A_Gate->Pininfo[1].Vmem;
	act->Iin = PfetVar->TauCgs * (VPin2 - VPin1 - VMem2 + VMem1) + act->Iin;
      }
      break;

    case 3:
      if (WITH1->attr[N_Cgd - 1].UU.r != 0) {
	VMem3 = A_Gate->Pininfo[2].Vmem;
	act->Iin = PfetVar->TauCgd * (VPin3 - VPin1 - VMem3 + VMem1) + act->Iin;
      }
      break;
    }
    Vds = VPin3 - VPin2;
    if (fabs(Vds) < 0.000708) {
      Vsg = VPin2 - VPin1;
      Vdg = VPin3 - VPin1;
      ExpVsg = Expo(PfetVar->Kp2 * Vsg);
      BlendVsg = 1 / (1 + PfetVar->Kp5 * ExpVsg);
      InvBlendVsg = 1 - BlendVsg;
      ExpVdg = Expo(PfetVar->Kp2 * Vdg);
      BlendVdg = 1 / (1 + PfetVar->Kp5 * ExpVdg);
      InvBlendVdg = 1 - BlendVdg;
      BlendVds = 1 / (1 + Expo(1000000L * Vds));
      InvBlendVds = 1 - BlendVds;
      Temp1 = Vds * PfetVar->InvVcross;
      Tempf = InvBlendVdg * Vdg;
      Tempb = InvBlendVsg * Vsg;
      Iforward = (1 + Temp1) * InvBlendVds *
	  (PfetVar->Kp1 * ExpVdg * BlendVdg + Tempf * PfetVar->Kp4 * Vdg);
      Ireverse = (1 - Temp1) * BlendVds *
	  (PfetVar->Kp1 * ExpVsg * BlendVsg + Tempb * PfetVar->Kp4 * Vsg);
      Ids = (Iforward + Ireverse) * Tanch(Vds * PfetVar->Kp3 /
		(PfetVar->V0 + Tempb * BlendVds + Tempf * InvBlendVds));
    } else {
      if (Vds > 0) {
	Vdg = VPin3 - VPin1;
	ExpVdg = Expo(PfetVar->Kp2 * Vdg);
	BlendVdg = 1 / (1 + PfetVar->Kp5 * ExpVdg);
	InvBlendVdg = 1 - BlendVdg;
	Temp1 = InvBlendVdg * Vdg;
	Ids = (1 + Vds * PfetVar->InvVcross) *
	      (PfetVar->Kp1 * ExpVdg * BlendVdg + PfetVar->Kp4 * Temp1 * Vdg) *
	      Tanch(Vds * PfetVar->Kp3 / (PfetVar->V0 + Temp1));
      } else {
	Vsg = VPin2 - VPin1;
	ExpVsg = Expo(PfetVar->Kp2 * Vsg);
	BlendVsg = 1 / (1 + PfetVar->Kp5 * ExpVsg);
	InvBlendVsg = 1 - BlendVsg;
	Temp1 = Vsg * InvBlendVsg;
	Ids = (1 - Vds * PfetVar->InvVcross) *
	      (PfetVar->Kp1 * ExpVsg * BlendVsg + Temp1 * PfetVar->Kp4 * Vsg) *
	      Tanch(Vds * PfetVar->Kp3 / (PfetVar->V0 + Temp1));
      }
    }
    switch (act->pin) {

    case 2:
      act->Iin -= Ids;
      break;

    case 3:
      act->Iin += Ids;
      break;
    }
    CapIin(act);
    break;

  default:
    printf("Data Structure Corruption\n");
    _Escape(1002);
    break;
  }
}  /*Iin_PFET4*/

Local Void GetCnf_Pfet4(NewIC)
ICPfet *NewIC;
{
  boolean Found;
  Char Arg[256], Keyword[256];
  long Dummy;
  ICPfet *WITH;
  Char *STR1;

  WITH = NewIC;
  do {
    Getcommand("PFET4", Arg, &Found);
    if (Found) {
      if (Logglobals->hook.getword.link != NULL)
	(*(Void(*) PP((Char *buf, Char *wrd, Anyptr _link)))
	  Logglobals->hook.getword.proc)(Arg, Keyword,
					 Logglobals->hook.getword.link);
      else
	(*(Void(*) PP((Char *buf, Char *wrd)))Logglobals->hook.getword.proc)(
	  Arg, Keyword);
      if (!strcmp(Keyword, "W/L")) {
	TRY(try1);
	  WITH->InitRatio = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try1);
	  WITH->InitRatio = 1.0;
	ENDTRY(try1);
      }
      if (!strcmp(Keyword, "VT")) {
	TRY(try2);
	  WITH->InitVt = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try2);
	  WITH->InitVt = 0.9;
	ENDTRY(try2);
      }
      if (!strcmp(Keyword, "I0")) {
	TRY(try3);
	  WITH->InitLeakage = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try3);
	  WITH->InitLeakage = 1.66e-17;
	ENDTRY(try3);
      }
      if (!strcmp(Keyword, "FITFACTOR")) {
	TRY(try4);
	  WITH->InitFitFactor = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try4);
	  WITH->InitFitFactor = 3.33;
	ENDTRY(try4);
      }
      if (!strcmp(Keyword, "KT/QN")) {
	TRY(try5);
	  WITH->InitKtqn = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try5);
	  WITH->InitKtqn = 0.04;
	ENDTRY(try5);
      }
      if (!strcmp(Keyword, "CGS")) {
	TRY(try6);
	  WITH->InitCgs = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try6);
	  WITH->InitCgs = 10e-15;
	ENDTRY(try6);
      }
      if (!strcmp(Keyword, "CGD")) {
	TRY(try7);
	  WITH->InitCgd = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try7);
	  WITH->InitCgd = 10e-15;
	ENDTRY(try7);
      }
      if (!strcmp(Keyword, "CGB")) {
	TRY(try8);
	  WITH->InitCgb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try8);
	  WITH->InitCgb = 10e-15;
	ENDTRY(try8);
      }
      if (!strcmp(Keyword, "CDB")) {
	TRY(try9);
	  WITH->InitCdb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try9);
	  WITH->InitCdb = 10e-15;
	ENDTRY(try9);
      }
      if (!strcmp(Keyword, "CSB")) {
	TRY(try10);
	  WITH->InitCsb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try10);
	  WITH->InitCsb = 10e-15;
	ENDTRY(try10);
      }
      if (!strcmp(Keyword, "EARLY")) {
	TRY(try11);
	  WITH->InitEarly = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try11);
	  WITH->InitEarly = 50.0;
	ENDTRY(try11);
      }
    }
  } while (Found);
}

/****************************************************************************/
/*                          Installation Procedure                           */
/****************************************************************************/

Local Void initlib_PFET4(act)
log_32_action *act;
{
  Cell_rec *NewCell;
  ICPfet *NewIC;

  NewCell = NULL;
  NewCell = (Cell_rec *)Malloc(sizeof(Cell_rec));
  NewCell->simulatable = true;
  NewCell->intr_cell = 0;
  NewCell->phase = simple;
  Newasm_new(NewCell->plist_cell, sizeof(pinrec) * act->kind->numpins);
/* p2c: gates.text, line 627:
 * Warning: Symbol 'NEWASM_NEW' is not defined [221] */
  NewCell->plist_cell[0].standalone = false;
  NewCell->plist_cell[0].active = true;
  NewCell->plist_cell[1].standalone = false;
  NewCell->plist_cell[1].active = true;
  NewCell->plist_cell[2].standalone = false;
  NewCell->plist_cell[2].active = true;
  NewIC = (ICPfet *)Malloc(sizeof(ICPfet));
  NewIC->InitRatio = 1.0;
  NewIC->InitVt = 0.9;
  NewIC->InitLeakage = 1.66e-17;
  NewIC->InitFitFactor = 3.33;
  NewIC->InitKtqn = 0.04;
  NewIC->InitCgs = 10e-15;
  NewIC->InitCgd = 10e-15;
  NewIC->InitCgb = 10e-15;
  NewIC->InitCdb = 10e-15;
  NewIC->InitCsb = 10e-15;
  NewIC->InitEarly = 50.0;
  GetCnf_Pfet4(NewIC);
  NewCell->Ainfo = (Anyptr)NewIC;
  act->kind->info = (Anyptr)NewCell;
}  /*Pfet4_Initlib*/



Local boolean Pfetcheck(Inst, Attrnum)
log_grec *Inst;
long Attrnum;
{
  boolean Result;
  PfetConst *PfetVar;
  instlist *A_Gate;

  Result = true;
  A_Gate = (instlist *)Inst->info;
  PfetVar = (PfetConst *)A_Gate->InstVar;
  switch (Attrnum) {

  case N_Ratio:
    if (Inst->attr[N_Ratio - 1].UU.r < 0 || Inst->attr[N_Ratio - 1].blnk ||
	Inst->attr[N_Ratio - 1].UU.r > 1e20)
      Result = false;
    else {
      PfetVar->Kp1 = Inst->attr[N_Ratio - 1].UU.r * Inst->attr[N_Leakage - 1].UU.r;
      PfetVar->Kp4 = PfetVar->Kp1 *
		     Expo(PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      PfetVar->TwoKp4 = 2 * PfetVar->Kp4;
    }
    break;

  case N_Vt:
    if (Inst->attr[N_Vt - 1].UU.r < -Vdd || Inst->attr[N_Vt - 1].UU.r > Vdd ||
	Inst->attr[N_Vt - 1].blnk || Inst->attr[N_Vt - 1].UU.r == 0)
      Result = false;
    else {
      PfetVar->Kp4 = PfetVar->Kp1 *
		     Expo(PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      PfetVar->TwoKp4 = 2 * PfetVar->Kp4;
      PfetVar->Kp5 = Expo(-PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r);
    }
    break;

  case N_Leakage:
    if ((unsigned)Inst->attr[N_Leakage - 1].UU.r > 1 ||
	Inst->attr[N_Leakage - 1].blnk)
      Result = false;
    else {
      PfetVar->Kp1 = Inst->attr[N_Ratio - 1].UU.r * Inst->attr[N_Leakage - 1].UU.r;
      PfetVar->Kp4 = PfetVar->Kp1 *
		     Expo(PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      PfetVar->TwoKp4 = 2 * PfetVar->Kp4;
    }
    break;

  case N_FitFactor:
    if ((unsigned)Inst->attr[N_FitFactor - 1].UU.r > 20 ||
	Inst->attr[N_FitFactor - 1].blnk)
      Result = false;
    else
      PfetVar->Kp3 = Inst->attr[N_FitFactor - 1].UU.r;
    break;

  case N_Ktqn:
    if ((unsigned)Inst->attr[N_Ktqn - 1].UU.r > 1 || Inst->attr[N_Ktqn - 1].blnk)
      Result = false;
    else {
      PfetVar->V0 = Inst->attr[N_Ktqn - 1].UU.r;
      PfetVar->Kp2 = 1 / PfetVar->V0;
      PfetVar->Kp4 = PfetVar->Kp1 *
		     Expo(PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      PfetVar->TwoKp4 = 2 * PfetVar->Kp4;
      PfetVar->Kp5 = Expo(-PfetVar->Kp2 * Inst->attr[N_Vt - 1].UU.r);
    }
    break;

  case N_Early:
    if (Inst->attr[N_Early - 1].UU.r <= 0 || Inst->attr[N_Early - 1].UU.r > VTooBig)
      Result = false;
    else {
      if (Inst->attr[N_Early - 1].blnk)
	PfetVar->InvVcross = 0.0;
      else
	PfetVar->InvVcross = 1 / Inst->attr[N_Early - 1].UU.r;
    }
    break;
  }
  return Result;
}


Local Void Attr_Pfet4(act)
log_32_action *act;
{
  long Attrnum;

  Attrnum = act->pin;
  if (Attrnum >= Cstart && Attrnum <= Cstart + 8) {
    Capattrinsert((long)Cstart, Attrnum, act->inst, &act->ok);
    return;
  }
  if ((unsigned long)Attrnum < 32 &&
      ((1L << Attrnum) & ((1L << N_Cgs) | (1L << N_Cgd))) != 0)
    act->ok = Crange(act->inst, Attrnum, true);
  else
    act->ok = Pfetcheck(act->inst, Attrnum);
}


Local Void Newgate_Pfet4(act)
log_32_action *act;
{
  PfetConst *PfetVar;
  instlist *A_Gate;
  Cell_rec *Cellptr;
  ICPfet *ICptr;
  log_grec *WITH;

  Cellptr = (Cell_rec *)act->inst->kind->info;
  ICptr = (ICPfet *)Cellptr->Ainfo;
  PfetVar = (PfetConst *)Malloc(sizeof(PfetConst));
  PfetVar->V0 = ICptr->InitKtqn;   /*kt/qn*/
  PfetVar->Kp1 = ICptr->InitRatio * ICptr->InitLeakage;
      /*Diode leakage current*(W/L)*/
  PfetVar->Kp2 = 1 / PfetVar->V0;   /*qn/kt*/
  PfetVar->Kp3 = ICptr->InitFitFactor;   /*Tanch(Kp3*x) ~=.99 when x=1*/
  PfetVar->Kp4 = PfetVar->Kp1 * Expo(PfetVar->Kp2 * ICptr->InitVt) /
		 (ICptr->InitVt * ICptr->InitVt);
      /*Kp1*Expo(Kp2*Vtp)/(Vtp*Vtp)*/
  PfetVar->TwoKp4 = 2 * PfetVar->Kp4;   /*2*Kp4*/
  PfetVar->Kp5 = Expo(-PfetVar->Kp2 * ICptr->InitVt);   /*Expo(-Kp2*Vtp)*/
  if (ICptr->InitEarly == 0)
    PfetVar->InvVcross = 0.0;
  else
    PfetVar->InvVcross = 1 / ICptr->InitEarly;
  PfetVar->TauCgs = 0.0;
  PfetVar->TauCgd = 0.0;
  A_Gate = (instlist *)act->inst->info;
  A_Gate->InstVar = (Anyptr)PfetVar;
  CapInit(act->inst);   /*6*/
  A_Gate->Pininfo[0].Cap = ICptr->InitCgb;
  A_Gate->Pininfo[1].Cap = ICptr->InitCdb;
  A_Gate->Pininfo[2].Cap = ICptr->InitCsb;
  WITH = act->inst;
  WITH->attr[N_Cgd - 1].UU.r = ICptr->InitCgd;   /*Cgd*/
  WITH->attr[N_Cgd - 1].blnk = false;
  WITH->attr[N_Cgd - 1].changed = true;
  WITH->attr[N_Cgs - 1].UU.r = ICptr->InitCgs;   /*Cgs*/
  WITH->attr[N_Cgs - 1].blnk = false;
  WITH->attr[N_Cgs - 1].changed = true;
  WITH->attr[N_Ratio - 1].UU.r = ICptr->InitRatio;   /*W/L*/
  WITH->attr[N_Ratio - 1].blnk = false;
  WITH->attr[N_Ratio - 1].changed = true;
  WITH->attr[N_Vt - 1].UU.r = ICptr->InitVt;   /*Vt*/
  WITH->attr[N_Vt - 1].blnk = false;
  WITH->attr[N_Vt - 1].changed = true;
  WITH->attr[N_Leakage - 1].UU.r = ICptr->InitLeakage;   /*leakage current*/
  WITH->attr[N_Leakage - 1].blnk = false;
  WITH->attr[N_Leakage - 1].changed = true;
  WITH->attr[N_FitFactor - 1].UU.r = ICptr->InitFitFactor;   /*fit factor*/
  WITH->attr[N_FitFactor - 1].blnk = false;
  WITH->attr[N_FitFactor - 1].changed = true;
  WITH->attr[N_Ktqn - 1].UU.r = ICptr->InitKtqn;   /*Kt/qn*/
  WITH->attr[N_Ktqn - 1].blnk = false;
  WITH->attr[N_Ktqn - 1].changed = true;
  WITH->attr[N_Early - 1].UU.r = ICptr->InitEarly;
  WITH->attr[N_Early - 1].blnk = false;
  WITH->attr[N_Early - 1].changed = true;
}

Local Void Copygate_Pfet4(act)
log_32_action *act;
{
  PfetConst *PfetVar, *Old_Pfetvar;
  instlist *A_Gate, *A_Oldgate;

  A_Oldgate = (instlist *)Logglobals->actgate2->info;
  Old_Pfetvar = (PfetConst *)A_Oldgate->InstVar;
  PfetVar = (PfetConst *)Malloc(sizeof(PfetConst));
  *PfetVar = *Old_Pfetvar;
  A_Gate = (instlist *)act->inst->info;
  A_Gate->InstVar = (Anyptr)PfetVar;
  CapCopy(act->inst);   /*6*/
}

Local Void Dispose_Pfet4(act)
log_32_action *act;
{
  PfetConst *PfetVar;
  instlist *A_Gate;

  A_Gate = (instlist *)act->inst->info;
  PfetVar = (PfetConst *)A_Gate->InstVar;
  Free(PfetVar);
  CapDispose(act->inst);   /*7*/
}

Local Void Readgate_Pfet4(act)
log_32_action *act;
{
  PfetConst *PfetVar;
  instlist *A_Gate;
  Cell_rec *Cellptr;
  ICPfet *ICptr;
  log_grec *WITH;

  Capattrread((long)Cstart, act->inst);
  A_Gate = (instlist *)act->inst->info;
  PfetVar = (PfetConst *)A_Gate->InstVar;
  if (!Oldmodel) {
    Cellptr = (Cell_rec *)act->inst->kind->info;
    ICptr = (ICPfet *)Cellptr->Ainfo;
    A_Gate->Pininfo[0].Cap = ICptr->InitCgb;
    A_Gate->Pininfo[1].Cap = ICptr->InitCdb;
    A_Gate->Pininfo[2].Cap = ICptr->InitCsb;
    WITH = act->inst;
    WITH->attr[N_Cgd - 1].UU.r = ICptr->InitCgd;   /*Cgd*/
    WITH->attr[N_Cgd - 1].blnk = false;
    WITH->attr[N_Cgd - 1].changed = true;
    WITH->attr[N_Cgs - 1].UU.r = ICptr->InitCgs;   /*Cgs*/
    WITH->attr[N_Cgs - 1].blnk = false;
    WITH->attr[N_Cgs - 1].changed = true;
    WITH->attr[N_Vt - 1].UU.r = ICptr->InitVt;   /*Vt*/
    WITH->attr[N_Vt - 1].blnk = false;
    WITH->attr[N_Vt - 1].changed = true;
    WITH->attr[N_Leakage - 1].UU.r = ICptr->InitLeakage;   /*leakage current*/
    WITH->attr[N_Leakage - 1].blnk = false;
    WITH->attr[N_Leakage - 1].changed = true;
    WITH->attr[N_FitFactor - 1].UU.r = ICptr->InitFitFactor;   /*fit factor*/
    WITH->attr[N_FitFactor - 1].blnk = false;
    WITH->attr[N_FitFactor - 1].changed = true;
    WITH->attr[N_Ktqn - 1].UU.r = ICptr->InitKtqn;   /*Kt/qn*/
    WITH->attr[N_Ktqn - 1].blnk = false;
    WITH->attr[N_Ktqn - 1].changed = true;
    WITH->attr[N_Early - 1].UU.r = ICptr->InitEarly;
    WITH->attr[N_Early - 1].blnk = false;
    WITH->attr[N_Early - 1].changed = true;
  }
  WITH = act->inst;
  PfetVar->V0 = WITH->attr[N_Ktqn - 1].UU.r;   /*kt/qn*/
  PfetVar->Kp1 = WITH->attr[N_Ratio - 1].UU.r * WITH->attr[N_Leakage - 1].UU.r;
      /*Diode leakage current*(W/L)*/
  PfetVar->Kp2 = 1 / PfetVar->V0;   /*qn/kt*/
  PfetVar->Kp3 = WITH->attr[N_FitFactor - 1].UU.r;
      /*Tanch(Kp3*x) ~=.99 when x=1*/
  PfetVar->Kp4 = PfetVar->Kp1 * Expo(PfetVar->Kp2 * WITH->attr[N_Vt - 1].UU.r) /
		 (WITH->attr[N_Vt - 1].UU.r * WITH->attr[N_Vt - 1].UU.r);
      /*Kp1*Expo(Kp2*Vtp)/(Vtp*Vtp)*/
  PfetVar->TwoKp4 = 2 * PfetVar->Kp4;   /*2*Kp4*/
  PfetVar->Kp5 = Expo(-PfetVar->Kp2 * WITH->attr[N_Vt - 1].UU.r);
      /*Expo(-Kp2*Vtp)*/
  if (WITH->attr[N_Early - 1].blnk)
    PfetVar->InvVcross = 0.0;
  else
    PfetVar->InvVcross = 1 / WITH->attr[N_Early - 1].UU.r;
  PfetVar->TauCgs = 0.0;
  PfetVar->TauCgd = 0.0;
}

Local Void Probe_Pfet4(act)
log_32_action *act;
{
  instlist *A_Gate;
  Ext_rec *Node1Ptr, *Node2Ptr, *Node3Ptr;
  double Node1, Node2, Node3, Pin1, Pin2, Pin3, d1, d2;
  Char Name2[256], Name3[256];
  log_grec *WITH;
  Char STR1[18];
  Char STR2[12];
  Char STR3[256];
  Char STR4[256];

  WITH = act->inst;
  A_Gate = (instlist *)WITH->info;
  Pin1 = A_Gate->Pininfo[0].Vmem;
  Pin2 = A_Gate->Pininfo[1].Vmem;
  Pin3 = A_Gate->Pininfo[2].Vmem;
  Node1Ptr = (Ext_rec *)WITH->pin[0]->info;
  Node2Ptr = (Ext_rec *)WITH->pin[1]->info;
  Node3Ptr = (Ext_rec *)WITH->pin[2]->info;
  Node1 = Node1Ptr->ltimestep;
  Node2 = Node2Ptr->ltimestep;
  Node3 = Node3Ptr->ltimestep;
  if (Node1 != Pin1 || Node2 != Pin2 || Node3 != Pin3 || Node1 == Notyet ||
      Node2 == Notyet || Node3 == Notyet) {
    sprintf(STR1, "%s <undefined>", GateName);
    Scoreboard(STR1, (long)MessGate1);
    Scoreboard("$", (long)MessGate2);
    Scoreboard("$", (long)MessGate3);
    return;
  }
  d1 = Pin2 - Pin3;
  if (d1 >= 0) {
    d2 = Pin2 - Pin1;
    strcpy(Name2, " (source)");
    strcpy(Name3, " (drain)");
  } else {
    d1 = -d1;
    d2 = Pin3 - Pin1;
    strcpy(Name2, " (drain)");
    strcpy(Name3, " (source)");
  }
  switch (Logglobals->probepin) {

  case 0:
    Scoreboard(GateName, (long)MessGate1);
    break;

  case 1:
    sprintf(STR2, "%s (gate)", GateName);
    Scoreboard(STR2, (long)MessGate1);
    break;

  case 2:
    sprintf(STR3, "%s%s", GateName, Name2);
    Scoreboard(STR3, (long)MessGate1);
    break;

  case 3:
    sprintf(STR3, "%s%s", GateName, Name3);
    Scoreboard(STR3, (long)MessGate1);
    break;
  }
  sprintf(STR3, "Vsd = %s", ProbeVoltStr(STR4, d1));
  Scoreboard(STR3, (long)MessGate2);
  sprintf(STR3, "Vsg = %s", ProbeVoltStr(STR4, d2));
  Scoreboard(STR3, (long)MessGate3);
}



/*==========================================================================*/
/*                 Library Listing For Cell PFET4                           */
/*==========================================================================*/


Void LOG_PFET4_INITLIB_32(act)
log_32_action *act;
{
  /*Main Procedure*/


  switch (act->action) {

  case log_act_newkind:
    initlib_PFET4(act);
    break;

  case log_act_ex:
    Ex_PFET4(act);
    break;

  case log_act_update:   /*8*/
    CapUpdate(act->inst);
    break;

  case log_act_pass1:
    Cappass1(act->inst);
    break;

  case log_act_pass2:
    Cappass2(act->inst);
    break;

  case log_act_attrchange:
    Attr_Pfet4(act);
    break;

  case log_act_reset:
    CapReset(act->inst);
    break;

  case log_act_newgate:
    Newgate_Pfet4(act);
    break;

  case log_act_copygate:
    Copygate_Pfet4(act);
    break;

  case log_act_disposegate:
    Dispose_Pfet4(act);
    break;

  case log_act_openconfig:
    Capattrload((long)Cstart, act->inst, act->ok);
    break;

  case log_act_readgate:
    Readgate_Pfet4(act);
    break;

  case log_act_writegate:
    Capattrwrite((long)Cstart, act->inst);
    break;

  case log_act_Iin:
    Iin_PFET4(act);
    break;

  case log_act_probe:
    Probe_Pfet4(act);
    break;
  }
}

#undef Cstart
#undef N_Cgd
#undef N_Cgs
#undef N_Ratio
#undef N_Vt
#undef N_Leakage
#undef N_FitFactor
#undef N_Ktqn
#undef N_Early
#undef GateName


#define NFET4_G
#include "nfet4.h"


#define Cstart          1
#define N_Cgd           10
#define N_Cgs           11
#define N_Ratio         12
#define N_Vt            13
#define N_Leakage       14
#define N_FitFactor     15
#define N_Ktqn          16
#define N_Early         17

#define Gatename        "NFET"


typedef struct NfetConst {
  double V0;   /*kt/qn*/
  double Kn1;   /*nfet diode leakage current*(W/L)*/
  double Kn2;   /*qn/kt*/
  double Kn3;   /*Tanch(kn3*x) ~=.99 when x=1*/
  double Kn4;   /*Kn1*Expo(Kn2*Vtn)/(Vtn*Vtn)*/
  double TwoKn4;   /*2*Kn4*/
  double Kn5;   /*Expo(-Kn2*Vtn)*/
  double InvVcross;   /*Inverse Early Voltage*/
  double TauCgd;   /*Tau for capacitor*/
  double TauCgs;   /*Tau for capacitor*/
} NfetConst;


typedef struct ICNfet {
  double InitRatio, InitVt, InitLeakage, InitFitFactor, InitKtqn, InitCgd,
	 InitCgs, InitCgb, InitCdb, InitCsb, InitEarly;
} ICNfet;



/****************************************************************************/
/*            Procedure To Simulate Cell In Algorithm                        */
/****************************************************************************/


Local Void Ex_NFET4(act)
log_32_action *act;
{
  double MinI2, dI2dV1, dI2dV2, dI2dV3, MinI3, dI3dV1, dI3dV2, dI3dV3, MinI1,
	 dI1dV1, dI1dV2, dI1dV3;
  Ext_rec *Pin1Ptr, *Pin2Ptr, *Pin3Ptr;
  long Pin1, Pin2, Pin3;
  double VPin1, VPin2, VPin3, VMem1, VMem2, VMem3;   /*1*/
  double ICap;
  NfetConst *NfetVar;
  instlist *A_Gate;
  double Vds, Vgs, Vgd, ExpVgs, ExpVgd, ExpVds, BlendVgs, BlendVgd, BlendVds,
	 InvBlendVgs, InvBlendVgd, InvBlendVds, dBlendVgsdS, dBlendVgddD,
	 dBlendVdsdS, Temp1, Temp2, Temp4, OldIforward, Iforward, OldIreverse,
	 Ireverse, dIforwarddD, dIforwarddG, dIforwarddS, dIreversedD,
	 dIreversedG, dIreversedS, Finv, F, dFinvdD, dFinvdS, dFinvdG,
	 Saturate, dTanch, Itotal, Ids, dIdsdG, dIdsdS, dIdsdD, Resforward,
	 Resreverse, dSaturatedD_Iforward, dSaturatedG_Iforward,
	 dSaturatedS_Iforward, dSaturatedS_Ireverse, dSaturatedG_Ireverse,
	 dSaturatedD_Ireverse, dIforwarddG_Saturate, dIforwarddD_Saturate,
	 dIforwarddS_Saturate, dIreversedG_Saturate, dIreversedD_Saturate,
	 dIreversedS_Saturate, Vgs_InvBlendVgs, Vgs_dBlendVgsds, Kn1_ExpVgs,
	 Kn1_ExpVgs_BlendVgs, Kn4_Vgs, Vgd_InvBlendVgd, Vgd_dBlendVgddD,
	 Kn1_ExpVgd, Kn1_ExpVgd_BlendVgd, Kn4_Vgd, ResForward_InvBlendVds,
	 BlendVds_ResReverse, Vgs_InvBlendVgs_dBlendVdsdS,
	 InvBlendVgd_BlendVds, Vgd_BlendVds_dBlendVgddD,
	 Vgd_InvBlendVgd_dBlendVdsdS, InvBlendVgs_InvBlendVds,
	 Vgs_InvBlendVds_dBlendVgsdS, Vds_dTanch_Temp3_Itotal,
	 F_dTanch_Itotal, dSaturatedD_Itotal, dSaturatedG_Itotal,
	 dSaturatedS_Itotal;
/* p2c: gates.text, line 1996: Note:
 * Line breaker spent 3.0+1.00 seconds, 5000 tries on line 1078 [251] */
  log_grec *WITH;
  MatRec *WITH1;

  WITH = act->inst;
  A_Gate = (instlist *)WITH->info;
  NfetVar = (NfetConst *)A_Gate->InstVar;
  Pin1Ptr = (Ext_rec *)WITH->pin[0]->info;
  Pin2Ptr = (Ext_rec *)WITH->pin[1]->info;
  Pin3Ptr = (Ext_rec *)WITH->pin[2]->info;
  Pin1 = Pin1Ptr->nodenum;
  Pin2 = Pin2Ptr->nodenum;
  Pin3 = Pin3Ptr->nodenum;
  VPin1 = Pin1Ptr->last;
  VPin2 = Pin2Ptr->last;
  VPin3 = Pin3Ptr->last;
  VMem1 = A_Gate->Pininfo[0].Vmem;   /*2*/
  VMem2 = A_Gate->Pininfo[1].Vmem;
  VMem3 = A_Gate->Pininfo[2].Vmem;
  Vds = VPin2 - VPin3;
  Vgs = VPin1 - VPin3;
  Vgd = VPin1 - VPin2;
  if (fabs(Vds) < 0.000708) {
    ExpVgs = Expo(NfetVar->Kn2 * Vgs);
    ExpVgd = Expo(NfetVar->Kn2 * Vgd);
    Temp1 = NfetVar->Kn5 * ExpVgs;
    BlendVgs = 1 / (1 + Temp1);
    InvBlendVgs = 1 - BlendVgs;
    dBlendVgsdS = Temp1 * NfetVar->Kn2 * BlendVgs * BlendVgs;
    Temp2 = NfetVar->Kn5 * ExpVgd;
    BlendVgd = 1 / (1 + Temp2);
    InvBlendVgd = 1 - BlendVgd;
    dBlendVgddD = Temp2 * NfetVar->Kn2 * BlendVgd * BlendVgd;
    ExpVds = Expo(1000000L * Vds);
    BlendVds = 1 / (1 + ExpVds);
    InvBlendVds = 1 - BlendVds;
    dBlendVdsdS = ExpVds * BlendVds * BlendVds;
    dBlendVdsdS = 1000000L * dBlendVdsdS;
    Temp4 = Vds * NfetVar->InvVcross;
    Resforward = 1 + Temp4;
    Resreverse = 1 - Temp4;
    Kn4_Vgs = NfetVar->Kn4 * Vgs;
    Kn1_ExpVgs = NfetVar->Kn1 * ExpVgs;
    Kn1_ExpVgs_BlendVgs = Kn1_ExpVgs * BlendVgs;
    Vgs_InvBlendVgs = Vgs * InvBlendVgs;
    OldIforward = Kn1_ExpVgs_BlendVgs + Vgs_InvBlendVgs * Kn4_Vgs;
    ResForward_InvBlendVds = Resforward * InvBlendVds;
    Iforward = ResForward_InvBlendVds * OldIforward;
    dIforwarddG = ResForward_InvBlendVds *
	(Kn1_ExpVgs_BlendVgs * NfetVar->Kn2 - Kn1_ExpVgs * dBlendVgsdS +
	 NfetVar->TwoKn4 * Vgs_InvBlendVgs + Kn4_Vgs * Vgs * dBlendVgsdS);
    dIforwarddD = (dBlendVdsdS * Resforward +
		   NfetVar->InvVcross * InvBlendVds) * OldIforward;
    dIforwarddS = -(dIforwarddG + dIforwarddD);
    Kn1_ExpVgd = NfetVar->Kn1 * ExpVgd;
    Kn1_ExpVgd_BlendVgd = Kn1_ExpVgd * BlendVgd;
    Kn4_Vgd = NfetVar->Kn4 * Vgd;
    Vgd_InvBlendVgd = Vgd * InvBlendVgd;
    OldIreverse = Kn1_ExpVgd_BlendVgd + Vgd_InvBlendVgd * Kn4_Vgd;
    BlendVds_ResReverse = BlendVds * Resreverse;
    Ireverse = BlendVds_ResReverse * OldIreverse;
    dIreversedG = BlendVds_ResReverse *
	(Kn1_ExpVgd_BlendVgd * NfetVar->Kn2 - Kn1_ExpVgd * dBlendVgddD +
	 NfetVar->TwoKn4 * Vgd_InvBlendVgd + Kn4_Vgd * Vgd * dBlendVgddD);
    dIreversedS = (dBlendVdsdS * Resreverse + BlendVds * NfetVar->InvVcross) *
		  OldIreverse;
    dIreversedD = -(dIreversedG + dIreversedS);
    Finv = NfetVar->V0 + Vgs_InvBlendVgs * InvBlendVds + Vgd_InvBlendVgd * BlendVds;
    F = NfetVar->Kn3 / Finv;
    Vgs_InvBlendVgs_dBlendVdsdS = Vgs_InvBlendVgs * dBlendVdsdS;
    InvBlendVgd_BlendVds = InvBlendVgd * BlendVds;
    Vgd_BlendVds_dBlendVgddD = Vgd * BlendVds * dBlendVgddD;
    Vgd_InvBlendVgd_dBlendVdsdS = Vgd_InvBlendVgd * dBlendVdsdS;
    InvBlendVgs_InvBlendVds = InvBlendVgs * InvBlendVds;
    Vgs_InvBlendVds_dBlendVgsdS = Vgs * InvBlendVds * dBlendVgsdS;
    dFinvdD = Vgs_InvBlendVgs_dBlendVdsdS - InvBlendVgd_BlendVds -
	      Vgd_BlendVds_dBlendVgddD - Vgd_InvBlendVgd_dBlendVdsdS;
    dFinvdS = Vgd_InvBlendVgd_dBlendVdsdS - Vgs_InvBlendVds_dBlendVgsdS -
	      Vgs_InvBlendVgs_dBlendVdsdS - InvBlendVgs_InvBlendVds;
    dFinvdG = InvBlendVgs_InvBlendVds + Vgs_InvBlendVds_dBlendVgsdS +
	      InvBlendVgd_BlendVds + Vgd_BlendVds_dBlendVgddD;
    Saturate = Tanch(Vds * F);
    Itotal = Iforward + Ireverse;
    F_dTanch_Itotal = F * Itotal * (1 - Saturate * Saturate);
    Vds_dTanch_Temp3_Itotal = -(Vds * F_dTanch_Itotal / Finv);
    dSaturatedD_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdD + F_dTanch_Itotal;
    dSaturatedG_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdG;
    dSaturatedS_Itotal = Vds_dTanch_Temp3_Itotal * dFinvdS - F_dTanch_Itotal;
    Ids = Itotal * Saturate;
    dIdsdG = (dIforwarddG + dIreversedG) * Saturate + dSaturatedG_Itotal;
    dIdsdS = (dIforwarddS + dIreversedS) * Saturate + dSaturatedS_Itotal;
    dIdsdD = (dIforwarddD + dIreversedD) * Saturate + dSaturatedD_Itotal;
  } else {
    if (Vds > 0) {
      ExpVgs = Expo(NfetVar->Kn2 * Vgs);
      Temp1 = NfetVar->Kn5 * ExpVgs;
      BlendVgs = 1 / (1 + Temp1);
      InvBlendVgs = 1 - BlendVgs;
      dBlendVgsdS = Temp1 * NfetVar->Kn2 * BlendVgs * BlendVgs;
      Vgs_InvBlendVgs = Vgs * InvBlendVgs;
      Vgs_dBlendVgsds = Vgs * dBlendVgsdS;
      Kn4_Vgs = NfetVar->Kn4 * Vgs;
      Kn1_ExpVgs = NfetVar->Kn1 * ExpVgs;
      Kn1_ExpVgs_BlendVgs = Kn1_ExpVgs * BlendVgs;
      Resforward = 1 + Vds * NfetVar->InvVcross;
      OldIforward = Kn1_ExpVgs_BlendVgs + Vgs_InvBlendVgs * Kn4_Vgs;
      Iforward = Resforward * OldIforward;
      Finv = 1 / (NfetVar->V0 + Vgs_InvBlendVgs);
      F = NfetVar->Kn3 * Finv;
      Saturate = Tanch(Vds * F);
      dIforwarddG_Saturate = Resforward * Saturate *
	  (Kn1_ExpVgs_BlendVgs * NfetVar->Kn2 - Kn1_ExpVgs * dBlendVgsdS +
	   NfetVar->TwoKn4 * Vgs_InvBlendVgs + Kn4_Vgs * Vgs_dBlendVgsds);
      dIforwarddD_Saturate = NfetVar->InvVcross * OldIforward * Saturate;
      dIforwarddS_Saturate = -(dIforwarddG_Saturate + dIforwarddD_Saturate);
      dTanch = 1 - Saturate * Saturate;
      dSaturatedD_Iforward = F * dTanch * Iforward;
      dSaturatedG_Iforward = -Vds * dSaturatedD_Iforward * Finv *
			      (InvBlendVgs + Vgs_dBlendVgsds);
      dSaturatedS_Iforward = -(dSaturatedG_Iforward + dSaturatedD_Iforward);
      Ids = Iforward * Saturate;
      dIdsdG = dIforwarddG_Saturate + dSaturatedG_Iforward;
      dIdsdS = dIforwarddS_Saturate + dSaturatedS_Iforward;
      dIdsdD = dIforwarddD_Saturate + dSaturatedD_Iforward;
    } else {
      ExpVgd = Expo(NfetVar->Kn2 * Vgd);
      Temp2 = NfetVar->Kn5 * ExpVgd;
      BlendVgd = 1 / (1 + Temp2);
      InvBlendVgd = 1 - BlendVgd;
      dBlendVgddD = Temp2 * NfetVar->Kn2 * BlendVgd * BlendVgd;
      Vgd_InvBlendVgd = Vgd * InvBlendVgd;
      Vgd_dBlendVgddD = Vgd * dBlendVgddD;
      Kn1_ExpVgd = NfetVar->Kn1 * ExpVgd;
      Kn1_ExpVgd_BlendVgd = Kn1_ExpVgd * BlendVgd;
      Kn4_Vgd = NfetVar->Kn4 * Vgd;
      Resreverse = 1 - Vds * NfetVar->InvVcross;
      OldIreverse = Kn1_ExpVgd_BlendVgd + Vgd_InvBlendVgd * Kn4_Vgd;
      Ireverse = Resreverse * OldIreverse;
      Finv = 1 / (NfetVar->V0 + Vgd_InvBlendVgd);
      F = NfetVar->Kn3 * Finv;
      Saturate = Tanch(Vds * F);
      dIreversedG_Saturate = Resreverse * Saturate *
	  (Kn1_ExpVgd_BlendVgd * NfetVar->Kn2 - Kn1_ExpVgd * dBlendVgddD +
	   NfetVar->TwoKn4 * Vgd_InvBlendVgd + Kn4_Vgd * Vgd_dBlendVgddD);
      dIreversedS_Saturate = NfetVar->InvVcross * OldIreverse * Saturate;
      dIreversedD_Saturate = -(dIreversedG_Saturate + dIreversedS_Saturate);
      dTanch = 1 - Saturate * Saturate;
      dSaturatedS_Ireverse = -F * dTanch * Ireverse;
      dSaturatedG_Ireverse = Vds * dSaturatedS_Ireverse * Finv *
			     (InvBlendVgd + Vgd_dBlendVgddD);
      dSaturatedD_Ireverse = -(dSaturatedG_Ireverse + dSaturatedS_Ireverse);
      Ids = Ireverse * Saturate;
      dIdsdG = dIreversedG_Saturate + dSaturatedG_Ireverse;
      dIdsdS = dIreversedS_Saturate + dSaturatedS_Ireverse;
      dIdsdD = dIreversedD_Saturate + dSaturatedD_Ireverse;
    }
  }
  MinI3 = Ids;
  MinI2 = -Ids;
  dI2dV1 = dIdsdG;
  dI3dV1 = -dIdsdG;
  dI2dV3 = dIdsdS;
  dI3dV3 = -dIdsdS;
  dI2dV2 = dIdsdD;
  dI3dV2 = -dIdsdD;
  MinI1 = 0.0;
  dI1dV1 = 0.0;
  dI1dV2 = 0.0;
  dI1dV3 = 0.0;
  if (WITH->attr[N_Cgd - 1].UU.r != 0) {   /*C gate-drain*/
    if (ChangedDt)
      NfetVar->TauCgd = WITH->attr[N_Cgd - 1].UU.r / Dt;
    ICap = NfetVar->TauCgd * (VMem1 - VMem2 - Vgd);   /*4*/
    MinI2 -= ICap;
    dI2dV1 -= NfetVar->TauCgd;
    dI2dV2 += NfetVar->TauCgd;
    MinI1 += ICap;
    dI1dV1 += NfetVar->TauCgd;
    dI1dV2 -= NfetVar->TauCgd;
  }
  /*3*/
  if (WITH->attr[N_Cgs - 1].UU.r != 0) {   /*C gate-source*/
    if (ChangedDt)
      NfetVar->TauCgs = WITH->attr[N_Cgs - 1].UU.r / Dt;
    ICap = NfetVar->TauCgs * (VMem1 - VMem3 - Vgs);
    MinI3 -= ICap;
    dI3dV1 -= NfetVar->TauCgs;
    dI3dV3 += NfetVar->TauCgs;
    MinI1 += ICap;
    dI1dV1 += NfetVar->TauCgs;
    dI1dV3 -= NfetVar->TauCgs;
  }
  WITH1 = System;
  if (Pin1 != 0) {
    WITH1->Mat[Pin1][SysCol] += MinI1;
    WITH1->Mat[Pin1][Pin1] += dI1dV1;
    WITH1->Mat[Pin1][Pin2] += dI1dV2;
    WITH1->Mat[Pin1][Pin3] += dI1dV3;
    WITH1->MatNZ[Pin1][SysCol] = true;
    WITH1->MatNZ[Pin1][Pin1] = true;
    WITH1->MatNZ[Pin1][Pin2] = true;
    WITH1->MatNZ[Pin1][Pin3] = true;
  }
  if (Pin2 != 0) {
    WITH1->Mat[Pin2][SysCol] += MinI2;
    WITH1->Mat[Pin2][Pin1] += dI2dV1;
    WITH1->Mat[Pin2][Pin2] += dI2dV2;
    WITH1->Mat[Pin2][Pin3] += dI2dV3;
    WITH1->MatNZ[Pin2][SysCol] = true;
    WITH1->MatNZ[Pin2][Pin1] = true;
    WITH1->MatNZ[Pin2][Pin2] = true;
    WITH1->MatNZ[Pin2][Pin3] = true;
  }
  if (Pin3 != 0) {
    WITH1->Mat[Pin3][SysCol] += MinI3;
    WITH1->Mat[Pin3][Pin1] += dI3dV1;
    WITH1->Mat[Pin3][Pin2] += dI3dV2;
    WITH1->Mat[Pin3][Pin3] += dI3dV3;
    WITH1->MatNZ[Pin3][SysCol] = true;
    WITH1->MatNZ[Pin3][Pin1] = true;
    WITH1->MatNZ[Pin3][Pin2] = true;
    WITH1->MatNZ[Pin3][Pin3] = true;
  }
  Capex(act->inst);   /*5*/
}  /*Ex_NFET4*/


/****************************************************************************/
/*          Procedure To Supply Current  Out of a Cell To Current Meters    */
/****************************************************************************/

Local Void Iin_NFET4(act)
log_32_action *act;
{
  Ext_rec *Pin1Ptr, *Pin2Ptr, *Pin3Ptr;
  double VPin1, VPin2, VPin3, VMem1, VMem2, VMem3;
  NfetConst *NfetVar;
  instlist *A_Gate;
  double Vds, Vgs, Vgd, ExpVgs, ExpVgd, BlendVgs, BlendVgd, BlendVds,
	 InvBlendVgs, InvBlendVgd, InvBlendVds, Temp1, Tempf, Tempb, Iforward,
	 Ireverse, Ids;
  log_grec *WITH1;

  switch (act->pin) {

  case 1:
    WITH1 = act->inst;
    Pin1Ptr = (Ext_rec *)WITH1->pin[0]->info;
    Pin2Ptr = (Ext_rec *)WITH1->pin[1]->info;
    Pin3Ptr = (Ext_rec *)WITH1->pin[2]->info;
    A_Gate = (instlist *)WITH1->info;
    NfetVar = (NfetConst *)A_Gate->InstVar;
    VMem1 = A_Gate->Pininfo[0].Vmem;
    VMem2 = A_Gate->Pininfo[1].Vmem;
    VMem3 = A_Gate->Pininfo[2].Vmem;
    VPin1 = Pin1Ptr->now;
    VPin2 = Pin2Ptr->now;
    VPin3 = Pin3Ptr->now;
    if (WITH1->attr[N_Cgd - 1].UU.r != 0)   /*C gate-drain*/
      act->Iin = NfetVar->TauCgd * (VPin1 - VPin2 - VMem1 + VMem2) + act->Iin;
    if (WITH1->attr[N_Cgs - 1].UU.r != 0)   /*C gate-source*/
      act->Iin = NfetVar->TauCgs * (VPin1 - VPin3 - VMem1 + VMem3) + act->Iin;
    CapIin(act);
    break;

  case 2:
  case 3:
    WITH1 = act->inst;
    Pin1Ptr = (Ext_rec *)WITH1->pin[0]->info;
    Pin2Ptr = (Ext_rec *)WITH1->pin[1]->info;
    Pin3Ptr = (Ext_rec *)WITH1->pin[2]->info;
    A_Gate = (instlist *)WITH1->info;
    NfetVar = (NfetConst *)A_Gate->InstVar;
    VMem1 = A_Gate->Pininfo[0].Vmem;
    VPin1 = Pin1Ptr->now;
    VPin2 = Pin2Ptr->now;
    VPin3 = Pin3Ptr->now;
    switch (act->pin) {

    case 2:
      if (WITH1->attr[N_Cgd - 1].UU.r != 0) {
	VMem2 = A_Gate->Pininfo[1].Vmem;
	act->Iin = NfetVar->TauCgd * (VPin2 - VPin1 - VMem2 + VMem1) + act->Iin;
      }
      break;

    case 3:
      if (WITH1->attr[N_Cgs - 1].UU.r != 0) {
	VMem3 = A_Gate->Pininfo[2].Vmem;
	act->Iin = NfetVar->TauCgs * (VPin3 - VPin1 - VMem3 + VMem1) + act->Iin;
      }
      break;
    }
    Vds = VPin2 - VPin3;
    if (fabs(Vds) < 0.000708) {
      Vgs = VPin1 - VPin3;
      Vgd = VPin1 - VPin2;
      ExpVgs = Expo(NfetVar->Kn2 * Vgs);
      BlendVgs = 1 / (1 + NfetVar->Kn5 * ExpVgs);
      InvBlendVgs = 1 - BlendVgs;
      ExpVgd = Expo(NfetVar->Kn2 * Vgd);
      BlendVgd = 1 / (1 + NfetVar->Kn5 * ExpVgd);
      InvBlendVgd = 1 - BlendVgd;
      BlendVds = 1 / (1 + Expo(1000000L * Vds));
      InvBlendVds = 1 - BlendVds;
      Temp1 = Vds * NfetVar->InvVcross;
      Tempf = InvBlendVgs * Vgs;
      Tempb = InvBlendVgd * Vgd;
      Iforward = (1 + Temp1) * InvBlendVds *
	  (NfetVar->Kn1 * ExpVgs * BlendVgs + Tempf * NfetVar->Kn4 * Vgs);
      Ireverse = (1 - Temp1) * BlendVds *
	  (NfetVar->Kn1 * ExpVgd * BlendVgd + Tempb * NfetVar->Kn4 * Vgd);
      Ids = (Iforward + Ireverse) * Tanch(Vds * NfetVar->Kn3 /
		(NfetVar->V0 + Tempb * BlendVds + Tempf * InvBlendVds));
    } else {
      if (Vds > 0) {
	Vgs = VPin1 - VPin3;
	ExpVgs = Expo(NfetVar->Kn2 * Vgs);
	BlendVgs = 1 / (1 + NfetVar->Kn5 * ExpVgs);
	InvBlendVgs = 1 - BlendVgs;
	Temp1 = InvBlendVgs * Vgs;
	Ids = (1 + Vds * NfetVar->InvVcross) *
	      (NfetVar->Kn1 * ExpVgs * BlendVgs + Temp1 * NfetVar->Kn4 * Vgs) *
	      Tanch(Vds * NfetVar->Kn3 / (NfetVar->V0 + Temp1));
      } else {
	Vgd = VPin1 - VPin2;
	ExpVgd = Expo(NfetVar->Kn2 * Vgd);
	BlendVgd = 1 / (1 + NfetVar->Kn5 * ExpVgd);
	InvBlendVgd = 1 - BlendVgd;
	Temp1 = InvBlendVgd * Vgd;
	Ids = (1 - Vds * NfetVar->InvVcross) *
	      (NfetVar->Kn1 * ExpVgd * BlendVgd + Temp1 * NfetVar->Kn4 * Vgd) *
	      Tanch(Vds * NfetVar->Kn3 / (NfetVar->V0 + Temp1));
      }
    }
    switch (act->pin) {

    case 2:
      act->Iin += Ids;
      break;

    case 3:
      act->Iin -= Ids;
      break;
    }
    CapIin(act);
    break;

  default:
    printf("Data Structure Corruption\n");
    _Escape(1002);
    break;
  }
}  /*Iin_NFET4*/


Local Void GetCnf_Nfet4(NewIC)
ICNfet *NewIC;
{
  boolean Found;
  Char Arg[256], Keyword[256];
  long Dummy;
  ICNfet *WITH;
  Char *STR1;

  WITH = NewIC;
  do {
    Getcommand("NFET4", Arg, &Found);
    if (Found) {
      if (Logglobals->hook.getword.link != NULL)
	(*(Void(*) PP((Char *buf, Char *wrd, Anyptr _link)))
	  Logglobals->hook.getword.proc)(Arg, Keyword,
					 Logglobals->hook.getword.link);
      else
	(*(Void(*) PP((Char *buf, Char *wrd)))Logglobals->hook.getword.proc)(
	  Arg, Keyword);
      if (!strcmp(Keyword, "W/L")) {
	TRY(try12);
	  WITH->InitRatio = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try12);
	  WITH->InitRatio = 1.0;
	ENDTRY(try12);
      }
      if (!strcmp(Keyword, "VT")) {
	TRY(try13);
	  WITH->InitVt = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try13);
	  WITH->InitVt = 0.9;
	ENDTRY(try13);
      }
      if (!strcmp(Keyword, "I0")) {
	TRY(try14);
	  WITH->InitLeakage = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try14);
	  WITH->InitLeakage = 5e-17;
	ENDTRY(try14);
      }
      if (!strcmp(Keyword, "FITFACTOR")) {
	TRY(try15);
	  WITH->InitFitFactor = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try15);
	  WITH->InitFitFactor = 3.33;
	ENDTRY(try15);
      }
      if (!strcmp(Keyword, "KT/QN")) {
	TRY(try16);
	  WITH->InitKtqn = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try16);
	  WITH->InitKtqn = 0.04;
	ENDTRY(try16);
      }
      if (!strcmp(Keyword, "CGS")) {
	TRY(try17);
	  WITH->InitCgs = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try17);
	  WITH->InitCgs = 10e-15;
	ENDTRY(try17);
      }
      if (!strcmp(Keyword, "CGD")) {
	TRY(try18);
	  WITH->InitCgd = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try18);
	  WITH->InitCgd = 10e-15;
	ENDTRY(try18);
      }
      if (!strcmp(Keyword, "CGB")) {
	TRY(try19);
	  WITH->InitCgb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try19);
	  WITH->InitCgb = 10e-15;
	ENDTRY(try19);
      }
      if (!strcmp(Keyword, "CDB")) {
	TRY(try20);
	  WITH->InitCdb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try20);
	  WITH->InitCdb = 10e-15;
	ENDTRY(try20);
      }
      if (!strcmp(Keyword, "CSB")) {
	TRY(try21);
	  WITH->InitCsb = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try21);
	  WITH->InitCsb = 10e-15;
	ENDTRY(try21);
      }
      if (!strcmp(Keyword, "EARLY")) {
	TRY(try22);
	  WITH->InitEarly = strtod(Arg, &STR1);
	  Dummy = STR1 - Arg + 1;
	RECOVER(try22);
	  WITH->InitEarly = 50.0;
	ENDTRY(try22);
      }
    }
  } while (Found);
}

/****************************************************************************/
/*                          Installation Procedure                           */
/****************************************************************************/

Local Void initlib_NFET4(act)
log_32_action *act;
{
  Cell_rec *NewCell;
  ICNfet *NewIC;

  NewCell = NULL;
  NewCell = (Cell_rec *)Malloc(sizeof(Cell_rec));
  NewCell->simulatable = true;
  NewCell->intr_cell = 0;
  NewCell->phase = simple;
  Newasm_new(NewCell->plist_cell, sizeof(pinrec) * act->kind->numpins);
/* p2c: gates.text, line 1625:
 * Warning: Symbol 'NEWASM_NEW' is not defined [221] */
  NewCell->plist_cell[0].standalone = false;
  NewCell->plist_cell[0].active = true;
  NewCell->plist_cell[1].standalone = false;
  NewCell->plist_cell[1].active = true;
  NewCell->plist_cell[2].standalone = false;
  NewCell->plist_cell[2].active = true;
  NewIC = (ICNfet *)Malloc(sizeof(ICNfet));
  NewIC->InitRatio = 1.0;
  NewIC->InitVt = 0.9;
  NewIC->InitLeakage = 5e-17;
  NewIC->InitFitFactor = 3.33;
  NewIC->InitKtqn = 0.04;
  NewIC->InitCgs = 10e-15;
  NewIC->InitCgd = 10e-15;
  NewIC->InitCgb = 10e-15;
  NewIC->InitCdb = 10e-15;
  NewIC->InitCsb = 10e-15;
  NewIC->InitEarly = 50.0;
  GetCnf_Nfet4(NewIC);
  NewCell->Ainfo = (Anyptr)NewIC;
  act->kind->info = (Anyptr)NewCell;
}  /*NFET4_Initlib*/



Local boolean Nfetcheck(Inst, Attrnum)
log_grec *Inst;
long Attrnum;
{
  boolean Result;
  NfetConst *NfetVar;
  instlist *A_Gate;

  Result = true;
  A_Gate = (instlist *)Inst->info;
  NfetVar = (NfetConst *)A_Gate->InstVar;
  switch (Attrnum) {

  case N_Ratio:
    if (Inst->attr[N_Ratio - 1].UU.r < 0 || Inst->attr[N_Ratio - 1].blnk ||
	Inst->attr[N_Ratio - 1].UU.r > 1e20)
      Result = false;
    else {
      NfetVar->Kn1 = Inst->attr[N_Ratio - 1].UU.r * Inst->attr[N_Leakage - 1].UU.r;
      NfetVar->Kn4 = NfetVar->Kn1 *
		     Expo(NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      NfetVar->TwoKn4 = 2 * NfetVar->Kn4;
    }
    break;

  case N_Vt:
    if (Inst->attr[N_Vt - 1].UU.r < -Vdd || Inst->attr[N_Vt - 1].UU.r > Vdd ||
	Inst->attr[N_Vt - 1].blnk || Inst->attr[N_Vt - 1].UU.r == 0)
      Result = false;
    else {
      NfetVar->Kn4 = NfetVar->Kn1 *
		     Expo(NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      NfetVar->TwoKn4 = 2 * NfetVar->Kn4;
      NfetVar->Kn5 = Expo(-NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r);
    }
    break;

  case N_Leakage:
    if ((unsigned)Inst->attr[N_Leakage - 1].UU.r > 1 ||
	Inst->attr[N_Leakage - 1].blnk)
      Result = false;
    else {
      NfetVar->Kn1 = Inst->attr[N_Ratio - 1].UU.r * Inst->attr[N_Leakage - 1].UU.r;
      NfetVar->Kn4 = NfetVar->Kn1 *
		     Expo(NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      NfetVar->TwoKn4 = 2 * NfetVar->Kn4;
    }
    break;

  case N_FitFactor:
    if ((unsigned)Inst->attr[N_FitFactor - 1].UU.r > 20 ||
	Inst->attr[N_FitFactor - 1].blnk)
      Result = false;
    else
      NfetVar->Kn3 = Inst->attr[N_FitFactor - 1].UU.r;
    break;

  case N_Ktqn:
    if ((unsigned)Inst->attr[N_Ktqn - 1].UU.r > 1 || Inst->attr[N_Ktqn - 1].blnk)
      Result = false;
    else {
      NfetVar->V0 = Inst->attr[N_Ktqn - 1].UU.r;
      NfetVar->Kn2 = 1 / NfetVar->V0;
      NfetVar->Kn4 = NfetVar->Kn1 *
		     Expo(NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r) /
		     (Inst->attr[N_Vt - 1].UU.r * Inst->attr[N_Vt - 1].UU.r);
      NfetVar->TwoKn4 = 2 * NfetVar->Kn4;
      NfetVar->Kn5 = Expo(-NfetVar->Kn2 * Inst->attr[N_Vt - 1].UU.r);
    }
    break;

  case N_Early:
    if (Inst->attr[N_Early - 1].UU.r <= 0 || Inst->attr[N_Early - 1].UU.r > VTooBig)
      Result = false;
    else {
      if (Inst->attr[N_Early - 1].blnk)
	NfetVar->InvVcross = 0.0;
      else
	NfetVar->InvVcross = 1 / Inst->attr[N_Early - 1].UU.r;
    }
    break;
  }
  return Result;
}


Local Void Attr_Nfet4(act)
log_32_action *act;
{
  long Attrnum;

  Attrnum = act->pin;
  if (Attrnum >= Cstart && Attrnum <= Cstart + 8) {
    Capattrinsert((long)Cstart, Attrnum, act->inst, &act->ok);
    return;
  }
  if ((unsigned long)Attrnum < 32 &&
      ((1L << Attrnum) & ((1L << N_Cgs) | (1L << N_Cgd))) != 0)
    act->ok = Crange(act->inst, Attrnum, true);
  else
    act->ok = Nfetcheck(act->inst, Attrnum);
}


Local Void Newgate_Nfet4(act)
log_32_action *act;
{
  NfetConst *NfetVar;
  instlist *A_Gate;
  Cell_rec *Cellptr;
  ICNfet *ICptr;
  log_grec *WITH;

  Cellptr = (Cell_rec *)act->inst->kind->info;
  ICptr = (ICNfet *)Cellptr->Ainfo;
  NfetVar = (NfetConst *)Malloc(sizeof(NfetConst));
  NfetVar->V0 = ICptr->InitKtqn;   /*kt/qn*/
  NfetVar->Kn1 = ICptr->InitRatio * ICptr->InitLeakage;
      /*nfet diode leakage current*(W/L)*/
  NfetVar->Kn2 = 1 / NfetVar->V0;   /*qn/kt*/
  NfetVar->Kn3 = ICptr->InitFitFactor;   /*Tanch(kn3*x) ~=.99 when x=1*/
  NfetVar->Kn4 = NfetVar->Kn1 * Expo(NfetVar->Kn2 * ICptr->InitVt) /
		 (ICptr->InitVt * ICptr->InitVt);
      /*Kn1*Expo(Kn2*Vtn)/(Vtn*Vtn)*/
  NfetVar->TwoKn4 = 2 * NfetVar->Kn4;   /*2*Kn4*/
  NfetVar->Kn5 = Expo(-NfetVar->Kn2 * ICptr->InitVt);   /*Expo(-Kn2*Vtn)*/
  if (ICptr->InitEarly == 0)
    NfetVar->InvVcross = 0.0;
  else
    NfetVar->InvVcross = 1 / ICptr->InitEarly;
  NfetVar->TauCgd = 0.0;
  NfetVar->TauCgs = 0.0;
  A_Gate = (instlist *)act->inst->info;
  A_Gate->InstVar = (Anyptr)NfetVar;
  CapInit(act->inst);   /*6*/
  A_Gate->Pininfo[0].Cap = ICptr->InitCgb;
  A_Gate->Pininfo[1].Cap = ICptr->InitCdb;
  A_Gate->Pininfo[2].Cap = ICptr->InitCsb;
  WITH = act->inst;
  WITH->attr[N_Cgd - 1].UU.r = ICptr->InitCgd;   /*Cgd*/
  WITH->attr[N_Cgd - 1].blnk = false;
  WITH->attr[N_Cgd - 1].changed = true;
  WITH->attr[N_Cgs - 1].UU.r = ICptr->InitCgs;   /*Cgs*/
  WITH->attr[N_Cgs - 1].blnk = false;
  WITH->attr[N_Cgs - 1].changed = true;
  WITH->attr[N_Ratio - 1].UU.r = ICptr->InitRatio;   /*W/L*/
  WITH->attr[N_Ratio - 1].blnk = false;
  WITH->attr[N_Ratio - 1].changed = true;
  WITH->attr[N_Vt - 1].UU.r = ICptr->InitVt;   /*Vt*/
  WITH->attr[N_Vt - 1].blnk = false;
  WITH->attr[N_Vt - 1].changed = true;
  WITH->attr[N_Leakage - 1].UU.r = ICptr->InitLeakage;   /*leakage current*/
  WITH->attr[N_Leakage - 1].blnk = false;
  WITH->attr[N_Leakage - 1].changed = true;
  WITH->attr[N_FitFactor - 1].UU.r = ICptr->InitFitFactor;   /*fit factor*/
  WITH->attr[N_FitFactor - 1].blnk = false;
  WITH->attr[N_FitFactor - 1].changed = true;
  WITH->attr[N_Ktqn - 1].UU.r = ICptr->InitKtqn;   /*Kt/qn*/
  WITH->attr[N_Ktqn - 1].blnk = false;
  WITH->attr[N_Ktqn - 1].changed = true;
  WITH->attr[N_Early - 1].UU.r = ICptr->InitEarly;
  WITH->attr[N_Early - 1].blnk = false;
  WITH->attr[N_Early - 1].changed = true;
}

Local Void Copygate_Nfet4(act)
log_32_action *act;
{
  NfetConst *NfetVar, *Old_Nfetvar;
  instlist *A_Gate, *A_