
/*
	xdrop.c

	This program allows a drag and drop or cut and paste interface
	to normal UNIX applications. When a file (or text) is dropped
	(or pasted) into xdrop, xdrop will execute a command line to
	start up an application.

	usage:
		xdrop [-i icon_file]
		      [-x xbmfile]
		      [-l icon_label] command

	xdrop, 7-12-90, M.Dobie.

	13-12-90	Inverting icon added
	13-12-90	Handles the case where the icon cannot be opened
	14-12-90	Argument processing added.
	15-07-93	Ported to xview 3.0
	06-04-94	Added -debug
*/

/*
	Copyright 1990,1993,1994 Mark Dobie

	Permission to use, copy, modify, distribute, and sell this software
	and its documentation for any purpose is hereby granted without fee,
	provided that the above copyright notice appear in all copies and
	that both that copyright notice and this permission notice appear
	in supporting documentation.
*/

/****************************************************************/

#include <stdio.h>

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/text.h>
#include <xview/xv_xrect.h>
#include <xview/seln.h>
#include <xview/icon_load.h>
#include <xview/dragdrop.h>

#include <X11/Xresource.h>

/****************************************************************/

/* Are we debugging? */
static	int	debugging ;

/* Main window data key */
static	int	win_data_key = 0 ;

/* This is the data associated with each window */
typedef	struct
{
	/* graphics stuff */
	Display			*dpy ;
	GC			gc ;
	Window			icon_window ;

	/* icon stuff */
	Server_image		base_icon_image ;
	int			icon_width, icon_height ;

	/* xview objects */
	Frame			base_frame ;
	Panel_item		base_image ;
	Frame          		command_frame ;
	Panel_item		command_line ;
	Xv_drop_site		dropsite ;

	/* options information */
	Bool			got_sun_icon, got_x_icon ;
	char			*icon_file ;
	Bool			got_label ;
	char			*icon_label ;
	char			*command_string ;
	XrmDatabase		commandlineDB ;
} WIN_DATA, *WIN_DATA_P ;

/* X resource variables and command line arguments */
static	XrmOptionDescRec	optable[] = {
{"-i",		".icon",		XrmoptionSepArg,	(caddr_t)NULL},
{"-x",		".xbm",			XrmoptionSepArg,	(caddr_t)NULL},
{"-l",		".label",		XrmoptionSepArg,	(caddr_t)NULL},
{"-debug",	".debug",		XrmoptionNoArg,		(caddr_t)"true"}
				    } ;
static	int			optable_n = sizeof(optable)/
					    sizeof(XrmOptionDescRec) ;

/* the default xdrop icon */
#include "xdrop.xbm"

/****************************************************************/

/*
	create_window_data

	Create a window data structure and initialise it.

	Returns a new structure or NULL if out of memory.
*/

static	WIN_DATA_P	create_window_data(void)
{
	WIN_DATA_P	win_data ;

	/* initialise the XV_KEY_DATA key */
	if (win_data_key == 0)
		win_data_key = xv_unique_key() ;

	/* create the window data */
	if ((win_data = (WIN_DATA_P)malloc(sizeof(WIN_DATA))) == NULL)
	{
		fprintf(stderr,"create_window_data : no memory for window data\n") ;
		return(NULL) ;
	}

	/* Initialise the fields */
	win_data->got_sun_icon		= False ;
	win_data->got_x_icon		= False ;
	win_data->icon_file		= NULL ;
	win_data->got_label		= False ;
	win_data->icon_label		= NULL ;
	win_data->command_string	= NULL ;

	return(win_data) ;
}

/****************************************************************/

/*
	destroy_window_data

	Notify procedure to destroy a window data structure and all its
	fields.

 -	object is the object with which the data is assiciated.
 -	key is the data's key (should be win_data_key)
 -	win_data is the structure to destroy.
*/

static	void	destroy_window_data(Xv_object object, int key,
				    WIN_DATA_P win_data)
{
	/* destory the fields */

	/* destroy the structure */
	free(win_data) ;
}

/****************************************************************/

/*
	object_frame

	Given an Xview object, return the frame that owns it.

 -	object is an xview object.

	Returns a frame, or NULL on faulure.
*/

Frame	object_frame(Xv_object object)
{
	while(object != (Xv_object)NULL)
	{
		Xv_object	owner ;

		if ((Xv_pkg *)xv_get(object, XV_TYPE) == FRAME)
			return(object) ;

		owner  = xv_get(object, XV_OWNER) ;
		object = owner ;
	}

	return ((Frame)NULL) ;
}

/****************************************************************/

/*
	get_window_data

	Get the window data corresponding to an object.

 -	object is an object.

	Returns the window data.
*/

static	WIN_DATA_P	get_window_data(Xv_object object)
{
	Frame		frame ;
	WIN_DATA_P	win_data ;

	frame	 = object_frame(object) ;
	win_data = (WIN_DATA_P)xv_get(frame, XV_KEY_DATA, win_data_key) ;

	return(win_data) ;
}

/****************************************************************/

/*
	invert_icon

	Invert an icon (assuming the window data is all set up).

 -	win_data is the window data.
*/

void	invert_icon(WIN_DATA_P win_data)
{
	/* Invert the server image */
	XCopyArea(win_data->dpy,
		  win_data->icon_window,
		  win_data->icon_window,
		  win_data->gc,
		  0,0,
		  win_data->icon_width,win_data->icon_height,
		  0,0) ;

	/* Re-assign the Server_image to the icon */
	xv_set(xv_get(win_data->base_frame, FRAME_ICON),
		ICON_IMAGE,		win_data->base_icon_image,
		NULL) ;

	/* and to the base window image */
	xv_set(win_data->base_image,
		PANEL_LABEL_IMAGE,	win_data->base_icon_image,
		NULL) ;
}

/****************************************************************/

/*
	escape_string

	Escape special shell characters in a string. This assumes
	sh and replaces \ ` " $ with \\ \` \" \$ respectively.

 -	dest is the destination string
 -	src is the source string
*/

static	void	escape_string(char *dest, char *src)
{
	char	*d, *s ;

	/* escape characters */
	s = src ;
	d = dest ;
	while (*s != '\0')
	{
		switch (*s)
		{
		case '\\':
		case '`':
		case '\"':
		case '$':
			*d++ = '\\' ;
		default:
			*d++ = *s++ ;
		}
	}
	*d = '\0' ;
}

/****************************************************************/

/*
	execute_drop

	Run the xdrop command with a given selection

 -	win_data is the window data
 -	filename is the name of a file
	NULL => this is not a file
 -	text is some text
	filename != NULL => this text is the contents of the file
*/

static	void	execute_drop(WIN_DATA_P win_data, char *filename, char *text)
{
	char	*command_line, *command, *buf ;
	int	file_length, text_length, command_length ;
	int	length ;

	/* get the command */
	command = (char *)xv_get(win_data->command_line, PANEL_VALUE) ;

	/* work out the string lengths */
	command_length = (command != NULL)  ? strlen(command)  : 0 ;
	file_length    = (filename != NULL) ? strlen(filename) : 0 ;
	text_length    = (text != NULL)	    ? strlen(text)     : 0 ;

	/* work out the whole command line length */
	length = file_length + text_length + command_length + 50 ;
	
	/* create space for the command line */
	if ((command_line = (char *)malloc(length)) == NULL)
	{
		fprintf(stderr,"xdrop : no memory for command line\n") ;
		return ;
	}

	/* and a buffer of twice the size (to allow for escaping) */
	if ((buf = (char *)malloc(2*length)) == NULL)
	{
		fprintf(stderr,"xdrop : no memory for command line buffer\n") ;
		return ;
	}

	/* construct the command */
	*command_line = '\0' ;

	/* is there a filename? */
	if (filename != NULL)
	{
		strcat(command_line, "FILE=\"") ;
		escape_string(buf, filename) ;
		strcat(command_line, buf) ;
		strcat(command_line, "\";export FILE;") ;
	}

	/* is there some text? */
	if (text != NULL)
	{
		strcat(command_line, "TEXT=\"") ;
		escape_string(buf, text) ;
		strcat(command_line, buf) ;
		strcat(command_line, "\";export TEXT;") ;
	}

	/* there is always a command */
	sprintf(buf, "%s\n", command) ;
	strcat(command_line, buf) ;

	/* see what we are doing */
	if (debugging)
		fprintf(stderr,"xdrop : executing <%s>\n", command_line) ;

	/* now execute the command */
	system(command_line) ;

	/* and clean up */
	free(command_line) ;
	free(buf) ;
}

/****************************************************************/

/*
	process_drop

	Handle drop events

 -	win is the window
 -	event is the event
 -	arg is an event arg
*/

static	void	process_drop(Xv_Window win, Event *event, Notify_arg arg)
{
	WIN_DATA_P		win_data ;
	void			*data ;
	char			*filename, *text ;
	unsigned int		n ;
	int			format ;
	Selection_requestor	request ;
static	char			*targets[] = {"STRING",
					      "FILE_NAME",
					      "NAME",
					      "TARGETS"} ;
	int			n_targets = sizeof(targets)/sizeof(char *) ;
	int			i ;

	if (debugging)
	{
		/* display the event */
		if (event_action(event) == ACTION_DRAG_LOAD)
			fprintf(stderr,"ACTION_DRAG_LOAD\n") ;
		if (event_action(event) == ACTION_DRAG_PREVIEW)
			fprintf(stderr,"ACTION_DRAG_PREVIEW\n") ;
		if (event_action(event) == ACTION_DRAG_COPY)
			fprintf(stderr,"ACTION_DRAG_COPY\n") ;
		if (event_action(event) == ACTION_DRAG_MOVE)
			fprintf(stderr,"ACTION_DRAG_MOVE\n") ;
		if (event_action(event) == ACTION_ADJUST)
			fprintf(stderr,"ACTION_ADJUST\n") ;
		if (event_action(event) == ACTION_PASTE)
			fprintf(stderr,"ACTION_PASTE\n") ;
	}

	/* only interested in up button and keyboard events */
	if ((event_action(event)==ACTION_PASTE) ||
	    (event_action(event)==ACTION_ADJUST))
		if (!event_is_up(event))
			return ;

	/* get the window data */
	win_data = get_window_data(win) ;

	switch(event_action(event))
	{
	case ACTION_DRAG_PREVIEW:
		break ;
	case ACTION_ADJUST:
	case ACTION_PASTE:
	case ACTION_DRAG_LOAD:
	case ACTION_DRAG_MOVE:
	case ACTION_DRAG_COPY:
		/* invert the icon */
		invert_icon(win_data) ;

		/* create a request for the selection */
		request = xv_create(win_data->base_frame, SELECTION_REQUESTOR,
					NULL) ;

		/* decode the drop if its a drag and drop event */
		if ((event_action(event)!=ACTION_PASTE) &&
		    (event_action(event)!=ACTION_ADJUST))
		{	/* drag and drop */
			if (dnd_decode_drop(request, event) == XV_ERROR)
			{
				fprintf(stderr, "xdrop : drop failed\n") ;
				invert_icon(win_data) ;
				xv_destroy(request) ;
				return ;
			}
		}

		/* find a target type that we can convert to */
		for (i=0 ; i<n_targets ; i++)
		{
			/* set the request type */
			xv_set(request, SEL_TYPE_NAME,targets[i], NULL) ;

			/* fetch the data */
			data = (void *)xv_get(request, SEL_DATA, &n, &format) ;
			if (n == SEL_ERROR)
			{
				/* cannot convert, ensure data will be NULL */
				data = NULL ;
			}

			/* keep the data pointer */
			if (i == 0)
				text = data ;		/* STRING */
			else if (i == 1)
				filename = data ;	/* FILE_NAME */
			else if ((i == 2) && debugging)
				/* display NAME target */
				fprintf(stderr,"NAME is %s\n", (char *)data) ;
			else if ((i == 3) && debugging)
			{	/* display TARGETS target */
				int	j ;

				fprintf(stderr,"There are %d targets\n",n) ;
				for (j=0 ; j<n ; j++)
					fprintf(stderr,"target %d is %d	%s\n",
						j, ((int *)data)[j],
						XGetAtomName(win_data->dpy,
							     ((Atom *)data)[j])) ;
			}
		}

		/* execute the command */
		execute_drop(win_data, filename, text) ;

		/* free the data */
		free(data) ;

		/* finish the drop */
		dnd_done(request) ;

		/* finish the request */
		xv_destroy(request) ;

		/* invert the icon */
		invert_icon(win_data) ;

		break ;
	}
}

/****************************************************************/

/*
	process_command_line

	Sets fields in the window data to reflect command line options.

 -	argc_ptr, argv are the command line arguments.
 -	win_data is the window data
*/

static	void	process_command_line(int *argc_ptr, char *argv[],
				     WIN_DATA_P win_data)
{
	char		*str_type[20] ;
	XrmValue	value ;

	/* initialise the resource manager */
	XrmInitialize() ;

	/* create a resource database from the command line */
	win_data->commandlineDB = NULL ;
	XrmParseCommand(&(win_data->commandlineDB), optable, optable_n,
			"xdrop", argc_ptr, argv) ;

	/* check for debugging */
	debugging = XrmGetResource(win_data->commandlineDB,
				  "xdrop.debug", "Xdrop.Debug",
				  str_type, &value) ;
	if (debugging)
		fprintf(stderr,"xdrop : debugging on\n") ;
	    
	/* check for sun icon settings */
	win_data->got_sun_icon = XrmGetResource(win_data->commandlineDB,
				  "xdrop.icon", "Xdrop.Icon",
				  str_type, &value) ;
	if (win_data->got_sun_icon)
		win_data->icon_file = value.addr ;

	/* check for X icon settings */
	win_data->got_x_icon = XrmGetResource(win_data->commandlineDB,
				  "xdrop.xbm", "Xdrop.Xbm",
				  str_type, &value) ;
	if (win_data->got_x_icon)
		win_data->icon_file = value.addr ;

	/* check for label settings */
	win_data->got_label = XrmGetResource(win_data->commandlineDB,
				  "xdrop.label", "Xdrop.Label",
				  str_type, &value) ;
	if (win_data->got_label)
		win_data->icon_label = value.addr ;

	/* There should be 2 arguments left, xdrop and command */
	if (*argc_ptr != 2)
	{
		fprintf(stderr, "usage : xdrop [-i iconfile][-x xbmfile][-l iconlabel] command\n") ;
		exit(1) ;
	}

	/* the command string is the second */
	win_data->command_string = argv[1] ;
}

/****************************************************************/

/*
	read_image_from_file

	Read either a Sun icon file or an X bitmap file and create a
	server image from the data.

 -	win_data is the window data.

	Sets the base_icon_image field of the window data.
	If an error occurs, win_data is as if there is no icon.
*/

static	void	read_image_from_file(WIN_DATA_P win_data)
{
	Server_image	image ;
static	char		buf[BUFSIZ] ;

	if (win_data->got_sun_icon)
	{
		image = icon_load_svrim(win_data->icon_file, buf) ;
	}
	else if (win_data->got_x_icon)
	{
		image = xv_create((Frame)NULL, SERVER_IMAGE,
				SERVER_IMAGE_BITMAP_FILE, win_data->icon_file,
				NULL) ;
	}
	else	/* use the xdrop icon */
		image = (Server_image)NULL ;

	/* check if it worked */
	if (image == (Server_image)NULL)
	{
		/* an error message if we couldn't read the file */
		if (win_data->icon_file != NULL)
			fprintf(stderr, "xdrop : could not read icon file %s\n",
					win_data->icon_file) ;

		win_data->got_sun_icon = win_data->got_x_icon = False ;

		/* use the default Xdrop icon */
		image = xv_create((Frame)NULL, SERVER_IMAGE,
				XV_WIDTH,		xdrop_width,
				XV_HEIGHT,		xdrop_height,
				SERVER_IMAGE_X_BITS,	xdrop_bits,
				NULL) ;
	}

	/* assign the image to the window data */
	win_data->base_icon_image = image ;
}

/****************************************************************/

/*
	window_title

	Make an appropriate window title from the window data

 -	win_data is the window data

	Return a window title.
*/

static	char	*window_title(WIN_DATA_P win_data)
{
static	char	title[BUFSIZ] ;

	/* create the title */
	sprintf(title, "%s", (win_data->got_label) ? win_data->icon_label
						   : "command") ;

	/* return it */
	return(title) ;
}

/****************************************************************/

/*
	create_frame_icon

	Create an icon for the frame if we supplied one, or take note
	of the default icon if we didn't.

 -	win_data is the window data
*/

static	void	create_frame_icon(WIN_DATA_P win_data)
{
	Icon	icon ;
	Rect	rect ;

	/* get the image dimensions */
	rect.r_left   = 0 ;
	rect.r_top    = 0 ;
	rect.r_width  = xv_get(win_data->base_icon_image, XV_WIDTH) ;
	rect.r_height = xv_get(win_data->base_icon_image, XV_HEIGHT) ;

	/* create an icon from an image */
	icon = xv_create((Frame)NULL, ICON,
			ICON_IMAGE,	win_data->base_icon_image,
			ICON_IMAGE_RECT,&rect,
			NULL) ;

	/* add a label to the icon */
	if (win_data->got_label)
	{	/* Add in the label if we've got one */
		xv_set(icon,
			ICON_LABEL,	win_data->icon_label,
			NULL) ;
	}

	/* note the size of the icon */
	win_data->icon_width  = rect.r_width  = xv_get(icon, XV_WIDTH) ;
	win_data->icon_height = rect.r_height = xv_get(icon, XV_HEIGHT) ;

	/* add the icon to the frame */
	xv_set(win_data->base_frame,
		FRAME_ICON,		icon,
		FRAME_CLOSED_RECT,	&rect,
		NULL) ;

	/* note graphics parameters */
	win_data->dpy		= (Display *)xv_get(win_data->base_frame,
						    XV_DISPLAY) ;
	win_data->icon_window	= xv_get(win_data->base_icon_image, XV_XID) ;
	win_data->gc		= XCreateGC(win_data->dpy,
					    win_data->icon_window,
					    0, NULL) ;
	XSetFunction(win_data->dpy, win_data->gc, GXinvert) ;
}

/****************************************************************/

/*
	edit_command

	Notify procedure to edit the xdrop command

 -	item is the button
 -	event is the event
*/

static	void	edit_command(Panel_item item, Event *event)
{
	WIN_DATA_P	win_data ;
	int		state ;

	/* get the window data */
	win_data = get_window_data(item) ;

	/* get the current state of the command frame */
	state = xv_get(win_data->command_frame, XV_SHOW) ;

	/* toggle it */
	xv_set(win_data->command_frame, XV_SHOW, !state, NULL) ;
}

/****************************************************************/

/*
	panel_event_proc

	Process events for the panel

 -	obj is the xview object
 -	event is the event
 -	arg is an arg
 -	type is the delivery type

	Returns a status value
*/

Notify_value	panel_event_proc(Notify_client obj, Notify_event event,
				 Notify_arg arg, Notify_event_type type)
{
	Notify_value	status ;

	/* send the event to the normal handler */
	status = notify_next_event_func(obj, event, arg, type) ;

	/* just call the main event procedure */
	process_drop(obj, (Event *)event, arg) ;

	/* return the status */
	return(status) ;
}

/****************************************************************/

/*
	create_frame_contents

	Create all the bits to go in the xdrop frame.

 -	win_data is the window data
*/

static	void	create_frame_contents(WIN_DATA_P win_data)
{
	Panel		base_panel, command_panel ;
	Panel_item	image, label ;
	int		label_width, label_height, win_width ;

	/* Create a panel */
	base_panel = xv_create(win_data->base_frame, PANEL, NULL) ;

	/* Create the icon */
	image = xv_create(base_panel, PANEL_MESSAGE,
			XV_X,			0,
			XV_Y,			0,
			PANEL_LABEL_IMAGE,	win_data->base_icon_image,
			PANEL_NOTIFY_PROC,	edit_command,
			NULL) ;

	/* create the label underneath (if we have one) */
	if (win_data->got_label)
	{
		label = xv_create(base_panel, PANEL_MESSAGE,
				XV_X,			0,
				XV_Y,			win_data->icon_height,
				PANEL_LABEL_STRING,	win_data->icon_label,
				NULL) ;
		label_height = xv_get(label, XV_HEIGHT) ;
		label_width  = xv_get(label, XV_WIDTH) ;

		/* centre the label if necessary */
		if (label_width < win_data->icon_width)
			xv_set(label, XV_X, (win_data->icon_width-label_width)/2,
				      NULL) ;
		else	/* centre the image */
			xv_set(image, XV_X, (label_width-win_data->icon_width)/2,
				      NULL) ;

	}
	else
		label_width = label_height = 0 ;

	/* calculate the window width */
	win_width = (win_data->icon_width > label_width) ? win_data->icon_width
							 : label_width ;

	/* intercept events to the panel */
	notify_interpose_event_func(base_panel, panel_event_proc, NOTIFY_SAFE) ;

	/* fit it */
	xv_set(base_panel, XV_WIDTH,  win_width,
			   XV_HEIGHT, win_data->icon_height + label_height,
			   NULL) ;
	window_fit(win_data->base_frame) ;

	/* save the image item */
	win_data->base_image = image ;

	/* Create a subframe */
	win_data->command_frame = xv_create(win_data->base_frame, FRAME_CMD,
			FRAME_LABEL,		window_title(win_data),
			XV_SHOW,		FALSE,
			NULL) ;

	/* get the panel */
	command_panel = xv_get(win_data->command_frame, FRAME_CMD_PANEL) ;

	/* Create the command line text */
	win_data->command_line = xv_create(command_panel, PANEL_TEXT,
			    PANEL_LABEL_STRING,		"command: ",
			    PANEL_VALUE,		win_data->command_string,
			    PANEL_VALUE_STORED_LENGTH,	5000,
			    NULL) ;

	/* fit it */
	window_fit(command_panel) ;
	window_fit(win_data->command_frame) ;
}

/****************************************************************/

/*
	create_drop_site

	Create a suitable drop site. It includes the whole window area
	plus the icon and borders. Also fixes the frame size.

 -	win_data is the window data
*/

#define	XDROP_SITE	41166

static	void	create_drop_site(WIN_DATA_P win_data)
{
	Rect	*frame_rect ;

	/* get a rect for the whole window */
	frame_rect = (Rect *)xv_get(win_data->base_frame, XV_RECT) ;

	/* create a dropsite to cover this and the icon */
	win_data->dropsite = xv_create(win_data->base_frame, DROP_SITE_ITEM,
				DROP_SITE_ID,		XDROP_SITE,
				DROP_SITE_REGION,	frame_rect,
				DROP_SITE_DEFAULT,	TRUE,
				DROP_SITE_EVENT_MASK,	DND_ENTERLEAVE | DND_MOTION,
				NULL) ;

	/* fix the frame to be this size */
	xv_set(win_data->base_frame,
		FRAME_MIN_SIZE,	frame_rect->r_width, frame_rect->r_height,
		FRAME_MAX_SIZE,	frame_rect->r_width, frame_rect->r_height,
		NULL) ;
}

/****************************************************************/

/*
	The main body
*/

int	main(int argc, char *argv[])
{
	WIN_DATA_P	win_data ;

	/* Initialise the toolkit */
	xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL) ;

	/* create the window data */
	win_data = create_window_data() ;

	/* Process the command line options */
	process_command_line(&argc, argv, win_data) ;

	/* Read in the icon to a Server_image */
	read_image_from_file(win_data) ;

	/* Create the base frame */
	win_data->base_frame = xv_create((Frame)NULL,	FRAME,
				FRAME_SHOW_RESIZE_CORNER,FALSE,
				FRAME_LABEL,		"Xdrop",
				FRAME_SHOW_LABEL,	FALSE,
				FRAME_NO_CONFIRM,	TRUE,
				WIN_EVENT_PROC,		process_drop,
				WIN_CONSUME_EVENTS,	ACTION_ADJUST,
							ACTION_PASTE,
							NULL,
				XV_KEY_DATA,		win_data_key,
							win_data,
				XV_KEY_DATA_REMOVE_PROC,win_data_key,
							destroy_window_data,
				NULL) ;

	/* Create the icon for the frame */
	create_frame_icon(win_data) ;

	/* create the contents of the frame */
	create_frame_contents(win_data) ;

	/* Create the drop site for xdrop */
	create_drop_site(win_data) ;

	/* fit it */
	window_fit(win_data->base_frame) ;

	/* Enter the notifier */
	xv_main_loop(win_data->base_frame) ;

	/* ok */
	exit(0) ;
}

/****************************************************************/
