/* > ffti.c */
/* Daniel F. Smith, 1994 */
/* Fast Fourier Transform routines in integers.
 * These are essentially the same as the routines in
 * Numerical Recipes in C Second Edition
 * by Press, Vetterling, Teukolsky and Flannery.
 * However, they have been modified so that arrays are numbered from 0
 * and fiddly complex arithmetic in reals has been replaced with fiddly
 * complex arithmetic in integers.
 * (A note on zero numbering: NRiC argues that it doesn't matter whether
 * arrays are numbered from 0 or 1.  They then proceed to do all their
 * calculations numbered from 0 and write their programs numbered from 1.
 * This is crazy.  In line with C, the routines here are 0 indexed.
 */
#include <math.h>
#include "ffti.h"

/* FP is the fixed point calculation */
/* FP is for the data, FPW for the rotating vector, FPWP for the rotation */
/* FP+FPW < log2(sizeof(long int)*8) (i.e., FP+FPW < bits in a long int */
/* and 2**FPW / n**2 should be > ~1/2 for precision */
/* (precisely, (int)2**FPW.sin**2(2.PI/n) should show structure ) */
/* e.g., sizeof(int)=4 (32 bits), data to be transformed is 8 bit */
/* then FP+FP<32, FP+FPW<32, 8+FP<32 */
#define FP 10
#define FPW 12
#define FPWP 18

#define swap(type,a,b) {type c; c=(a); (a)=(b); (b)=c;}

/* FFT an array of reals.
 * This is equivalent to realft() in NRiC using cints.
 * The data[0...n-1] are transformed in place leaving complex
 * pairs of frequencies.  Since the zeroth and nth frequency are
 * real (and independent) they are squahed into data[0] and data[1].
 * If isign==1,  the data are transformed.
 * If isign==-1, the data are inversely transformed and multiplied by n/2.
 * n must be a power of two, although this is not checked for.
 */
void ffti_real(cint data[],unsigned long n,int isign) {
	unsigned long i,j;
	double theta,wtemp;
	cint c;
	complexi w,wp,h1,h2,temp;
	complexi *cdata;

	/* The arrays are compatible and this arrangement is handy
	 * for the forward direction.  Confusing for the inverse though.
	 */
	cdata=(complexi *)data;

	theta=2.0*PI/n;
	if (isign>0) {
		c=-1;
		ffti_four(cdata,n/2,isign);
		}
	else {
		c=1;
		theta=-theta;
		}
	/* wp is the rotation difference vector * 2**FP */
	wtemp=sin(0.5*theta);
	wp.x=(1<<FPWP)*(-2.0*wtemp*wtemp);
	wp.y=(1<<FPWP)*sin(theta);
	w.x=(wp.x>>(FPWP-FPW))+(1<<FPW);
	w.y=(wp.y>>(FPWP-FPW));

	for(i=1,j=n/2-1;i<n/4;i++,j--) /* i=0 done later */ {
		h1.x=   (cdata[i].x+cdata[j].x)/2;
		h1.y=   (cdata[i].y-cdata[j].y)/2;
		h2.x=-c*(cdata[i].y+cdata[j].y)/2;
		h2.y= c*(cdata[i].x-cdata[j].x)/2;
		temp.x=(w.x*h2.x-w.y*h2.y)>>(FPW-FP); /* temp is <<(FP*2) */
		temp.y=(w.x*h2.y+w.y*h2.x)>>(FPW-FP);
		cdata[i].x=(( h1.x<<FP)+temp.x)>>FP;
		cdata[i].y=(( h1.y<<FP)+temp.y)>>FP;
		cdata[j].x=(( h1.x<<FP)-temp.x)>>FP;
		cdata[j].y=((-h1.y<<FP)+temp.y)>>FP;
		temp.x=(w.x*wp.x-w.y*wp.y)>>(FPWP-FPW); /* temp is <<(FPW*2) */
		temp.y=(w.x*wp.y+w.y*wp.x)>>(FPWP-FPW);
		w.x=((w.x<<FPW)+temp.x)>>FPW;
		w.y=((w.y<<FPW)+temp.y)>>FPW;
		}

	if (isign>0) {
		cint t;
		t=data[0];
		data[0]=t+data[1];
		data[1]=t-data[1];
		}
	else {
		cint t;
		t=data[0];
		data[0]=(t+data[1])/2;
		data[1]=(t-data[1])/2;
		ffti_four(cdata,n/2,isign);
		}
	}

/* FFT a complex array */
/* This is an equivalent to four1() in NRiC using cints.
 * The complex data[0...n-1] are transformed in place leaving complex
 * pairs of frequencies.
 * If isign==1,  the data are transformed.
 * If isign==-1, the data are inversely transformed and multiplied by nn.
 * n must be a power of two, although this is not checked for.
 */
void ffti_four(complexi cdata[],unsigned long n,int isign) {
	unsigned long mmax,m,j,istep,i;
	double theta,wtemp;
	complexi temp,w,wp;

	/* Bit reverse the index (in situ) */
	for(j=0,i=0;i<n;i++) {
		if (j>i)
			swap(complexi,cdata[i],cdata[j]);
		/* This is an addition backwards, from the top bit to bottom */
		m=n>>1;
		while(m>0 && j>=m) {
			j-=m;
			m/=2;
			}
		j+=m;
		}

	/* Routine proper */
	mmax=1;
	while(mmax<n) {
		istep=mmax*2;
		theta=isign*(2.0*PI/istep);
		wtemp=sin(0.5*theta);
		wp.x=(1<<FPWP)*(-2.0*wtemp*wtemp);
		wp.y=(1<<FPWP)*sin(theta);
		w.x=(1<<FPW);
		w.y=(0<<FPW);
		for(m=0;m<mmax;m++) {
			for(i=m;i<n;i+=istep) {
				j=i+mmax;
				temp.x=(w.x*cdata[j].x-w.y*cdata[j].y)>>(FPW-FP);
				temp.y=(w.x*cdata[j].y+w.y*cdata[j].x)>>(FPW-FP);
				cdata[j].x=((cdata[i].x<<FP)-temp.x)>>FP;
				cdata[j].y=((cdata[i].y<<FP)-temp.y)>>FP;
				cdata[i].x=((cdata[i].x<<FP)+temp.x)>>FP;
				cdata[i].y=((cdata[i].y<<FP)+temp.y)>>FP;
				}
			temp.x=(w.x*wp.x-w.y*wp.y)>>(FPWP-FPW);
			temp.y=(w.x*wp.y+w.y*wp.x)>>(FPWP-FPW);
			w.x=((w.x<<FPW)+temp.x)>>FPW;
			w.y=((w.y<<FPW)+temp.y)>>FPW;
			}
		mmax=istep;
		}
	}

/* Build a power spectral density into a complex array
 * as given back from fft_real().  Note that this routine
 * generates a cfloat array over the top of the complex
 * array so that data[0..n]=power at cdata[0..n-1].
 * (This is a two-sided PSD.)
 * It returns the maximum PSD value.
 */
#define rmax(a,b) (((a)>(b))?(a):(b))
cint ffti_power(complexi cdata[],unsigned long n) {
	unsigned long i;
	cint temp,*data,max;

	data=(cint *)cdata;
	data[0]=data[0]*data[0];
	temp   =data[1]*data[1];

	max=rmax(temp,data[0]);

	for(i=1;i<n;i++) {
		data[i]=cdata[i].x*cdata[i].x+cdata[i].y*cdata[i].y;
		max=rmax(data[i],max);
		}
	data[n]=temp;
	return max;
	}

/* Build a power spectral density into a complex array
 * as given back from fft_four().
 * data[0..n/2-1]=power at cdata[n/2..n-1] (negative frequencies)
 * data[n/2]     =power at cdata[0]        (zero frequency)
 * data[n/2+1..n]=power at cdata[1..n/2]   (positive frequencies)
 * (This is a two-sided PSD.)
 * It returns the maximum PSD value.
 */
cint ffti_cpower(complexi cdata[],unsigned long n) {
	cint max,*data;
	unsigned long i;

	data=(cint *)cdata;
	/* Convert them all to powers. */
	max=0.0;
	i=n;
	do{	i--;
		data[n+i]=cdata[i].x*cdata[i].x+cdata[i].y*cdata[i].y;
		max=rmax(data[n+i],max);
		}while(i>0);
	/* Now sort them into a nice(r) order. */
	for(i=0;i<n/2;i++)
		data[i]=data[n+n/2+i];
	data[n/2]=data[n];
	for(i=n/2+1;i<=n;i++)
		data[i]=data[n-n/2+i];

	return max;
	}

void ffti_window(cint data[],unsigned long n_data) {
	cint scale,window;
	long int i,m;

	/* Welch window */
	m=n_data/2;
	scale=(1<<FP)*(4.0/((double)n_data*n_data));
	for(i=0;i<m;i++) {
		window=(1<<FP)-scale*((m-i)*(m-i));
		data[i         ]=(data[i         ]*window)>>FP;
		data[n_data-1-i]=(data[n_data-1-i]*window)>>FP;
		}
	}

void ffti_cwindow(complexi data[],unsigned long n_data) {
	cint scale,window;
	long int i,m;

	/* Welch window */
	m=n_data/2;
	scale=(1<<!5)*(4.0/((double)n_data*n_data));
	for(i=0;i<m;i++) {
		window=(1<<FP)-scale*((m-i)*(m-i));
		data[i         ].x=(data[i         ].x*window)>>FP;
		data[i         ].y=(data[i         ].y*window)>>FP;
		data[n_data-1-i].x=(data[n_data-1-i].x*window)>>FP;
		data[n_data-1-i].y=(data[n_data-1-i].y*window)>>FP;
		}
	}
