/* > fft.c */
/* Daniel F. Smith, 1993 */
/* Fast Fourier Transform routines.
 * These are essentially the same as the routines in
 * Numerical Recipes in C Second Edition
 * by Press, Vetterling, Teukolsky and Flannery.
 * However, they have been modified so that arrays are numbered from 0
 * and fiddly complex arithmetic in reals has been replaced with complex
 * arithmetic.
 * (A note on zero numbering: NRiC argues that it doesn't matter whether
 * arrays are numbered from 0 or 1.  They then proceed to do all their
 * calculations numbered from 0 and write their programs numbered from 1.
 * This is crazy.  In line with C, the routines here are 0 indexed.
 */
#include <math.h>
#include "complex.h"
#include "fft.h"

#define swap(type,a,b) {type c; c=(a); (a)=(b); (b)=c;}

/* FFT an array of reals.
 * This is equivalent to realft() in NRiC using cfloats.
 * The data[0...n-1] are transformed in place leaving complex
 * pairs of frequencies.  Since the zeroth and nth frequency are
 * real (and independent) they are squahed into data[0] and data[1].
 * If isign==1,  the data are transformed.
 * If isign==-1, the data are inversely transformed and multiplied by n/2.
 * n must be a power of two, although this is not checked for.
 */
void fft_real(cfloat data[],unsigned long n,int isign) {
	unsigned long i,j;
	double c1=0.5,c2,wtemp,theta;
	complex w,wp,h1,h2;
	complex *cdata;

	/* The arrays are compatible and this arrangement is handy
	 * for the forward direction.  Confusing for the inverse though.
	 */
	cdata=(complex *)data;

	theta=2.0*PI/n;
	if (isign>0) {
		c2=-0.5;
		fft_four(cdata,n/2,isign);
		}
	else {
		c2=0.5;
		theta=-theta;
		}
	/* wp is the rotation difference vector */
	wtemp=sin(0.5*theta);
	wp=cgen(-2.0*wtemp*wtemp,sin(theta));
	w =cadd(cgen(1.0,0),wp);

	for(i=1,j=n/2-1;i<n/4;i++,j--) /* i=0 done later */ {
		h1=cmul( cgen(c1,0.0), cadd(cdata[i],conj(cdata[j])) );
		h2=cmul( cgen(0.0,c2), csub(cdata[i],conj(cdata[j])) );
		cdata[i]=     cadd(h1,cmul(w,h2));
		cdata[j]=conj(csub(h1,cmul(w,h2)));
		w=cadd(w,cmul(w,wp));
		}

	if (isign>0) {
		cfloat t;
		t=data[0];
		data[0]=t+data[1];
		data[1]=t-data[1];
		}
	else {
		cfloat t;
		t=data[0];
		data[0]=c1*(t+data[1]);
		data[1]=c1*(t-data[1]);
		fft_four(cdata,n/2,isign);
		}
	}

/* FFT a complex array */
/* This is an equivalent to four1() in NRiC using cfloats.
 * The complex data[0...n-1] are transformed in place leaving complex
 * pairs of frequencies.
 * If isign==1,  the data are transformed.
 * If isign==-1, the data are inversely transformed and multiplied by nn.
 * n must be a power of two, although this is not checked for.
 */
void fft_four(complex cdata[],unsigned long n,int isign) {
	unsigned long mmax,m,j,istep,i;
	double wtemp,theta;
	complex temp,w,wp;

	/* Bit reverse the index (in situ) */
	for(j=0,i=0;i<n;i++) {
		if (j>i)
			swap(complex,cdata[i],cdata[j]);
		/* This is an addition backwards, from the top bit to bottom */
		m=n>>1;
		while(m>0 && j>=m) {
			j-=m;
			m/=2;
			}
		j+=m;
		}

	/* Routine proper */
	mmax=1;
	while(mmax<n) {
		istep=mmax*2;
		theta=isign*(2.0*PI/istep);
		wtemp=sin(0.5*theta);
		wp=cgen(-2.0*wtemp*wtemp,sin(theta));
		w =cgen(1.0,0.0);
		for(m=0;m<mmax;m++) {
			for(i=m;i<n;i+=istep) {
				j=i+mmax;
				temp=cmul(w,cdata[j]);
				cdata[j]=csub(cdata[i],temp);
				cdata[i]=cadd(cdata[i],temp);
				}
			w=cadd(w,cmul(w,wp));
			}
		mmax=istep;
		}
	}

/* Build a power spectral density into a complex array
 * as given back from fft_real().  Note that this routine
 * generates a cfloat array over the top of the complex
 * array so that data[0..n]=power at cdata[0..n-1].
 * (This is a two-sided PSD.)
 * It returns the maximum PSD value.
 */
#define rmax(a,b) (((a)>(b))?(a):(b))
cfloat fft_power(complex cdata[],unsigned long n) {
	unsigned long i;
	cfloat temp,*data,max;

	data=(cfloat *)cdata;
	data[0]=data[0]*data[0];
	temp   =data[1]*data[1];

	max=rmax(temp,data[0]);

	for(i=1;i<n;i++) {
		data[i]=cmod2(cdata[i]);
		max=rmax(data[i],max);
		}
	data[n]=temp;
	return max;
	}

/* Build a power spectral density into a complex array
 * as given back from fft_four().
 * data[0..n/2-1]=power at cdata[n/2..n-1] (negative frequencies)
 * data[n/2]     =power at cdata[0]        (zero frequency)
 * data[n/2+1..n]=power at cdata[1..n/2]   (positive frequencies)
 * (This is a two-sided PSD.)
 * It returns the maximum PSD value.
 */
cfloat fft_cpower(complex cdata[],unsigned long n) {
	cfloat max,*data;
	unsigned long i;

	data=(cfloat *)cdata;
	/* Convert them all to powers. */
	max=0.0;
	i=n;
	do{	i--;
		data[n+i]=cmod2(cdata[i]);
		max=rmax(data[n+i],max);
		}while(i>0);
	/* Now sort them into a nice(r) order. */
	for(i=0;i<n/2;i++)
		data[i]=data[n+n/2+i];
	data[n/2]=data[n];
	for(i=n/2+1;i<=n;i++)
		data[i]=data[n-n/2+i];

	return max;
	}

void fft_window(cfloat data[],unsigned long n_data) {
	cfloat scale,window;
	int i,m;

	/* Welch window */
	m=n_data/2;
	scale=4.0/((cfloat)n_data*n_data);
	for(i=0;i<m;i++) {
		window=1.0-scale*((long)(m-i)*(m-i));
		data[i         ]*=window;
		data[n_data-1-i]*=window;
		}
	}

void fft_cwindow(complex data[],unsigned long n_data) {
	cfloat scale,window;
	int i,m;

	/* Welch window */
	m=n_data/2;
	scale=4.0/((cfloat)n_data*n_data);
	for(i=0;i<m;i++) {
		window=1.0-scale*((long)(m-i)*(m-i));
		data[i         ].x*=window;
		data[i         ].y*=window;
		data[n_data-1-i].x*=window;
		data[n_data-1-i].y*=window;
		}
	}
