/* > Chart.c */
/* Daniel F. Smith, 1994 */
/* Chart widget, based on Template.c */

#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "ChartP.h"

#define DEFAULT_X_SCALE 1
#define DEFAULT_Y_SCALE 4

#define offset(field) XtOffsetOf(ChartRec, chart.field)
static XtResource resources[]={
        { /* sets the colour of the chart's line */
		XtNforeground,XtCForeground,
		XtRPixel,sizeof(Pixel),offset(fg),
		XtRString,XtDefaultForeground
		},
	{ /* sets the colour of the grid's dotted line */
	        XtNgridColour,XtCGridColour,
		XtRPixel,sizeof(Pixel),offset(gridcolour),
		XtRString,XtDefaultForeground
		},
	{ /* sets the spacing (in pixels) of the grid lines */
	        XtNgridSize,XtCGridSize,
		XtRInt,sizeof(int),offset(grid),
		XtRImmediate,(XtPointer)0
		},
	{ /* sets the divisor of the x values */
	        XtNscaleX,XtCScaleX,
		XtRInt,sizeof(int),offset(xscale),
		XtRImmediate,(XtPointer)DEFAULT_X_SCALE
		},
	{ /* sets the divisor of the y values */
	        XtNscaleY,XtCScaleY,
		XtRInt,sizeof(int),offset(yscale),
		XtRImmediate,(XtPointer)DEFAULT_Y_SCALE
		},
	{ /* the pointer to the applications chart data as unsigned char * */
	        XtNdatap,XtCDatap,
		XtRPointer,sizeof(XtPointer),offset(data),
		XtRImmediate,NULL
		},
        { /* the number of data points */
	        XtNndata,XtCNdata,
		XtRInt,sizeof(int),offset(ndata),
		XtRImmediate,(XtPointer)0
		},
	{ /* set True to redraw---will always be reset to False */
          	XtNredraw,XtCRedraw,
		XtRBoolean,sizeof(Boolean),offset(redraw),
		XtRImmediate,(XtPointer)False
		},
	{ /* how many points to skip between x values */
		XtNskip,XtCSkip,
		XtRInt,sizeof(int),offset(skip),
		XtRImmediate,(XtPointer)1
		}
	};
#undef offset

/* Methods declaration */
static void Initialise(Widget,Widget,ArgList,Cardinal *);
static void Destroy(Widget);
static void Redisplay(Widget,XEvent *,Region);
static Boolean SetValues(Widget,Widget,Widget,ArgList,Cardinal *);
static XtGeometryResult QueryGeometry(Widget,
				      XtWidgetGeometry *,
				      XtWidgetGeometry *);

ChartClassRec chartClassRec={
	{ /* core fields */
		/* superclass		*/	(WidgetClass)&coreClassRec,
		/* class_name		*/	"Chart",
		/* widget_size		*/	sizeof(ChartRec),
		/* class_initialize	*/	NULL,
		/* class_part_initialize*/	NULL,
		/* class_inited		*/	False,
		/* initialize		*/	Initialise,
		/* initialize_hook	*/	NULL,
		/* realize		*/	XtInheritRealize,
		/* actions		*/	NULL,
		/* num_actions		*/	0,
		/* resources		*/	resources,
		/* num_resources	*/	XtNumber(resources),
		/* xrm_class		*/	NULLQUARK,
		/* compress_motion	*/	True,
		/* compress_exposure	*/	True,
		/* compress_enterleave	*/	True,
		/* visible_interest	*/	False,
		/* destroy		*/	Destroy,
		/* resize		*/	NULL,
		/* expose		*/	Redisplay,
		/* set_values		*/	SetValues,
		/* set_values_hook	*/	NULL,
		/* set_values_almost	*/	XtInheritSetValuesAlmost,
		/* get_values_hook	*/	NULL,
		/* accept_focus		*/	NULL,
		/* version		*/	XtVersion,
		/* callback_private	*/	NULL,
		/* tm_table		*/	NULL,
		/* query_geometry	*/	QueryGeometry,
		/* display_accelerator	*/	XtInheritDisplayAccelerator,
		/* extension		*/	NULL
		},
	{ /* chart fields */
	        /* empty		*/	0
		}
        };

WidgetClass chartWidgetClass=(WidgetClass)&chartClassRec;

static GC getlineGC(ChartWidget new) {
	XGCValues gcval;
	XtGCMask gcmask;
	/* set up line's GC */
	gcmask=GCForeground | GCBackground;
	gcval.foreground=new->chart.fg;
	gcval.background=new->core.background_pixel;
	return XtGetGC((Widget)new,gcmask,&gcval);
	}

static GC getgridGC(ChartWidget new) {
	XGCValues gcval;
	XtGCMask gcmask;
	/* set up grid's GC */
	gcmask=GCForeground | GCBackground | \
	       GCDashOffset | GCDashList | GCLineStyle;
	gcval.foreground=new->chart.gridcolour;
	gcval.background=new->core.background_pixel;
	gcval.dashes=1;
	gcval.dash_offset=0;
	gcval.line_style=LineOnOffDash;
	return XtGetGC((Widget)new,gcmask,&gcval);
	}

static void Initialise(Widget treq,Widget tnew,ArgList args,Cardinal *nargs) {
	ChartWidget new=(ChartWidget)tnew;
	
	/* Check ranges of resources */
	if (new->chart.grid<0)
		new->chart.grid=0;
	if (new->chart.xscale<1) {
		XtWarning("Chart: bad x scale; set to 1.");
		new->chart.xscale=1;
		}
	if (new->chart.yscale<1) {
		XtWarning("Chart: bad y scale; set to 1.");
		new->chart.yscale=1;
		}
	if (new->chart.ndata<0) {
		XtWarning("Chart: number of data points negative, zeroed.");
		new->chart.ndata=0;
		}
	if (new->chart.skip<1) {
		XtWarning("Chart: skip value is less than 1.");
		new->chart.skip=1;
		}
	new->core.width=new->chart.ndata/new->chart.xscale;
	new->core.height=            256/new->chart.yscale;
	new->chart.gridgc=getgridGC(new);
	new->chart.gc    =getlineGC(new);
	new->chart.redraw=False;
	}

static void Redisplay(Widget gw,XEvent *ge,Region R) {
	XExposeEvent *e=(XExposeEvent *)ge;
	ChartWidget w=(ChartWidget)gw;
	int x,width,xx,ww; /* values in number, not pixels */
	int yt,yb,yy;
	XPoint points[100];
	int i,grid,xscale,yscale,ndata,yoff,skip;

	if (!XtIsRealized(gw) || w->chart.ndata==0 || w->chart.data==NULL)
		return;
	skip=w->chart.skip;
	xscale=w->chart.xscale;
	yscale=w->chart.yscale;
	grid=w->chart.grid;
	ndata=w->chart.ndata;
	yy=256/yscale;                /* height (in pixels) of y chart */
	yoff=w->core.height/2-yy/2+1; /* y offset from zero to centre chart */
	                              /* yoff+yy/2 is centre of the chart */
	if (e) /* normal event */ {
		/* caution over endpoints */
		x    =(e->x      )*xscale-1;
		width=(e->width+2)*xscale;
		yt   =yoff+(e->y)/yscale;
		yb   =yoff+(e->y+e->height)/yscale;
		}
	else /* complete redraw */ {
		x=0;
		width=ndata;
		yt=yoff+yy/2;
		yb=yoff+yy/2;
		}

	if (x<0)
		x=0;
	if (x+width>=ndata)
		width=ndata-x-1;

	/* draw the grid lines */
	if (grid>0) {
		xx=x-(x%grid)+grid;
		ww=width/grid;
		for(;ww>=0;ww--,xx+=grid)
			XDrawLine(XtDisplay(w),XtWindow(w),w->chart.gridgc,
				  xx/xscale,0,xx/xscale,w->core.height);
		}

	/* is the y exposed area beyond the chart? */
	if ((yt>yy && yb>yy) || (yt<0 && yb<0))
		return;
	/* draw the chart */
	for(i=0,xx=x,ww=width;ww>=0;ww--,i++,xx++) {
		if (i>=100 || ww<=0) /* time to draw */ {
			XDrawLines(XtDisplay(gw),XtWindow(gw),
				   w->chart.gc,points,i,CoordModeOrigin);
			i=0;
			if (ww>0) ww++;
			xx--;
			if (xx<x) xx=x;
			}
		/* start point (not drawn) covered */
		points[i].x=xx/xscale;
		points[i].y=yoff+(int)(255-(w->chart.data[skip*xx]))/yscale;
		}
	/* that's it */
	}

static Boolean SetValues(Widget current,Widget request,Widget wnew,
			 ArgList args,Cardinal *nargs) {
	Boolean redraw=False;
	ChartWidget curw=(ChartWidget)current;
	ChartWidget neww=(ChartWidget)wnew;

	if ((neww->chart.fg != curw->chart.fg) ||
	    (neww->core.background_pixel != curw->core.background_pixel)) {
		redraw=True;
		XtReleaseGC(current,curw->chart.gc);
		neww->chart.gc=getlineGC(neww);
		}
	if ((neww->chart.gridcolour != curw->chart.gridcolour) ||
	    (neww->core.background_pixel != curw->core.background_pixel)) {
		redraw=True;
		XtReleaseGC(current,curw->chart.gridgc);
		neww->chart.gridgc=getgridGC(neww);
		}

	if (neww->chart.ndata != curw->chart.ndata ||
	    neww->chart.data  != curw->chart.data) {
		redraw=True;
		if (neww->chart.ndata<0) {
			neww->chart.ndata=0;
			neww->chart.data=NULL;
			}
		}
	if (neww->chart.xscale<1)
		neww->chart.xscale=1;
	if (neww->chart.yscale<1)
		neww->chart.yscale=1;
	neww->core.width=neww->chart.ndata/neww->chart.xscale;
	neww->core.height=             256/neww->chart.yscale;

	if ((neww->chart.grid   != curw->chart.grid) ||
	    (neww->chart.xscale != curw->chart.xscale) ||
	    (neww->chart.yscale != curw->chart.yscale) ||
	    (neww->chart.data   != curw->chart.data) ||
	    (neww->chart.redraw != curw->chart.redraw))
		redraw=True;
	neww->chart.redraw=False;
	return redraw;
	}

static void Destroy(Widget gw) {
	ChartWidget w=(ChartWidget)gw;
	if (w->chart.gc) {
		XtReleaseGC(gw,w->chart.gc);
		w->chart.gc=0;
		}
	if (w->chart.gridgc) {
		XtReleaseGC(gw,w->chart.gridgc);
		w->chart.gridgc=0;
		}
	}

static XtGeometryResult QueryGeometry(Widget gw,
				      XtWidgetGeometry *prop,
				      XtWidgetGeometry *ret) {
	ChartWidget w=(ChartWidget)gw;
	/* the size is in fact fixed by the scales and number of points */
	ret->request_mode=0;
	ret->width =w->chart.ndata/w->chart.xscale;
	ret->height=           256/w->chart.yscale;

	/* Oddly, prop->request_mode has bits set where the fields aren't */
	if (!(prop->request_mode & CWWidth) && prop->width!=ret->width)
		ret->request_mode|=CWWidth;
	if (!(prop->request_mode & CWHeight) && prop->height!=ret->height)
		ret->request_mode|=CWHeight;
	if (!ret->request_mode) return XtGeometryYes;
	if (prop->width==w->core.width && prop->height==w->core.height)
		return XtGeometryNo;
	return XtGeometryAlmost;
	}
