#include "surface.h"
#include <math.h>

/* Scope of transformation limited to this source file */
static matrix transformation;

/* calculate transformation matrix */
void calctransform(objectdata * object)
{
   extern FLOATTYPE r, theta, phi;
   extern matrix transformation;
   extern Itemsmode itemsmode;
   extern vector R;

   int i, j;
   vector n, V, u, v, worldcentre;
   FLOATTYPE xmin, xmax, ymin, ymax, zmin, zmax;
   patch *patchptr;

   if (itemsmode == ALL) {
      /* bounding box of whole object already stored! */
      for (i = 0; i < 3; i++)
         worldcentre[i] = 0.5 * (object->bounds.max[i] + object->bounds.min[i]);
   } else {
      /* find bounding box of current patch */
      patchptr = object->currentpatch;
      xmin = ymin = zmin = BIG;
      xmax = ymax = zmax = -BIG;
      for (i = 0; i < 4; i++)
         for (j = 0; j < 4; j++) {
            xmin = min(patchptr->data[i][j]->world[0], xmin);
            xmax = max(patchptr->data[i][j]->world[0], xmax);
            ymin = min(patchptr->data[i][j]->world[1], ymin);
            ymax = max(patchptr->data[i][j]->world[1], ymax);
            zmin = min(patchptr->data[i][j]->world[2], zmin);
            zmax = max(patchptr->data[i][j]->world[2], zmax);
         }
      worldcentre[0] = (xmin + xmax) / 2;
      worldcentre[1] = (ymin + ymax) / 2;
      worldcentre[2] = (zmin + zmax) / 2;
   }

   /* n is normalised VPN */
   n[0] = (-SIN(phi) * COS(theta));
   n[1] = (-SIN(phi) * SIN(theta));
   n[2] = (-COS(phi));

   /* R is position vector of COP */
   for (i = 0; i < 3; i++)
      R[i] = -r * n[i] + worldcentre[i];

   /* V is normalised VUV */
   V[0] = -COS(phi) * COS(theta);
   V[1] = -COS(phi) * SIN(theta);
   V[2] = SIN(phi);

   /* calculate u,v vectors */
   vecprod(n, V, u);
   normalise(u);
   vecprod(u, n, v);

   /* form view transformation matrix */
   transformation[0][0] = u[0];
   transformation[1][0] = u[1];
   transformation[2][0] = u[2];
   transformation[3][0] = -scalarprod(R, u);
   transformation[0][1] = v[0];
   transformation[1][1] = v[1];
   transformation[2][1] = v[2];
   transformation[3][1] = -scalarprod(R, v);
   transformation[0][2] = n[0];
   transformation[1][2] = n[1];
   transformation[2][2] = n[2];
   transformation[3][2] = -scalarprod(R, n);
   transformation[0][3] = 0.0;
   transformation[1][3] = 0.0;
   transformation[2][3] = 0.0;
   transformation[3][3] = 1.0;
}

/*
 * Transform point from world coordinates to screen coordinates, and generate
 * its viewcode for Cohen-Sutherland clipping
 */
void transformpoint(vertex * point, windowdata * window)
{
   extern matrix transformation;
   extern FLOATTYPE vpdist;

   int i, j;
   vector viewpoint;

   /* Transform to view coordinates - note tweak to increase speed */
   for (i = 0; i < 3; i++) {
      viewpoint[i] = 0;
      for (j = 0; j < 3; j++) {
         viewpoint[i] += point->world[j] * transformation[j][i];
      }
      viewpoint[i] += transformation[3][i];
   }

   /* Project onto viewplane and scale */
   if ((point->screen.z = viewpoint[2]) != 0) {
      point->screen.x = (Position) window->width * 0.5 
                         + window->scale * viewpoint[0] * vpdist / viewpoint[2];
      point->screen.y = (Position) window->height *0.5 
                         - window->scale * viewpoint[1] * vpdist / viewpoint[2];
   } else {
      point->screen.x = 0;
      point->screen.y = 0;
   }
   point->screen.viewcode = calcviewcode(point->screen, window);
}

/*
 * Draws clipped line joining point1 and point2, using Cohen-Sutherland
 * algorithm.  Code is written for speed rather than readability...
 */
void clipline(screenpoint * point1, screenpoint * point2, 
              Display * display, Drawable drawable, GC gc, windowdata * window)
{
   bool finished = FALSE, notvertical;
   screenpoint pt1, pt2;
   FLOATTYPE grd, dy, dx;

   if ((point1->viewcode == 0) && (point2->viewcode == 0))  /* Wholly visible */
       XDrawLine(display, drawable, gc, point1->x, point1->y, 
                point2->x, point2->y); 
   else if (!(point1->viewcode & point2->viewcode)) {  /* Partially visible */
      pt1 = *point1;
      pt2 = *point2;
      notvertical = (pt1.x != pt2.x);
      if (notvertical) {
         dx = pt2.x - pt1.x;
         dy = pt2.y - pt1.y;
         grd = dy / dx;
      }
      while (!finished) {
         if (pt1.viewcode != 0) {
            if (pt1.viewcode & 1) {              /* Point offscreen left */
               pt1.y = pt1.y + (CLIPMARGIN - pt1.x) * grd;
               pt1.x = CLIPMARGIN;
            } else if (pt1.viewcode & 2) {       /* Point offscreen right */
               pt1.y = pt1.y + (window->width-CLIPMARGIN - pt1.x) * grd;
               pt1.x = window->width - CLIPMARGIN;
            } else if (pt1.viewcode & 4) {       /* Point offscreen down */
               if (notvertical)
                  pt1.x = pt1.x + (CLIPMARGIN - pt1.y) / grd;
               pt1.y = CLIPMARGIN;
            } else if (pt1.viewcode & 8) {       /* Point offscreen up */
               if (notvertical)
                  pt1.x = pt1.x + (window->height-CLIPMARGIN - pt1.y) / grd;
               pt1.y = window->height - CLIPMARGIN;
            } else
               /* Point must be offscreen front */
            {
               if (pt1.z != pt2.z) {
                  pt1.x = pt1.x + (FRONTCLIP - pt1.z) * (pt2.x - pt1.x) 
                          / (pt2.z - pt1.z);
                  pt1.y = pt1.y + (FRONTCLIP - pt1.z) * (pt2.y - pt1.y) 
                          / (pt2.z - pt1.z);
               }
               pt1.z = FRONTCLIP;
            }

            pt1.viewcode = calcviewcode(pt1, window);
         } else {
            if (pt2.viewcode & 1) {              /* Point offscreen left */
               pt2.y = pt1.y + (CLIPMARGIN - pt1.x) * grd;
               pt2.x = CLIPMARGIN;
            } else if (pt2.viewcode & 2) {       /* Point offscreen right */
               pt2.y = pt1.y + (window->width - CLIPMARGIN - pt1.x) * grd;
               pt2.x = window->width - CLIPMARGIN;
            } else if (pt2.viewcode & 4) {       /* Point offscreen down */
               if (notvertical)
                  pt2.x = pt1.x + (CLIPMARGIN - pt1.y) / grd;
               pt2.y = CLIPMARGIN;
            } else if (pt2.viewcode & 8) {       /* Point offscreen up */
               if (notvertical)
                  pt2.x = pt1.x + (window->height - CLIPMARGIN - pt1.y) / grd;
               pt2.y = window->height - CLIPMARGIN;
            } else
               /* Point must be offscreen front */
            {
               if (pt1.z != pt2.z) {
                  pt2.x = pt1.x + (FRONTCLIP - pt1.z) * (pt2.x - pt1.x) 
                          / (pt2.z - pt1.z);
                  pt2.y = pt1.y + (FRONTCLIP - pt1.z) * (pt2.y - pt1.y) 
                          / (pt2.z - pt1.z);
               }
               pt2.z = FRONTCLIP;
            }
            pt2.viewcode = calcviewcode(pt2, window);
         }
         if ((pt1.viewcode == 0) && (pt2.viewcode == 0)) { /* Wholly visible */
            finished = TRUE;
            XDrawLine(display, drawable, gc, pt1.x, pt1.y, pt2.x, pt2.y); 
         } else if (pt1.viewcode & pt2.viewcode) {      /* Wholly invisible */
            finished = TRUE;
         }
      }
   }
}
