
#include "surface.h"
#include <stdio.h>
#include <stdlib.h>


bool
saveobject(char *name)
{
   extern objectdata *object;

   FILE *fp;
   patch *patchptr;
   vertex *vertexptr;
   int i;

   /* Renumber Object */
   renumber(object);

   /* Open file */
   fp = fopen(name, "w");
   if (fp == NULL) {
      printstatus("SaveObject: could not open file.");
      return (FALSE);
   }
   /* Write header */
   fprintf(fp, "Object made from %u patches, using %u vertices\n\n", 
           object->numpatches, object->numvertices);

   /* Write vertices */
   fprintf(fp, "Vertices:\n---------\n");
   vertexptr = object->firstvertex;
   do {
      fprintf(fp, "%u:  %+f, %+f, %+f\n", vertexptr->number, 
              vertexptr->world[0], vertexptr->world[1], vertexptr->world[2]);
      vertexptr = vertexptr->next;
   } while (vertexptr != NULL);

   /* Write patches */
   fprintf(fp, "\nPatches:\n--------\n");
   patchptr = object->firstpatch;
   do {
      fprintf(fp, "%u:  ", patchptr->number);
      for (i = 0; i < 15; i++)
         fprintf(fp, "%u, ", patchptr->data[i % 4][i / 4]->number);
      fprintf(fp, "%u\n", patchptr->data[3][3]->number);
      patchptr = patchptr->next;
   } while (patchptr != NULL);

   /* Close file */
   fclose(fp);
   return (TRUE);
}


bool
loadobject(char *name)
{
   extern objectdata *object;

   FILE *fp;
   int i, j, numpatches, numvertices, number;
   vertex **vertices, *vertexptr;
   patch *patchptr;

   /* Open file */
   fp = fopen(name, "r");
   if (fp == NULL) {
      printstatus("LoadObject: Could not open file.");
      return FALSE;
   }
   /* Read the number of vertices and patches from file */
   fscanf(fp, "Object made from %u patches, using %u vertices\n\n", 
          &numpatches, &numvertices);

   /* Check these are sensible */
   if (numpatches < 1) {
      printstatus("LoadObject: Could not read sensible number of patches.");
      return FALSE;
   }
   if ((numvertices < 9 * numpatches) && (numvertices > 12 * numpatches + 4)) {
      printstatus("LoadObject: Could not read sensible number of vertices.");
      return FALSE;
   }
   /* Delete old object */
   wipeobject(object);

   /* Create new object */
   object = createobject(numpatches, numvertices);

   /* Malloc an array in which to temporarily store pointers to vertices */
   vertices = (vertex **) malloc(numvertices * sizeof(vertex *));
   if (vertices == NULL)
      error("Malloc failed in LoadObject");

   /* Read vertices */
   fscanf(fp, "Vertices:\n---------\n");
   vertexptr = object->firstvertex;
   do {
      fscanf(fp, "%u:  %lf, %lf, %lf\n", &vertexptr->number, 
             &(vertexptr->world[0]), &(vertexptr->world[1]), 
             &(vertexptr->world[2]));
      /* Store pointer in array vertices */
      vertices[vertexptr->number - 1] = vertexptr;
      vertexptr->world[3] = 1.0;
      vertexptr->occurrences = 0;
      vertexptr = vertexptr->next;
   } while (vertexptr != NULL);

   /* Read patches */
   fscanf(fp, "\nPatches:\n--------\n");
   patchptr = object->firstpatch;
   do {
      fscanf(fp, "%u:  ", &(patchptr->number));
      for (i = 0; i < 15; i++) {
         fscanf(fp, "%u, ", &number);
         patchptr->data[i % 4][i / 4] = vertices[number - 1];
      }
      fscanf(fp, "%u\n", &number);
      patchptr->data[3][3] = vertices[number - 1];
      patchptr->visited = FALSE;
      patchptr = patchptr->next;
   } while (patchptr != NULL);

   /* Close the file */
   fclose(fp);

   /* Free the vertices array */
   free(vertices);

   /* Generate occurrence data - required by MendObject */
   patchptr = object->firstpatch;
   do {
      for (i = 0; i < 4; i++)
         for (j = 0; j < 4; j++)
            patchptr->data[i][j]->occurrences += 1;
      patchptr = patchptr->next;
   } while (patchptr != NULL);

   /* Now make the connections between patches with common vertices */
   if (!mendobject(object)) {
      /* MendObject produced torn or degenerate object */
      printstatus("LoadObject: Object invalid.");
      /* Need to replace duff object with valid one */
      wipeobject(object);
      object = initobject();
      return FALSE;
   }
   /* Generate coordinates and bounds for connectivity diagram */
   getconnectcoords(object);
   getconnectivitybounds(object);

   /* Generate the object bounds */
   findworldbounds(object);

   /* Set the current patch pointer */
   object -> currentpatch = object -> firstpatch;

   /* That's all folks */
   return TRUE;
}

/* Initialise object, patchlist, vertexlist, and set up default patch */
objectdata * initobject(void)
{
   int i, j;
   patch *patchptr;
   vertex *vertexptr, *oldvtxptr;
   objectdata *object;

   /* Malloc Object */
   object = (objectdata *) malloc(sizeof(objectdata));
   if (object == NULL)
      error("Malloc failed in initobject\n");
   object->numpatches = 1;
   object->numvertices = 16;

   /* Form patch */
   patchptr = (patch *) malloc(sizeof(patch));
   if (patchptr == NULL)
      error("Malloc failed in initobject\n");
   patchptr->next = patchptr->previous = NULL;
   patchptr->n = patchptr->s = patchptr->e = patchptr->w = NULL;
   patchptr->connectpos.x = patchptr->connectpos.y = 0;
   patchptr->visited = FALSE;
   patchptr->number = 1;
   object->currentpatch = object->firstpatch = object->lastpatch = patchptr;

   /* Now vertexlist */
   oldvtxptr = NULL;
   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++) {
         vertexptr = (vertex *) malloc(sizeof(vertex));
         if (vertexptr == NULL)
            error("Malloc failed in initobject\n");

         vertexptr->previous = oldvtxptr;
         if (oldvtxptr != NULL)
            oldvtxptr->next = vertexptr;

         vertexptr->occurrences = 1;
         vertexptr->number = 4 * i + j + 1;
         /* World Coords */
         vertexptr->world[0] = i - 1.5;
         vertexptr->world[1] = j - 1.5;
         vertexptr->world[2] = 0.0;
         vertexptr->world[3] = 1.0;

         patchptr->data[i][j] = vertexptr;
         oldvtxptr = vertexptr;
      }
   vertexptr->next = NULL;

   object->firstvertex = patchptr->data[0][0];
   object->lastvertex = patchptr->data[3][3];

   /* Set bounds */
   object->bounds.min[0] = -1.5;
   object->bounds.min[1] = -1.5;
   object->bounds.min[2] = 0.0;
   object->bounds.min[3] = 0.0;
   object->bounds.max[0] = 1.5;
   object->bounds.max[1] = 1.5;
   object->bounds.max[2] = 0.0;
   object->bounds.max[3] = 0.0;
   object->connect.xmax = object->connect.xmin = 0;
   object->connect.ymax = object->connect.ymin = 0;

   /* Return pointer to object */
   return object;
}

void wipeobject(objectdata * object)
{
   patch *patchptr, *nextpatch;
   vertex *vertexptr, *nextvertex;

   /* Free the vertices first */
   vertexptr = object->firstvertex;
   do {
      nextvertex = vertexptr->next;
      free(vertexptr);
      vertexptr = nextvertex;
   } while (vertexptr != NULL);

   /* Free the patches */
   patchptr = object->firstpatch;
   do {
      nextpatch = patchptr->next;
      free(patchptr);
      patchptr = nextpatch;
   } while (patchptr != NULL);

   /* Free object itself */
   free(object);
}
