#include "surface.h"
#include <stdio.h>
#include <stdlib.h>


/* Draws the connectivity diagram */
void drawconnectivity(objectdata * object)
{
   extern Display *connectivitydisplay;
   extern Window connectivitywindow;
   extern GC gc;
   extern windowdata cnwindow;

   int loops = 0;
   patch *patchptr;
   FLOATTYPE scale;
   Position quarterscale, x, y, x2, y2, xcentre, ycentre, halfbox;
   connectdata *connect = &object->connect;
   char label[2];

   /* Calculate scaling and offsets */
   scale = min(cnwindow.width / (connect->xmax - connect->xmin + 3), 
               cnwindow.height / (connect->ymax - connect->ymin + 3));
   halfbox = (Position) scale / 2 - BOXGAP;
   quarterscale = (Position) scale / 4;
   xcentre = (Position) (cnwindow.width - scale * 
                                           (connect->xmax + connect->xmin)) / 2;
   ycentre = (Position) (cnwindow.height - scale * 
                                           (connect->ymax + connect->ymin)) / 2;

   /* Clear window */
   XClearWindow(connectivitydisplay, connectivitywindow);

   /* Draw diagram */
   patchptr = object->firstpatch;
   do {
      x = (Position) scale *patchptr->connectpos.x + xcentre;
      y = (Position) scale *patchptr->connectpos.y + ycentre;

      /* Draw filled box for currentpatch, open otherwise */
      if (patchptr == object->currentpatch)
         XFillRectangle(connectivitydisplay, connectivitywindow, gc, x - halfbox, y - halfbox, 2 * halfbox, 2 * halfbox);
      else
         XDrawRectangle(connectivitydisplay, connectivitywindow, gc, x - halfbox, y - halfbox, 2 * halfbox, 2 * halfbox);

      /* Draw numbered lines to indicate loop connections */
      if (patchptr->n != NULL)
         if (patchptr->connectpos.y > patchptr->n->connectpos.y) {
            sprintf(label, "%u", ++loops);
            XDrawLine(connectivitydisplay, connectivitywindow, gc, x, y + quarterscale, x, y + 3 * quarterscale);
            XDrawImageString(connectivitydisplay, connectivitywindow, gc, x - CHARX, y + 3 * quarterscale + CHARY, label, 2);
            x2 = (Position) scale *patchptr->n->connectpos.x + xcentre;
            y2 = (Position) scale *patchptr->n->connectpos.y + ycentre;
            XDrawLine(connectivitydisplay, connectivitywindow, gc, x2, y2 - quarterscale, x2, y2 - 3 * quarterscale);
            XDrawImageString(connectivitydisplay, connectivitywindow, gc, x2 - CHARX, y2 - 3 * quarterscale + CHARY, label, 2);
         }
      if (patchptr->e != NULL)
         if (patchptr->connectpos.x > patchptr->e->connectpos.x) {
            sprintf(label, "%u", ++loops);
            XDrawLine(connectivitydisplay, connectivitywindow, gc, x + quarterscale, y, x + 3 * quarterscale, y);
            XDrawImageString(connectivitydisplay, connectivitywindow, gc, x + 3 * quarterscale - CHARX, y + CHARY, label, 2);
            x2 = (Position) scale *patchptr->e->connectpos.x + xcentre;
            y2 = (Position) scale *patchptr->e->connectpos.y + ycentre;

            XDrawLine(connectivitydisplay, connectivitywindow, gc, x2 - quarterscale, y2, x2 - 3 * quarterscale, y2);
            XDrawImageString(connectivitydisplay, connectivitywindow, gc, x2 - 3 * quarterscale - CHARX, y2 + CHARY, label, 2);
         }
      patchptr = patchptr->next;
   } while (patchptr != NULL);

   /* Store scale and centre for connectivityclicked */
   connect->scale = scale;
   connect->xcentre = xcentre;
   connect->ycentre = ycentre;
}

/* Form parent array - needed in MendObject, ConnectPatches etc. */
patch *** formparent(objectdata * object)
{
   patch *patchptr, ***parent;
   int i;

   /* Malloc and initialise array of pointers to parent patches */
   parent = (patch ***) malloc(object->numvertices * sizeof(patch **));
   if (parent == NULL)
      error("Malloc failed in FormParent");
   for (i = 0; i < object->numvertices; i++) {
      parent[i] = (patch **) calloc(4, sizeof(patch *));
      if (parent[i] == NULL)
         error("Calloc failed in FormParent");
   }

   /* Renumber object - may be gaps due to deletions */
   renumber(object);

   #ifdef DEBUG
     check(object);
   #endif

   /* Run through the patch, and fill up the parent array */
   patchptr = object->firstpatch;
   do {
      /* Only consider corners for the time being */
      parent[patchptr->data[0][0]->number - 1][0] = patchptr;
      parent[patchptr->data[0][3]->number - 1][1] = patchptr;
      parent[patchptr->data[3][3]->number - 1][2] = patchptr;
      parent[patchptr->data[3][0]->number - 1][3] = patchptr;

      patchptr = patchptr->next;
   } while (patchptr != NULL);

   return parent;
}

/* MendObject fixes connectivity of an object. */
bool
mendobject(objectdata * object)
{
   int i, oldnumvertices;
   bool success;
   patch ***parent;      /* Whoooooaaah ! */

   /* ConnectShared may change numvertices, so save a copy */
   oldnumvertices = object->numvertices;

   /* Form array of parent patches to each vertex */
   parent = formparent(object);

   /* Call ConnectShared to fix object */
   success = connectshared(object, parent);

   /* Free the parent array */
   for (i = 0; i < oldnumvertices; i++) {
      free(parent[i]);
   }
   free(parent);

   /* Pass back result */
   return success;
}

/* ConnectShared is called recursively to join patches with shared vertices */
bool
connectshared(objectdata * object, patch *** parent)
{
   vertex *vertexptr;
   patch **daddy;
   bool changed = FALSE;

   /* Check for shared vertices */
   vertexptr = object->firstvertex;
   do {
      if (vertexptr->occurrences > 1) {
         /*
          * Use daddy[?] as shorthand for
          * parent[vertexptr->number-1][?]
          */
         daddy = parent[vertexptr->number - 1];

         /*
          * If vertex has NE & SE parent, then check patches
          * are connected
          */
         if ((daddy[0] != NULL) && (daddy[1] != NULL))
            /* Only bother checking connection one way */
            if (daddy[0]->s != daddy[1]) {
               /*
                * Call JoinPatches to make
                * connection
                */
               if (!joinpatches(object, NS, daddy[0], daddy[1], parent))
                  return FALSE;
               /*
                * Set flag to show object has been
                * altered
                */
               changed = TRUE;
            }
         /* Now check SE and SW parents... */
         if ((daddy[1] != NULL) && (daddy[2] != NULL))
            if (daddy[1]->w != daddy[2]) {
               if (!joinpatches(object, EW, daddy[1], daddy[2], parent))
                  return FALSE;
               changed = TRUE;
            }
         if ((daddy[2] != NULL) && (daddy[3] != NULL))
            if (daddy[2]->n != daddy[3]) {
               if (!joinpatches(object, NS, daddy[3], daddy[2], parent))
                  return FALSE;
               changed = TRUE;
            }
         if ((daddy[3] != NULL) && (daddy[0] != NULL))
            if (daddy[3]->e != daddy[0]) {
               if (!joinpatches(object, EW, daddy[0], daddy[3], parent))
                  return FALSE;
               changed = TRUE;
            }
      }
      vertexptr = vertexptr->next;
   } while (vertexptr != NULL);

   /*
    * If object has been changed, call routine recursively to make joins
    * that may have been implied by changes. Otherwise terminate
    * succesfully
    */
   if (changed)
      return connectshared(object, parent);
   else
      return TRUE;
}

/*
 * Brings together two edges of patches to be joined. Compare with
 * ConnectPatches.
 */
bool
joinpatches(objectdata * object, jointype join, patch * patchptr1, 
            patch * patchptr2, patch *** parent)
{
   extern Continuitymode continuitymode;
   vector edge, grad1, grad2, edgedir, graddir;
   FLOATTYPE magedge, maggrad;
   int i, j, number1, number2;
   vertex *vertexptr1, *vertexptr2;

   /* Make either a NS or an EW connection */
   switch (join) {
     case NS:

        /* Check validity of join before doing anything */
        for (i = 0; i < 4; i++) {
           number1 = patchptr1->data[i][0]->number - 1;
           number2 = patchptr2->data[i][3]->number - 1;
           for (j = 0; j < 4; j++)
              if ((parent[number1][j] != NULL) && (parent[number2][j] != NULL))
                 if (parent[number1][j] != parent[number2][j]) {
                    printstatus("JoinPatches: Operation results in degenerate object.");
                    return FALSE;
                 }
        }
        /* For each point along the edge... */
        for (i = 0; i < 4; i++) {
           /* A bit of shorthand */
           vertexptr1 = patchptr1->data[i][0];
           vertexptr2 = patchptr2->data[i][3];
           number1 = vertexptr1->number - 1;
           number2 = vertexptr2->number - 1;

           /* If the vertices are distinct... */
           if (vertexptr1 != vertexptr2) {
              /* Move the vertices */
              for (j = 0; j < 3; j++) {
                 /* edge will be new position of edge */
                 edge[j] = (patchptr1->data[i][0]->world[j] 
                           + patchptr2->data[i][3]->world[j]) / 2;
                 /*
                  * vectors required for smooth join
                  * calculations
                  */
                 edgedir[j] = (patchptr1->data[i][0]->world[j] 
                              - patchptr2->data[i][3]->world[j])/2;
                 graddir[j] = (patchptr1->data[i][0]->world[j] 
                              - patchptr1->data[i][1]->world[j] 
                              + patchptr2->data[i][2]->world[j]
                              - patchptr2->data[i][3]->world[j])/2; 
              }

              /* Make a smooth join */
              if (continuitymode == FIRST) {
                 magedge = magnitude(edgedir);
                 maggrad = magnitude(graddir);
                 /* Attempt to make `intuitive' join */
                 if (magedge > maggrad) {
                    /* Set gradient vectors */
                    for (j = 0; j < 3; j++) {
                       grad1[j] = -edgedir[j];
                       grad2[j] = -grad1[j];
                    }
                 } else {
                    /* Set gradient vectors */
                    for (j = 0; j < 3; j++) {
                       grad1[j] = graddir[j];
                       grad2[j] = -grad1[j];
                    }
                 }
              }
              /* Set the coordinates of the control points */
              for (j = 0; j < 3; j++) {
                 patchptr1->data[i][0]->world[j] = edge[j];
                 patchptr2->data[i][3]->world[j] = edge[j];
                 patchptr1->data[i][1]->world[j] = edge[j] - grad1[j];
                 patchptr2->data[i][2]->world[j] = edge[j] - grad2[j];
              }

              /* Now sort out pointers */
              for (j = 0; j < 4; j++)
                 if ((parent[number1][j] == NULL) && 
                      (parent[number2][j] != NULL)) {
                    /*
                     * Set pointers to second
                     * vertex to point to first
                     */
                    parent[number2][j]->data[3 * (j / 2)][(j % 3 == 0) ? 0 : 3]
                      = vertexptr1;
                    /*
                     * Merge entries in parent
                     * array
                     */
                    parent[number1][j] = parent[number2][j];
                    parent[number2][j] = NULL;
                 }
              /*
               * Parent array only holds data on corner
               * vertices. Fix pointers for vertices in
               * middle of edge
               */
              patchptr2->data[i][3] = vertexptr1;
              /* Change occurrence count for vertex */
              vertexptr1->occurrences += vertexptr2->occurrences;
              /* Delete vertex from second patch */
              killvertex(object, vertexptr2);
           }
        }

        /*
         * Make a crude attempt to maintain continuity with
         * neighbours
         */ 
        if (continuitymode == FIRST) {
           for (j = 0; j < 3; j++) {
              if (patchptr1->w != NULL) {
                 patchptr1->w->data[3][0]->world[j] = 
                   patchptr1->data[0][0]->world[j];
                 patchptr1->w->data[2][0]->world[j] = 
                   2 * patchptr1->data[0][0]->world[j] 
                   - patchptr1->data[1][0]->world[j];
                 patchptr1->w->data[3][1]->world[j] = 
                   patchptr1->data[0][1]->world[j];
                 patchptr1->w->data[2][1]->world[j] = 
                   2 * patchptr1->data[0][1]->world[j] 
                   - patchptr1->data[1][1]->world[j];
              }
              if (patchptr1->e != NULL) {
                 patchptr1->e->data[0][0]->world[j] = 
                   patchptr1->data[3][0]->world[j];
                 patchptr1->e->data[1][0]->world[j] = 
                   2 * patchptr1->data[3][0]->world[j] 
                   - patchptr1->data[2][0]->world[j];
                 patchptr1->e->data[0][1]->world[j] = 
                   patchptr1->data[3][1]->world[j];
                 patchptr1->e->data[1][1]->world[j] = 
                   2 * patchptr1->data[3][1]->world[j] 
                   - patchptr1->data[2][1]->world[j];
              }
              if (patchptr2->w != NULL) {
                 patchptr2->w->data[3][3]->world[j] = 
                   patchptr2->data[0][3]->world[j];
                 patchptr2->w->data[2][3]->world[j] = 
                   2 * patchptr2->data[0][3]->world[j] 
                   - patchptr2->data[1][3]->world[j];
                 patchptr2->w->data[3][2]->world[j] = 
                   patchptr2->data[0][2]->world[j];
                 patchptr2->w->data[2][2]->world[j] = 
                   2 * patchptr2->data[0][2]->world[j] 
                   - patchptr2->data[1][2]->world[j];
              }
              if (patchptr2->e != NULL) {
                 patchptr2->e->data[0][3]->world[j] = 
                   patchptr2->data[3][3]->world[j];
                 patchptr2->e->data[1][3]->world[j] = 
                   2 * patchptr2->data[3][3]->world[j] 
                   - patchptr2->data[2][3]->world[j];
                 patchptr2->e->data[0][2]->world[j] = 
                   patchptr2->data[3][2]->world[j];
                 patchptr2->e->data[1][2]->world[j] = 
                   2 * patchptr2->data[3][2]->world[j] 
                   - patchptr2->data[2][2]->world[j];
              }
           }
        }
       
        /* Change the connectivity */
        patchptr1->s = patchptr2;
        patchptr2->n = patchptr1;

        break;

     case EW:

        /* Check validity of join before doing anything */
        for (i = 0; i < 4; i++) {
           number1 = patchptr1->data[0][i]->number - 1;
           number2 = patchptr2->data[3][i]->number - 1;
           for (j = 0; j < 4; j++)
              if ((parent[number1][j] != NULL) && (parent[number2][j] != NULL))
                 if (parent[number1][j] != parent[number2][j]) {
                    printstatus("JoinPatches: Operation results in degenerate object.");
                    return FALSE;
                 }
        }

        /* For each point along the edge... */
        for (i = 0; i < 4; i++) {
           /* A bit of shorthand */
           vertexptr1 = patchptr1->data[0][i];
           vertexptr2 = patchptr2->data[3][i];
           number1 = vertexptr1->number - 1;
           number2 = vertexptr2->number - 1;

           /* If the vertices are distinct... */
           if (vertexptr1 != vertexptr2) {
              /* Move the vertices */
              for (j = 0; j < 3; j++) {
                 /* edge will be new position of edge */
                 edge[j] = (patchptr1->data[0][i]->world[j] 
                            + patchptr2->data[3][i]->world[j]) / 2;
                 /*
                  * vectors required for smooth join
                  * calculations
                  */
                 edgedir[j] = (patchptr1->data[0][i]->world[j] 
                              - patchptr2->data[3][i]->world[j])/2;
                 graddir[j] = (patchptr1->data[0][i]->world[j] 
                              - patchptr1->data[1][i]->world[j] 
                              + patchptr2->data[2][i]->world[j]
                              - patchptr2->data[3][i]->world[j])/2;
              }

              /* Make a smooth join */
              if (continuitymode == FIRST) {
                 magedge = magnitude(edgedir);
                 maggrad = magnitude(graddir);
                 /* Attempt to make `intuitive' join */
                 if (magedge > maggrad) {
                    /* Set gradient vectors */
                    for (j = 0; j < 3; j++) {
                       grad1[j] = -edgedir[j];
                       grad2[j] = -grad1[j];
                    }
                 } else {
                    /* Set gradient vectors */
                    for (j = 0; j < 3; j++) {
                       grad1[j] = graddir[j];
                       grad2[j] = -grad1[j];
                    }
                 }
              }
              /* Set the coordinates of the control points */
              for (j = 0; j < 3; j++) {
                 patchptr1->data[0][i]->world[j] = edge[j];
                 patchptr2->data[3][i]->world[j] = edge[j];
                 patchptr1->data[1][i]->world[j] = edge[j] - grad1[j];
                 patchptr2->data[2][i]->world[j] = edge[j] - grad2[j];
              }

              /* Now sort out pointers */
              for (j = 0; j < 4; j++)
                 if ((parent[number1][j] == NULL) && 
                      (parent[number2][j] != NULL)) {
                    /*
                     * Set pointers to second
                     * vertex to point to first
                     */
                    parent[number2][j]->data[3 * (j / 2)][(j % 3 == 0) ? 0 : 3] 
                      = vertexptr1;
                    /*
                     * Merge the entries in
                     * parent array
                     */
                    parent[number1][j] = parent[number2][j];
                    parent[number2][j] = NULL;
                 }
              /*
               * Parent array only holds data on corner
               * vertices. Fix pointers for vertices in
               * middle of edge
               */
              patchptr2->data[3][i] = vertexptr1;
              /* Change occurrence count for vertex */
              vertexptr1->occurrences += vertexptr2->occurrences;
              /* Delete vertex from second patch */
              killvertex(object, vertexptr2);
           }
        }

        /*
         * Make a crude attempt to maintain continuity with
         * neighbours
         */ 
        if (continuitymode == FIRST) {
           for (j = 0; j < 3; j++) {
              if (patchptr1->s != NULL) {
                 patchptr1->s->data[0][3]->world[j] = 
                   patchptr1->data[0][0]->world[j];
                 patchptr1->s->data[0][2]->world[j] = 
                   2 * patchptr1->data[0][0]->world[j] 
                   - patchptr1->data[0][1]->world[j];
                 patchptr1->s->data[1][3]->world[j] =
                   patchptr1->data[1][0]->world[j];
                 patchptr1->s->data[1][2]->world[j] = 
                   2 * patchptr1->data[1][0]->world[j] 
                   - patchptr1->data[1][1]->world[j];
              }
              if (patchptr1->n != NULL) {
                 patchptr1->n->data[0][0]->world[j] = 
                   patchptr1->data[0][3]->world[j];
                 patchptr1->n->data[0][1]->world[j] = 
                   2 * patchptr1->data[0][3]->world[j] 
                   - patchptr1->data[0][2]->world[j];
                 patchptr1->n->data[1][0]->world[j] = 
                   patchptr1->data[1][3]->world[j];
                 patchptr1->n->data[1][1]->world[j] = 
                   2 * patchptr1->data[1][3]->world[j] 
                   - patchptr1->data[1][2]->world[j];
              }
              if (patchptr2->s != NULL) {
                 patchptr2->s->data[3][3]->world[j] = 
                   patchptr2->data[3][0]->world[j];
                 patchptr2->s->data[3][2]->world[j] = 
                   2 * patchptr2->data[3][0]->world[j] 
                   - patchptr2->data[3][1]->world[j];
                 patchptr2->s->data[2][3]->world[j] = 
                   patchptr2->data[2][0]->world[j];
                 patchptr2->s->data[2][2]->world[j] = 
                   2 * patchptr2->data[2][0]->world[j] 
                   - patchptr2->data[2][1]->world[j];
              }
              if (patchptr2->n != NULL) {
                 patchptr2->n->data[3][0]->world[j] = 
                   patchptr2->data[3][3]->world[j];
                 patchptr2->n->data[3][1]->world[j] = 
                   2 * patchptr2->data[3][3]->world[j] 
                   - patchptr2->data[3][2]->world[j];
                 patchptr2->n->data[2][0]->world[j] = 
                   patchptr2->data[2][3]->world[j];
                 patchptr2->n->data[2][1]->world[j] = 
                   2 * patchptr2->data[2][3]->world[j] 
                   - patchptr2->data[2][2]->world[j];
              }
           }
        }
 
        /* Change the connnctivity */
        patchptr1->w = patchptr2;
        patchptr2->e = patchptr1;

        break;
   }

   /* Live happily ever after */
   return TRUE;
}

/*
 * TidyObject runs through the patchlist, and removes any duplicated vertices
 * that lie along the edges of joined patches
 */
bool tidyobject(objectdata * object)
{
   patch *patchptr;
   int i;

   /* Run throught patch */
   patchptr = object->firstpatch;
   do {
      /* If patch has a north neighbour */
      if (patchptr->n != NULL) {
         /*
          * Check that all vertices along north edge are
          * shared
          */
         for (i = 0; i < 4; i++)
            if (patchptr->data[i][3] != patchptr->n->data[i][0])
               /* If not, then merge the two */
               if (!mergevertices(object, &patchptr->data[i][3], 
                                  &patchptr->n->data[i][0]))
                  /*
                   * If mergevertices failed,
                   * then return FALSE
                   */
                  return FALSE;
      }
      /* If patch has a south neighbour... */
      if (patchptr->s != NULL) {
         for (i = 0; i < 4; i++)
            if (patchptr->data[i][0] != patchptr->s->data[i][3])
               if (!mergevertices(object, &patchptr->data[i][0], 
                   &patchptr->s->data[i][3]))
                  return FALSE;
      }
      if (patchptr->e != NULL) {
         for (i = 0; i < 4; i++)
            if (patchptr->data[3][i] != patchptr->e->data[0][i])
               if (!mergevertices(object, &patchptr->data[3][i], 
                                  &patchptr->e->data[0][i]))
                  return FALSE;
      }
      if (patchptr->w != NULL) {
         for (i = 0; i < 4; i++)
            if (patchptr->data[0][i] != patchptr->w->data[3][i])
               if (!mergevertices(object, &patchptr->data[0][i], 
                                  &patchptr->w->data[3][i]))
                  return FALSE;
      }
      patchptr = patchptr->next;
   } while (patchptr != NULL);

   return TRUE;
}

/*
 * MergeVertices takes the addresses of two pointers to vertices, and
 * replaces the value of those variables with pointers to the vertex with the
 * smaller number.
 */
bool
mergevertices(objectdata * object, vertex ** var1, vertex ** var2)
{
   vertex **temp;
   int i;

   /* Quick check that routine is not going to do something silly */
   if (*var1 == *var2)
      error("MergeVertices: Cannot merge identical vertices.");

   /*
    * Should never be called to merge vertices with different
    * coordinates
    */
   for (i = 0; i < 3; i++)
      if (difference((*var1)->world[i], (*var2)->world[i]) > ERRORMARGIN) {
         printstatus("MergeVertices: Warning - vertices spatially separate.");
         return FALSE;
      }
   /* Make sure that var1 holds pointer to vertex with lower number */
   if ((*var1)->number > (*var2)->number) {
      /* Swap them over */
      temp = var1;
      var1 = var2;
      var2 = temp;
   }
   /* Decrement occurrence count for 2nd vertex, remove if zero */
   if (--((*var2)->occurrences) == 0)
      killvertex(object, (*var2));

   /* Increase occurrence count for 1st vertex */
   (*var1)->occurrences += 1;

   /* Make second variable hold same pointer as first */
   (*var2) = (*var1);

   return TRUE;
}

/*
 * Attempts to generate patch that would fit in position (pos->x, pos->y) 
 * in the connectivity diagram
 */
bool
createpatch(objectdata ** objectaddress, connectpoint *pos)
{
   patch *newpatch, *patchptr, *np = NULL, *sp = NULL, *ep = NULL, *wp = NULL;
   int i, j;
   bool n, s, e, w;
   objectdata *object = *objectaddress, *copy;

   /* Look for immediate neighours in connectivity diagram */
   patchptr = object->firstpatch;
   do {
      if ((patchptr->connectpos.x == pos->x) && 
           (patchptr->connectpos.y == pos->y + 1))
         np = patchptr;
      if ((patchptr->connectpos.x == pos->x) && 
           (patchptr->connectpos.y == pos->y - 1))
         sp = patchptr;
      if ((patchptr->connectpos.x == pos->x + 1) && 
           (patchptr->connectpos.y == pos->y))
         ep = patchptr;
      if ((patchptr->connectpos.x == pos->x - 1) && 
           (patchptr->connectpos.y == pos->y))
         wp = patchptr;

      patchptr = patchptr->next;
   } while (patchptr != NULL);

   /* If the patch has no neighbours, then quit */
   if (!(np || sp || ep || wp))
      return FALSE;

   /*
    * Now check neighbours' neighbour's neighbours (!) for consistency.
    * If these are inconsistent, then creation of patch will result in a
    * degenerate surface.  Ugly? - Tell me about it.
    */
   if (np) {
      if ((np->e) && (np->e->s)) {
         if (!ep)
            ep = np->e->s;
         else if (ep != np->e->s) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
      if ((np->w) && (np->w->s)) {
         if (!wp)
            wp = np->w->s;
         else if (wp != np->w->s) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
   }
   if (sp) {
      if ((sp->e) && (sp->e->n)) {
         if (!ep)
            ep = sp->e->n;
         else if (ep != sp->e->n) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
         return FALSE;
         }
      }
      if ((sp->w) && (sp->w->n)) {
         if (!wp)
            wp = sp->w->n;
         else if (wp != sp->w->n) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
   }
   if (ep) {
      if ((ep->n) && (ep->n->w)) {
         if (!np)
            np = ep->n->w;
         else if (np != ep->n->w) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
      if ((ep->s) && (ep->s->w)) {
         if (!sp)
            sp = ep->s->w;
         else if (sp != ep->s->w) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
   }
   if (wp) {
      if ((wp->n) && (wp->n->e)) {
         if (!np)
            np = wp->n->e;
         else if (np != wp->n->e) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
      if ((wp->s) && (wp->s->e)) {
         if (!sp)
            sp = wp->s->e;
         else if (sp != wp->s->e) {
            printstatus("CreatePatch: Connectivity conflict - cannot proceed.");
            return FALSE;
         }
      }
   }

   /*
    * Check that there is not already a patch in that space - this can
    * occur with looped objects.
    */
   if ((np) && (np->s)) {
      printstatus("CreatePatch: Patch exists already.");
      return FALSE;
   }
   if ((sp) && (sp->n)) {
      printstatus("CreatePatch: Patch exists already.");
      return FALSE;
   }
   if ((ep) && (ep->w)) {
      printstatus("CreatePatch: Patch exists already.");
      return FALSE;
   }
   if ((wp) && (wp->e)) {
      printstatus("CreatePatch: Patch exists already.");
      return FALSE;
   }

   /* Now malloc new patch */
   newpatch = (patch *) malloc(sizeof(patch));
   if (newpatch == NULL)
      error("Malloc failed in CreatePatch.");

   /* Insert into patchlist */
   object->lastpatch->next = newpatch;
   newpatch->previous = object->lastpatch;
   object->lastpatch = newpatch;
   object->currentpatch = newpatch;
   object->connect.xmin = min(object->connect.xmin, pos->x);
   object->connect.xmax = max(object->connect.xmax, pos->x);
   object->connect.ymin = min(object->connect.ymin, pos->y);
   object->connect.ymax = max(object->connect.ymax, pos->y);
   newpatch->next = NULL;
   newpatch->connectpos.x = pos->x;
   newpatch->connectpos.y = pos->y;
   newpatch->visited = FALSE;
   newpatch->number = ++object->numpatches;
   newpatch->n = np;
   newpatch->s = sp;
   newpatch->e = ep;
   newpatch->w = wp;
   if (np)
      np->s = newpatch;
   if (sp)
      sp->n = newpatch;
   if (ep)
      ep->w = newpatch;
   if (wp)
      wp->e = newpatch;

   /* Malloc vertices and insert into vertexlist */
   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++)
         newpatch->data[i][j] = createvertex(object);
 
   /* Define n,s,e,w shorthand */ 
   n = (np != NULL);
   s = (sp != NULL);
   e = (ep != NULL);
   w = (wp != NULL);

   /*
    * Set world coordinates of the vertices - make compromise
    * when extrapolating from neighbouring patches 
    */ 
   for (i = 0; i < 3; i++) {
      newpatch->data[0][0]->world[i] = 
          ((n*!s*!w)?(4*np->data[0][0]->world[i]-3*np->data[0][1]->world[i])/(n+e):0)
        + ((s)?(sp->data[0][3]->world[i])/(s+w):0)
        + ((e*!s*!w)?(4*ep->data[0][0]->world[i]-3*ep->data[1][0]->world[i])/(n+e):0)
        + ((w)?(wp->data[3][0]->world[i])/(s+w):0);
      newpatch->data[1][0]->world[i] = 
          ((n*!s*!w)?(4*np->data[1][0]->world[i]-3*np->data[1][1]->world[i])/(n+e):0)
        + ((s)?(sp->data[1][3]->world[i])/(s+w):0)
        + ((e*!s*!w)?(3*ep->data[0][0]->world[i]-2*ep->data[1][0]->world[i])/(n+e):0)
        + ((w)?(2*wp->data[3][0]->world[i]-wp->data[2][0]->world[i])/(s+w):0);
      newpatch->data[2][0]->world[i] = 
          ((n*!s*!e)?(4*np->data[2][0]->world[i]-3*np->data[2][1]->world[i])/(n+w):0)
        + ((s)?(sp->data[2][3]->world[i])/(s+e):0)
        + ((e)?(2*ep->data[0][0]->world[i]-ep->data[1][0]->world[i])/(s+e):0)
        + ((w*!s*!e)?(3*wp->data[3][0]->world[i]-2*wp->data[2][0]->world[i])/(n+w):0);
      newpatch->data[3][0]->world[i] = 
          ((n*!s*!e)?(4*np->data[3][0]->world[i]-3*np->data[3][1]->world[i])/(n+w):0)
        + ((s)?(sp->data[3][3]->world[i])/(s+e):0)
        + ((e)?(ep->data[0][0]->world[i])/(s+e):0)
        + ((w*!s*!e)?(4*wp->data[3][0]->world[i]-3*wp->data[2][0]->world[i])/(n+w):0);
      newpatch->data[0][1]->world[i] = 
          ((n*!s*!w)?(3*np->data[0][0]->world[i]-2*np->data[0][1]->world[i])/(n+e):0)
        + ((s)?(2*sp->data[0][3]->world[i]-sp->data[0][2]->world[i])/(s+w):0)
        + ((e*!s*!w)?(4*ep->data[0][1]->world[i]-3*ep->data[1][1]->world[i])/(n+e):0)
        + ((w)?(wp->data[3][1]->world[i])/(s+w):0);
      newpatch->data[1][1]->world[i] = 
          ((n*!s*!w)?(3*np->data[1][0]->world[i]-2*np->data[1][1]->world[i])/(n+e):0)
        + ((s)?(2*sp->data[1][3]->world[i]-sp->data[1][2]->world[i])/(s+w):0)
        + ((e*!s*!w)?(3*ep->data[0][1]->world[i]-2*ep->data[1][1]->world[i])/(n+e):0)
        + ((w)?(2*wp->data[3][1]->world[i]-wp->data[2][1]->world[i])/(s+w):0);
      newpatch->data[2][1]->world[i] = 
          ((n*!s*!e)?(3*np->data[2][0]->world[i]-2*np->data[2][1]->world[i])/(n+w):0)
        + ((s)?(2*sp->data[2][3]->world[i]-sp->data[2][2]->world[i])/(s+e):0)
        + ((e)?(2*ep->data[0][1]->world[i]-ep->data[1][1]->world[i])/(s+e):0)
        + ((w*!s*!e)?(3*wp->data[3][1]->world[i]-2*wp->data[2][1]->world[i])/(n+w):0);
      newpatch->data[3][1]->world[i] = 
          ((n*!s*!e)?(3*np->data[3][0]->world[i]-2*np->data[3][1]->world[i])/(n+w):0)
        + ((s)?(2*sp->data[3][3]->world[i]-sp->data[3][2]->world[i])/(s+e):0)
        + ((e)?(ep->data[0][1]->world[i])/(s+e):0)
        + ((w*!s*!e)?(4*wp->data[3][1]->world[i]-3*wp->data[2][1]->world[i])/(n+w):0);
      newpatch->data[0][2]->world[i] = 
          ((n)?(2*np->data[0][0]->world[i]-np->data[0][1]->world[i])/(n+w):0)
        + ((s*!n*!w)?(3*sp->data[0][3]->world[i]-2*sp->data[0][2]->world[i])/(s+e):0)
        + ((e*!n*!w)?(4*ep->data[0][2]->world[i]-3*ep->data[1][2]->world[i])/(s+e):0)
        + ((w)?(wp->data[3][2]->world[i])/(n+w):0);
      newpatch->data[1][2]->world[i] = 
          ((n)?(2*np->data[1][0]->world[i]-np->data[1][1]->world[i])/(n+w):0)
        + ((s*!n*!w)?(3*sp->data[1][3]->world[i]-2*sp->data[1][2]->world[i])/(s+e):0)
        + ((e*!n*!w)?(3*ep->data[0][2]->world[i]-2*ep->data[1][2]->world[i])/(s+e):0)
        + ((w)?(2*wp->data[3][2]->world[i]-wp->data[2][2]->world[i])/(n+w):0);
      newpatch->data[2][2]->world[i] = 
          ((n)?(2*np->data[2][0]->world[i]-np->data[2][1]->world[i])/(n+e):0)
        + ((s*!n*!e)?(3*sp->data[2][3]->world[i]-2*sp->data[2][2]->world[i])/(s+w):0)
        + ((e)?(2*ep->data[0][2]->world[i]-ep->data[1][2]->world[i])/(n+e):0)
        + ((w*!n*!e)?(3*wp->data[3][2]->world[i]-2*wp->data[2][2]->world[i])/(s+w):0);
      newpatch->data[3][2]->world[i] = 
          ((n)?(2*np->data[3][0]->world[i]-np->data[3][1]->world[i])/(n+e):0)
        + ((s*!n*!e)?(3*sp->data[3][3]->world[i]-2*sp->data[3][2]->world[i])/(s+w):0)
        + ((e)?(ep->data[0][2]->world[i])/(n+e):0)
        + ((w*!n*!e)?(4*wp->data[3][2]->world[i]-3*wp->data[2][2]->world[i])/(s+w):0);
      newpatch->data[0][3]->world[i] = ((n)?(np->data[0][0]->world[i])/(n+w):0)
        + ((s*!n*!w)?(4*sp->data[0][3]->world[i]-3*sp->data[0][2]->world[i])/(s+e):0)
        + ((e*!n*!w)?(4*ep->data[0][3]->world[i]-3*ep->data[1][3]->world[i])/(s+e):0)
        + ((w)?(wp->data[3][3]->world[i])/(n+w):0);
      newpatch->data[1][3]->world[i] = ((n)?(np->data[1][0]->world[i])/(n+w):0)
        + ((s*!n*!w)?(4*sp->data[1][3]->world[i]-3*sp->data[1][2]->world[i])/(s+e):0)
        + ((e*!n*!w)?(3*ep->data[0][3]->world[i]-2*ep->data[1][3]->world[i])/(s+e):0)
        + ((w)?(2*wp->data[3][3]->world[i]-wp->data[2][3]->world[i])/(n+w):0);
      newpatch->data[2][3]->world[i] = ((n)?(np->data[2][0]->world[i])/(n+e):0)
        + ((s*!n*!e)?(4*sp->data[2][3]->world[i]-3*sp->data[2][2]->world[i])/(s+w):0)
        + ((e)?(2*ep->data[0][3]->world[i]-ep->data[1][3]->world[i])/(n+e):0)
        + ((w*!n*!e)?(3*wp->data[3][3]->world[i]-2*wp->data[2][3]->world[i])/(s+w):0);
      newpatch->data[3][3]->world[i] = ((n)?(np->data[3][0]->world[i])/(n+e):0)
        + ((s*!n*!e)?(4*sp->data[3][3]->world[i]-3*sp->data[3][2]->world[i])/(s+w):0)
        + ((e)?(ep->data[0][3]->world[i])/(n+e):0)
        + ((w*!n*!e)?(4*wp->data[3][3]->world[i]-3*wp->data[2][3]->world[i])/(s+w):0);
   } 

   /* Now call TidyObject to remove duplicate vertices along the edges */
   if (!tidyobject(object)) {
      /* If this fails, then undo all the work we have just done */
      printstatus("CreatePatch: Neighbour constraints inconsistent.");
      deletepatch(object, newpatch);
      return FALSE;
   }

   /* Make copy of object in case MendObject fails */
   /* Note: copy missing connectivity data, screen coords, & differences  */
   copy = createobject(object->numpatches, object->numvertices);
   copyobject(object, copy);

   /* Call MendObject */
   if (!mendobject(object)) {
      /* If MendObject fails, object is degenerate */
      printstatus("CreatePatch: operation results in degenerate object.");
      /* Replace object by copy */
      wipeobject(object);
      *objectaddress = (object = copy);
      /* Delete the patch we just created */
      deletepatch(object, object->lastpatch);
      /* Call MendObject again to connect up the copied object */
      if (!mendobject(object))
         /*
          * If MendObject fails again, then evince mild
          * surprise
          */
         error("CreatePatch: MendObject failed to connect object copy.");
      /* Generate screencoords and difference matrices */
      calccontrol(object);
      calcbezier(object);
      /* Unsuccessful */
      return FALSE;
   }
   /* Delete copy of object */
   wipeobject(copy);

   /* Do the hand jive */
   return TRUE;
}

/*
 * ConnectPatches connects the relevant edges of two patches.  It is similar
 * to MendObject in structure, as it uses JoinPatches to make the join, and
 * which requires the parent array to have been formed.
 */
void connectpatches(objectdata ** objaddress, jointype join, 
                    patch * patchptr1, patch * patchptr2)
{
   int i, oldnumvertices;
   patch ***parent;
   objectdata *object = *objaddress, *copy;

   /* ConnectShared may change numvertices - save a copy */
   oldnumvertices = object->numvertices;

   /* Make copy of object in case ConnectShared fails */
   /* Note: copy missing connectivity data, screen coords, & differences */
   copy = createobject(object->numpatches, object->numvertices);
   copyobject(object, copy);

   /* Form array of parent patches to each vertex */
   parent = formparent(object);

   /* Call JoinPatches to join the patches */
   if (joinpatches(object, join, patchptr1, patchptr2, parent) && 
       !connectshared(object, parent)) {
      /* Replace object with backup copy */
      wipeobject(object);
      *objaddress = (object = copy);
      /* Connect up copied object with MendObject */
      if (!mendobject(object))
         error("ConnectPatches: MendObject failed to connect backup object.");
      /*
       * Call CalcControl to generate screen coordinates of
       * vertices
       */
      calccontrol(object);
      calcbezier(object);
   } else {
      /* no need to keep copy */
      wipeobject(copy);
      /*
       * update displays - not connectivity as that is done by
       * parent
       */
      findworldbounds(object);
      calctransform(object);
      calccontrol(object);
      drawcontrol(object);
      calcbezier(object);
      displaybezier(object);
   }

   /* Free parent array */
   for (i = 0; i < oldnumvertices; i++) {
      free(parent[i]);
   }
   free(parent);
}

/*
 * Form stucture that will contain all the data relevant to the join at the
 * corner of the current patch closest to selected point
 */
void formjoinstruct(patch * current, int col, int row, joindata * join)
{
   patch *north, *south, *east, *west;
   int i, j, k;

   for (i = 0; i < 3; i++)
      for (j = 0; j < 3; j++)
         join->vertexptr[i][j] = NULL;

   north = current->n;
   south = current->s;
   east = current->e;
   west = current->w;

   if (row <= 1) {
      if (col <= 1) {           /* Selected point in SW corner */
         join->vertexptr[0][0] = current->data[1][1];
         join->vertexptr[0][1] = current->data[0][1];
         join->vertexptr[1][0] = current->data[1][0];
         join->vertexptr[1][1] = current->data[0][0];
         if (south != NULL) {
            join->vertexptr[2][0] = south->data[1][2];
            join->vertexptr[2][1] = south->data[0][2];
            if (south->w != NULL) {
               join->vertexptr[2][2] = south->w->data[2][2];
               join->vertexptr[1][2] = south->w->data[2][3];
            }
         }
         if (west != NULL) {
            join->vertexptr[0][2] = west->data[2][1];
            join->vertexptr[1][2] = west->data[2][0];
            if (west->s != NULL) {
               join->vertexptr[2][2] = west->s->data[2][2];
               join->vertexptr[2][1] = west->s->data[3][2];
            }
         }
      } else
         /* Selected point in SE corner */
      {
         join->vertexptr[0][0] = current->data[2][1];
         join->vertexptr[0][1] = current->data[2][0];
         join->vertexptr[1][0] = current->data[3][1];
         join->vertexptr[1][1] = current->data[3][0];
         if (east != NULL) {
            join->vertexptr[2][0] = east->data[1][1];
            join->vertexptr[2][1] = east->data[1][0];
            if (east->s != NULL) {
               join->vertexptr[2][2] = east->s->data[1][2];
               join->vertexptr[1][2] = east->s->data[0][2];
            }
         }
         if (south != NULL) {
            join->vertexptr[0][2] = south->data[2][2];
            join->vertexptr[1][2] = south->data[3][2];
            if (south->e != NULL) {
               join->vertexptr[2][2] = south->e->data[1][2];
               join->vertexptr[2][1] = south->e->data[1][3];
            }
         }
      }
   } else {
      if (col <= 1) {            /* Selected point in NW corner */
         join->vertexptr[0][0] = current->data[1][2];
         join->vertexptr[0][1] = current->data[1][3];
         join->vertexptr[1][0] = current->data[0][2];
         join->vertexptr[1][1] = current->data[0][3];
         if (west != NULL) {
            join->vertexptr[2][0] = west->data[2][2];
            join->vertexptr[2][1] = west->data[2][3];
            if (west->n != NULL) {
               join->vertexptr[2][2] = west->n->data[2][1];
               join->vertexptr[1][2] = west->n->data[3][1];
            }
         }
         if (north != NULL) {
            join->vertexptr[0][2] = north->data[1][1];
            join->vertexptr[1][2] = north->data[0][1];
            if (north->w != NULL) {
               join->vertexptr[2][2] = north->w->data[2][1];
               join->vertexptr[2][1] = north->w->data[2][0];
            }
         }
      } else
         /* Selected point in NE corner */
      {
         join->vertexptr[0][0] = current->data[2][2];
         join->vertexptr[0][1] = current->data[3][2];
         join->vertexptr[1][0] = current->data[2][3];
         join->vertexptr[1][1] = current->data[3][3];
         if (north != NULL) {
            join->vertexptr[2][0] = north->data[2][1];
            join->vertexptr[2][1] = north->data[3][1];
            if (north->e != NULL) {
               join->vertexptr[2][2] = north->e->data[1][1];
               join->vertexptr[1][2] = north->e->data[1][0];
            }
         }
         if (east != NULL) {
            join->vertexptr[0][2] = east->data[1][2];
            join->vertexptr[1][2] = east->data[1][3];
            if (east->n != NULL) {
               join->vertexptr[2][2] = east->n->data[1][1];
               join->vertexptr[2][1] = east->n->data[0][1];
            }
         }
      }
   }

   /* Initialise move array */
   for (i = 0; i < 3; i++)
      for (j = 0; j < 3; j++)
         for (k = 0; k < 3; k++)
            join->move[i][j][k] = 0.0;
}


/*
 * Sets the move vectors for each point in join->struct so as to maintain
 * continuity of surface normal.  Implements continuity conditions h=1 and
 * k=0. See report for explanation.
 */
void fixcontinuity(int col, int row, vector change, joindata * join)
{
   int i, j, k;

   if (!(col % 3) && !(row % 3)) {               /* Corner point moved */
      for (i = 0; i < 3; i++)
         for (j = 0; j < 3; j++)
            for (k = 0; k < 3; k++)
               join->move[i][j][k] = change[k];
   } else if ((col % 3) && (row % 3)) {          /* Interior point moved */
      for (k = 0; k < 3; k++) {
         join->move[0][0][k] = change[k];
         join->move[0][2][k] = -change[k];
         join->move[2][0][k] = -change[k];
         join->move[2][2][k] = change[k];
      }
   } else if ((2 * col + row) % 5 == 2) {        /* Edge point moved */
      for (i = 0; i < 3; i++)
         for (k = 0; k < 3; k++) {
            join->move[i][0][k] = change[k];
            join->move[i][2][k] = -change[k];
         }
   } else if ((col + 2 * row) % 5 == 2) {        /* Other edge point moved */
      for (j = 0; j < 3; j++)
         for (k = 0; k < 3; k++) {
            join->move[0][j][k] = change[k];
            join->move[2][j][k] = -change[k];
         }
   }
}

