/* descr.h - configuration description handling */

/* Written 1994,1995 by Werner Almesberger */


#ifndef DESCR_H
#define DESCR_H


#include "expr.h"


#define MAX_BUFFER 1024 /* path buffer */

#define FLAG_SEEN	1	/* the cursor was already on the variable */
#define FLAG_PROTECTED	2	/* this field shouldn't be changed */
#define FLAG_ABSENT	4	/* this field may be seen but is absent */


/*
 * This is a typical case where classes would be extremely useful and elegant.
 * Unfortunately, it would be impossible to make C++ a required part of a
 * minimal self-contained Linux system, so let's stay in the caves of ANSI C ...
 */

typedef enum { it_comment,it_bool,it_number,it_choice,it_set,it_if } ITEM_TYPE;

typedef enum { act_none,act_select,act_toggle,act_edit } ACTION;

typedef enum { acc_invalid,acc_more,acc_accept } ACCEPT;

typedef struct {
    const char *long_name;
    const char *short_name; /* points to long_name if unused */
} NAME;

typedef struct _value {
    NAME name;
    const char *value; /* points to name.long_name if omitted */
    struct _item *section;
    int like;
    int visible;
    const struct _value *prev; /* used during construction */
} VALUE;

typedef struct _item {
    ITEM_TYPE type;
    NAME name;
    const char *var;
    union {
	struct {
	    int visible;
	    struct _item *section;
	} comment;
	struct {
	    int on;
	    int visible;
	    struct _item *section;
	} bool;
	struct {
	    int off; /* value indicating absence of feature */
	    int base;
	    int min,max;
	    int value;
	    int visible;
	    struct _item *section;
	} number;
	struct {
	    int values;
	    VALUE *value;
	    int curr;
	} choice;
	struct {
	    int values;
	    VALUE *value;
	    int curr;
	    unsigned long set;
	} set;
	struct {
	    EXPR *expr;
	    int busy; /* prevent recursive evaluation */
	    struct _item *true,*false;
	} cond;
    } u;
    const char *help;
    unsigned char flags;
    struct _item *parent,*prev,*next;
} ITEM;


int (*ask_protected)(void);


#define CURR_VALUE(t) (((t)->type == it_choice ? (t)->u.choice.value : \
  (t)->u.set.value)[(t)->type == it_choice ? (t)->u.choice.curr : \
  (t)->u.set.curr])


ITEM *get_prev(ITEM *this);

/* Search for a non-if clause starting at THIS. The search follows the PREV
   pointer. */

#define get(t) ((t) && (t)->type == it_if ? _get(t) : (t))
ITEM *_get(ITEM *this);

/* Search for a non-if clause starting at THIS. The search follows the NEXT
   pointer. Macro reduces call frequency of get by about 97% */

void initialize(const char *class,ITEM *item);

/* Initialize the new object ITEM of class CLASS with class-specific data. */

ITEM *get_section(const ITEM *item);

/* Returns the currently active section of ITEM or NULL if ITEM has no
   section. Skips if clauses. */

int visible(ITEM *item,int true);

/* Sets the visibility of the current section of ITEM to TRUE. Returns 1 if
   the visibility already was TRUE before, 0 otherwise. */

int is_visible(const ITEM *item);

/* Returns 1 if the current section of ITEM is visible, 0 otherwise. */

int is_unseen(const ITEM *item);

/* Returns 1 if ITEM has the unseen marker set, 0 otherwise. */

int num_fields(const ITEM *item);

/* Returns the number of fields of ITEM. */

int curr_field(const ITEM *item);

/* Returns the zero-based number of the current field of ITEM. */

int field_active(const ITEM *item,int i);

/* Returns 1 if the field I of ITEM is active, 0 otherwise. Note that the
   result of FIELD_ACTIVE differs from the result of  i == curr_field  if
   ITEM is a set. */

int field_size(const ITEM *item,int i);

/* Returns the maximum length (in characters) of the Ith value field of
   ITEM. */

void field_content(const ITEM *item,int i,char *buffer);

/* Copies the current value of the Ith field of ITEM into BUFFER. */

const char *valid_range(const ITEM *item);

/* Returns a pointer to a (static) string indicating the valid value range of
   ITEM. */

int summarize(const ITEM *item,char *buffer,int left);

/* Summarizes the subtree starting at ITEM into BUFFER. At most LEFT characters
   are used. Returns the length of the resulting summary or 0 if no summary
   has been generated. */

void default_action(ITEM *item);

/* Performs the default action ([Space]) on ITEM. */

void field_select(ITEM *item,int i);

/* Selects the Ith field of ITEM. */

ACTION field_action(const ITEM *item);

/* Returns the default action of ITEM. */

ACCEPT check_field(const ITEM *item,int i,const char *value);

/* Checks whether VALUE would conform to the syntax rules of the Ith field of
   ITEM. Returns ACC_ACCEPT if VALUE would be a valid value, ACC_MORE if VALUE
   is a prefix of a valid value, or ACC_INVALID if VALUE is entirely
   incompatible. */

int set_field(ITEM *item,int i,const char *value);

/* Stores VALUE in the Ith field of ITEM. Returns 1 if VALUE was a valid
   value, 0 otherwise */

ITEM *lookup(const char *name,int *ind,int all);

/* Looks up the variable named NAME. If ALL is non-zero, even disabled sections
   are searched. If a variable is found, LOOKUP returns a non-NULL pointer.
   Additionally, if IND is non-NULL, -1 is stored in *IND if NAME is a variable
   name. If NAME is a value name, the value index is stored in *IND. If no
   variable or value has been found, LOOKUP returns NULL. */

int is_active(ITEM *this);

/* Returns 0 if the item pointed to by THIS is in a disabled section, 1
   otherwise. This is used to check for cyclic dependencies in if clauses. */

void check_unique(ITEM *area);

/* Checks uniqueness of all names in the name space pointed to by AREA. */

#endif
