/* descr.c - configuration description handling */

/* Written 1994,1995 by Werner Almesberger */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "scend.h"
#include "common.h"
#include "parse.h"
#include "expr.h"
#include "descr.h"


int (*ask_protected)(void) = NULL;


ITEM *get_prev(ITEM *this)
{
    ITEM *temp;

    if (!this || this->type != it_if) return this;
    if (this->u.cond.busy) return get_prev(this->prev);
    this->u.cond.busy = 1;
    temp = get(eval(this->u.cond.expr) ? this->u.cond.true :
      this->u.cond.false);
    this->u.cond.busy = 0;
    if (temp) return temp;
    return get_prev(this->prev);
}


ITEM *_get(ITEM *this)
{
    ITEM *temp;

#ifndef get
    if (!this || this->type != it_if) return this;
#endif
    if (this->u.cond.busy) return get(this->next);
    this->u.cond.busy = 1;
    temp = get(eval(this->u.cond.expr) ? this->u.cond.true :
      this->u.cond.false);
    this->u.cond.busy = 0;
    if (temp) return temp;
    return get(this->next);
}


static int modify(ITEM *this)
{
    if (!(this->flags & FLAG_PROTECTED)) {
	this->flags &= ~FLAG_ABSENT;
	changed = 1;
	return 1;
    }
    if (!ask_protected()) return 0;
    this->flags &= ~(FLAG_PROTECTED | FLAG_ABSENT);
    changed = 1;
    return 1;
}


static void default_choice(ITEM *this,int i)
{
    if (this->u.choice.curr != -1) parse_error("multiple defaults in choice");
    this->u.choice.curr = i;
}


static void default_set(ITEM *this,int i)
{
    if (i == sizeof(unsigned long)*8) parse_error("too many values in set");
    this->u.set.set |= 1 << i;
}


void initialize(const char *class,ITEM *item)
{
        if (!strcmp(class,"comment")) item->type = it_comment;
    else if (!strcmp(class,"bool")) item->type = it_bool;
    else if (!strcmp(class,"int")) item->type = it_number;
    else if (!strcmp(class,"choice")) item->type = it_choice;
    else if (!strcmp(class,"set")) item->type = it_set;
    else if (!strcmp(class,"if")) item->type = it_if;
    else parse_error("Unknown class \"%s\"",class);
    switch (item->type) {
	case it_comment:
	    item->flags &= ~FLAG_ABSENT;
	    item->u.comment.visible = demo;
	    item->help = parse_help();
	    if (next() == '{') item->u.comment.section = section(item);
	    else {
		back();
		item->u.comment.section = NULL;
	    }
	    break;
	case it_bool:
	    item->u.bool.visible = demo;
	    if (next() == '*') item->u.bool.on = 1;
	    else if (ch || (strcmp(token,"0") && strcmp(token,"1"))) back();
		else item->u.bool.on = atoi(token);
	    if (demo) item->u.bool.on = 1;
	    item->help = parse_help();
	    if (next() == '{') item->u.bool.section = section(item);
	    else {
		back();
		item->u.bool.section = NULL;
	    }
	    break;
	case it_number:
	    item->u.number.visible = demo;
	    item->u.number.base = 0;
	    item->u.number.value = number(&item->u.number.base);
	    if (non_eof() != ':') item->u.number.min = item->u.number.value;
	    else {
		item->u.number.min = number(&item->u.number.base);
		(void) non_eof();
	    }
	    if (ch != '-') parse_error("- expected");
	    item->u.number.max = number(item->u.number.base ? NULL :
	      &item->u.number.base);
	    if (item->u.number.min > item->u.number.max)
		parse_error("lower bound is above upper bound");
	    if (next() == ',') {
		item->u.number.off = number(item->u.number.base ? NULL :
		  &item->u.number.base);
		if (item->u.number.off < item->u.number.min ||
		  item->u.number.off > item->u.number.max)
		    parse_error("null value is outside valid range");
	    }
	    else {
		back();
		item->u.number.off = item->u.number.min-1;
	    }
	    if (!item->u.number.base) item->u.number.base = 10;
	    item->help = parse_help();
	    if (next() == '{') item->u.number.section = section(item);
	    else {
		back();
		item->u.number.section = NULL;
	    }
	    break;
	case it_choice:
	    item->help = parse_help();
	    item->u.choice.curr = -1;
	    item->u.choice.value = values(item,&item->u.choice.values,
	      default_choice);
	    if (item->u.choice.curr == -1) item->u.choice.curr = 0;
	    break;
	case it_set:
	    item->help = parse_help();
	    item->u.set.set = demo ? ~0 : 0;
	    item->u.set.value = values(item,&item->u.set.values,default_set);
	    break;
	case it_if:
	    item->flags &= ~FLAG_ABSENT;
	    item->u.cond.expr = expr_parse();
	    item->u.cond.busy = 0;
	    if (non_eof() != '{') parse_error("\"{\" expected");
	    item->u.cond.true = section(item);
	    if (!next() && !strcmp(token,"else")) {
		if (non_eof() != '{') parse_error("\"{\" expected");
		item->u.cond.false = section(item);
	    }
	    else {
		item->u.cond.false = NULL;
		back();
	    }
	    break;
    }
}


ITEM *get_section(const ITEM *item)
{
    switch (item->type) {
	case it_comment:
	    return get(item->u.comment.section);
	case it_bool:
	    return item->u.bool.on ? get(item->u.bool.section) : NULL;
	case it_number:
	    return item->u.number.value == item->u.number.off ? NULL :
	      get(item->u.number.section);
	case it_choice:
	    return get(CURR_VALUE(item).section);
	case it_set:
	    return item->u.set.set & (1 << item->u.set.curr) ?
	      get(item->u.set.value[item->u.set.curr].section) : NULL;
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return NULL;
}


int visible(ITEM *item,int true)
{
#define SET(f) \
  if (item->u.f == true) return 0; \
  item->u.f = true; \
  break

    switch (item->type) {
	case it_comment:
	    SET(comment.visible);
	case it_bool:
	    SET(bool.visible);
	case it_number:
	    SET(number.visible);
	case it_choice:
	    SET(choice.value[item->u.choice.curr].visible);
	case it_set:
	    SET(set.value[item->u.set.curr].visible);
	case it_if:
	    abort();
    }
    return 1;
#undef SET
}


int is_visible(const ITEM *item)
{
    switch (item->type) {
	case it_comment:
	    return item->u.comment.visible;
	case it_bool:
	    return item->u.bool.visible;
	case it_number:
	    return item->u.number.visible;
	case it_choice:
	    /* fall through */
	case it_set:
	    return CURR_VALUE(item).visible;
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return 0;
}


static int any_unseen(const ITEM *this)
{
    while (this) {
	if (this->type != it_if) {
	    if (is_unseen(this)) return 1;
	}
	else if (any_unseen(get((ITEM *) this))) return 1;
	this = this->next;
    }
    return 0;
}


int is_unseen(const ITEM *item)
{
    int i;

    if (!item) return 0;
    if (item->var && !(item->flags & FLAG_SEEN)) return 1;
    if (item->type != it_set) return any_unseen(get_section(item));
    for (i = 0; i < item->u.set.values; i++)
	if (item->u.set.set & (1 << i))
	    if (any_unseen(get(item->u.set.value[i].section))) return 1;
    return 0;
}


int num_fields(const ITEM *item)
{
    switch (item->type) {
	case it_comment:
	    return 0;
	case it_bool:
	    return 2;
	case it_number:
	    return 1;
	case it_choice:
	    return item->u.choice.values;
	case it_set:
	    return item->u.set.values;
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return 0;
}


int curr_field(const ITEM *item)
{
    switch (item->type) {
	case it_comment:
	    abort();
	case it_bool:
	    return !item->u.bool.on;
	case it_number:
	    return 0;
	case it_choice:
	    return item->u.choice.curr;
	case it_set:
	    return item->u.set.curr;
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return 0;
}


int field_active(const ITEM *item,int i)
{
    switch (item->type) {
	case it_comment:
	    abort();
	case it_bool:
	    return i == !item->u.bool.on;
	case it_number:
	    break;
	case it_choice:
	    return item->u.choice.curr == i;
	case it_set:
	    return !!(item->u.set.set & (1 << i));
	case it_if:
	    abort();
    }
    return 1;
}


int field_size(const ITEM *item,int i)
{
    switch (item->type) {
	case it_comment:
	    abort();
	case it_bool:
	    return i ? 2 : 3;
	case it_number:
	    {
		int neg,pos,tmp;

		if ((tmp = -item->u.number.min) < 0) neg = 0;
		else for (neg = 0; tmp; neg++) tmp /= item->u.number.base;
		tmp = item->u.number.max;
		for (pos = 0; tmp; pos++) tmp /= item->u.number.base;
		return (pos > neg+1 ? pos : neg+1)+(item->u.number.base == 8 ?
		  1 : item->u.number.base == 16 ? 2 : 0);
	    }
	case it_choice:
	    return strlen(item->u.choice.value[i].name.long_name);
	case it_set:
	    return strlen(item->u.set.value[i].name.long_name);
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return 0;
}


static void fmt_number(char *buffer,int number,int base)
{
    int neg;

    neg = number < 0;
    sprintf(buffer,base == 8 ? number ? "%s0%o" : "%s%o" : base == 10 ? "%s%d" :
      "%s0x%X",neg ? "-" : "",neg ? -number : number);
}


void field_content(const ITEM *item,int i,char *buffer)
{
    switch (item->type) {
	case it_comment:
	    abort();
	case it_bool:
	    strcpy(buffer,i ? "No" : "Yes");
	    return;
	case it_number:
	    fmt_number(buffer,item->u.number.value,item->u.number.base);
	    return;
	case it_choice:
	    strcpy(buffer,item->u.choice.value[i].name.long_name);
	    return;
	case it_set:
	    strcpy(buffer,item->u.set.value[i].name.long_name);
	    return;
	case it_if:
	    abort();
    }
}


const char *valid_range(const ITEM *item)
{
    static char buffer[50]; /* ugly */
    char min[20],max[20]; /* even uglier */

    switch (item->type) {
	case it_number:
	    fmt_number(min,item->u.number.min,item->u.number.base);
	    fmt_number(max,item->u.number.max,item->u.number.base);
	    sprintf(buffer,"Range is %s to %s",min,max);
	    return buffer;
	default:
	    return NULL;
    }
}


static int empty_section(const ITEM *item)
{
    if (!item) return 1;
    while (item) {
	switch (item->type) {
	    case it_comment:
		if (!empty_section(item->u.comment.section)) return 0;
		break;
	    case it_bool:
		if (item->u.bool.on) return 0;
		break;
	    case it_number:
		if (item->u.number.value != item->u.number.off) return 0;
		break;
	    case it_choice:
		return 0;
	    case it_set:
		if (item->u.set.set) return 0;
		break;
	    case it_if:
		if (!empty_section(get((ITEM *) item))) return 0;
		break;
	}
	item = item->next;
    }
    return 1;
}


static int sum(const ITEM *item,char *buffer,int *left,int levels);


static int sum_list(const ITEM *item,char *buffer,int *left,int levels)
{
    int prev,old;

    if (levels < 0 || !item) return 1;
    if (*left < 1) return 0;
    prev = *left;
    if (buffer) strcpy(buffer++,"(");
    (*left)--;
    while (item) {
	old = *left;
	if (!sum(item,buffer,left,levels)) return 0;
	if (buffer) buffer += old-*left;
	if (old != *left) {
	    if (--(*left) < 0) return 0;
	    if (buffer) strcpy(buffer++,",");
	}
	while (1) {
	    if (get(item->next)) {
		item = get(item->next);
		break;
	    }
	    if (!(item = item->parent)) break;
	    if (item->type != it_if) {
		item = NULL;
		break;
	    }
	}
    }
    if (*left == prev-1) (*left)++;
    else if (buffer) strcpy(buffer-1,")");
    return 1;
}


static int sum(const ITEM *item,char *buffer,int *left,int levels)
{
    const char *name,*value;
    char num_buf[20]; /* ugly */
    int i,size,old;

    name = item->type == it_if ? NULL : item->name.short_name;
    value = NULL;
    switch (item->type) {
	case it_comment:
	    if (empty_section(item->u.comment.section)) return 1;
	    break;
	case it_bool:
	    if (!item->u.bool.on) return 1;
	    break;
	case it_number:
	    if (item->u.number.value == item->u.number.off) return 1;
	    field_content(item,0,num_buf);
	    value = num_buf;
	    break;
	case it_choice:
	    value = CURR_VALUE(item).name.short_name;
	    break;
	case it_set:
	    if (!item->u.set.set) return 1;
	    break;
	default:
	    abort();
    }
    if ((size = strlen(name)+(value ? strlen(value)+1 : 0)) > *left) return 0;
    if (buffer) buffer += sprintf(buffer,"%s%s%s",name,value ? "=" : "",value ?
       value : "");
    *left -= size;
    if (item->type != it_set)
	return sum_list(get_section(item),buffer,left,levels-1);
    if (*left < 1) return 0;
    if (buffer) *buffer++ = '(';
    (*left)--;
    for (i = 0; i < item->u.set.values; i++) {
	if (*(value = item->u.set.value[i].name.short_name)) {
	    if ((*left -= strlen(value)+1) < 0) return 0;
	    if (buffer) buffer += sprintf(buffer,"%s,",value);
	}
	old = *left;
	if (!sum_list(get(item->u.set.value[i].section),buffer,left,
	  levels-1)) return 0;
	if (buffer) buffer += old-*left;
    }
    if (buffer) strcpy(buffer-1,")");
    return 1;
}


int summarize(const ITEM *item,char *buffer,int size)
{
    const ITEM *section;
    int i,last,left;

    if (!(section = get_section(item))) return 0;
    i = -1;
    if (buffer) *buffer = 0;
    last = -1;
    while (1) {
	left = size;
        if (!sum_list(section,buffer,&left,++i)) break;
	if (left == last) break;
	last = left;
    }
    if (!i) return 0;
    left = size;
    (void) sum_list(section,buffer,&left,i-1);
    return size-left;
}


void default_action(ITEM *item)
{
    if (item->type != it_comment && (item->flags & FLAG_ABSENT)) {
	item->flags &= ~FLAG_ABSENT;
	return;
    }
    switch (item->type) {
	case it_comment:
	    item->u.comment.visible = !item->u.comment.visible;
	    return;
	case it_bool:
	    if (item->u.bool.on)
		if (item->u.bool.visible || !item->u.bool.section) {
		    if (modify(item)) item->u.bool.on = 0;
		}
		else item->u.bool.visible = 1;
	    else if (modify(item))
		    item->u.bool.visible = item->u.bool.on = 1;
	    return;
	case it_number:
	    item->u.number.visible = !item->u.number.visible;
	    return;
	case it_choice:
	    if (CURR_VALUE(item).visible || !CURR_VALUE(item).section) {
		    if (modify(item)) {
			item->u.choice.curr = (item->u.choice.curr+1) %
			  item->u.choice.values;
			CURR_VALUE(item).visible = 1;
		    }
		}
	    else CURR_VALUE(item).visible = 1;
	    return;
	case it_set:
	    if (item->u.set.set & (1 << item->u.set.curr))
		if (CURR_VALUE(item).visible || !CURR_VALUE(item).section) {
		    if (modify(item)) item->u.set.set ^= 1 << item->u.set.curr;
		}
		else CURR_VALUE(item).visible = 1;
	    else if (modify(item)) {
		    item->u.set.set ^= 1 << item->u.set.curr;
		    CURR_VALUE(item).visible = 1;
		}
	    return;
	case it_if:
	    abort();
    }
}


void field_select(ITEM *item,int i)
{
    switch (item->type) {
	case it_comment:
	    abort();
	case it_bool:
	    if (modify(item))
		if (item->u.bool.on = !i) item->u.bool.visible = 1;
	    return;
	case it_number:
	    abort();
	case it_choice:
	    if (modify(item)) {
		item->u.choice.curr = i;
		item->u.choice.value[i].visible = 1;
	    }
	    return;
	case it_set:
	    item->u.set.curr = i;
	    return;
	case it_if:
	    abort();
    }
}


ACTION field_action(const ITEM *item)
{
    switch (item->type) {
	case it_comment:
	    return act_none;
	case it_bool:
	    return act_select;
	case it_number:
	    return act_edit;
	case it_choice:
	    return act_select;
	case it_set:
	    return act_toggle;
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return act_none;
}


static ACCEPT process_number(ITEM *item,const char *value,int set)
{
    int sign,base,number;
    int min,max;
    char *end;

    sign = number = 0;
    if (*value == '-') {
	sign = 1;
	value++;
    }
    min = item->u.number.min;
    max = item->u.number.max;
    if ((min >= 0 && sign) || (max < 0 && !sign)) return acc_invalid;
    base = item->u.number.base;
    if (*value == '0')
	if (value[1] == 'x' || value[1] == 'X') {
	    base = 16;
	    value += 2;
	}
	else if (value[1]) {
		base = 8;
		value++;
	    }
    if (!*value) return acc_more;
    number = strtol(value,&end,base);
    if (*end) return acc_invalid;
    if (sign) number = -number;
    if (number < min || number > max) return acc_invalid;
    if ((sign && number > min) || (!sign && number < min)) return acc_more;
    if (set) item->u.number.value = number;
    return acc_accept;
}


ACCEPT check_field(const ITEM *item,int i,const char *value)
{
    switch (item->type) {
	case it_comment:
	    /* fall through */
	case it_bool:
	    abort();
	case it_number:
	    return process_number((ITEM *) item,value,0);
	case it_choice:
	    /* fall through */
	case it_set:
	    /* fall through */
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return acc_invalid;
}


int set_field(ITEM *item,int i,const char *value)
{
    switch (item->type) {
	case it_comment:
	    /* fall through */
	case it_bool:
	    abort();
	case it_number:
	    if (!modify(item)) return 0;
	    return process_number(item,value,1) == acc_accept;
	case it_choice:
	    /* fall through */
	case it_set:
	    /* fall through */
	case it_if:
	    abort();
    }
    /*NOTREACHED*/
    return 0;
}


static ITEM *find_var(ITEM *curr,const char *name,int length,int all)
{
    ITEM *temp,*next;

    while (curr) {
	if (curr->var && strlen(curr->var) == length &&
	  !strncmp(curr->var,name,(unsigned) length)) return curr;
	if (!all)
	    if (curr->type == it_choice || curr->type == it_set) temp = NULL;
	    else temp = get_section(curr);
	else switch (curr->type) {
		case it_comment:
		    temp = curr->u.comment.section;
		    break;
		case it_bool:
		    temp = curr->u.bool.section;
		    break;
		case it_number:
		    temp = curr->u.number.section;
		    break;
		case it_if:
		    if (temp = find_var(get(curr->u.cond.true),name,length,all))
			return temp;
		    temp = curr->u.cond.false;
		    break;
		default:
		    temp = NULL;
	    }
	if (temp)
	    if (temp = find_var(get(temp),name,length,all)) return temp;
	if (!(next = get(curr->next)))
	    for (next = NULL; curr->parent && curr->parent->type == it_if;
	      curr = curr->parent)
		if (next = get(curr->parent->next)) break;
	curr = next;
    }
    return NULL;
}


ITEM *lookup(const char *name,int *ind,int all)
{
    ITEM *curr;
    VALUE *values;
    char *end;
    int var,length,i;

    curr = get(descr);
    i = -1;
    if (ind) *ind = -1;
    var = 1;
    do {
	if (!(end = strchr(name,var ? ':' : '.'))) end = strchr(name,0);
	length = end-name;
	if (var) curr = find_var(curr,name,length,all);
	else {
	    if (curr->type != it_choice && curr->type != it_set) return NULL;
	    values = curr->type == it_choice ? curr->u.choice.value :
	      curr->u.set.value;
	    for (i = 0; i < num_fields(curr); i++)
		if (strlen(values[i].value) == length && !strncmp(name,
		  values[i].value,(unsigned) length)) break;
	    if (i == num_fields(curr)) return NULL;
	    curr = get(values[i].section);
	}
	var = !var;
	name = end+1;
    }
    while (*end);
    if (var && ind) *ind = i;
    return curr;
}


int is_active(ITEM *this)
{
    if (!this || !this->parent) return 1;
    while (this->prev) this = this->prev;
    if (this->parent->type == it_if && (eval(this->parent->u.cond.expr) ?
      this->parent->u.cond.true : this->parent->u.cond.false) != this)
	return 0;
    return is_active(this->parent);
}


static void check_subtree(ITEM *top,ITEM *curr,const char *name)
{
    int i;

    while (top && curr) {
	if (curr->var)
	    if (!name) check_subtree(curr,top,curr->var);
	    else if (top != curr && !strcmp(curr->var,name))
		die("variable %s is not unique",name);
	switch (curr->type) {
	    case it_comment:
		check_subtree(top,curr->u.comment.section,name);
		break;
	    case it_bool:
		check_subtree(top,curr->u.bool.section,name);
		break;
	    case it_number:
		check_subtree(top,curr->u.number.section,name);
		break;
	    case it_if:
		check_subtree(top,curr->u.cond.true,name);
		check_subtree(top,curr->u.cond.false,name);
		break;
	    case it_choice:
		/* fall through */
	    case it_set:
		for (i = (curr->type == it_choice ? curr->u.choice.values :
		  curr->u.set.values)-1; i > -1; i--)
		    check_unique((curr->type == it_choice ?
		      curr->u.choice.value : curr->u.set.value)[i].section);
	}
	curr = curr->next;
    }
}


void check_unique(ITEM *area)
{
    check_subtree(area,area,NULL);
}
