////////////////////////////////////////////////////////////////////////////////
//  Implementations of camera classes. some platform dependend stuff is       //  
//  placed in the system-sub directory. The core raytracing stuff is taken    //  
//  from Paul S. Heckberts Article in "An Introduction to Raytracing". But,   //  
//  computation of refraction was replaced by an own algorithm                //
//  LAST EDIT: Wed Jan 11 09:58:02 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "scene.h"
#include "camera.h"
#include "light.h"
#include "matrix.h"

const char *RTN_CAMERA = "Camera";

void RT_IntersectPrimitiveFunc::exec(RT_GeneralListEntry *e, void *) {
    if ( !(e->isA( RTN_PRIMITIVE ))) return;
    RT_Primitive *p = (RT_Primitive*)e;
    if (!p->get_visible()) return;
    if (p->intersectHierarchical(ray, inter)) interSect = 1;
}

void RT_Camera::scene(RT_Scene *sc) { 
    if (xscene) {
	if ((long)sc == -1) sc = 0;
	else xscene->removeRelatedObject( this );
    }
    xscene = sc; 
    if (xscene) {
	xscene->addRelatedObject( this );
	addRelatedObject( xscene );
    }
}

void RT_Camera::rendering() {
    if (xscene) xscene->update();
    feedback->update();
    if (xpixmap) { 
	xpixmap->activate(); 
	if (xscene) xpixmap->clear( xscene->get_background() ); 
    } 
}

class RT_PrintDevFunc:public RT_GeneralListFunctoid {
    char *name;
    void exec(RT_GeneralListEntry *e, void * f) {
	if (e->isA( RTN_INPUT_DEVICE )) fprintf((FILE*)f, "%s -father %s\n", 
					 ((RT_InputDevice*)e)->get_name(),
					 name );
    }
  public:
    RT_PrintDevFunc(char *_name) { name = _name; }
};

void RT_Camera::print( FILE *f ) const {
    printCon( f );
    if (xpixmap || xscene) {
	fprintf( f, "\n%s ", get_name() );
	if (xpixmap) fprintf( f, "-pixmap %s ", xpixmap->get_name() );
	if (xscene) fprintf( f, "-scene %s ", xscene->get_name() );
	fprintf( f, "\n" );
    }
    // connections to the pixmaps:
    RT_PrintDevFunc func( (char*)get_name());
    relObjects.doWithElements( &func, f );
}

void RT_Camera::pixmap(RT_Pixmap *px) { 
    if (xpixmap) {
	if ((long)px == -1) px = 0;
	else {
	    xpixmap->setCamera( 0 );
	    xpixmap->removeRelatedObject( this );
	}
    }
    xpixmap = px; 
    if (xpixmap) {
	xpixmap->setCamera( this );
	xpixmap->addRelatedObject( this );
	addRelatedObject( xpixmap );
    }
}

int RT_Camera::pixF, RT_Camera::pixG; 
int RT_Camera::rendF, RT_Camera::scnF, RT_Camera::scnG;
int RT_Camera::refrF; char *RT_Camera::pixV, *RT_Camera::scnV; 

RT_ParseEntry RT_Camera::table[] = {
    { "-pixmap", RTP_STRING, (char*)&pixV, &pixF, "Specify a {ARG 1 Pixmap} device.", "Object" },
    { "-get_pixmap", RTP_NONE, 0, &pixG, "Return the current pixmap.", RTPS_NONE },
    { "-scene", RTP_STRING, (char*)&scnV, &scnF, "Specify a {ARG 1 Scene} object.", "Object" },
    { "-get_scene", RTP_NONE, 0, &scnG, "Return the current scene.", RTPS_NONE },
    { "-rendering", RTP_NONE, 0, &rendF, "Render the scene and print the output to the pixmap.", RTPS_NONE },
    { "-refresh", RTP_NONE, 0, &refrF, "Refresh the pixmap using the shading presentation instead of ray-tracing.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Camera::objectCMD(char *argv[]) {
    RT_parseTable( argv, table );
    int ret = 0;

    if (pixF) {
	RT_Object *obj = RT_Object::getObject( pixV );
	if (!obj || !(obj->isA( RTN_PIXMAP ))) 
	    rt_Output->errorVar( get_name(), ": No such pixmap object: ", pixV, "!", 0 );
	else { pixmap( (RT_Pixmap*)obj ); ret++;}
    }
    if (pixG) {
	if (get_pixmap()) RT_Object::result( get_pixmap()->get_name() );
	ret++;
    }

    if (scnF) {
	RT_Object *obj = RT_Object::getObject( scnV );
	if (!obj || !(obj->isA( RTN_SCENE)) ) 
	    rt_Output->errorVar( get_name(), ": No such scene object: ", scnV, "!", 0 );
	else { scene( (RT_Scene*)obj ); ret++; }
    }
    if (scnG) {
	if (get_scene()) RT_Object::result( get_scene()->get_name() );
	ret++;
    }

    if (rendF) { rendering(); ret++; }
    if (refrF) { refresh(); ret++; }

    return ret;
}

void RT_Camera::event(RT_Event &ev) { 
    class LFunctoid:public RT_GeneralListFunctoid {
	RT_Event *ev;
      public:
	LFunctoid( RT_Event *_ev ) { ev = _ev; }
	void exec(RT_GeneralListEntry *e, void * = 0) {
	    if (e->isA( RTN_INPUT_DEVICE)) {
		RT_InputDevice *dv = (RT_InputDevice*)e;
		if ( dv->get_on() ) dv->event( ev );
	    }
	}
    } func( &ev );
    
    relObjects.doWithElements( &func );
}

const char *RTN_RAY_CAMERA = "RayCamera";

RT_Primitive *RT_RayCamera::getPrimitive(const RT_Ray &ray) {
    if (get_scene()) {
	RT_InterSectionList inter;
	RT_PrimitiveBoundingFunc func;
	get_scene()->doWithElements( &func);
	RT_IntersectPrimitiveFunc func1( ray, inter);
	get_scene()->doWithElements( &func1);
	if (func1.getInterSect() ) {
	    if (rt_PickDebug) {
		ray.print( stderr );
		inter.print( stderr );
	    }
	    return inter.first()->getPrim();
	}
    }
    return 0;
}

void RT_RayCamera::trace(int level, double weight, const RT_Ray &ray, RT_Color &c) {
    if (rt_RayDebug) {
	fprintf( stderr, "trace ray: ");
	ray.print( stderr );
    }
    RT_InterSectionList inter;
    RT_IntersectPrimitiveFunc func( ray, inter);
    get_scene()->doWithElements( &func);
    if (func.getInterSect() ) {
	RT_InterSection *top = inter.first();
	RT_Primitive *prim = top->getPrim();
	RT_Vector p = ray.pt + ray.dir * top->getT(); 
	RT_Vector n;

	prim->mappedNormal( p, n );
	
	if (rt_RayDebug) {
	    fprintf( stderr, "level: %i\nIntersections: \n", level );
	    inter.print( stderr );
	    p.print( stderr, "\nPoint: " );
	    n.print( stderr, "\nNormal: " );
	    fprintf( stderr, "\n" );
	}

	shade( level, weight, p, n, ray.dir, inter, c );
    } 
    else c = get_scene()->get_background();
    
    // clip color to [0,1]:
    c.correct();
}

// a functoid class used for ray-tracing:
class RT_ShadeFunc: public RT_GeneralListFunctoid {
    RT_Color &col;
    const RT_Vector &p, &n, &r;
    RT_RayCamera *camera;
    const RT_Surface &sf;
  public:
    RT_ShadeFunc(RT_Color &c, const RT_Vector &_p, const RT_Vector &_n, 
		 const RT_Vector &_r, const RT_Surface &_sf, RT_RayCamera *ca ): 
    col( c ), p( _p ), n( _n ), r( _r ), sf( _sf ) { camera = ca; }
    
    void exec(RT_GeneralListEntry *, void *);
};

void RT_ShadeFunc::exec(RT_GeneralListEntry *e, void *) {
    if (e->isA( RTN_LIGHT )) 
	col = col + ((RT_Light*)e)->shade( p, n, r, sf, camera );
}

void RT_RayCamera::shade( int level, double weight, const RT_Vector &p, const RT_Vector &no, 
			 const RT_Vector &i, const RT_InterSectionList &inter, RT_Color &col) {

    // shade a surface point ( recursively if necessary), stop recursing when level > RT_MAX_LEVEL
    // or when weight < RT_MIN_WEIGHT, WEIGTH is the cumulative weight, PT is surface point, 
    // NO is normal vector, I is incident ray, HIT is the intersection list

    RT_Surface surface = *(inter.first()->getSurface());
    inter.first()->getPrim()->mappedSurface( p, surface );
    int enter = inter.first()->enter;

    col = RT_Color (0, 0, 0);

    RT_Ray tray;
    tray.pt = p;

    // computing the light contributed
    // by the primary light sources:
    // (and emitting primitives)

    specularDirection( i, no, tray.dir );

    // if we are inside an object than invert the normal
    // to get the effect of a lamp:
    RT_Vector n2 = no;
    if (!enter && (i.DOT( n2 ) > 0 )) n2 = n2.NEGATE();

    RT_ShadeFunc func( col, p, n2, tray.dir, surface, this  );
    emiScene->doWithElements( &func );

    RT_Vector n = no;
    if ( i.DOT( n ) > 0 ) n = n.NEGATE();
    // flip normal if necessary
	
    // if we're not too deep then recurse
    if (level + 1 < rt_MaxLevel) {
	// recurse on specular reflection ray if significant  
	if (enter && surface.spec.ABS() * weight > rt_MinWeight) {
	    RT_Color tcol;

	    if (rt_RayDebug) fprintf( stderr, "specular: \n" );
	    
	    trace( level + 1, surface.spec.ABS() * weight, tray, tcol );
	    col = col + tcol * surface.spec;
	}

	// recurse on transmission ray if significant
	if ( surface.tran * weight > rt_MinWeight ) {
	    RT_Surface *senter, *sexit;
	    static RT_Surface *_surface; 
	    
	    if (inter.first()->enter) {
		sexit = _surface; _surface = senter = inter.first()->getSurface();
	    }
	    else {
		senter = _surface; _surface = sexit = inter.first()->getSurface();
	    }
	    
	    // coming from air:
	    if (traceCnt == 0 && inter.first()->enter) sexit = 0; 

	    // going into air:
	    if (traceCnt == 1 && !inter.first()->enter) senter = 0;

	    if (transmissionDirection( sexit, senter, i, n, tray.dir )) {
		
		if (rt_RayDebug) fprintf( stderr, "transmission: \n" );
		
		if (inter.first()->enter) traceCnt++; else traceCnt--;
		
		RT_Color tcol;  
		trace( level + 1, surface.tran * weight, tray, tcol );
		col = col + tcol * surface.tran;
	    }
	}
    }
}

void RT_RayCamera::specularDirection(const RT_Vector &i, const RT_Vector &n, RT_Vector &r) {

    // compute specular direction R from incident direction I and normal N
    // all vectors unit

    double f = -2.0 * i.DOT( n );
    r = n.SCALE( f ) + i;
}

int RT_RayCamera::transmissionDirection(const RT_Surface *m1, const RT_Surface *m2, 
					const RT_Vector &i, const RT_Vector &n, RT_Vector &t) {

    // compute transmission direction T from incident direction I, normal N, 
    //  going from medium M1 to M2, with refraction governed by the relative index
    // of refraction according to Snell's law:
    //     n1 * sin O1 = n2 * sin O2
    // If there is total internal reflection, return 0, else set T and return 1.
    // all vectors unit

    double n1 = m1 ? m1->refr : 1;
    double n2 = m2 ? m2->refr : 1;
    
    double eta = n1 / n2; // relative index of refraction
    double cs1 = - i.DOT( n ); // cos O1
    double cs2_2 = 1 - eta * eta * (1 - cs1 * cs1 ); // cos2 O2
    
    if (cs2_2 < 0) return 0; // total internal refraction
    t = i * eta + n * (eta * cs1 - sqrt( cs2_2 ));
    return 1;
}

void RT_RayCamera::rendering() {
    RT_Camera::rendering();
    delete emiScene;

    class LFunctoid: public RT_GeneralListFunctoid {
	RT_Scene *sc;
      public:
	LFunctoid( RT_Scene *_sc) { sc = _sc; } 
	void exec(RT_GeneralListEntry *e, void *) {
	    if (e->isA( RTN_LIGHT )) sc->insert( (RT_Light*)e ); 
	}
    } func( emiScene = new RT_Scene );
    get_scene()->doWithElements( &func );
}

// functoids:

void RT_RenderLightFunc::exec(RT_GeneralListEntry *e, void * ) {
    if (e->isA( RTN_LIGHT )) ((RT_Light*)e)->render();
}

void RT_RenderPrimitiveFunc::exec(RT_GeneralListEntry *e, void * ) {
    if (e->isA( RTN_PRIMITIVE )) ((RT_Primitive*)e)->renderHierarchical();
} 




