/* Setup loading/saving.
   Copyright (C) 1994 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>		/* Needed to include local .h files */
#include <sys/stat.h>
#include <sys/param.h>
#include <string.h>
#include <ncurses.h>
#include <stdlib.h>

#include "mad.h"
#include "dir.h"
#include "file.h"
#include "global.h"
#include "panel.h"
#include "util.h"		/* Functions and externs */
#include "main.h"
#include "tree.h"
#include "profile.h"
#include "setup.h"
#include "mouse.h"		/* To make view.h happy */
#include "view.h"		/* For the externs */
#include "key.h"		/* For the externs */
#include "hotlist.h"		/* load/save/done hotlist */
#include "layout.h"

static char rcsid [] = "$Id: setup.c,v 1.8 1995/01/27 02:36:18 miguel Exp $";

char *profile_name;

char setup_color_string [4096];
char term_color_string [4096];

static struct {
    char *key;
    sortfn *sort_type;
} sort_names [] = {
    "name",      (sortfn *) sort_name,
    "extension", (sortfn *) sort_ext,
    "time",      (sortfn *) sort_time,
    "size",      (sortfn *) sort_size,
    "inode",     (sortfn *) sort_inode,
    "unsorted",  (sortfn *) unsorted,
    0, 0
};

static struct {
    char *key;
    int  view_type;
} view_types [] = {
    "full",  view_full,
    "brief", view_brief,
    "long",  view_long,
    "info",  view_info,
    "tree",  view_tree,
    "quick", view_quick,
    "user",  view_user,
    0, 0
}; 	

void panel_save (char *profile, Panel *panel, char *section)
{
    char buffer [6];
    int  i;
    
    sprintf (buffer, "%d", panel->reverse);
    WritePrivateProfileString (section, "reverse", buffer, profile);
    for (i = 0; sort_names [i].key; i++)
	if (sort_names [i].sort_type == (sortfn *) panel->sort_type){
	    WritePrivateProfileString (section, "sort_order",
				       sort_names [i].key, profile);
	    break;
	}
    for (i = 0; view_types [i].key; i++)
	if (view_types [i].view_type == panel->view_type){
	    WritePrivateProfileString (section, "display",
				       view_types [i].key, profile);
	    break;
	}
    WritePrivateProfileString (section, "user_format",
			       panel->user_format, profile);
    WritePrivateProfileString (section, "mini_status_format",
			       panel->mini_status_format, profile);
    sprintf (buffer, "%d", panel->user_mini_status);
    WritePrivateProfileString (section, "user_mini_status", buffer, profile);
}

static struct {
    char *opt_name;
    int *opt_addr;
} layout [] = {
    { "horizontal_split", &horizontal_split },
    { "equal_split", &equal_split },
    { "first_panel_size", &first_panel_size },
    { "menubar_visible", &menubar_visible },
    { "command_prompt", &command_prompt },
    { "keybar_visible", &keybar_visible },
    { "output_lines", &output_lines },
    { 0, 0 }
};

void save_layout (void)
{
    char *profile;
    int  i;
    char buffer [6];

    profile = copy_strings (home_dir, "/.mc.ini", 0);

    /* Save integer options */
    for (i = 0; layout [i].opt_name; i++){
	sprintf (buffer, "%d", *layout [i].opt_addr);
	WritePrivateProfileString ("Layout", layout [i].opt_name, buffer, profile);
    }

    free (profile);
}

static struct {
    char *opt_name;
    int  *opt_addr;
} options [] = {
    { "show_backups", &show_backups },
    { "show_dot_files", &show_dot_files },
    { "verbose", &verbose },
    { "mark_moves_down", &mark_moves_down },
    { "pause_after_run", &pause_after_run },
    { "show_mini_info", &show_mini_info },
    { "shell_patterns", &easy_patterns },
    { "auto_save_setup", &auto_save_setup },
    { "align_extensions", &align_extensions },
    { "auto_menu", &auto_menu },
    { "auto_mount", &auto_mount },
    { "use_internal_view", &use_internal_view },
    { "clear_before_exec", &clear_before_exec },
    { "mix_all_files", &mix_all_files },
    { "fast_reload", &fast_reload },
    { "fast_reload_msg_shown", &fast_reload_w },
    { "confirm_delete", &confirm_delete },
    { "mouse_repeat_rate", &mou_auto_repeat },
    { "default_list_perm", &extra_info },
    { "double_click_speed", &double_click_speed },
    { "eight_bit_clean", &eight_bit_clean },
    { "confirm_view_dir", &confirm_view_dir },
    { "tree_navigation_flag", &tree_navigation_flag },
    { "mouse_move_pages", &mouse_move_pages },
    { "fast_refresh", &fast_refresh },
    { "navigate_with_arrows", &navigate_with_arrows },
    { "advanced_chown", &advanced_chfns },
    { "drop_menus", &drop_menus },
    { "wrap_mode",  &wrap_mode},
    { "old_esc_mode", &old_esc_mode },
    { "nice_rotating_dash", &nice_rotating_dash },
    { 0, 0 }
};

void save_configure (void)
{
    char *profile;
    int  i;

    profile = copy_strings (home_dir, "/.mc.ini", 0);

    /* Save integer options */
    for (i = 0; options [i].opt_name; i++)
	set_int (profile, options [i].opt_name, *options [i].opt_addr);

    free (profile);
}

void save_setup (void)
{
    char *profile;

    profile = copy_strings (home_dir, "/.mc.ini", 0);

    save_layout ();
    save_configure ();
    WritePrivateProfileString ("Dirs", "other_dir", opanel->cwd, profile);
    panel_save (profile, &left_panel,  "Left Panel");
    panel_save (profile, &right_panel, "Right Panel");
    save_hotlist ();
    free (profile);
}

static void panel_load (char *profile, Panel *panel, char *section)
{
    int i;
    char buffer [40];
    
    panel->reverse = GetPrivateProfileInt (section, "reverse", 0, profile);
    
    GetPrivateProfileString (section, "sort_order", "name", buffer,
			     sizeof (buffer), profile);

    for (i = 0; sort_names [i].key; i++)
	if (strcasecmp (sort_names [i].key, buffer) == 0){
	    panel->sort_type = sort_names [i].sort_type;
	    break;
	}
    GetPrivateProfileString (section, "display", "full", buffer,
			     sizeof (buffer), profile);
    for (i = 0; view_types [i].key; i++)
	if (strcasecmp (view_types [i].key, buffer) == 0){
	    panel->view_type = view_types [i].view_type;
	    break;
	}
    
    if (panel->user_format)
	free (panel->user_format);
    
    panel->user_format = strdup (get_profile_string (section, "user_format",
						     DEFAULT_USER_FORMAT,
						     profile));
    if (panel->mini_status_format)
	free (panel->mini_status_format);
    
    panel->mini_status_format =
	strdup (get_profile_string (section, "mini_status_format",
				    DEFAULT_USER_FORMAT, profile));
    panel->user_mini_status =
	GetPrivateProfileInt (section, "user_mini_status", 0, profile);
}

static void load_layout (char *profile)
{
    int i;
    
    for (i = 0; layout [i].opt_name; i++)
	*layout [i].opt_addr =
	    GetPrivateProfileInt ("Layout", layout [i].opt_name,
				  *layout [i].opt_addr, profile);
}

void load_setup (void)
{
    static char *buffer;
    char   *profile;
    int    i;

    /* Other modules expect load_setup to set this variables */
    left_panel.sort_type = right_panel.sort_type = (sortfn *) sort_name;
    left_panel.view_type = right_panel.view_type = view_full;

    buffer = copy_strings (home_dir, "/.mc.ini", 0);
    
    if (exist_file (buffer)){
	profile = buffer;
    } else if (exist_file (LIBDIR "mc.ini")){
	profile = strdup (LIBDIR "mc.ini");
	free (buffer);
    } else {
	profile = buffer;
    }
    
    profile_name = profile;

    /* Load integer boolean options */
    for (i = 0; options [i].opt_name; i++)
	*options [i].opt_addr =
	    get_int (profile, options [i].opt_name, *options [i].opt_addr);

    load_layout (profile);
    panel_load (profile, &left_panel, "Left Panel");
    panel_load (profile, &right_panel,"Right Panel");

    /* The directory hot list */
    load_hotlist ();
    
    /* Only one panel can be in information mode */
    if (left_panel.view_type == view_info &&
	right_panel.view_type== view_info)
	left_panel.view_type = view_full;

    /* Only one panel can be in tree mode */
    if (left_panel.view_type == view_tree &&
	right_panel.view_type== view_tree)
	right_panel.view_type = view_full;

    /* Only one panel can be in quick view mode */
    if (left_panel.view_type == view_quick &&
	right_panel.view_type== view_quick)
	left_panel.view_type = view_full;

    if (!other_dir){
	buffer = (char*) malloc (MAXPATHLEN);
	GetPrivateProfileString ("Dirs", "other_dir", ".", buffer,
			     MAXPATHLEN, profile);
	other_dir = buffer;
    }

    /* The default color and the terminal dependent color */
    GetPrivateProfileString ("Colors", "base_color", "", setup_color_string,
			     sizeof (setup_color_string), profile);
    GetPrivateProfileString ("Colors", getenv ("TERM"), "",
			     term_color_string, sizeof (term_color_string),
			     profile);
}

void done_setup (void)
{
    free (profile_name);
    done_hotlist ();
}
